(ns component.metrics
  (:require [integrant.core :as ig]
            [clojure.tools.logging :as log]
            [malli.core :as m]
            [medley.core :refer [deep-merge]])
  (:import io.micrometer.core.instrument.Metrics
           io.micrometer.prometheus.PrometheusConfig
           io.micrometer.prometheus.PrometheusMeterRegistry
           io.micrometer.core.instrument.Counter
           io.micrometer.core.instrument.Counter$Builder
           io.micrometer.core.instrument.Gauge
           io.micrometer.core.instrument.MeterRegistry
           io.micrometer.core.instrument.MeterRegistry$Config
           io.micrometer.core.instrument.Metrics
           io.micrometer.core.instrument.Timer
           io.micrometer.core.instrument.Timer$Builder
           io.micrometer.core.instrument.binder.jvm.ClassLoaderMetrics
           io.micrometer.core.instrument.binder.jvm.JvmGcMetrics
           io.micrometer.core.instrument.binder.jvm.JvmMemoryMetrics
           io.micrometer.core.instrument.binder.jvm.JvmThreadMetrics
           io.micrometer.core.instrument.binder.system.FileDescriptorMetrics
           io.micrometer.core.instrument.binder.system.UptimeMetrics
           io.micrometer.core.instrument.binder.system.ProcessorMetrics
           java.util.concurrent.TimeUnit
           java.util.function.Supplier)
  (:gen-class))

(defn- ->tags
  "Converts a map of tags to an array of string"
  {:added "1.1.0"}
  [tags]
  (into-array String
              (->> tags
                   (map (fn [[k v]] [(name k) (name v)]))
                   flatten)))

(defn mk-registry
  "Build a Micrometer premetheus registry"
  {:added "1.1.0"}
  [tags]
  (let [^PrometheusMeterRegistry registry (PrometheusMeterRegistry. PrometheusConfig/DEFAULT)]
    (.commonTags ^MeterRegistry$Config
     (.config registry)
                 ^"[Ljava.lang.String;" (->tags tags))
    (Metrics/addRegistry registry)
    (.bindTo (ClassLoaderMetrics.) registry)
    (.bindTo (JvmGcMetrics.) registry)
    (.bindTo (JvmMemoryMetrics.) registry)
    (.bindTo (JvmThreadMetrics.) registry)
    (.bindTo (FileDescriptorMetrics.) registry)
    (.bindTo (UptimeMetrics.) registry)
    (.bindTo (ProcessorMetrics.) registry)
    registry))

(defn get-timer!
  "get a timer by name and tags"
  {:added "1.1.0"}
  [^MeterRegistry registry n tags]
  (.register ^Timer$Builder (doto (Timer/builder (name n))
                              (.publishPercentiles (double-array [0.5 0.75 0.98 0.99]))
                              (.tags ^"[Ljava.lang.String;" (->tags tags)))
             registry))

(defn record
  {:added "1.1.0"}
  [^MeterRegistry registry n tags duration]
  (when registry
    (let [timer (get-timer! registry n tags)]
      (.record ^Timer timer duration TimeUnit/MILLISECONDS))))

(defmacro with-time
  {:added "1.1.0"}
  [^MeterRegistry registry n tags & body]
  `(if ~registry
     (let [^Timer timer# (get-timer! ~registry ~n ~tags)
           current# (java.time.Instant/now)]
       (try
         (do ~@body)
         (finally
           (let [end# (java.time.Instant/now)]
             (.record timer# (java.time.Duration/between current# end#))))))
     (do ~@body)))

(defn get-counter!
  {:added "1.1.0"}
  [^MeterRegistry registry n tags]
  (.register ^Counter$Builder
   (doto (Counter/builder (name n))
     (.tags ^"[Ljava.lang.String;" (->tags tags)))
             registry))

(defn increment!
  "increments a counter"
  {:added "1.1.0"}
  ([^MeterRegistry registry counter tags]
   (increment! registry counter tags 1))
  ([^MeterRegistry registry counter tags n]
   (when registry
     (let [builder (doto (Counter/builder (name counter))
                     (.tags ^"[Ljava.lang.String;" (->tags tags)))
           counter (.register builder registry)]
       (.increment counter n)))))

(defn- gauge-fn
  [producer-fn]
  (reify Supplier
    (get [_]
      (producer-fn))))

(defn gauge!
  [^MeterRegistry registry gauge tags producer-fn]
  (when registry
    (doto (Gauge/builder (name gauge) (gauge-fn producer-fn))
      (.strongReference true)
      (.tags ^"[Ljava.lang.String;" (->tags tags))
      (.register registry))))

(defn scrape
  [^PrometheusMeterRegistry registry]
  (some-> registry .scrape))

(defn- register-customs-metrics!
  [registry tags custom]
  (doseq [[type data] (apply deep-merge custom)]
    (condp = type
      :gauges (run! (fn [[n gfun]]
                      (let [m (if (string? n) (keyword n) n)]
                        (log/tracef "registering gauge %s" m)
                        (gauge! registry m tags (gfun))))
                    data))))

(defn register-probe!
  "Add probe to the system"
  {:added "1.1.0"}
  [^clojure.lang.Keyword probes-type probes]
  {[:component.metrics/custom-probes (keyword "component.metrics.custom" (name probes-type))]
   {probes-type probes}})

;;; Custom Probes injection mechanism
(defmethod ig/init-key :component.metrics/custom-probes [_ probes] probes)

;; ___________                                .___
;; \_   _____/__  ______________    ____    __| _/
;;  |    __)_\  \/  /\____ \__  \  /    \  / __ |
;;  |        \>    < |  |_> > __ \|   |  \/ /_/ |
;; /_______  /__/\_ \|   __(____  /___|  /\____ |
;;         \/      \/|__|       \/     \/      \/

(defmethod ig/expand-key :component/metrics
  [k v]
  {k (deep-merge
      {:route "/metrics"
       :tags {}
       ::custom-probes (ig/refset :component.metrics/custom-probes)}
      v)})

;;    _____                               __
;;   /  _  \   ______ ______ ____________/  |_
;;  /  /_\  \ /  ___//  ___// __ \_  __ \   __\
;; /    |    \\___ \ \___ \\  ___/|  | \/|  |
;; \____|__  /____  >____  >\___  >__|   |__|
;;         \/     \/     \/     \/

(def spec
  [:map [:route {:title "Metrics route"
                 :description "The metrics route exposed by pedestal"
                 :json-schema/example "/metrics"} string?
         :tags {:title "Prometheus Tags"
                :description "Default prometheus tags injected in each metrics"
                :json-schema/example {:component "api"}}]])

(defmethod ig/assert-key :component/metrics
  [_ system]
  (assert (m/validate spec system)
          (m/explain spec system)))

;; .___       .__  __
;; |   | ____ |__|/  |_
;; |   |/    \|  \   __\
;; |   |   |  \  ||  |
;; |___|___|  /__||__|
;;          \/

(defmethod ig/init-key :component/metrics
  [_ {:keys [tags component.metrics/custom-probes] :as system}]
  (let [registry (mk-registry tags)]
    (register-customs-metrics! registry tags custom-probes)
    (log/info "starting metrics component")
    {:registry registry}))

;;   ___ ___        .__   __
;;  /   |   \_____  |  |_/  |_
;; /    ~    \__  \ |  |\   __\
;; \    Y    // __ \|  |_|  |
;;  \___|_  /(____  /____/__|
;;        \/      \/

(defmethod ig/halt-key! :component/metrics
  [_ {:keys [registry] :as system}]
  (when registry
    (.close registry))
  (log/info "stopping metrics component")
  (dissoc system :registry))
