(ns component.logging
  (:require [integrant.core :as ig]
            [clojure.tools.logging :as log]
            [unilog.config :as ul]
            [malli.core :as m]
            [medley.core :refer [deep-merge]])
  (:import org.slf4j.LoggerFactory)
  (:gen-class))

(defonce ex-handler
  (reify Thread$UncaughtExceptionHandler
    (uncaughtException [_ t e]
      (log/error e "Uncaught exception in" (.getName t)))))

(defn- get-logger
  "Retrieve logging factory instance"
  {:added "1.1.0"}
  [app]
  (LoggerFactory/getLogger (name app)))

;; ___________                                .___
;; \_   _____/__  ______________    ____    __| _/
;;  |    __)_\  \/  /\____ \__  \  /    \  / __ |
;;  |        \>    < |  |_> > __ \|   |  \/ /_/ |
;; /_______  /__/\_ \|   __(____  /___|  /\____ |
;;         \/      \/|__|       \/     \/      \/

(defmethod ig/expand-key :component/logging
  [k v]
  {k (deep-merge
      ul/default-configuration
      {:console {:encoder :pattern
                 :pattern "%d{yyyy-MM-dd HH:mm:ss.SSS} %highlight(%-5level) %logger{36} - %msg%n"
                 :filters [[:logstash :deny :neutral]]}
       :level :debug
       :app-level :debug
       :overrides {"io.pedestal"    :warn
                   "org.eclipse"    :warn
                   "org.apache"     :warn}}
      v)})

;;    _____                               __
;;   /  _  \   ______ ______ ____________/  |_
;;  /  /_\  \ /  ___//  ___// __ \_  __ \   __\
;; /    |    \\___ \ \___ \\  ___/|  | \/|  |
;; \____|__  /____  >____  >\___  >__|   |__|
;;         \/     \/     \/     \/

(def spec
  [:map
   [:app-name {:title "Application Name"
               :description "The application name"
               :json-schema/example "acme"}
    string?]])

(defmethod ig/assert-key :component/logging
  [_ system]
  (assert (m/validate spec system)
          (m/explain spec system)))

;; .___       .__  __
;; |   | ____ |__|/  |_
;; |   |/    \|  \   __\
;; |   |   |  \  ||  |
;; |___|___|  /__||__|
;;          \/

(defmethod ig/init-key :component/logging
  [_ {:keys [app-name app-level] :as system}]
  (let [config (cond-> system
                 app-level (update :overrides assoc (name app-name) app-level))]
    (ul/start-logging! config)
    (log/info "starting logging component")
    (Thread/setDefaultUncaughtExceptionHandler ex-handler)
    {:app-name app-name
     :logger (get-logger app-name)}))

;;   ___ ___        .__   __
;;  /   |   \_____  |  |_/  |_
;; /    ~    \__  \ |  |\   __\
;; \    Y    // __ \|  |_|  |
;;  \___|_  /(____  /____/__|
;;        \/      \/

(defmethod ig/halt-key! :component/logging
  [_ {:keys [] :as system}]
  (log/info "stopping logging component")
  (Thread/setDefaultUncaughtExceptionHandler nil)
  (dissoc system :logger))
