(ns lib.security.oidc.interceptor
  (:require [lib.security.oidc.jwt :as jwt]
            [clojure.tools.logging :as log]
            [clojure.string :as str]
            [clojure.core.async :as a]))


;; .___  __
;; |   |/  |_  ____ ______
;; |   \   __\/ ___\\____ \
;; |   ||  | \  \___|  |_> >
;; |___||__|  \___  >   __/
;;                \/|__|

(defn default-unauthorized
  "Default handler for any failure.
  Will receive a failure keyword and possibly an exception"
  {:added "1.1.0"}
  [{:keys [request] :as ctx} failure & [?ex]]
  (assoc ctx
         :response {:status 401
                    :body {:error "Unauthorized"
                           :message (condp = failure
                                      :header-missing "Bearer token header is missing in the request."
                                      :keyset-error "Keyset error."
                                      (str "OIDC " (name failure) " error."))
                           :infos (cond-> {}
                                    (ex-cause ?ex) (assoc :cause (ex-cause ?ex))
                                    (ex-data  ?ex) (assoc :data  (ex-data ?ex)))
                           :request (select-keys request
                                                 [:remote-addr
                                                  :form-params
                                                  :query-params
                                                  :request-method
                                                  :context-path])}
                    :headers {}}
         :security.oidc/failure failure))

(defn decode-enter-sync
  "Adapted:
   from https://auth0.com/blog/secure-a-clojure-web-api-with-auth0/"
  {:added "1.1.0"}
  [ctx
   required?
   check-header
   unauthorized
   keyset
   unsign-opts]
  (try
    (if-let [auth-header (get-in ctx
                                 [:request
                                  :headers
                                  check-header])]
      (if (str/starts-with? auth-header "Bearer ")
        (let [access-token (subs auth-header 7)]
          (try
            (-> ctx
                (assoc :security.oidc/token access-token)
                (assoc-in [:request
                           :security.oidc/claims]
                          (jwt/unsign
                           keyset
                           access-token
                           unsign-opts)))
            (catch clojure.lang.ExceptionInfo exi
              (case (some-> exi
                            ex-data
                            :type)
                ;; Unknown Key
                ::jwt/kid-not-found
                (unauthorized ctx :kid-not-found exi)
                ;; Invalid Key per buddy-sign
                :validation
                (unauthorized ctx :token-invalid exi)
                ;; Throw to top-level handling
                (throw exi)))))
        ;; bad auth header
        (if required?
          (unauthorized ctx :header-invalid)
          ctx))
      ;; no auth header
      (if required?
        (unauthorized ctx :header-missing)
        ctx))
    (catch Exception ex
      (log/warn "Unknown failure yielded a 401")
      (unauthorized ctx :unknown ex))))

(defn oidc-auth-interceptor
  "Given a function that returns a map of public keys, return an interceptor
  that decodes claims and stores them on the context as
  :com.yetanalytics.pedestal-oidc/claims.

  If :async? is true, the function is expected to return a channel unless
  :keyset-blocking? is also true in which case it will be run in a thread.

  Other options:

    :required? - Return a 401 unless valid claims are present.
    :unauthorized - A function that will receive the context map to handle a 401,
      a failure keyword and possibly an exception.
    :check-header - the header to check for the access token.
    :unsign-opts - extra options to pass to buddy-sign.
  "
  {:added "1.1.0"}
  [get-keyset-fn
   & {:keys [required?
             check-header
             unauthorized
             async?
             keyset-blocking?
             unsign-opts]
      :or {required? true
           check-header "authorization"
           unauthorized default-unauthorized
           async? false
           keyset-blocking? false
           unsign-opts {}}}]
  {:name ::oidc-auth-interceptor
   :enter
   (fn [ctx]
     (if async?
       (a/go
         (if-some [keyset (a/<!
                           (if keyset-blocking?
                             (a/thread (get-keyset-fn ctx))
                             (get-keyset-fn ctx)))]
           (decode-enter-sync
            ctx
            required?
            check-header
            unauthorized
            keyset
            unsign-opts)
           (unauthorized ctx :keyset-invalid)))
       (try
         (if-some [keyset (get-keyset-fn ctx)]
           (decode-enter-sync
            ctx
            required?
            check-header
            unauthorized
            keyset
            unsign-opts)
           (unauthorized ctx :keyset-invalid))
         (catch Exception ex
           (unauthorized ctx :keyset-error ex)))))})
