(ns commons.components.kafka
  "Kafka Client component"
  {:added "0.0.2"}
  (:require [integrant.core :as ig]
            [clojure.string :as s]
            [commons.forms :refer :all]
            [jsonista.core :as json])
  (:import [org.apache.kafka.clients.consumer KafkaConsumer]
           [org.apache.kafka.clients.producer KafkaProducer ProducerRecord]
           [org.apache.kafka.common.serialization Serializer Deserializer]))


(def default-bootstrap-endpoint
  "127.0.0.1:9092")

(def json-mapper
  (json/object-mapper {:encode-key-fn name
                       :decode-key-fn keyword
                       :date-format "yyyy-MM-dd'T'HH:mm:ss.SSSX"}))


;;; Unsafe deserializers

(defn json-serializer []
  (reify
    Serializer
    (close [this])
    (configure [this config is-key?])
    (serialize [this topic payload]
      (.getBytes (json/write-value-as-string payload json-mapper)))))

(defn json-deserializer []
  (reify
    Deserializer
    (close [this])
    (configure [this config is-key?])
    (deserialize [this topic payload]
      (let [as-string (String. payload)]
        (try (json/read-value as-string json-mapper)
             (catch Exception e
               (throw (ex-info "malformed json"
                               {:cause e
                                :content as-string
                                :topic topic}))))))))

(def config
  "Default component configuration
   that need to be deepmerged with value
   override in order to have an calibrated
   at project level."
  {:component/kafka
   {:logger (ig/ref :component/logging)
    ;; http://kafka.apache.org/documentation/#producerconfigs
    :producer {:key.serializer org.apache.kafka.common.serialization.StringSerializer ;json-serializer
               :value.serializer org.apache.kafka.common.serialization.StringSerializer ; json-serializer
               :bootstrap.servers default-bootstrap-endpoint}

    ;; http://kafka.apache.org/documentation/#consumerconfigs
    :consumer {:key.deserializer org.apache.kafka.common.serialization.StringDeserializer
               :value.deserializer org.apache.kafka.common.serialization.StringDeserializer
               :bootstrap.servers default-bootstrap-endpoint}
    }})

(defn- stringify-configmap
  "Convert clojure map to stringmap
   compliant to kafka java API."
  [config]
  (into {}
        (for [[k v] config]
          [(name k) v])))


(defn- build-consumer
  "Build kafka consumer" ^KafkaConsumer
  [configmap]
  (when configmap
    (if-let [props (stringify-configmap configmap)]
      (KafkaConsumer. props))))


(defn- build-producer
  "Build kafka producer" ^KafkaProducer
  [configmap]
  (when configmap
    (if-let [props (stringify-configmap configmap)]
      (KafkaProducer. props))))


(defn- send!
  "Send message to topic"
  [producer topic payload]
  (when producer
    (.send producer (ProducerRecord. topic "a" payload))))

;;;
;;; Integrant setup
;;;

(defmethod ig/init-key :component/kafka
  [_ {:keys [consumer producer] :as sys}]
  (let [*consumer (build-consumer consumer)
        *producer (build-producer producer)]
    (assoc sys
           :*producer *producer
           :*consumer *consumer
           :send! send!)))


(defmethod ig/halt-key! :component/kafka
  [_ {:keys [*producer *consumer] :as sys}]
  (when (some? *producer)
    (.close *producer))
  (when (some? *consumer)
    (.close *consumer)))
