(ns thdr.kfk.core
  (:require [camel-snake-kebab.core :refer [->snake_case ->kebab-case]]))

(defmacro defkafkamessage
  "Syntactic sugar for generating messages for Kafka.
   Sets the topic same to `event`, fetches `key` from
   provided object if `key-fn` is set and transforms a value
   with provided `serialize-fn`.

   Example:

   (defkafkamessage user-event [event-type]
     :topic :user-events ;; a Kafka topic message will be sent to
     :key-fn #(-> % :id str) ;; applied to obj before serialization
     :serialize-fn #(merge % {:type event-type :id (uuid-to-bytes (:id %))})
     :additional-key :some-value)

   This macro generates two functions:
   1. make-:event-name:-message. This function returns
   a message which is ready to be sent to Kafka if there's no
   need for additional transformation and serialization.
   (make-user-created-message {:id (java.util.UUID/randomUUID)}
                              {:partition 0});; {:topic ..., :key ..., :value ..., ...}

   2. make-:event-name:-message*. Returns a map with provided additional keys
   and a :message key with a generated message. May be useful when
   those additional keys are needed for further serialization/transformation.
   For example, you can provide :key-schema and :value-schema keys in
   `defkafkamessage` and then pass a map generated by this function
   to some `avro-seralize` function, which will extract all schemas,
   serialize the message and finally return a map which can be used
   with `kfk.producer/send`.

   (make-user-created-message* {:id (java.util.UUID/randomUUID)}
                               {:partition 0})
   "
  [event args & {:keys [topic key-fn serialize-fn deserialize-fn]
                 :or {serialize-fn identity
                      deserialize-fn identity}
                 :as opts}]
  {:pre [(not (nil? topic))
         (vector? args)]}
  (let [fn-name (str "make-" event "-message")
        fn-name* (symbol (str fn-name "*"))
        fn-name (symbol fn-name)]
    `(do
       (defn ~fn-name*
         ([~@args obj#]
          (~fn-name* ~@args obj# {}))
          ([~@args obj# opts#]
           (let [{partition# :partition
                  key# :key
                  timestamp# :timestamp} opts#
                 opts# (dissoc opts# :topic :key-fn :serialize-fn :deserialize-fn)]
             (merge opts#
                    {:message
                     {:topic ~(-> topic name ->snake_case)
                      :partition partition#
                      :timestamp timestamp#
                      :key (cond key# key# ~key-fn (~key-fn obj#) :else nil)
                      :value (~serialize-fn obj#)}}))))

       (defn ~fn-name
         [~@args & rest#]
         (:message (apply ~fn-name* ~@args rest#))))))
