(ns tenfold.pull-stream
  (:refer-clojure :exclude [new empty take map filter flatten count cat])
  (:require ["pull-stream" :as pull-stream]
            ["pull-cat" :as pull-cat]
            ["pull-catch" :as pull-catch]
            ["pull-zip" :as pull-zip]
            ["pull-many" :as pull-many]
            ["pull-promise" :as pull-promise]
            ["pull-defer" :as pull-defer]
            ["pull-tee" :as pull-tee]
            ["pull-tap" :as pull-tap]
            ["pull-pair" :as pull-pair]
            ["pull-notify" :as pull-notify]
            ["pull-pushable" :as pull-pushable]
            ["stream-to-pull-stream" :as stream-to-pull-stream]
            ["pull-high-watermark" :as pull-high-watermark]
            ["pull-cache" :as pull-cache]
            ["pull-flat-merge" :as pull-flat-merge]))

(def stream pull-stream)
(def empty pull-stream/empty)
(def drain pull-stream/drain)
(def on-end pull-stream/onEnd)
(def collect pull-stream/collect)
(def take pull-stream/take)
(def map pull-stream/map)
(def filter pull-stream/filter)
(def async-map pull-stream/asyncMap)
(def flatten pull-stream/flatten)
(def once pull-stream/once)
(def through pull-stream/through)
(def infinite pull-stream/infinite)
(def count pull-stream/count)
(def error pull-stream/error)

(def zip pull-zip)
(def tee pull-tee)
(def tap pull-tap/tap)
(def catch pull-catch)
(def cache pull-cache)
(def high-watermark pull-high-watermark)
(def flatmerge pull-flat-merge)

(def read-stream-to-source (-> stream-to-pull-stream .-source))
(def write-stream-to-source (-> stream-to-pull-stream .-sink))

(defn act
  "Yields from the given sources (in sequence) but also discard their values.
  This is useful for sources with side effects."
  [& sources]
  (stream
   (pull-cat (apply array sources))
   (filter (constantly false))))

(defn pushable
  "Returns a `[push end source]` tuple where:

  - `push` pushes an event into the source.
  - `end` ends the source.
  - `source` is the source."
  ([]
   (pushable nil))
  ([done]
   (let [p (pull-pushable true done)]
     [(-> p .-push) (-> p .-end) (-> p .-source)])))

(defn notify []
  (let [n (pull-notify)]
    [(.listen n) n (.-end n)]))

(defn pair []
  (let [p (pull-pair)]
    [(-> p .-sink) (-> p .-source)]))

(defn flatmap
  [f]
  (stream
   (map f)
   (flatten)))

(defn values [xs]
  (->> xs
       (apply array)
       (pull-stream/values)))

(defn cat
  [& args]
  (->> args
       (apply array)
       (pull-cat)))

(defn many
  [& args]
  (->> args
       (apply array)
       (pull-many)))

(def defer-source (-> pull-defer .-source))
(def defer-through (-> pull-defer .-through))
(defn defer
  "Returns `[resolve, source]`.  In the following example, the second stream
  starts only after the first ends.

  (let [[resolve source] (p/defer)]
     (p/stream
      (p/on-end resolve))
     (p/stream
      (source (p/values [1 2 3]))))"
  []
  (let [*stream (atom nil)
        deferred (defer-source)]
    [(fn []
       (-> deferred (.resolve @*stream)))
     (fn [s]
       (reset! *stream s)
       deferred)]))

(defn timeout-source
  "Returns a source that behaves like `once` except that it only yields the
  value after `n` milliseconds."
  [val n]
  (let [deferred (defer-source)]
    (js/setTimeout #(-> deferred (.resolve (once val))) n)
    deferred))

(defn timeout-through
  "Returns a through that blocks initially, but functions as a noop after `n`
  milliseconds."
  [n]
  (let [deferred (defer-through)
        *triggered (atom false)]
    (stream
     (through #(when-not @*triggered
                 (do (js/setTimeout
                      (fn [] (-> deferred (.resolve (through identity)))))
                     (reset! *triggered true))))
     deferred)))

(defn promise-through
  "Takes a function that takes a value and returns a promise."
  [f]
  (pull-promise/through f))

(defn promise-source
  "Takes a function that returns a promise."
  [f]
  (stream
   (once true)
   (promise-through f)))

(defn combine
  "Combine multiple sources into one source, emitting values as they come.

  When one stream errors, all streams are aborted and the error is propagated
  downstream.

  When all streams end, the combined stream ends.
  
  This function does not respect back pressure.  That is, it drains values
  from the sources as soon as they come, regardless of whether values are
  being read from the combined stream."
  [& sources]
  (let [source-count (cljs.core/count sources)
        *ended-source-count (atom 0)
        *aborted? (atom false)
        [push end source] (pushable)]
    (doseq [source sources]
      (stream
       source
       (drain
        (fn [x]
          (if @*aborted?
            false
            (push x)))
        (fn [err]
          (if err
            (do
              ;; Abort all streams when one stream errors
              (reset! *aborted? true)
              (end err))
            (when (>= (swap! *ended-source-count inc) source-count)
              ;; End the combined stream when all streams have ended
              (end)))))))
    source))