# Gregor-Lite

[![Clojars Project](https://img.shields.io/clojars/v/io.synple/gregor-lite.svg)](https://clojars.org/io.synple/gregor-lite)

Lightweight Clojure bindings for [Apache Kafka](http://kafka.apache.org/) `0.9.X` and up.

Currently targeting Kafka `0.10.0.1`.

Gregor wraps most of the Java API for the Kafka [Producer](http://kafka.apache.org/0100/javadoc/index.html?org/apache/kafka/clients/producer/KafkaProducer.html) and [New Consumer](http://kafka.apache.org/0100/javadoc/index.html?org/apache/kafka/clients/consumer/KafkaConsumer.html) and is almost feature complete as of `0.2.0`. The intent of this project is to stay very close to the Kafka API instead of adding more advanced features.

Gregor Lite differs from Gregor in that it only depends on the kafka-clients library, instead of the full kafka library, making the package significantly smaller. The trade-off is that the topic management is sacrificed.

## Example

Here's an example of at-least-once processing (using the excellent [`mount`](https://github.com/tolitius/mount)):

```clojure
(ns gregor-lite-sample-app.core
  (:gen-class)
  (:require [clojure.repl :as repl]
            [gregor-lite.core :as gregor-lite]
            [mount.core :as mount :refer [defstate]]))

(def run (atom true))

(defstate consumer
  :start (gregor-lite/consumer "localhost:9092"
                          "testgroup"
                          ["test-topic"]
                          {"auto.offset.reset" "earliest"
                           "enable.auto.commit" "false"})
  :stop (gregor-lite/close consumer))

(defstate producer
  :start (gregor-lite/producer "localhost:9092")
  :stop (gregor-lite/close producer))

(defn -main
  [& args]
  (mount/start)
  (repl/set-break-handler! (fn [sig] (reset! run false)))
  (while @run
    (let [consumer-records (gregor-lite/poll consumer)
          values (process-records consumer-records)]
      (doseq [v values]
        (gregor-lite/send producer "other-topic" v))
      (gregor-lite/commit-offsets! consumer)))
  (mount/stop))
```

Transformations over consumer records are applied in `process-records`. Each record in
the `seq` returned by `poll` is a map. Here's an example with a JSON object as the
`:value`:

```clojure
{:value "{\"foo\":42}"
 :key nil
 :partition 0
 :topic "test-topic"
 :offset 939}
```

## Producing

Gregor provides the `send` function for asynchronously sending a record to a topic. There
are multiple arities which correspond to those of the `ProducerRecord`
[Java constructor](https://kafka.apache.org/0100/javadoc/org/apache/kafka/clients/producer/ProducerRecord.html). If
you'd like to provide a callback to be invoked when the send has been acknowledged use
`send-then` instead.
