create schema if not exists schemamap;

create or replace function schemamap.trggr_set_update_common_fields()
returns trigger as $$
begin
  -- allow setting updated_at explicitly
  if new.updated_at is not distinct from old.updated_at then
    new.updated_at = now();
  end if;
  new.version = old.version + 1;

  return new;
end; $$ language plpgsql stable;

create or replace function schemamap.trggr_optimistic_update_guard()
returns trigger as $$
begin
  if new.version != old.version + 1 then
    raise exception 'Optimistic update failed' using hint = 'try again';
  end if;

  -- decrement new version so the trggr_set_update_common_fields trigger doesn't bump it twice
  new.version = old.version;

  return new;
end; $$ language plpgsql stable;

create or replace function schemamap.add_common_triggers(table_name text)
returns void as $$
begin
  execute format (
    'drop trigger if exists aaa_sm_io_optimistic_locking_update_guard on %s', table_name
  );

  execute format('
    create trigger aaa_sm_io_optimistic_locking_update_guard
    before update of version on %s for each row
    execute procedure schemamap.trggr_optimistic_update_guard();
  ', table_name);

  execute format (
    'drop trigger if exists aab_sm_io_maintain_update_fields on %s;', table_name
  );

  execute format('
  create trigger aab_sm_io_maintain_update_fields
  before update on %s for each row
  execute procedure schemamap.trggr_set_update_common_fields();
  ', table_name);
end; $$ language plpgsql volatile;

create table if not exists schemamap.table_metadata (
  id bigint primary key generated by default as identity,
  table_name text not null unique,
  natural_key_constraint_name text,
  created_at timestamptz not null default now(),
  updated_at timestamptz not null default now(),
  version bigint not null default 0 check (version >= 0)
);

comment on table schemamap.table_metadata is 'Categorizes tables to be mapped from/to.';
comment on column schemamap.table_metadata.table_name is 'The fully qualified table name as described, as per search_path.';
comment on column schemamap.table_metadata.natural_key_constraint_name is 'The unique constraint name that is used to support bidirectional mapping.';

select schemamap.add_common_triggers('schemamap.table_metadata');

create or replace function schemamap.trim_str(text)
returns text as $$
  select trim($1);
$$ language sql immutable strict parallel safe;

create or replace function schemamap.identity(anyelement)
returns anyelement as $$
  select $1;
$$ language sql immutable strict parallel safe;

create or replace function schemamap.trim_str(text)
returns text as $$
  select trim($1);
$$ language sql immutable strict parallel safe;

-- TODO: handle escapes
create or replace function schemamap.split_comma_sep_str(text)
returns text[] as $$
  select string_to_array($1, ',');
$$ language sql immutable strict parallel safe;

create or replace function schemamap.join_array_to_comma_sep_str(anyarray)
returns text as $$
  select array_to_string($1, ',');
$$ language sql immutable strict parallel safe;

create table if not exists schemamap.bidi_mapping_fns (
  name text primary key,
  i18n jsonb not null,
  forward_fn_name text not null,
  backward_fn_name text not null,
  input_type text not null,
  exact boolean not null,
  created_at timestamptz not null default now(),
  updated_at timestamptz not null default now(),
  version bigint not null default 0 check (version >= 0)
);

select schemamap.add_common_triggers('schemamap.bidi_mapping_fns');

insert into schemamap.bidi_mapping_fns
(name, i18n, forward_fn_name, backward_fn_name, input_type, exact)
values
('trim_str', '{"name": {"en": "Trim"}}'::jsonb, 'trim_str', 'identity', 'text', false),
('identity', '{"name": {"en": "Identity"}}'::jsonb, 'identity', 'identity', 'anyelement', true),
('split_comma_array', '{"name": {"en": "Split Commas To Array"}}'::jsonb, 'split_comma_sep_str', 'join_array_to_comma_sep_str', 'text', true)
on conflict (name) do update set
  i18n = excluded.i18n,
  forward_fn_name = excluded.forward_fn_name,
  backward_fn_name = excluded.backward_fn_name,
  input_type = excluded.input_type,
  exact = excluded.exact;

create table if not exists schemamap.column_mappings(
  id bigint primary key generated by default as identity,
  source_table text not null,
  source_column_name text not null,
  target_table text not null,
  target_column_name text not null,
  bidi_mapping_fn text not null references schemamap.bidi_mapping_fns,
  created_at timestamptz not null default now(),
  updated_at timestamptz not null default now(),
  version bigint not null default 0 check (version >= 0),
  unique (source_table, source_column_name, target_table, target_column_name)
);

create table if not exists schemamap.external_source_types (
  value text primary key,
  i18n jsonb,
  description text
);

insert into schemamap.external_source_types
(value, i18n, description)
values
('LOCAL_CSV', '{"name": {"en": "Local CSV"}}'::jsonb, 'Local comma separated values file, either a relative or absolute path'),
('LOCAL_SSV', '{"name": {"en": "Local SSV"}}'::jsonb, 'Local semicolon separated values (EU) file, either a relative or absolute path'),
('LOCAL_XLSX', '{"name": {"en": "Local XLSX"}}'::jsonb, 'Local xlsx file, either a relative or absolute path'),
('GOOGLE_SHEET', '{"name": {"en": "Google Sheet"}}'::jsonb, 'Google Sheet docs')
on conflict (value) do update set
  i18n = excluded.i18n,
  description = excluded.description;

create table if not exists schemamap.external_sources(
  id bigint primary key generated by default as identity,
  url text,
  sheet_id text,
  type text not null references schemamap.external_source_types,
  created_at timestamptz not null default now(),
  updated_at timestamptz not null default now(),
  version bigint not null default 0 check (version >= 0),
  constraint sheet_id_or_url_must_be_present check (
    case when type = 'GOOGLE_SHEET' then
      sheet_id is not null
    else
      url is not null
    end)
);

select schemamap.add_common_triggers('schemamap.external_sources');

create or replace function schemamap.get_function_definition(function_name text)
returns text as $$
  select pg_catalog.pg_get_functiondef(pp.oid)
  from pg_proc pp
  join pg_namespace pn on pn.oid = pp.pronamespace
  where
    pn.nspname = 'schemamap' and
    pp.proname = function_name;
$$ language sql stable;

create or replace function schemamap.update_function_definition
(function_name text, new_body text)
returns void as $$
declare
  v_function_oid oid;
  v_function_args text;
  v_function_returns text;
  v_function_lang text;
  v_function_volatile text;
  volatile_verbose text;
begin
  select pp.oid,
         pg_catalog.pg_get_function_arguments(pp.oid),
         pg_catalog.pg_get_function_result(pp.oid),
         pl.lanname,
         pp.provolatile
  into v_function_oid, v_function_args, v_function_returns, v_function_lang, v_function_volatile
  from pg_proc pp
  join pg_namespace pn on pn.oid = pp.pronamespace
  join pg_language pl on pl.oid = pp.prolang
  where pn.nspname = 'schemamap' and pp.proname = $1;

  volatile_verbose := case
    when v_function_volatile = 's' then 'stable'
    when v_function_volatile = 'i' then 'immutable' end;

  if v_function_volatile = 'v' then
    raise exception 'function %.% is volatile. update not allowed.', 'schemamap', $2;
  end if;

  execute format('create or replace function schemamap.%I(%s) returns %s as $fn$ %s $fn$ language %s %s',
  $1, v_function_args, v_function_returns, new_body, v_function_lang, volatile_verbose);
  raise notice 'Updated schemamap UDF definition for %', $1;
end; $$ language plpgsql volatile security definer;

-- https://www.postgresql.org/docs/current/sql-createfunction.html#SQL-CREATEFUNCTION-SECURITY
revoke all on function schemamap.update_function_definition(text, text) from public;

create or replace function schemamap.verify_installation()
returns table(tenants_defined boolean,
              mdes_defined boolean,
              external_sources_defined boolean) as $$
  select
    exists(select 1 from schemamap.list_tenants() where tenant_id is not null) as tenants_defined,
    false as mdes_defined,
    false as external_sources_defined
$$ language sql stable;

create or replace function schemamap.define_master_data_entity
(mde_name text, new_body text)
returns void as $$
begin
  execute format('create or replace view schemamap.mde_%I as %s', $1, $2);
  raise notice '(Re-)defined schemamap MDE definition for %', $1;
end; $$ language plpgsql volatile security definer;


-- https://www.postgresql.org/docs/current/sql-createfunction.html#SQL-CREATEFUNCTION-SECURITY
revoke all on function schemamap.define_master_data_entity(text, text) from public;

create or replace function schemamap.list_mdes()
returns table(mde_name text) as $$
  select substring(table_name from 5) as mde_name
  from information_schema.views
  where table_schema = 'schemamap' and table_name like 'mde\_%' escape '\';
$$ language sql stable;

create or replace function schemamap.dependent_views(p_schema_name text, p_view_name text)
returns table (
  view text,
  level int
) as $$
with recursive views as (
-- get the directly depending views
select distinct
  v.oid::regclass as view,
  1 as level
from pg_depend as d
join pg_rewrite as r on r.oid = d.objid
join pg_class as v on v.oid = r.ev_class and v.relkind = 'v' and v.oid != d.refobjid
where
  d.classid = 'pg_rewrite'::regclass and
  d.refclassid = 'pg_class'::regclass and
  d.deptype = 'n' and
  d.refobjid = (quote_ident(p_schema_name) || '.' || quote_ident(p_view_name))::regclass

union all

select distinct
  v.oid::regclass,
  views.level + 1
from views
join pg_depend as d on d.refobjid = views.view
join pg_rewrite as r on r.oid = d.objid
join pg_class as v on v.oid = r.ev_class and v.relkind = 'v' and v.oid != views.view -- avoid loop
where
  d.classid = 'pg_rewrite'::regclass and
  d.refclassid = 'pg_class'::regclass and
  d.deptype = 'n')
select view, level from views;
$$ language sql stable;

create or replace function schemamap.view_dependencies(p_schema_name text, p_view_name text)
returns table (
  object_type text,
  schema_name text,
  object_name text,
  level int
) as $$
with recursive dependencies as (
    -- get the objects directly depended on by the view
    select
        case when c.relkind = 'v' then 'view'
             when c.relkind = 'r' then 'table'
             else c.relkind::text
        end as object_type,
        n.nspname as schema_name,
        c.oid::regclass::text as object_name,
        1 as level
    from pg_depend as d
    join pg_class as c on c.oid = d.refobjid
    join pg_rewrite as r on r.oid = d.objid
    join pg_namespace as n on n.oid = c.relnamespace
    where d.classid = 'pg_rewrite'::regclass
    and d.refclassid = 'pg_class'::regclass
    and d.deptype = 'n'
    and r.ev_class = (quote_ident(p_schema_name) || '.' || quote_ident(p_view_name))::regclass
    and c.oid != r.ev_class -- avoid self
    and n.nspname = p_schema_name
    union all
    -- add objects that these objects depend on
    select
        case when c.relkind = 'v' then 'view'
             when c.relkind = 'r' then 'table'
             else c.relkind::text
        end as object_type,
        n.nspname as schema_name,
        c.oid::regclass::text,
        dependencies.level + 1
    from dependencies
    join pg_depend as d on d.objid = dependencies.object_name::regclass
    join pg_class as c on c.oid = d.refobjid
    join pg_namespace as n on n.oid = c.relnamespace
    where d.refclassid = 'pg_class'::regclass
    and d.deptype = 'n'
    and c.oid::regclass::text <> dependencies.object_name  -- Avoid processing the same object again
)
select distinct object_type, schema_name, object_name, level from dependencies order by level, object_name;
$$ language sql stable;

create or replace function schemamap.master_date_entity_candidates()
returns
  table(schema_name text,
        table_name text,
        approx_rows bigint,
        foreign_key_count bigint,
        probability_master_data real)
as $$
with tablestats as (
    select
        ns.nspname as schema,
        cls.relname as tablename,
        cls.reltuples::bigint as approx_rows,
        count(con.*) as foreign_key_count
    from pg_catalog.pg_class cls
    join pg_catalog.pg_namespace ns on ns.oid = cls.relnamespace
    left join pg_catalog.pg_constraint con on con.confrelid = cls.oid
    where cls.relkind = 'r' and ns.nspname not in (select nspname from schemamap.ignored_schemas())
    group by 1, 2, 3
), minmax as (
    select
        min(approx_rows) as min_rows,
        max(approx_rows) as max_rows,
        min(foreign_key_count) as min_fk,
        max(foreign_key_count) as max_fk
    from tablestats
)
select
    schema as schema_name,
    tablename as table_name,
    approx_rows,
    foreign_key_count::bigint as foreign_key_count,
    coalesce(
        case
            when max_fk = min_fk and max_fk = 0 then
                (max_rows - approx_rows)::real / nullif((max_rows - min_rows), 0)::real
            else
                (0.5 * ((max_rows - approx_rows)::real / nullif((max_rows - min_rows), 0)::real)) +
                (0.5 * ((foreign_key_count - min_fk)::real / nullif((max_fk - min_fk), 0)::real))
        end,
        0
    ) as probability_master_data
from tablestats, minmax
order by probability_master_data desc
$$ language sql stable;
