(ns io.rkn.conformity
  (:require [datomic.api :refer [q db] :as d]
            [clojure.java.io :as io]))

(def default-conformity-attribute :confirmity/conformed-norms)
(def conformity-ensure-norm-tx :conformity/ensure-norm-tx)

(def ensure-norm-tx-txfn
  "Transaction function to ensure each norm tx is executed exactly once"
  (d/function
   '{:lang :clojure
     :params [db norm-attr norm index-attr index tx]
     :code (when-not (seq (q '[:find ?tx
                               :in $ ?na ?nv ?ia ?iv
                               :where [?tx ?na ?nv ?tx] [?tx ?ia ?iv ?tx]]
                             db norm-attr norm index-attr index))
             (cons {:db/id (d/tempid :db.part/tx)
                    norm-attr norm
                    index-attr index}
                   tx))}))

(defn read-resource
  "Reads and returns data from a resource containing edn text. An
  optional argument allows specifying opts for clojure.edn/read"
  ([resource-name]
   (read-resource {:readers *data-readers*} resource-name))
  ([opts resource-name]
   (->> (io/resource resource-name)
        (io/reader)
        (java.io.PushbackReader.)
        (clojure.edn/read opts))))

(defn index-attr
  "Returns the index-attr corresponding to a conformity-attr"
  [conformity-attr]
  (keyword (namespace conformity-attr)
           (str (name conformity-attr) "-index")))

(defn has-attribute?
  "Returns true if a database has an attribute named attr-name"
  [db attr-name]
  (-> (d/entity db attr-name)
      :db.install/_attribute
      boolean))

(defn has-function?
  "Returns true if a database has a function named fn-name"
  [db fn-name]
  (-> (d/entity db fn-name)
      :db/fn
      boolean))

(defn ensure-conformity-schema
  "Ensure that the two attributes and one transaction function
  required to track conformity via the conformity-attr keyword
  parameter are installed in the database."
  [conn conformity-attr]
  (when-not (has-attribute? (db conn) conformity-attr)
    (d/transact conn [{:db/id (d/tempid :db.part/db)
                       :db/ident conformity-attr
                       :db/valueType :db.type/keyword
                       :db/cardinality :db.cardinality/one
                       :db/doc "Name of this transaction's norm"
                       :db/index true
                       :db.install/_attribute :db.part/db}]))
  (when-not (has-attribute? (db conn) (index-attr conformity-attr))
    (d/transact conn [{:db/id (d/tempid :db.part/db)
                       :db/ident (index-attr conformity-attr)
                       :db/valueType :db.type/long
                       :db/cardinality :db.cardinality/one
                       :db/doc "Index of this transaction within its norm"
                       :db/index true
                       :db.install/_attribute :db.part/db}]))
  (when-not (has-function? (db conn) conformity-ensure-norm-tx)
    (d/transact conn [{:db/id (d/tempid :db.part/user)
                       :db/ident conformity-ensure-norm-tx
                       :db/doc "Ensures each norm tx is executed exactly once"
                       :db/fn ensure-norm-tx-txfn}])))

(defn conforms-to?
  "Does database have a norm installed?

      conformity-attr  (optional) the keyword name of the attribute used to
                       track conformity
      norm             the keyword name of the norm you want to check
      tx-count         the count of transactions for that norm"
  ([db norm tx-count]
   (conforms-to? db default-conformity-attribute norm tx-count))
  ([db conformity-attr norm tx-count]
   (and (has-attribute? db conformity-attr)
        (pos? tx-count)
        (-> (q '[:find ?tx
                 :in $ ?na ?nv
                 :where [?tx ?na ?nv ?tx]]
               db conformity-attr norm)
            count
            (= tx-count)))))

(defn reduce-txes
  "Reduces the seq of transactions for a norm into a transaction
  result accumulator"
  [acc conn norm-attr norm-name txes]
  (reduce
   (fn [acc [tx-index tx]]
     (try
       (let [safe-tx [conformity-ensure-norm-tx
                      norm-attr norm-name
                      (index-attr norm-attr) tx-index
                      tx]
             tx-result @(d/transact conn [safe-tx])]
         (if (next (:tx-data tx-result))
           (conj acc {:norm-name norm-name
                      :tx-index tx-index
                      :tx-result tx-result})
           acc))
       (catch Throwable t
         (let [reason (.getMessage t)
               data {:succeeded acc
                     :failed {:norm-name norm-name
                              :tx-index tx-index
                              :reason reason}}]
           (throw (ex-info reason data t))))))
   acc (map-indexed vector txes)))

(defn reduce-norms
  "Reduces norms from a norm-map specified by a seq of norm-names into
  a transaction result accumulator"
  [acc conn norm-attr norm-map norm-names]
  (reduce
   (fn [acc norm-name]
     (let [{:keys [txes requires]} (get norm-map norm-name)]
       (cond (conforms-to? (db conn) norm-attr norm-name (count txes))
             acc
             (empty? txes)
             (let [reason (str "No transactions provided for norm " norm-name)
                   data {:succeeded acc
                         :failed {:norm-name norm-name
                                  :reason reason}}]
               (throw (ex-info reason data)))
             :else
             (-> acc
                 (reduce-norms conn norm-attr norm-map requires)
                 (reduce-txes conn norm-attr norm-name txes)))))
   acc norm-names))

(defn ensure-conforms
  "Ensure that norms represented as datoms are conformed-to (installed), be they
  schema, data or otherwise.

      conformity-attr  (optional) the keyword name of the attribute used to
                       track conformity
      norm-map         a map from norm names to data maps.
                       a data map contains:
                         :txes     - the data to install
                         :requires - (optional) a list of prerequisite norms
                                     in norm-map.
      norm-names       (optional) A collection of names of norms to conform to.
                       Will use keys of norm-map if not provided.

  On success, returns a vector of maps with values for :norm-name, :tx-index,
  and :tx-result for each transaction that improved the db's conformity.

  On failure, throws an ex-info with a reason and data about any partial
  success before the failure."
  ([conn norm-map]
   (ensure-conforms conn norm-map (keys norm-map)))
  ([conn norm-map norm-names]
   (ensure-conforms conn default-conformity-attribute norm-map norm-names))
  ([conn conformity-attr norm-map norm-names]
   (ensure-conformity-schema conn conformity-attr)
   (reduce-norms [] conn conformity-attr norm-map norm-names)))
