/*
 * Decompiled with CFR 0.152.
 */
package io.dropwizard.revolver.core.resilience;

import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.Maps;
import io.dropwizard.revolver.core.config.HystrixCommandConfig;
import io.dropwizard.revolver.core.config.RevolverConfig;
import io.dropwizard.revolver.core.config.RevolverServiceConfig;
import io.dropwizard.revolver.core.config.ThreadPoolGroupConfig;
import io.dropwizard.revolver.core.config.hystrix.ThreadPoolConfig;
import io.dropwizard.revolver.core.resilience.ResilienceHttpContext;
import io.dropwizard.revolver.http.config.RevolverHttpServiceConfig;
import io.github.resilience4j.bulkhead.Bulkhead;
import io.github.resilience4j.bulkhead.BulkheadConfig;
import io.github.resilience4j.bulkhead.BulkheadRegistry;
import io.github.resilience4j.circuitbreaker.CircuitBreaker;
import io.github.resilience4j.circuitbreaker.CircuitBreakerRegistry;
import io.github.resilience4j.micrometer.tagged.TaggedBulkheadMetrics;
import io.github.resilience4j.micrometer.tagged.TaggedCircuitBreakerMetrics;
import io.micrometer.core.instrument.Clock;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.dropwizard.DropwizardConfig;
import io.micrometer.core.instrument.dropwizard.DropwizardMeterRegistry;
import io.micrometer.core.instrument.util.HierarchicalNameMapper;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResilienceUtil {
    private static final Logger log = LoggerFactory.getLogger(ResilienceUtil.class);
    private static final Map<String, String> DEFAULT_KEY_VALUE_MAP = Maps.newHashMap();
    private static final String METRIC_PREFIX = "resilience";
    private static final String DEFAULT_CIRCUIT_BREAKER = "revolver";
    private static BulkheadRegistry bulkheadRegistry;
    private static CircuitBreakerRegistry circuitBreakerRegistry;

    private ResilienceUtil() {
    }

    public static void initializeResilience(RevolverConfig revolverConfig, ResilienceHttpContext resilienceHttpContext, MetricRegistry metrics) {
        log.info("Initializing resilience util");
        DropwizardMeterRegistry metricRegistry = new DropwizardMeterRegistry(new DropwizardConfig(){

            public String prefix() {
                return ResilienceUtil.METRIC_PREFIX;
            }

            public String get(String s) {
                return (String)DEFAULT_KEY_VALUE_MAP.get(s);
            }
        }, metrics, HierarchicalNameMapper.DEFAULT, Clock.SYSTEM){

            protected Double nullGaugeValue() {
                return null;
            }
        };
        TaggedBulkheadMetrics.ofBulkheadRegistry((BulkheadRegistry)bulkheadRegistry).bindTo((MeterRegistry)metricRegistry);
        TaggedCircuitBreakerMetrics.ofCircuitBreakerRegistry((CircuitBreakerRegistry)circuitBreakerRegistry).bindTo((MeterRegistry)metricRegistry);
        ResilienceUtil.initializeBulkHeads(revolverConfig, resilienceHttpContext);
        ResilienceUtil.initializeCircuitBreakers(revolverConfig, resilienceHttpContext);
        ResilienceUtil.initializeTimeout(revolverConfig, resilienceHttpContext);
    }

    private static void initializeCircuitBreakers(RevolverConfig revolverConfig, ResilienceHttpContext resilienceHttpContext) {
        HashMap poolVsCircuitBreaker = Maps.newHashMap();
        resilienceHttpContext.setDefaultCircuitBreaker(circuitBreakerRegistry.circuitBreaker(DEFAULT_CIRCUIT_BREAKER));
        for (RevolverServiceConfig revolverServiceConfig : revolverConfig.getServices()) {
            ResilienceUtil.updateCBForThreadPools(poolVsCircuitBreaker, revolverServiceConfig);
            ResilienceUtil.updateCBForApiConfigs(poolVsCircuitBreaker, revolverServiceConfig);
            ResilienceUtil.updateCBForDefaultServiceConfig(poolVsCircuitBreaker, revolverServiceConfig);
        }
        poolVsCircuitBreaker.forEach((s, circuitBreaker) -> log.info("Resilience circuit breaker : {}, circuit break config : {} ", s, (Object)Float.valueOf(circuitBreaker.getCircuitBreakerConfig().getFailureRateThreshold())));
        resilienceHttpContext.setPoolVsCircuitBreaker(poolVsCircuitBreaker);
    }

    private static void initializeBulkHeads(RevolverConfig revolverConfig, ResilienceHttpContext resilienceHttpContext) {
        log.info("Initializing resilience bulk heads");
        HashMap poolVsBulkHead = Maps.newHashMap();
        for (RevolverServiceConfig revolverServiceConfig : revolverConfig.getServices()) {
            ResilienceUtil.updateBulkheadsForThreadPools(poolVsBulkHead, revolverServiceConfig);
            ResilienceUtil.updateBulkheadsForApiConfigs(poolVsBulkHead, revolverServiceConfig);
            ResilienceUtil.updateBulkHeadsForDefaultServiceConfig(poolVsBulkHead, revolverServiceConfig);
        }
        poolVsBulkHead.forEach((s, bulkhead) -> log.info("Resilience bulk head Key : {}, bulk head value : {} ", s, (Object)bulkhead.getBulkheadConfig().getMaxConcurrentCalls()));
        resilienceHttpContext.setPoolVsBulkHeadMap(poolVsBulkHead);
    }

    private static void initializeTimeout(RevolverConfig revolverConfig, ResilienceHttpContext resilienceHttpContext) {
        log.info("Initializing resilience time out");
        HashMap poolVsTimeout = Maps.newHashMap();
        for (RevolverServiceConfig revolverServiceConfig : revolverConfig.getServices()) {
            ResilienceUtil.updateTimeoutsForThreadPools(poolVsTimeout, revolverServiceConfig);
            ResilienceUtil.updateTimeoutsForApiConfigs(poolVsTimeout, revolverServiceConfig);
            ResilienceUtil.updateTimeoutsForDefaultServiceConfig(poolVsTimeout, revolverServiceConfig);
        }
        poolVsTimeout.forEach((s, timeout) -> log.info("Resilience timeout  Key : {}, timeout value : {} ", s, timeout));
        resilienceHttpContext.setPoolVsTimeout(poolVsTimeout);
    }

    private static void updateCBForApiConfigs(Map<String, CircuitBreaker> poolVsCircuitBreaker, RevolverServiceConfig revolverServiceConfig) {
        if (revolverServiceConfig instanceof RevolverHttpServiceConfig) {
            ((RevolverHttpServiceConfig)revolverServiceConfig).getApis().forEach(revolverHttpApiConfig -> {
                if (revolverHttpApiConfig.getRuntime() != null) {
                    HystrixCommandConfig hystrixCommandConfig = revolverHttpApiConfig.getRuntime();
                    if (hystrixCommandConfig == null || hystrixCommandConfig.getThreadPool() == null) {
                        return;
                    }
                    String threadPoolName = ResilienceUtil.getThreadPoolName(revolverServiceConfig, hystrixCommandConfig.getThreadPool());
                    if (StringUtils.isEmpty((CharSequence)threadPoolName)) {
                        threadPoolName = revolverServiceConfig.getService() + "." + revolverHttpApiConfig.getApi();
                    }
                    poolVsCircuitBreaker.putIfAbsent(threadPoolName, circuitBreakerRegistry.circuitBreaker(threadPoolName));
                }
            });
        }
    }

    private static void updateCBForThreadPools(Map<String, CircuitBreaker> poolVsCircuitBreaker, RevolverServiceConfig revolverServiceConfig) {
        ThreadPoolGroupConfig threadPoolGroupConfig = revolverServiceConfig.getThreadPoolGroupConfig();
        if (threadPoolGroupConfig != null) {
            threadPoolGroupConfig.getThreadPools().forEach(threadPoolConfig -> {
                String threadPoolName = ResilienceUtil.getThreadPoolName(revolverServiceConfig, threadPoolConfig);
                poolVsCircuitBreaker.put(threadPoolName, circuitBreakerRegistry.circuitBreaker(threadPoolName));
            });
        }
    }

    private static void updateCBForDefaultServiceConfig(Map<String, CircuitBreaker> poolVsCircuitBreaker, RevolverServiceConfig revolverServiceConfig) {
        if (revolverServiceConfig instanceof RevolverHttpServiceConfig) {
            ThreadPoolConfig threadPoolConfig = revolverServiceConfig.getRuntime().getThreadPool();
            if (threadPoolConfig == null) {
                return;
            }
            poolVsCircuitBreaker.put(revolverServiceConfig.getService(), circuitBreakerRegistry.circuitBreaker(revolverServiceConfig.getService()));
        }
    }

    private static void updateBulkheadsForApiConfigs(Map<String, Bulkhead> poolVsBulkHead, RevolverServiceConfig revolverServiceConfig) {
        if (revolverServiceConfig instanceof RevolverHttpServiceConfig) {
            ((RevolverHttpServiceConfig)revolverServiceConfig).getApis().forEach(revolverHttpApiConfig -> {
                if (revolverHttpApiConfig.getRuntime() != null) {
                    HystrixCommandConfig hystrixCommandConfig = revolverHttpApiConfig.getRuntime();
                    if (hystrixCommandConfig == null || hystrixCommandConfig.getThreadPool() == null) {
                        return;
                    }
                    String threadPoolName = ResilienceUtil.getThreadPoolName(revolverServiceConfig, hystrixCommandConfig.getThreadPool());
                    if (StringUtils.isEmpty((CharSequence)threadPoolName)) {
                        threadPoolName = revolverServiceConfig.getService() + "." + revolverHttpApiConfig.getApi();
                    }
                    poolVsBulkHead.putIfAbsent(threadPoolName, bulkheadRegistry.bulkhead(threadPoolName, BulkheadConfig.custom().maxConcurrentCalls(hystrixCommandConfig.getThreadPool().getConcurrency()).build()));
                }
            });
        }
    }

    private static void updateBulkheadsForThreadPools(Map<String, Bulkhead> poolVsBulkHead, RevolverServiceConfig revolverServiceConfig) {
        ThreadPoolGroupConfig threadPoolGroupConfig = revolverServiceConfig.getThreadPoolGroupConfig();
        if (threadPoolGroupConfig != null) {
            threadPoolGroupConfig.getThreadPools().forEach(threadPoolConfig -> {
                String threadPoolName = ResilienceUtil.getThreadPoolName(revolverServiceConfig, threadPoolConfig);
                log.info("ThreadPool Name : {} ", (Object)threadPoolName);
                poolVsBulkHead.put(threadPoolName, bulkheadRegistry.bulkhead(threadPoolName, BulkheadConfig.custom().maxConcurrentCalls(threadPoolConfig.getConcurrency()).build()));
            });
        }
    }

    private static String getThreadPoolName(RevolverServiceConfig revolverServiceConfig, ThreadPoolConfig threadPoolConfig) {
        return revolverServiceConfig.getService() + "." + threadPoolConfig.getThreadPoolName();
    }

    private static void updateTimeoutsForDefaultServiceConfig(Map<String, Integer> poolVsTimeout, RevolverServiceConfig revolverServiceConfig) {
        if (revolverServiceConfig instanceof RevolverHttpServiceConfig) {
            ThreadPoolConfig threadPoolConfig = revolverServiceConfig.getRuntime().getThreadPool();
            if (threadPoolConfig == null) {
                return;
            }
            poolVsTimeout.put(revolverServiceConfig.getService(), threadPoolConfig.getTimeout());
        }
    }

    private static void updateTimeoutsForThreadPools(Map<String, Integer> poolVsTimeout, RevolverServiceConfig revolverServiceConfig) {
        ThreadPoolGroupConfig threadPoolGroupConfig = revolverServiceConfig.getThreadPoolGroupConfig();
        if (threadPoolGroupConfig != null) {
            threadPoolGroupConfig.getThreadPools().forEach(threadPoolConfig -> {
                String threadPoolName = ResilienceUtil.getThreadPoolName(revolverServiceConfig, threadPoolConfig);
                poolVsTimeout.put(threadPoolName, threadPoolConfig.getTimeout());
            });
        }
    }

    private static void updateTimeoutsForApiConfigs(Map<String, Integer> poolVsTimeout, RevolverServiceConfig revolverServiceConfig) {
        if (revolverServiceConfig instanceof RevolverHttpServiceConfig) {
            ((RevolverHttpServiceConfig)revolverServiceConfig).getApis().forEach(revolverHttpApiConfig -> {
                if (revolverHttpApiConfig.getRuntime() != null) {
                    HystrixCommandConfig hystrixCommandConfig = revolverHttpApiConfig.getRuntime();
                    if (hystrixCommandConfig == null || hystrixCommandConfig.getThreadPool() == null) {
                        return;
                    }
                    String threadPoolName = ResilienceUtil.getThreadPoolName(revolverServiceConfig, hystrixCommandConfig.getThreadPool());
                    if (StringUtils.isEmpty((CharSequence)threadPoolName)) {
                        threadPoolName = revolverServiceConfig.getService() + "." + revolverHttpApiConfig.getApi();
                    }
                    poolVsTimeout.putIfAbsent(threadPoolName, hystrixCommandConfig.getThreadPool().getTimeout());
                }
            });
        }
    }

    private static void updateBulkHeadsForDefaultServiceConfig(Map<String, Bulkhead> poolVsBulkHead, RevolverServiceConfig revolverServiceConfig) {
        if (revolverServiceConfig instanceof RevolverHttpServiceConfig) {
            ThreadPoolConfig threadPoolConfig = revolverServiceConfig.getRuntime().getThreadPool();
            if (threadPoolConfig == null) {
                return;
            }
            poolVsBulkHead.put(revolverServiceConfig.getService(), bulkheadRegistry.bulkhead(revolverServiceConfig.getService(), BulkheadConfig.custom().maxConcurrentCalls(threadPoolConfig.getConcurrency()).build()));
        }
    }

    static {
        DEFAULT_KEY_VALUE_MAP.put("resilience.step", "PT1M");
        DEFAULT_KEY_VALUE_MAP.put("resilience.enabled", "true");
        bulkheadRegistry = BulkheadRegistry.ofDefaults();
        circuitBreakerRegistry = CircuitBreakerRegistry.ofDefaults();
    }
}

