(ns kabel.middleware.auth
  (:require [kabel.platform-log :refer [debug info warn error]]
            [konserve.core :as k]
            [konserve.memory :refer [new-mem-store]]
            [hasch.core :refer [uuid]]
            [full.async :refer [<? go-try go-loop-try alt?]]
            #?(:clj [clojure.core.async :as async
                     :refer [<! >! >!! <!! timeout chan alt! go put!
                             go-loop pub sub unsub close!]]
               :cljs [cljs.core.async :as async
                      :refer [<! >! timeout chan put! pub sub unsub close!]])))

;; https://medium.com/@ninjudd/passwords-are-obsolete-9ed56d483eb
;; Passwordless Authentication

;; Here’s how passwordless authentication works in more detail:

;;     Instead of asking users for a password when they try to log in to your app or website, just ask them for their username (or email or mobile phone number).
;;     Create a temporary authorization code on the backend server and store it in your database.
;;     Send the user an email or SMS with a link that contains the code.
;;     The user clicks the link which opens your app or website and sends the authorization code to your server.
;;     On your backend server, verify that the code is valid and exchange it for a long-lived token, which is stored in your database and sent back to be stored on the client device as well.
;;     The user is now logged in, and doesn’t have to repeat this process again until their token expires or they want to authenticate on a new device.


;; next steps
;; - model proper peer identities?
;; - IO
;;   - provide mailer for urls with token on out-auth
;;   - collect tokens activated on in-auth

(defn now [] #?(:clj (java.util.Date.)
                :cljs (js/Date.)))

(def in-auth (chan))
(def out-auth #_(chan 10) in-auth)

(def p-auth (pub in-auth :token))

(def trusted-peers (atom #{"127.0.0.1"}))

(defn auth-request [auth-users id]
  (let [[_ type user] (re-seq #"(.+):(.+)" id)
        token (uuid)
        a-ch (chan)]
    (sub p-auth token a-ch)
    (go-try
     (debug "REQUESTING AUTH" id)
     (>! out-auth {:token token})
     (alt? a-ch
           (do
             (debug "AUTHENTICATED" id token)
             (swap! auth-users assoc token (now)))

           (timeout 100)
           (debug "TIMEOUT" id)))))

(defn authenticate [server-token-store auth-ch new-in]
  (go-loop-try []
               ;; TODO parametrize
               (let [{:keys [peer downstream user] :as a-msg} (<? auth-ch)
                     authed-msg (assoc a-msg ::authenticated :true)
                     token-timeout (* 10 60 1000)]
                 (debug "AUTHENTICATING" user)
                 (cond (@trusted-peers peer)
                       (>! new-in authed-msg)

                       (let [{:keys [time token]} (<? (k/get-in server-token-store [peer user]))]
                         (and token
                              (< (- (.getTime (now)) time)
                                 token-timeout)))
                       (>! new-in authed-msg)

                       (<? (auth-request server-token-store user))
                       (>! new-in authed-msg)

                       :default (throw (ex-info "Authentication timeout:" {:msg a-msg})))
                 (when a-msg (recur)))))


(defn store-token [remote token-store store-token-ch]
  (go-loop-try [{:keys [peer user token]} (<? store-token-ch)]
               (when token
                 (<? (k/update-in token-store [remote user] token))
                 (recur (<? store-token-ch)))))

;; TODO only append necessary tokens for messages needing auth
(defn add-tokens-to-out [remote client-token-store out new-out]
  (go-loop-try [o (<? out)]
               (when o
                 (>! new-out (if-let [t (<? (k/get-in client-token-store [remote]))]
                               (assoc o ::auth-tokens t)
                               o))
                 (recur (<? out)))))

(defn auth-required [auth-required-ch auth-fn out]
  (go-loop-try [{:keys [user]} (<? auth-required-ch)]
               (when user
                 (<? (auth-fn user))
                 (recur (<? auth-required-ch)))))


(defn auth [server-token-store
            client-token-store
            remote dispatch-fn auth-fn [peer [in out]]]
  (let [ ;; core.async bug: cannot use channel size 0 (?)
        new-in (chan 1 (map #(assoc % :authenticated false))) ;; why NOT workz?
        new-out (chan)
        p (pub in (fn [{:keys [type] :as m}]
                    (case type
                      ::auth-required ::auth-required
                      ::auth-token ::auth-token
                      (dispatch-fn m))))
        auth-ch (chan)
        auth-required-ch (chan)
        store-token-ch (chan)]
    (sub p :auth auth-ch)
    (authenticate server-token-store auth-ch new-in)

    (sub p :unrelated new-in) ;; pass-through

    (sub p ::auth-token store-token-ch)
    (store-token remote client-token-store store-token-ch)

    (sub p ::auth-required auth-required-ch)
    (auth-required auth-required-ch auth-fn)

    (add-tokens-to-out remote client-token-store out new-out)
    [peer [new-in new-out]]))


(comment
  (go-try
   (let [mapping {:pub/downstream :auth}
         dispatch-fn (fn [m] (or (mapping (:type m)) :unrelated))
         auth-fn (fn [user] (println "Check channel " user))
         in (chan)
         out (chan)
         [_ [new-in _]] (auth (<? (new-mem-store))
                              (<? (new-mem-store))
                              "wss://topiq.es:8080/replikativ/ws"
                              dispatch-fn [nil [in out]])]
     (put! in {:type :pub/downstream
               :downstream {"mail:eve@topiq.es" {1 {:foo :bar}}}
               #_:peer #_"127.0.0.1"})
     (go-loop [i (<! new-in)]
       (debug "PASSED:" i)
       (recur (<! new-in))))))
