(ns conductor.client
  (:import (com.netflix.conductor.client.worker Worker)
           (io.orkes.conductor.client.http OrkesTaskClient)
           (com.netflix.conductor.client.automator TaskRunnerConfigurer$Builder)
           (com.netflix.conductor.common.metadata.tasks TaskResult TaskResult$Status))

  (:require [clojure.tools.logging :as log]
            [conductor.mapper-utils :as mapperutils]
            [conductor.metadata :as metadata]))

(defn task-client
  "Returns an instance of TaskClient. when app-key and app-secret are provided
  Then returned instance will be Orkes Compatible"
  [{:keys [app-key app-secret url] :or {url "http://localhost:8080/api"}} ]
  (let [client (OrkesTaskClient. )]
    (.setRootURI client url)
    (when app-key
      (.withCredentials client app-key app-secret)
      (log/debug "Creating client with authentication"))
    client))


(defn task-runner-configurer
  "Returns a TaskRunnerConfigurer instance for given client and workers"
  ([client workers thread-count]
   (log/debug "Creating TaskRunnerConfigurer with thread-count" thread-count)
   (-> (TaskRunnerConfigurer$Builder. client workers)
       (.withThreadCount thread-count)
       (.build)) )
  ([client workers] (task-runner-configurer client workers (count workers))))

(defn runner-executor-for-workers
  "Takes a list of workers and connection options. returnes an initiated
  TaskRunnerConfigurer instance"
  [workers options]
  (-> (task-client options)
      (task-runner-configurer
       (map mapperutils/clj-worker->Worker workers)
       (->> options (merge {:thread-count (count workers)}) :thread-count))
      (doto (.init))))


(comment
;; Given the options create-task and create-workflow
(def options {
                  :url  "http://localhost:8080/api/"
                  ;; :app-key "e6513e29-f07c-40ce-8217-ab7785de10b1"
                  ;; :app-secret "9KSAJchHKvMwLV1aGUDvz9bSkBShOTLeNkK3lyFYm7J1dD9r"
              } )
;; Programatically Create a task
(metadata/register-tasks options [{
                         :name "cool_clj_task_b"
                         :description "some description"
                         :owner-email "mail@gmail.com"
                         :retry-count 3
                         :timeout-seconds 300
                         :response-timeout-seconds 180 },
{
                         :name "cool_clj_task_z"
                         :description "some description"
                         :owner-email "mail@gmail.com"
                         :retry-count 3
                         :timeout-seconds 300
                         :response-timeout-seconds 180 }
{
                         :name "cool_clj_task_x"
                         :description "some description"
                         :owner-email "mail@gmail.com"
                         :retry-count 3
                         :timeout-seconds 300
                         :response-timeout-seconds 180 }
                                  ])
;; Programatically create a workflow
(metadata/register-workflow-def options {
                                         :name "cool_clj_workflow_2"
                                         :description "created programatically from clj"
                                         :version 1
                                         :tasks [ {
                                                   :name "cool_clj_task_b"
                                                   :task-reference-name "cool_clj_task_ref"
                                                   :input-parameters {}
                                                   :type :simple
                                                   },
                                                 {
                                                      :name "someting",
                                                      :task-reference-name "other"
                                                      :input-parameters {}
                                                      :type :fork-join
                                                      :fork-tasks [[
                                                                    {
                                                                     :name "cool_clj_task_z"
                                                                     :task-reference-name "cool_clj_task_z_ref"
                                                                     :input-parameters {}
                                                                     :type :simple
                                                                     }
                                                                    ]
                                                                   [
                                                                    {
                                                                     :name "cool_clj_task_x"
                                                                     :task-reference-name "cool_clj_task_x_ref"
                                                                     :input-parameters {}
                                                                     :type :simple
                                                                     }
                                                                    ]
                                                                   ]
                                                      }
                                                 {
                                                  :name "join"
                                                  :type :join
                                                  :task-reference-name "join_ref"
                                                  :join-on [ "cool_clj_task_z", "cool_clj_task_x"]
                                                  }
                                                 ]
                                         :input-parameters []
                                         :output-parameters {:message "${clj_prog_task_ref.output.:message}"}
                                         :schema-version 2
                                         :restartable true
                                         :owner-email "mail@yahoo.com"
                                         :timeout-seconds 0
                                         :timeout-policy :alert-only
                                         })

(metadata/register-workflow-def options {
                                         :name "exclusive_join"
                                         :description "Exclusive Join Example"
                                         :version 1
                                         :tasks [ {
                                                   :name "api_decision"
                                                   :task-reference-name "api_decision_ref"
                                                   :input-parameters {
                                                                      "case_value_param" "${workflow.input.type}"
                                                                      }
                                                   :type :decision
                                                   :case-value-param "case_value_param"
                                                   :default-case []
                                                   :decision-cases {
                                                                    "POST" [{
                                                                             :name "get-posts"
                                                                             :task-reference-name "get_posts_ref"
                                                                             :input-parameters {
                                                                                                "http_request" {
                                                                                                                "uri" "https://jsonplaceholder.typicode.com/posts/1"
                                                                                                                "method" "GET"
                                                                                                                }
                                                                                                }
                                                                             :type :http
                                                                             }]
                                                                    "COMMENT" [{
                                                                                :name "get_posts_comments"
                                                                                :task-reference-name "get_post_comments_ref"
                                                                                :input-parameters {
                                                                                                   "http_request" {
                                                                                                                   "uri" "https://jsonplaceholder.typicode.com/comments?postId=1"
                                                                                                                   "method" "GET"
                                                                                                                   }

                                                                                                   }
                                                                                :type :http
                                                                                }]
                                                                    "USER" [{
                                                                             :name "get_user_posts"
                                                                             :task-reference-name "get_user_posts_ref"
                                                                             :input-parameters {
                                                                                                "http_request" {
                                                                                                                "uri" "https://jsonplaceholder.typicode.com/posts?userId=1"
                                                                                                                "method" "GET"
                                                                                                                }

                                                                                                }

                                                                             :type :http
                                                                             }]
                                                                    }
                                                   },
                                                 {
                                                  :name "notification_join",
                                                  :task-reference-name "notification_join_ref"
                                                  :input-parameters {}
                                                  :type :exclusive-join
                                                  :join-on ["get_posts_ref" "get_post_comments_ref" "get_user_posts_ref"]
                                                  }

                                                 ]
                                         :input-parameters []
                                         :output-parameters {:message "${clj_prog_task_ref.output.:message}"}
                                         :schema-version 2
                                         :restartable true
                                         :owner-email "mail@yahoo.com"
                                         :timeout-seconds 0
                                         :timeout-policy :alert-only
                                         })

(metadata/register-workflow-def options {
                                         :name "port_in_wf"
                                         :description "Port In workflow"
                                         :version 1
                                         :output-parameters {:message "${clj_prog_task_ref.output.:message}"}
                                         :schema-version 2
                                         :restartable true
                                         :owner-email "mail@yahoo.com"
                                         :timeout-seconds 0
                                         :timeout-policy :alert-only
                                         :tasks [
                                                 {
                                                  :name "Submit to itg"
                                                  :task-reference-name  "submit_to_itg_with_retry"
                                                  :input-parameters {
                                                                     "value"  "${workflow.input.iterations}"
                                                                     "terminate"  "${workflow.variables.terminate_loop}"
                                                                     }
                                                  :type :do-while
                                                  :loop-condition "if ( ($.submit_to_itg_with_retry['iteration'] < $.value) && !$.terminate) { true; } else { false; }"
                                                  :loop-over [{
                                                               :name  "Submit to ITG"
                                                               :task-reference-name  "submit_to_itg"
                                                               :input-parameters {
                                                                                  "http_request" {
                                                                                                  "uri"  "https://jsonplaceholder.typicode.com/todos/${$.workflow.input.iterations}",
                                                                                                  "method"  "GET",
                                                                                                  },

                                                                                  }
                                                               :type :http

                                                               }
                                                              {
                                                               :name "Check Status"
                                                               :task-reference-name "check_status"
                                                               :input-parameters {
                                                                                  "prev_task_result"  "${submit_to_itg.output}"
                                                                                  "switchCaseValue"  "${submit_to_itg.status}"
                                                                                  }
                                                               :type :decision
                                                               :case-value-param "switchCaseValue"
                                                               :decision-cases {
                                                                                "COMPLETED" [{
                                                                                              :name  "Complete request loop"
                                                                                              :task-reference-name  "complete_loop_success"
                                                                                              :input-parameters {
                                                                                                                 "terminate_loop"  true
                                                                                                                 "success" true
                                                                                                                 }
                                                                                              :type :set-variable
                                                                                              }]

                                                                                "COMPLETED_WITH_ERRORS" [{
                                                                                                          :name  "Retry Http"
                                                                                                          :task-reference-name  "retry_http_request"
                                                                                                          :input-parameters {
                                                                                                                             "update_records_on_retry"  2
                                                                                                                             }
                                                                                                          :type :set-variable
                                                                                                          }]

                                                                                }
                                                               :default-case [{
                                                                               :name  "Permanent Failure"
                                                                               :task-reference-name  "terminate_loop"
                                                                               :input-parameters {
                                                                                                  "update_records_on_retry"  2
                                                                                                  }
                                                                               :type :set-variable
                                                                                         }]
                                                               }
                                                              ]
                                                  }
                                                 {
                                                  :name "Wait for async message"
                                                  :task-reference-name "wait_for_response"
                                                  :input-parameters {}
                                                  :type :wait
                                                  }
                                                 ]
                                         })




;; Programatically create a worker and run it to pool
(def instance (runner-executor-for-workers
               (list {
                      :name "cool_clj_task_b"
                      :execute (fn [someData]
                                 [:completed {:message "Hi From Clj i was created programatically"}])
                      })
               options ))
(.shutdown instance)

)
