The EVM is backed by a data structure best described as a _Merkle Radix Trie_.  Let's go through
each of those concepts in reverse order.

## "Trie"

Tries are an intuitive means of representing associative data as trees, leveraging
shared structure (common prefixes) in the key space.  Let's look at an example which
uses a character trie to count the occurrences of each word in an input string:

```clojure
(defn build-trie [s]
  (reduce
   (fn [t chars]
     (update-in t chars (fnil inc 0)))
   {}
   (re-seq #"\w+" s)))

(build-trie "car cat cat cot cot cult")
;; => {\c {\a {\t 2 \r 1}
           \o {\t 2}
           \u {\l {\t 1}}}}
```

## "Radix"

Infobox: Nomenclature
Ethereum prefers the phrase Merkle _Patricia_ Trie.  While _patricia trie_ is occasionally
used interchangeably with _radix trie_, it often refers exclusively  to _binary_ radix tries.  As Ethereum's tries are not binary, I will use _radix trie_ exclusively.

Radix tries can be summed up into three main, uh.

Though not the most intuitive if we're thinking about strings, binary radix tries
are conceptually the simplest.

That said, while radix tries can be used in this way, we're going to be considering
those with smaller alphabets, starting with _binary_ radix tries --- tries in which each
node has at most two children.

###

Prefix trees are a family collision-free, associative data structures which represent their
keys structurally, rather than explicitly.  Compact prefix trees are those employ the space-optimization of explicitly representing unique segments of the keyspace.

To try and get an intuitive foothold, let's consider storing strings taken from an
alphabet having only the letters A, E, R & T.  A generic prefix tree might store the
single mapping `TEAR` -> 1` as something equivalent to:

{\T {\E\ {\A {\R 1}}}}

While a compact prefix tree containing only the same value would yield:

{"TEAR" 1}

After inserting the mapping `TARE -> 2` into the compact tree:

{\T {"ARE" 2}
    {"EAR" 1}}

Followed by `TAT - 3`:

{\T {\A    {"RE" 2, \T 3}
     "EAR" 1}}

You'll notice all of the compacted nodes happen to be leaves - the tree we're
building will also compact inner nodes, where possible.  After inserting `TEAT -> 4`:

{\T {\A    {"RE" 2, \T 3}
     "EA"  {\R   1, \T 4}}}

There's fundamentally three types of nodes:
 - Branches
 - Extension nodes
 - Leaf nodes

Branches hold as many child nodes as there are entities in our alphabet, and a value,
in the event there's a key that terminates at the same level as the branch node.

Extension nodes and leaf nodes are very similar - they both hold a single compacted key,
and point to either a child node (in the case of extension nodes) or hold a value (leaf nodes).

Let's look at our abstractions, and some helper functions:

```clojure
(ns sputter.state.trie.node)

(defprotocol Node
  (terminates? [node k]
    "Does this node represent a successful termination of a search
     for the given key remainder.")
  (attach [node k v]
    "Attach the given value to this node, at the given key remainder.
     Return a new node.")
  (value [node]
    "The value of this node.")
  (descend [node candidate-k]
    "Locate the immediate child pointed to by the given key.
     Return vector of `[child remaining-k child-tag]` or `nil`."))

(defn walk
  "Analogous to [[clojure.walk/walk]], only operating on nodes
   pointed at by the given key."
  [node k outer inner]
  (if-not node
    (outer k nil #{:terminal?})
    (if-let [[child tail tag] (descend node k)]
      (outer k (assoc node tag (inner tail child)))
      (outer k node #{:terminal?}))))
```

Note we support `descend`, rather than, say, `search` --- we want an interface
which could accomodate an implementation in which children are not
immediately available, and we don't really want `Node` to be concerned with
the details of realizing them.

```clojure
(ns sputter.state.trie
 (:require [sputter.state.trie.node :as node]
           [sputter.util.nibbles    :as nibbles]))

(defprotocol Trie
  (-insert [trie k v]
   "Return a new Trie containing the supplied mapping.")
  (-search [trie node k]
   "Consume as much of the given key as possible descending beneath `node`.
    Return a sequence containing a `[remaining-key node]` pair for each level."))

...

(defn insert [trie k v]
  (-insert trie (key->path k) v))

(defn search
  "Return the value of the node at the given key, or `nil`."
  [trie k]
  (when-let [[node rest-k] (last (-search trie (:root t) (key->path k)))]
    (when (node/terminates? node rest-k)
      (node/value node))))
```

And the trie implementation itself:

```clojure
(defn- attach-walker [v k node & [opts]]
  (cond (not node)        (node/->LeafNode k v)
        (:terminal? opts) (node/attach node k v)
        :else             node))

(defn- walk-path [f k node]
  (node/walk node k f (partial walk-path f)))

(defrecord KVTrie [root]
  Trie
  (-insert [this k v]
    (let [root (walk-path (partial attach-walker v) k root)]
      (assoc this :root root)))
  (-search [this node k]
    (lazy-seq
     (cons [node k]
           (when-let [[child rest-k] (node/descend node k)]
             (-search this node rest-k))))))
```

There's not yet much need for our trie to be a record, but it'll become clearer
once we add a little more functionality.  Finally, let's implement the three node types, back in `sputter.state.trie.node`.  First, branch nodes, which we'll implement as vectors
of 17 elements: 16 branches and a value.

```clojure
(def ^:private empty-branch (into [] (take 17 (repeat nil))))

(declare ->LeafNode ->ExtensionNode)

(extend-type clojure.lang.IPersistentVector
  Node
  (descend [this [nibble & rest-k]]
    (when nibble
      [(this (int nibble)) rest-k (int nibble)]))
  (terminates? [_ k]  (empty? k))
  (value       [this] (peek this))
  (attach [this [nibble & rest-k] v]
    (if nibble
      (assoc this (int nibble)       (->LeafNode rest-k v))
      (assoc this (dec (count this)) v))))
```

Elsewhere, we'll be required to repeatedly flatten and reconstitute nodes
to/from arrays --- given it's not a totally unnatural representation for
branch nodes, we'll stick with it.

`attach` is straightforward: if the key has been exhausted, we replace our
value, else we attach a leaf at the appropriate slot with the remainder of the
key, if any.

```clojure
(defrecord LeafNode [key value]
  Node
  (descend     [_ _] nil)
  (terminates? [_ k] (= k key))
  (value       [_]   value)
  (attach [this attach-k v]
    (if (= attach-k key)
      (assoc node :value v)
      (let [[prefix key attach-k] (split-prefix key attach-k)]
        (-> empty-branch
            (attach attach-k v)
            (attach key      value)
            (cond->> prefix (->ExtensionNode prefix)))))))
```

When attaching, if the remaining key isn't a perfect match, we create a branch
and attach both the input value (`v`) at the distinct suffix of the input key
(`attach-k`), and the leaf's `value` at the distinct suffix of the leaf's `key`.
Finally, if there was a common prefix between the two keys, we stuff it into an
extension node that holds the branch.

```clojure
(defrecord ExtensionNode [key child]
  Node
  (descend [this candidate-k]
    (let [[prefix tail] (split-at (count key) candidate-k)]
      (when (= prefix key)
        [child tail :child])))
  (terminates? [_ _] false)
  (value       [_]   nil)
  (attach [this attach-k v]
    (let [[prefix key attach-k] (split-prefix key attach-k)
          [nibble & rest-k]     key
          me                    (if rest-k
                                  (assoc this :key rest-k)
                                  child)]
        (-> empty-branch
            (attach attach-k v)
            (assoc nibble    me)
            (cond->> prefix (->ExtensionNode prefix))))))
```

`attach` here is similar to `LeafNode`, though in addition to attaching the input
value to the empty branch node, we also directly assoc a `Node` --- either an abbreviated
extension node (`assoc this :key rest-k`) or the extension's child directly, if `rest-k` is empty.


```clojure
(def ^:private branches     4)
(def ^:private empty-branch (into [] (take (inc branches) (repeat nil))))

(defprotocol Node
  (descend [node k]))

(extend-type clojure.lang.IPersistentVector
  Node
  (descend [this [idx & rest-k]]
    (when idx
      [(this idx) rest-k]))
  (attach [this [idx & rest-k] v]
    (if idx
      (assoc this idx      (->LeafNode rest-k v))
      (assoc this branches v))))

(defrecord LeafNode [key value]
  Node
  (attach [this attach-k v]
    ))

(extend-type nil
  Node
  (attach [_ k v]
    (->LeafNode k v)))
```

Let's build a using a couple of toy definitions:

```
(defn str->key [x]
  (map {\A 0 \E 1 \R 3 \T 4} (seq x)))

(defn map->trie [root entries]
  (reduce-kv (fn [root k v]
               (attach root (str->key k) v)) root entries))

(map->trie
  empty-branch
  {"TEAR" "TEAR"
   "TARE" "TARE"
   "TAT"  "TAT"
   "TEAT" "TEAT"})
```

Drum roll:

```clojure
[nil
 nil
 nil
 nil
 [[nil
   nil
   nil
   [nil [nil nil nil nil "TARE"] nil nil nil]
   [nil nil nil nil "TAT"]]
  [[nil nil nil [nil nil nil nil "TEAR"] [nil nil nil nil "TEAT"]]
   nil
   nil
   nil
   nil]
  nil
  nil
  nil]]
```

Clearly that's no good.  Let's implement leaf nodes:

```clojure

```


```clojure
(defprotocol Node
  (terminates? [node k]
    "Does this node represent a successful termination of a search
     for the given key remainder?")
  (value [node]
    "The value of this node, if any.")
  (descend [node candidate-k]
    "Locate the immediate child pointed to by the given key.
     Return vector of `[child remaining-k]` or `nil`."))

(extend-type clojure.lang.IPersistentVector
  Node
  (descend [this [nibble & rest-k]]
    (when nibble
      [(this (int nibble)) rest-k]))
  (terminates? [_ k]  (empty? k))
  (value       [this] (peek this)))

(defrecord LeafNode [key value]
  Node
  (descend     [_ _] nil)
  (terminates? [_ k] (= k key))
  (value       [_]   value))

(defrecord ExtensionNode [key child]
  Node
  (descend [this candidate-k]
    (let [[prefix tail] (split-at (count key) candidate-k)]
      (when (= prefix key)
        [child tail :child])))
  (terminates? [_ _] false)
  (value       [_]   nil))
```
