(ns fink-nottle.examples.bridge
  (:require [clojure.core.async :as a :refer [go <! >!]]
            [clojure.set :as set]
            [fink-nottle.sns :as sns]
            [fink-nottle.sqs :as sqs]
            [clojure.pprint :as pprint]
            [fink-nottle.sqs.channeled :as channeled]
            [fink-nottle.sqs.tagged :as tagged]
            [glossop :as g :refer [go-catching <?]]))

(def creds
  {:access-key (get (System/getenv) "AWS_ACCESS_KEY")
   :secret-key (get (System/getenv) "AWS_SECRET_KEY")})

(def gcm-api-key (get (System/getenv) "GCM_API_KEY"))

(defmethod tagged/message-in  :edn [_ body] (clojure.edn/read-string body))
(defmethod tagged/message-out :edn [_ body] (pr-str body))

(defn send-loop [creds queue-url]
  (let [{:keys [in-chan]} (channeled/batching-sends creds queue-url)]
    (go
      (loop [i 0]
        (>! in-chan {:body {:event :increment :value i}
                     :fink-nottle/tag :edn})
        (<! (a/timeout (rand-int 300)))
        (recur (inc i))))))

(defn receive-loop [id creds queue-url]
  (let [message-chan (channeled/receive! creds queue-url)
        delete-chan  (channeled/batching-deletes creds queue-url)]
    (go
      (loop []
        (let [{:keys [body attrs] :as message} (<! message-chan)]
          (log/info {:id id :body body :attrs attrs})
          (>! delete-chan message)
          (recur))))))


(let [app-arn (sns/create-platform-application!!
               creds :GCM "fink-nottle-gcm"
               {:platform-credential gcm-api-key})
      end-arn (sns/create-platform-endpoint!!
               creds app-arn gcm-registration-id)]
  (sns/publish-endpoint!!
   creds end-arn
   {:GCM {:data {:message "Hello world!"}}}))

(def app-arn
  (sns/create-platform-application!!
   creds :GCM "fink-nottle-gcm" {:platform-credential gcm-api-key}))
(def endpoint-arn
  (sns/create-platform-endpoint!! creds app-arn "eMHN0dahlPo:APA91bGU-7xnBsWaamRUDqEYjzfXnwSK7yzdCig-hEJYgDzwcgp3Su4OqFUJwSQ_O6CDj_q2MFENRAUVyGV1fu-aMGBmWaP02VXkN4rUUawWYogPjW5lZ_tKHUM4PfwD_Od1mZalJvXA"))
(sns/subscribe!! )
(sns/publish-endpoint!! creds endpoint-arn {:GCM {:data {:message "Hello world!"}}})

;; (let [{:keys [in-chan error-chan]}
;;       (sqs.channeled/batching-channel creds queue-url)]
;;   (>! in-chan {:body {:event :increment :value i}
;;                :fink-nottle/tag :edn})
;;   ;; ...
;; )

(defn batch-send [creds queue-url]
  (let [{:keys [in-chan error-chan]}
        (channeled/batching-sends creds queue-url)]
    (go-catching
      (loop [i 0]
        (a/>! in-chan
              {:body  {:event :increment :value i}
               :fink-nottle/tag :edn})
        (a/<! (a/timeout (rand-int 300)))
        (recur (inc i))))))

(def creds-west (assoc creds :region :us-west-2))

(defn go []
  (def queue-url (sqs/create-queue!! creds "fink-nottle-tasks-demo"))
  (def topic-arn (sns/create-topic!! creds "fink-nottle-tasks-demo"))
  (sqs/set-queue-attribute!!
   creds queue-url :policy
   {:statement
    [{:sid "fink-nottle-sqs-sns-bridge",
      :effect :allow
      :principal {:AWS "*"}
      :action [:sqs/send-message],
      :resource queue-url,
      :condition {:arn-equals {:aws/source-arn topic-arn}}}]})

  (let [subscription-arn (sns/subscribe!!
                          creds topic-arn :sqs (sqs/queue-arn!! creds queue-url))]
    (sns/set-subscription-attribute!!
     creds subscription-arn :raw-message-delivery true))
  (sns/publish-topic!! creds topic-arn "Hello")
  (sqs/receive-message!! creds queue-url {:wait-seconds 2}))

(defn penetrate []
  (def queue-url (sqs/create-queue!! creds "fink-nottle-events"))

  (sqs/send-message!! creds queue-url {:body {:x 1 #{:y} 2} :fink-nottle/tag :edn})
  (sqs/receive-message!! creds queue-url {:wait-seconds 2 :attrs :all}))

