(ns mattilsynet.design
  (:require [clojure.java.io :as io]
            [clojure.string :as str]))

(def class-idx (read-string (slurp (io/resource "mattilsynet-design/css-modules.edn"))))
(def illustrations-dir "illustrations")
(def icons-dir "icons")

(defmacro ^:export c [& classes]
  (->> classes
       (mapcat
        (fn [class]
          (get class-idx (if (keyword? class) (name class) class) [class])))
       (cons 'list)))

(defmacro ^:export classes [& classes]
  `(c ~@classes))

(defn ^:export load-svg
  "Loads an SVG from the design system"
  [path]
  (slurp (io/resource (str "public/mtds/" path ".svg"))))

(def base-path "mattilsynet-design")

(defn get-svg-path [id]
  (case (namespace id)
    "icon"
    (str base-path "/" icons-dir "/" (name id) ".edn")

    "illustration"
    (str base-path "/" illustrations-dir "/"
         (str/replace (name id) #"\." "/") ".edn")

    (throw (ex-info "Unknown svg kind, use either :icon/* or :illustration/*" {:id id}))))

(defn load-svg-resource [id]
  (-> (get-svg-path id)
      io/resource
      slurp
      read-string))

(defn ^:export render-svg [id & [attrs]]
  (if-let [svg (load-svg-resource id)]
    (update svg 1 merge attrs)
    (throw (ex-info (str "SVG " id " does not exist") {:id id}))))

(defn load-svg! [_id _resource])

(defmacro ^:export svg [id]
  (if (:ns &env)
    ;; ClojureScript må laste svg-en inn i bygget
    `(do
       (mattilsynet.design/load-svg! ~id ~(load-svg-resource id))
       ~id)
    ;; Clojure laster svg-en dynamisk ved bruk
    id))

(defn get-ids [path ns]
  (let [all-your-base (str base-path "/" path)]
    (->> (io/resource all-your-base)
         io/file
         file-seq
         (map str)
         (filter #(re-find #"\.edn$" %))
         (map #(second (str/split % (re-pattern (str all-your-base "/")))))
         (map #(keyword ns (-> %
                               (str/replace #"\.edn$" "")
                               (str/replace #"/" "."))))
         sort
         vec)))

(defmacro ^:export get-icon-ids []
  (get-ids icons-dir "icon"))

(defmacro ^:export get-illustration-ids []
  (get-ids illustrations-dir "illustration"))

(defn parse-tag [^clojure.lang.Keyword tag]
  (let [ns ^String (namespace tag)
        tag ^String (name tag)
        id-index (let [index (.indexOf tag "#")] (when (pos? index) index))
        class-index (let [index (.indexOf tag ".")] (when (pos? index) index))
        tag-name (cond->> (cond
                            id-index (.substring tag 0 id-index)
                            class-index (.substring tag 0 class-index)
                            :else tag)
                   ns (keyword ns)
                   (nil? ns) keyword)
        id (when id-index
             (if class-index
               (.substring tag (unchecked-inc-int id-index) class-index)
               (.substring tag (unchecked-inc-int id-index))))
        classes (when class-index
                  (seq (.split (.substring tag (unchecked-inc-int class-index)) "\\.")))]
    [tag-name id classes]))

(defn ->classes [class-attribute]
  (cond
    (nil? class-attribute) []
    (keyword? class-attribute) [(name class-attribute)]
    (string? class-attribute) [class-attribute]
    (sequential? class-attribute) (mapv #(if (keyword? %) (name %) %) class-attribute)
    :else []))

(defn rewrite [form]
  (cond
    (vector? form)
    (let [[tag maybe-attrs & body] form
          [tag id tag-classes] (parse-tag tag)]
      (if (map? maybe-attrs)
        (let [all-classes (concat tag-classes (->classes (:class maybe-attrs)))
              attrs (cond-> (dissoc maybe-attrs :class)
                      (seq all-classes)
                      (assoc :class (list* `classes all-classes))

                      id
                      (assoc :id id))]
          (into [tag attrs] (map rewrite body)))
        (let [attrs (cond-> {}
                      (seq tag-classes) (assoc :class (list* `classes tag-classes))
                      id (assoc :id id))]
          (into (cond-> [tag] attrs (conj attrs))
                (map rewrite (cons maybe-attrs body))))))

    (seq? form) (map rewrite form)
    :else form))

(defmacro h [hiccup]
  (rewrite hiccup))
