/**
 * A database transaction.
 * @example
 * [{ todo: { name: "a new todo" } },
 *  { todo: { id: 1, name: "an update (see the id)", project: -1 } },
 *  { project: { id: -1, name: "a new project with a temp id" } }]
 */
export type Transaction = Array<object>;

/**
 * Add schema to simplify relational queries. Define relationships and uniqueness constraints in just one place.
 * @example
 * { todo: { project: { type: 'ref' } 
 *           name: { unique: 'identity' } } }
 */
export type Schema = object;

/**
 * Add lookup helpers to simplify relational queries. Define relationships and uniqueness constraints in just one place.
 * @example
 * { todo: { project: { type: 'ref' } 
 *           name: { unique: 'identity' } } }
 */
export type LookupHelpers = object;

/**
 * A homebase configuration.
 * @typedef {Object} config
 * @property {?object} schema - an optional schema
 * @property {?object} lookupHelpers - optional lookupHelpers
 * @property {?array} initialData - an optional initial transaction
 */
export type config = {schema?: Schema, lookupHelpers?: LookupHelpers, initialData?: Transaction };

/**
 * A reactive reference to an entity's data.
 */
export type Entity = {
  /**
   * Retrieves an attribute from the entity. Traverse arbitrarily deep relationships between entities by passing multiple attributes.
   * @param {...string} attribute - an attribute of the entity
   * @example
   * // returns 'a todo name'
   * aTodoEntity.get('name')
   * @example
   * // returns 'a project name'
   * aTodoEntity.get('project', 'name')
   */
  get: (...attribute:string[]) => any;
}

/**
 * Datoms are the smallest unit of data in the database. They are key-value pairs with extra information like entity id, transaction id, and if this key was added or deleted from the database.
 * @example
 * [10, ":todo/name", "some todo", 536870922, true]
 */
export type Datom = [number, string, string | number | object | Array<any>, number, boolean];

/**
 * The homebase client. Provides additional functions to read and write data. It's primarily used when synchronizing data with a backend.
 */
export type homebaseClient = {
  /**
   * Serializes the whole db including the lookupHelpers to a string.
   * @returns {string} Returns the whole db as a string
   */
  dbToString: () => string,
  /**
   * Replaces the current db with one generated by `homebaseClient.dbToString()`.
   * @param {string} dbString - a serialized db string
   */
  dbFromString: (dbString: string) => void,
  /**
   * Datoms are the smallest unit of data in the database, similar to a key-value pair with extra info.
   * @returns {Array.<Datom>} Returns all the datoms in the database.
   */
  dbToDatoms: () => Datom[],
  /**
   * Adds a listener callback that fires after every transaction. Typically used to save data to a backend. Only one transact listener is supported per homebaseClient instance.
   * @param {transactListener} listener - A callback that provides an array of changedDatoms.
   */
  addTransactListener: (listener: (changedDatoms: Datom[]) => void) => void,
  
  /**
   * This callback is displayed as part of the Requester class.
   * @callback transactListener
   * @param {Array.<Datom>} changedDatoms - The datoms that were added and removed in a transaction.
   */

  /**
   * Removes the transact listener. Only one transact listener is supported per homebaseClient instance.
   */
  removeTransactListener: () => void,
  /**
   * Transacts data without triggering any listeners. Typically used to sync data from your backend into the client.
   * @param transaction - A database transaction.
   */
  transactSilently: (transaction: Transaction) => any
}

/**
 * The Homebase React context component. It creates a local database and feeds it to child hooks. Put it high in your component tree.
 * @param props.config - an object with optional lookupHelpers and initialData parameters.
 * @param props.children - children elements
 */
export function HomebaseProvider(props: {config?:config, children:React.ReactNode}): React.ReactElement;

/**
 * React hook to transact data to the local homebase database.
 * @returns [transact] - A tuple with a transact function.
 * @example
 * const [transact] = useTransact()
 * transact([{ todo: { name: "a new todo" } }])
 */
export function useTransact(): [(transaction:Transaction) => void];

/**
 * React hook to return a single entity by `lookup`.
 * @param lookup - an entity id or lookup object.
 * @returns [entity] - A tuple with an entity.
 * @example const [entity] = useEntity(10)
 * @example const [entity] = useEntity({ identity: "a unique lookup key" })
 * @example 
 * const [project] = useEntity({ project: { name: "a unique name" }})
 * project.get('name')
 */
export function useEntity(lookup: object | number): [Entity];

/**
 * React hook to return a collection of entities by `query`.
 * @param query - a query object or datalog string.
 * @param args - optional query arguments.
 * @returns [entities] - A tuple with an array of entities.
 * @example 
 * const [todos] = useQuery({ 
 *   $find: 'todo', 
 *   $where: { todo: { name: '$any' } } 
 * })
 * todos.map(todo => todo.get('name'))
 */
export function useQuery(query: object | string, ...args: any): [Array<Entity>];

/**
 * React hook to return a homebaseClient.
 * @returns [client] - A tuple with a homebaseClient
 * @example 
 * const [client] = useClient()
 * client.dbToString()
 * client.dbToDatoms()
 */
export function useClient(): [homebaseClient];
