(ns zeph.ring
  "Ring adapter for Zeph HTTP server."
  (:import [zeph.http HttpRequest HttpResponse]
           [clojure.lang IPersistentMap]))

(defn- build-request-map
  "Convert HttpRequest to Ring request map."
  [^HttpRequest req]
  {:server-port (.getServerPort req)
   :server-name (.getServerName req)
   :remote-addr (.getRemoteAddr req)
   :uri (.getPath req)
   :query-string (.getQueryString req)
   :scheme (if (.isSecure req) :https :http)
   :request-method (keyword (.toLowerCase (.name (.getMethod req))))
   :protocol (.getProtocol req)
   :headers (into {} (.getHeaders req))
   ;; Prefer streaming body (InputStream) over buffered body (byte[])
   :body (or (.getBodyStream req)
             (when-let [body (.getBody req)]
               (java.io.ByteArrayInputStream. body)))})

(defn- build-response
  "Convert Ring response map to HttpResponse."
  [ring-resp]
  (let [status (or (:status ring-resp) 200)
        headers (or (:headers ring-resp) {})
        body (:body ring-resp)
        resp (doto (HttpResponse.)
               (.setStatus status))]
    ;; Set headers
    (doseq [[k v] headers]
      (.setHeader resp (name k) (str v)))
    ;; Set body
    (cond
      (string? body)
      (.setBody resp ^String body)

      (bytes? body)
      (.setBody resp ^bytes body)

      (instance? java.io.InputStream body)
      (.setBody resp ^java.io.InputStream body)

      (instance? java.io.File body)
      (.setBody resp ^java.io.File body)

      ;; Function as body writer - for streaming responses
      (fn? body)
      (.setBodyWriter resp
        (reify java.util.function.Consumer
          (accept [_ out] (body out))))

      (nil? body)
      nil

      :else
      (.setBody resp (str body)))
    resp))

(defn- error-response
  "Create a 500 Internal Server Error response."
  []
  (doto (HttpResponse.)
    (.setStatus 500)
    (.setContentType "text/plain; charset=utf-8")
    (.setBody "Internal Server Error")))

(defn wrap-ring-handler
  "Wrap a Ring handler for use with Zeph HttpServer.
   Returns a java.util.function.Function.

   Handlers can return:
   - Ring response map {:status :headers :body}
   - Pre-built HttpResponse (for zero-overhead static responses)"
  [handler]
  (let [error-resp (error-response)]
    (reify java.util.function.Function
      (apply [_ req]
        (try
          (let [ring-req (build-request-map ^HttpRequest req)
                ring-resp (handler ring-req)]
            (cond
              ;; Pre-built HttpResponse - return directly
              (instance? HttpResponse ring-resp)
              ring-resp

              ;; Ring response map - convert to HttpResponse
              (instance? IPersistentMap ring-resp)
              (build-response ring-resp)

              ;; Fallback for invalid responses
              :else
              error-resp))
          (catch Throwable t
            (binding [*out* *err*]
              (println "Handler error:" (.getMessage t)))
            error-resp))))))
