(ns zeph.metrics
  "Simple metrics collection for Zeph.

  Provides counters, gauges, and histograms with no external dependencies.

  Usage:
    (require '[zeph.metrics :as m])

    ;; Increment a counter
    (m/inc! :http/requests)
    (m/inc! :http/requests 5)

    ;; Register a gauge
    (m/gauge! :connections/active #(count @connections))

    ;; Record histogram values
    (m/record! :request/latency-ms elapsed)

    ;; Get all metrics
    (m/get-all)

    ;; Time a block of code
    (m/with-timer :db/query-ms
      (execute-query!))"
  (:import [zeph.metrics Metrics Metrics$Counter Metrics$Histogram Metrics$Histogram$Stats]))

(defn inc!
  "Increment a counter by 1 or delta."
  ([name]
   (inc! name 1))
  ([name delta]
   (let [counter (Metrics/counter (clojure.core/name name))]
     (.increment counter (long delta)))))

(defn dec!
  "Decrement a counter by 1."
  [name]
  (let [counter (Metrics/counter (clojure.core/name name))]
    (.decrement counter)))

(defn counter
  "Get current counter value."
  [name]
  (.get (Metrics/counter (clojure.core/name name))))

(defn gauge!
  "Register a gauge with a supplier function."
  [name supplier-fn]
  (Metrics/gauge (clojure.core/name name)
                 (reify java.util.function.Supplier
                   (get [_] (supplier-fn)))))

(defn record!
  "Record a value in a histogram."
  [name value]
  (let [histogram (Metrics/histogram (clojure.core/name name))]
    (.record histogram (long value))))

(defn histogram
  "Get histogram stats as a map."
  [name]
  (let [h (Metrics/histogram (clojure.core/name name))
        ^Metrics$Histogram$Stats stats (.getStats h)]
    {:count (.count stats)
     :min (.min stats)
     :max (.max stats)
     :mean (.mean stats)
     :p50 (.p50 stats)
     :p95 (.p95 stats)
     :p99 (.p99 stats)}))

(defmacro with-timer
  "Time a block of code and record to histogram."
  [name & body]
  `(let [start# (System/nanoTime)
         result# (do ~@body)
         elapsed# (/ (- (System/nanoTime) start#) 1000000.0)]
     (record! ~name (long elapsed#))
     result#))

(defn get-all
  "Get all metric values as a map."
  []
  (into {} (Metrics/getAll)))

(defn get-counters
  "Get all counter values."
  []
  (into {} (Metrics/getCounters)))

(defn get-gauges
  "Get all gauge values."
  []
  (into {} (Metrics/getGauges)))

(defn get-histograms
  "Get all histogram stats."
  []
  (into {}
    (map (fn [[k ^Metrics$Histogram$Stats v]]
           [k {:count (.count v)
               :min (.min v)
               :max (.max v)
               :mean (.mean v)
               :p50 (.p50 v)
               :p95 (.p95 v)
               :p99 (.p99 v)}])
         (Metrics/getHistograms))))

(defn reset!
  "Reset all metrics (counters and histograms)."
  []
  (Metrics/reset))

(defn clear!
  "Clear all metrics completely."
  []
  (Metrics/clear))

;; Convenience functions for common server metrics

(defn record-request!
  "Record an HTTP request with optional latency."
  ([]
   (inc! :http/requests.total))
  ([status-code]
   (inc! :http/requests.total)
   (inc! (keyword (str "http/status." status-code))))
  ([status-code latency-ms]
   (inc! :http/requests.total)
   (inc! (keyword (str "http/status." status-code)))
   (record! :http/latency-ms latency-ms)))

(defn record-error!
  "Record an error."
  ([]
   (inc! :errors/total))
  ([error-type]
   (inc! :errors/total)
   (inc! (keyword (str "errors/" (clojure.core/name error-type))))))

(defn record-connection!
  "Record a connection event."
  [event]
  (case event
    :open (inc! :connections/opened)
    :close (inc! :connections/closed)
    :error (inc! :connections/errors)
    nil))

(defn format-prometheus
  "Format metrics in Prometheus text format."
  []
  (let [counters (get-counters)
        gauges (get-gauges)
        histograms (get-histograms)]
    (str
      ;; Counters
      (apply str
        (for [[k v] counters]
          (str "# TYPE " (clojure.string/replace k "/" "_") " counter\n"
               (clojure.string/replace k "/" "_") " " v "\n")))
      ;; Gauges
      (apply str
        (for [[k v] gauges]
          (str "# TYPE " (clojure.string/replace (clojure.core/name k) "/" "_") " gauge\n"
               (clojure.string/replace (clojure.core/name k) "/" "_") " " v "\n")))
      ;; Histograms (as summary)
      (apply str
        (for [[k v] histograms]
          (let [base (clojure.string/replace k "/" "_")]
            (str "# TYPE " base " summary\n"
                 base "_count " (:count v) "\n"
                 base "{quantile=\"0.5\"} " (:p50 v) "\n"
                 base "{quantile=\"0.95\"} " (:p95 v) "\n"
                 base "{quantile=\"0.99\"} " (:p99 v) "\n")))))))
