(ns zeph.logging
  "Logging integration for Zeph using Timbre.

  Usage:
    (require '[zeph.logging :as log])

    ;; Initialize Timbre as the logger
    (log/init!)

    ;; Use standard Timbre logging
    (log/info \"Server started\")
    (log/error \"Connection failed\" {:error e})

    ;; Configure log level
    (log/set-level! :debug)"
  (:require [taoensso.timbre :as timbre])
  (:import [zeph.logging Logger Logger$Level Logger$LogHandler]))

(defn- level->timbre
  "Convert Java Logger.Level to Timbre level keyword."
  [^Logger$Level level]
  (condp = level
    Logger$Level/TRACE :trace
    Logger$Level/DEBUG :debug
    Logger$Level/INFO  :info
    Logger$Level/WARN  :warn
    Logger$Level/ERROR :error
    :info))

(defn- timbre-handler
  "Log handler that forwards to Timbre."
  [^Logger$Level level ^String component ^String message ^Throwable error]
  (let [lvl (level->timbre level)
        msg (if component
              (str "[" component "] " message)
              message)]
    (if error
      (timbre/log! lvl :p [msg] {:?err error})
      (timbre/log! lvl :p [msg]))))

(defn init!
  "Initialize Timbre as the log handler for Java code.
   Call this at application startup."
  []
  (Logger/setHandler
    (reify Logger$LogHandler
      (log [_ level component message error]
        (timbre-handler level component message error))))
  (timbre/info "Zeph logging initialized with Timbre"))

(defn reset!
  "Reset to default System.out/err logging."
  []
  (Logger/resetHandler))

(defn set-level!
  "Set the minimum log level."
  [level]
  (timbre/set-min-level! level))

(defn set-config!
  "Set Timbre configuration map."
  [config]
  (timbre/merge-config! config))

;; Re-export common Timbre logging macros for convenience

(defmacro trace
  "Log at TRACE level."
  [& args]
  `(timbre/trace ~@args))

(defmacro debug
  "Log at DEBUG level."
  [& args]
  `(timbre/debug ~@args))

(defmacro info
  "Log at INFO level."
  [& args]
  `(timbre/info ~@args))

(defmacro warn
  "Log at WARN level."
  [& args]
  `(timbre/warn ~@args))

(defmacro error
  "Log at ERROR level."
  [& args]
  `(timbre/error ~@args))

(defmacro with-context
  "Execute body with additional logging context."
  [context & body]
  `(timbre/with-context ~context ~@body))

;; Output format helpers

(defn console-appender
  "Create a console appender with custom options."
  [{:keys [output-fn] :or {output-fn :inherit}}]
  {:enabled? true
   :async? false
   :min-level nil
   :output-fn output-fn
   :fn (fn [data]
         (let [{:keys [output_]} data]
           (println (force output_))))})

(defn file-appender
  "Create a file appender."
  [path & {:keys [output-fn] :or {output-fn :inherit}}]
  {:enabled? true
   :async? true
   :min-level nil
   :output-fn output-fn
   :fn (fn [data]
         (let [{:keys [output_]} data]
           (spit path (str (force output_) "\n") :append true)))})

(def default-output-fn
  "Default output format: timestamp level namespace - message"
  (fn [data]
    (let [{:keys [level ?ns-str msg_ ?err]} data
          ts (java.time.LocalDateTime/now)
          msg (force msg_)]
      (str ts " " (name level) " " (or ?ns-str "zeph") " - " msg
           (when ?err (str "\n" (with-out-str (.printStackTrace ?err))))))))

(defn configure-production!
  "Configure logging for production use."
  []
  (timbre/merge-config!
    {:min-level :info
     :output-fn default-output-fn
     :appenders {:console (console-appender {:output-fn default-output-fn})}})
  (init!))

(defn configure-development!
  "Configure logging for development use."
  []
  (timbre/merge-config!
    {:min-level :debug
     :output-fn default-output-fn
     :appenders {:console (console-appender {:output-fn default-output-fn})}})
  (init!))
