(ns zeph.handlers
  "Built-in handlers for zeph server."
  (:import [java.io InputStream FileInputStream FileOutputStream File]))

(def ^:private root-dir (atom "."))

(defn set-root-dir!
  "Set the root directory for file operations."
  [dir]
  (reset! root-dir dir))

(defn get-root-dir []
  @root-dir)

(defn- safe-path?
  "Check if path is safe (no directory traversal)."
  [^String path]
  (not (or (.contains path "..")
           (.startsWith path "/")
           (.contains path "~"))))

(defn- resolve-path
  "Resolve file path relative to root directory."
  [uri]
  (let [path (if (.startsWith ^String uri "/")
               (subs uri 1)
               uri)]
    (when (safe-path? path)
      (File. ^String @root-dir ^String path))))

(defn- content-type
  "Guess content type from file extension."
  [^File file]
  (let [name (.getName file)
        ext (when-let [i (.lastIndexOf name ".")]
              (when (pos? i)
                (.toLowerCase (subs name (inc i)))))]
    (case ext
      "html" "text/html"
      "htm"  "text/html"
      "css"  "text/css"
      "js"   "application/javascript"
      "json" "application/json"
      "xml"  "application/xml"
      "txt"  "text/plain"
      "csv"  "text/csv"
      "md"   "text/markdown"
      "png"  "image/png"
      "jpg"  "image/jpeg"
      "jpeg" "image/jpeg"
      "gif"  "image/gif"
      "svg"  "image/svg+xml"
      "ico"  "image/x-icon"
      "pdf"  "application/pdf"
      "zip"  "application/zip"
      "gz"   "application/gzip"
      "tar"  "application/x-tar"
      "application/octet-stream")))

(defn file-handler
  "File server handler.

   Routes:
     GET /health     -> Health check
     GET /<path>     -> Download file (streaming)
     POST /<path>    -> Upload file (streaming)
     PUT /<path>     -> Upload file (streaming)"
  [req]
  (let [uri (:uri req)
        method (:request-method req)]
    (cond
      ;; Health check
      (and (= method :get) (= uri "/health"))
      {:status 200
       :headers {"Content-Type" "application/json"}
       :body "{\"status\":\"ok\"}\n"}

      ;; Root path - show info
      (and (= method :get) (= uri "/"))
      {:status 200
       :headers {"Content-Type" "application/json"}
       :body (format "{\"root\":\"%s\",\"endpoints\":[\"/health\",\"GET /<path>\",\"POST /<path>\"]}\n"
                     @root-dir)}

      ;; GET - Download file
      (= method :get)
      (if-let [file (resolve-path uri)]
        (cond
          (not (.exists file))
          {:status 404
           :headers {"Content-Type" "application/json"}
           :body "{\"error\":\"not found\"}\n"}

          (.isDirectory file)
          {:status 400
           :headers {"Content-Type" "application/json"}
           :body "{\"error\":\"is a directory\"}\n"}

          :else
          {:status 200
           :headers {"Content-Type" (content-type file)
                     "Content-Length" (str (.length file))}
           :body (FileInputStream. file)})
        {:status 400
         :headers {"Content-Type" "application/json"}
         :body "{\"error\":\"invalid path\"}\n"})

      ;; POST/PUT - Upload file
      (contains? #{:post :put} method)
      (if-let [file (resolve-path uri)]
        (let [body (:body req)
              start-time (System/currentTimeMillis)]
          (if (instance? InputStream body)
            (do
              ;; Create parent directories
              (when-let [parent (.getParentFile file)]
                (.mkdirs parent))
              (with-open [fos (FileOutputStream. file)]
                (let [buf (byte-array (* 64 1024))
                      total (loop [total 0]
                              (let [n (.read ^InputStream body buf)]
                                (if (pos? n)
                                  (do (.write fos buf 0 n)
                                      (recur (+ total n)))
                                  total)))]
                  (.flush fos)
                  (let [elapsed (- (System/currentTimeMillis) start-time)]
                    {:status 201
                     :headers {"Content-Type" "application/json"}
                     :body (format "{\"status\":\"created\",\"path\":\"%s\",\"bytes\":%d,\"ms\":%d}\n"
                                   (.getPath file) total elapsed)}))))
            {:status 400
             :headers {"Content-Type" "application/json"}
             :body "{\"error\":\"streaming body required\"}\n"}))
        {:status 400
         :headers {"Content-Type" "application/json"}
         :body "{\"error\":\"invalid path\"}\n"})

      ;; Other methods
      :else
      {:status 405
       :headers {"Content-Type" "application/json"}
       :body "{\"error\":\"method not allowed\"}\n"})))
