(ns zeph.client
  "Cross-platform HTTP client using Java NIO.
   API compatible with http-kit client."
  (:refer-clojure :exclude [get])
  (:import [zeph.client HttpClientNio HttpClientNio$HttpClientOptions
            HttpClientRequest HttpClientResponse]
           [java.util.concurrent CompletableFuture TimeUnit TimeoutException ExecutionException]))

;; ============================================================
;; Default Client (lazy initialization)
;; ============================================================

(defonce ^:private default-client
  (delay (HttpClientNio.)))

(def ^:dynamic *force-http1* false)
(def ^:dynamic *trace* false)
(def ^:dynamic *trace-detail* false)

(defn- get-client []
  @default-client)

;; ============================================================
;; Response Conversion
;; ============================================================

(defn- response->map
  "Convert HttpClientResponse to Clojure map.
   Supports :as option for body coercion:
   - :text (default) - return body as String
   - :byte-array - return body as byte[]
   - :stream - return body as InputStream (streaming from server)"
  [^HttpClientResponse response]
  (if (.hasError response)
    {:error (.getError response)
     :opts (.getOpts response)}
    (let [opts (.getOpts response)
          as-type (clojure.core/get opts :as :text)
          ;; Check for streaming response (body stream from NIO)
          body-stream (.getBodyStream response)
          body (if body-stream
                 ;; Streaming response - return stream directly
                 body-stream
                 ;; Non-streaming response - convert based on :as option
                 (let [body-bytes (.getBody response)]
                   (case as-type
                     :byte-array body-bytes
                     :stream (when body-bytes
                               (java.io.ByteArrayInputStream. body-bytes))
                     ;; :text or default
                     (.getBodyAsString response))))]
      {:status (.getStatus response)
       :headers (into {} (.getHeaders response))
       :body body
       :protocol (.getProtocol response)
       :opts opts})))

;; ============================================================
;; Promise Type (http-kit compatible)
;; ============================================================

(deftype HttpPromise [^CompletableFuture fut callback opts]
  clojure.lang.IDeref
  (deref [_]
    (try
      (let [^HttpClientResponse response (.get fut)]
        (response->map response))
      (catch ExecutionException e
        (let [cause (.getCause e)]
          (if (instance? TimeoutException cause)
            {:error :timeout :error-msg "Request timed out"}
            {:error :exception :error-msg (.getMessage (or cause e))})))))

  clojure.lang.IBlockingDeref
  (deref [_ timeout-ms timeout-val]
    (try
      (let [^HttpClientResponse response (.get fut timeout-ms TimeUnit/MILLISECONDS)]
        (response->map response))
      (catch TimeoutException _
        timeout-val)
      (catch ExecutionException e
        (let [cause (.getCause e)]
          (if (instance? TimeoutException cause)
            timeout-val
            {:error :exception :error-msg (.getMessage (or cause e))})))))

  clojure.lang.IPending
  (isRealized [_]
    (.isDone fut))

  clojure.lang.IFn
  (invoke [this callback]
    ;; Allow setting callback after creation (http-kit style)
    (.thenAccept fut
      (reify java.util.function.Consumer
        (accept [_ response]
          (callback (response->map response)))))
    this))

;; ============================================================
;; Request Building
;; ============================================================

(defn- build-request
  "Build HttpClientRequest from options map."
  ^HttpClientRequest
  [{:keys [url method headers body query-params form-params multipart
           timeout keepalive follow-redirects max-redirects
           insecure? basic-auth trace trace-detail trace-limit progress as]
    :or {method :get
         timeout 30000
         keepalive 120000
         follow-redirects true
         max-redirects 5
         insecure? false
         trace-limit 2000
         progress false}
    :as opts}]
  (let [req (doto (HttpClientRequest.)
              (.url url)
              (.method (name method))
              (.timeout timeout)
              (.keepalive keepalive)
              (.followRedirects follow-redirects)
              (.maxRedirects max-redirects)
              (.insecure insecure?)
              (.trace (boolean (or trace *trace*)))
              (.traceDetail (boolean (or trace-detail *trace-detail*)))
              (.traceLimit (long trace-limit))
              (.progress (boolean progress))
              (.opts opts))]
    (when headers
      (.headers req headers))
    (when body
      (cond
        (bytes? body) (.body req ^bytes body)
        (instance? java.io.InputStream body) (.body req ^java.io.InputStream body)
        (instance? java.io.File body) (.body req ^java.io.File body)
        :else (.body req (str body))))
    (when query-params
      (.queryParams req query-params))
    (when form-params
      (.formParams req form-params))
    (when multipart
      (.multipart req multipart))
    (when basic-auth
      (let [[user pass] basic-auth]
        (.basicAuth req user pass)))
    req))

;; ============================================================
;; Public API
;; ============================================================

(defn request
  "Send an HTTP request.

   Options:
     :url              - Request URL (required)
     :method           - HTTP method (:get, :post, :put, :delete, :head)
     :headers          - Request headers map
     :body             - Request body (string or bytes)
     :query-params     - Query parameters map
     :form-params      - Form parameters map (sets Content-Type)
     :multipart        - Multipart form data as vector of maps:
                         [{:name \"field\" :content \"value\"}
                          {:name \"file\" :content (io/file \"path\") :filename \"name.txt\"}]
                         Each map supports :name, :content, :filename, :content-type
     :timeout          - Request timeout in ms (default: 30000)
     :keepalive        - Keep-alive time in ms (default: 120000, -1 to disable)
     :follow-redirects - Follow redirects (default: true)
     :max-redirects    - Maximum redirects to follow (default: 5)
     :insecure?        - Skip SSL certificate validation (default: false)
     :basic-auth       - Basic auth as [user password]
     :trace            - Print request/response flow to stderr (default: false)
     :trace-detail     - Print headers and body details to stderr (default: false)
     :as               - Response body coercion (:text, :stream, :byte-array)

   Returns a promise that can be:
     - Dereferenced with @ for synchronous access
     - Used with deref with timeout
     - Called as a function with callback

   Examples:
     ;; Synchronous
     @(request {:url \"https://example.com\" :method :get})

     ;; With timeout
     (deref (request {:url \"https://example.com\"}) 5000 :timeout)

     ;; With callback
     (request {:url \"https://example.com\"}
              (fn [{:keys [status body error]}]
                (println status body)))

     ;; With trace
     @(request {:url \"https://example.com\" :trace true})

     ;; With detailed trace
     @(request {:url \"https://example.com\" :trace-detail true})"
  ([opts]
   (request opts nil))
  ([opts callback]
   (let [client (get-client)
         req (build-request opts)
         ^CompletableFuture fut (.request client req)
         promise (->HttpPromise fut callback opts)]
     (when callback
       (.thenAccept fut
         (reify java.util.function.Consumer
           (accept [_ response]
             (callback (response->map response))))))
     promise)))

(defn get
  "Send a GET request. Returns promise (http-kit compatible).

   Usage:
     @(get \"https://example.com\")
     @(get \"https://example.com\" {:timeout 5000})
     (get \"https://example.com\" {:timeout 5000} callback)"
  ([url]
   (request {:url url :method :get}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :get} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :get))))
  ([url opts callback]
   (request (assoc opts :url url :method :get) callback)))

(defn post
  "Send a POST request. Returns promise (http-kit compatible).

   Usage:
     @(post \"https://example.com\" {:body \"data\"})
     (post \"https://example.com\" {:body \"data\"} callback)"
  ([url]
   (request {:url url :method :post}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :post} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :post))))
  ([url opts callback]
   (request (assoc opts :url url :method :post) callback)))

(defn put
  "Send a PUT request. Returns promise (http-kit compatible)."
  ([url]
   (request {:url url :method :put}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :put} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :put))))
  ([url opts callback]
   (request (assoc opts :url url :method :put) callback)))

(defn delete
  "Send a DELETE request. Returns promise (http-kit compatible)."
  ([url]
   (request {:url url :method :delete}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :delete} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :delete))))
  ([url opts callback]
   (request (assoc opts :url url :method :delete) callback)))

(defn patch
  "Send a PATCH request. Returns promise (http-kit compatible)."
  ([url]
   (request {:url url :method :patch}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :patch} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :patch))))
  ([url opts callback]
   (request (assoc opts :url url :method :patch) callback)))

(defn head
  "Send a HEAD request. Returns promise (http-kit compatible)."
  ([url]
   (request {:url url :method :head}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :head} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :head))))
  ([url opts callback]
   (request (assoc opts :url url :method :head) callback)))

(defn execute
  "Execute a raw HttpClientRequest object.

   Usage:
     @(execute req)
     @(execute req {:timeout 30000})

   The request object can be created via Java interop:
     (-> (HttpClientRequest/post \"https://example.com/\")
         (.multipart \"field\" \"value\")
         (.multipartFile \"file\" (io/file \"path\") \"application/octet-stream\"))"
  ([^zeph.client.HttpClientRequest req]
   (execute req {}))
  ([^zeph.client.HttpClientRequest req opts]
   (let [client (get-client)
         ^CompletableFuture fut (.request client req)
         promise (->HttpPromise fut nil opts)]
     promise)))

;; ============================================================
;; Client Management
;; ============================================================

(defn create-client
  "Create a new HTTP client with custom options.

   Options:
     :backend                - Backend to use (:auto, :io-uring, :nio)
                               :auto selects io_uring on Linux, NIO on Windows/macOS
     :workers                - Number of worker threads (default: CPU/2)
     :ring-size              - io_uring ring size (default: 256, io_uring only)
     :max-connections-per-host - Max connections per host (default: 20)
     :max-total-connections  - Max total connections (default: 200)
     :idle-timeout           - Idle connection timeout in ms (default: 60000)
     :connection-timeout     - Connection timeout in ms (default: 30000)

   Returns: HttpClientNio instance (call .close when done)"
  ([]
   (create-client {}))
  ([opts]
   (let [options (HttpClientNio$HttpClientOptions.)]
     (when-let [v (:workers opts)]
       (.workers options v))
     (when-let [v (:max-connections-per-host opts)]
       (.maxConnectionsPerHost options v))
     (when-let [v (:max-total-connections opts)]
       (.maxTotalConnections options v))
     (when-let [v (:idle-timeout opts)]
       (.idleTimeout options v))
     (when-let [v (:connection-timeout opts)]
       (.connectionTimeout options v))
     (HttpClientNio. options))))

(defn close-client
  "Close an HTTP client."
  [client]
  (.close ^java.lang.AutoCloseable client))

(defn pool-stats
  "Get connection pool statistics for the default client."
  []
  (let [stats (.getPoolStats (get-client))]
    {:hosts (.hosts stats)
     :idle (.idle stats)
     :in-use (.inUse stats)
     :total (.total stats)}))

;; ============================================================
;; Shutdown hook
;; ============================================================

(defonce ^:private shutdown-hook-registered
  (delay
    (.addShutdownHook
      (Runtime/getRuntime)
      (Thread.
        (fn []
          (when (realized? default-client)
            (.close ^HttpClientNio @default-client)))))))

;; Force registration
@shutdown-hook-registered
