(ns zeph.cli
  "Zeph CLI - Cross-platform HTTP server & client."
  (:require [zeph.server :as server]
            [zeph.client :as client]
            [zeph.handlers :as handlers]
            [clojure.string :as str]
            [clojure.tools.cli :refer [parse-opts]])
  (:gen-class))

(def version "0.2.1")

;; ============================================================
;; Server Command
;; ============================================================

(def ^:private server-cli-opts
  [["-p" "--port PORT" "Port to listen on"
    :default 8080
    :parse-fn #(Long/parseLong %)]
   [nil "--host HOST" "Host/IP to bind"
    :default "0.0.0.0"]
   ["-t" "--thread N" "Number of worker threads"
    :default (.availableProcessors (Runtime/getRuntime))
    :parse-fn #(Long/parseLong %)]
   ["-r" "--root DIR" "Root directory for files (default: current dir)"
    :default "."]
   [nil "--ssl" "Enable HTTPS"]
   [nil "--cert FILE" "SSL certificate PEM file"]
   [nil "--key FILE" "SSL private key PEM file"]
   ["-h" "--help" "Show this help"]])

(defn- server-usage [summary]
  (str/join \newline
    ["Usage: zeph server [OPTIONS]"
     ""
     "Start an HTTP/HTTPS file server."
     ""
     "Endpoints:"
     "  GET  /health   Health check"
     "  GET  /<path>   Download file"
     "  POST /<path>   Upload file (streaming)"
     "  PUT  /<path>   Upload file (streaming)"
     ""
     "Options:"
     summary
     ""
     "Examples:"
     "  zeph server -p 8080"
     "  zeph server -p 8080 -r /data"
     "  zeph server --ssl -p 8443"
     "  zeph server --ssl --cert cert.pem --key key.pem"]))

(defn- run-server-cmd
  "Run the server command."
  [args]
  (let [{:keys [options summary errors]} (parse-opts args server-cli-opts)]
    (cond
      errors
      (do (doseq [e errors] (println e))
          1)

      (:help options)
      (do (println (server-usage summary))
          0)

      :else
      (let [{:keys [port host thread root ssl cert key]} options
            root-abs (.getAbsolutePath (java.io.File. ^String root))
            _ (handlers/set-root-dir! root-abs)
            server-opts (cond-> {:port port :ip host :thread thread
                                 :streaming? true :max-body -1}
                          ssl (assoc :ssl? true)
                          cert (assoc :cert cert)
                          key (assoc :key key))]
        (println (str "Starting Zeph " (if ssl "HTTPS" "HTTP") " server..."))
        (println (str "  Listening: " (if ssl "https://" "http://") host ":" port))
        (println (str "  Root: " root-abs))
        (println (str "  Workers: " thread))
        (when ssl
          (if (and cert key)
            (println (str "  SSL: " cert ", " key))
            (println "  SSL: built-in localhost certificate")))
        (println)
        (let [stop (server/run-server handlers/file-handler server-opts)]
          (.addShutdownHook
            (Runtime/getRuntime)
            (Thread. (fn []
                       (println "\nShutting down...")
                       (stop))))
          ;; Block forever
          @(promise))))))

;; ============================================================
;; Client Command (HTTPie-style)
;; ============================================================

(def ^:private client-cli-opts
  [["-v" "--verbose" "Show request headers"]
   ["-V" "--http-trace" "Show HTTP request/response flow"]
   ["-X" "--http-trace-detail" "Show HTTP request/response with headers and body"]
   ["-L" "--trace-limit BYTES" "Max body size to show in trace (0=unlimited, default 2000)"
    :default 2000
    :parse-fn #(Long/parseLong %)]
   ["-b" "--body" "Only show response body"]
   ["-H" "--headers-only" "Only show response headers"]
   ["-d" "--data DATA" "Raw request body (JSON, text, or @file)"]
   ["-k" "--insecure" "Skip SSL certificate verification"]
   ["-F" "--follow" "Follow redirects"]
   ["-1" "--http1" "Force HTTP/1.1 (disable HTTP/2)"]
   [nil "--timeout MS" "Request timeout in ms"
    :default 30000
    :parse-fn #(Long/parseLong %)]
   ["-h" "--help" "Show this help"]])

(defn- read-raw-body
  "Read raw body from string, file (@filename), or stdin (-).
   Returns [body content-type-hint]."
  [^String data]
  (cond
    ;; stdin
    (= data "-")
    (let [content (slurp *in*)]
      [content (when (or (str/starts-with? (str/trim content) "{")
                         (str/starts-with? (str/trim content) "["))
                 "application/json")])

    ;; file reference
    (str/starts-with? data "@")
    (let [filename (subs data 1)
          file (java.io.File. filename)]
      (if (.exists file)
        (let [content (slurp file)
              json? (or (str/ends-with? filename ".json")
                        (str/starts-with? (str/trim content) "{")
                        (str/starts-with? (str/trim content) "["))]
          [content (when json? "application/json")])
        (throw (ex-info (str "File not found: " filename) {:file filename}))))

    ;; inline data - auto-detect JSON
    :else
    (let [json? (or (str/starts-with? (str/trim data) "{")
                    (str/starts-with? (str/trim data) "["))]
      [data (when json? "application/json")])))

(defn- http-method? [s]
  (contains? #{"GET" "POST" "PUT" "DELETE" "PATCH" "HEAD" "OPTIONS"} (str/upper-case s)))

(defn- url? [s]
  (or (str/starts-with? s "http://")
      (str/starts-with? s "https://")
      (str/starts-with? s "localhost")
      ;; hostname:port or hostname:port/path (port must be 1-5 digits followed by nothing or /)
      (re-matches #"[\w.-]+:\d{1,5}(/.*)?$" s)
      ;; hostname/path or hostname with at least one dot (domain name)
      (re-matches #"[\w.-]+/.*" s)
      (re-matches #"[\w-]+\.[\w.-]+" s)))

(defn- parse-request-item
  "Parse HTTPie-style request item."
  [^String item]
  (cond
    ;; JSON data (key:=value)
    (str/includes? item ":=")
    (let [idx (.indexOf item ":=")]
      (when (pos? idx)
        {:type :json
         :key (subs item 0 idx)
         :value (subs item (+ idx 2))}))

    ;; Header: value
    (str/includes? item ":")
    (let [idx (.indexOf item ":")]
      (when (pos? idx)
        (let [raw-value (str/trim (subs item (inc idx)))
              ;; Strip surrounding quotes if present
              value (if (and (>= (count raw-value) 2)
                             (or (and (str/starts-with? raw-value "\"")
                                      (str/ends-with? raw-value "\""))
                                 (and (str/starts-with? raw-value "'")
                                      (str/ends-with? raw-value "'"))))
                      (subs raw-value 1 (dec (count raw-value)))
                      raw-value)]
          {:type :header
           :key (subs item 0 idx)
           :value value})))

    ;; key=value (string data)
    (str/includes? item "=")
    (let [idx (.indexOf item "=")]
      (when (pos? idx)
        {:type :data
         :key (subs item 0 idx)
         :value (subs item (inc idx))}))

    :else nil))

(defn- parse-client-positional
  "Parse positional arguments for client command."
  [positional]
  (loop [args positional
         result {:headers {} :data-items [] :json-items []}]
    (if (empty? args)
      result
      (let [[arg & more] args]
        (cond
          ;; HTTP method
          (http-method? arg)
          (recur more (assoc result :method (keyword (str/lower-case arg))))

          ;; URL
          (url? arg)
          (let [url (if (str/starts-with? arg "http://") arg
                         (if (str/starts-with? arg "https://") arg
                           (str "https://" arg)))]
            (recur more (assoc result :url url)))

          ;; Request item
          :else
          (if-let [item (parse-request-item arg)]
            (case (:type item)
              :header (recur more (update result :headers assoc (:key item) (:value item)))
              :data (recur more (update result :data-items conj item))
              :json (recur more (update result :json-items conj item)))
            (recur more result)))))))

(defn- parse-key-path
  "Parse a key path like 'a.b.0.c' into path segments.
   Supports escape with backslash: 'a\\.b' keeps dot as part of key.
   Numeric segments become integers (for array indexing)."
  [^String key]
  (loop [chars (seq key)
         current (StringBuilder.)
         result []]
    (if (empty? chars)
      (let [s (.toString current)]
        (if (empty? s)
          result
          (conj result (if (re-matches #"\d+" s)
                         (Integer/parseInt s)
                         s))))
      (let [[c & more] chars]
        (cond
          ;; Escaped character
          (and (= c \\) (seq more))
          (recur (rest more) (.append current (first more)) result)

          ;; Dot separator
          (= c \.)
          (let [s (.toString current)]
            (recur more
                   (StringBuilder.)
                   (conj result (if (re-matches #"\d+" s)
                                  (Integer/parseInt s)
                                  s))))

          ;; Regular character
          :else
          (recur more (.append current c) result))))))

(defn- ensure-array-size
  "Ensure vector has at least n+1 elements, padding with nil."
  [v n]
  (if (> (inc n) (count v))
    (into v (repeat (- (inc n) (count v)) nil))
    v))

(defn- assoc-in-path
  "Associate a value at a nested path, creating intermediate structures.
   Integer path segments create vectors, string segments create maps."
  [data path value]
  (if (empty? path)
    value
    (let [[k & ks] path
          current (if (integer? k)
                    (or data [])
                    (or data {}))]
      (if (integer? k)
        ;; Array index
        (let [arr (ensure-array-size (if (vector? current) current []) k)
              existing (get arr k)]
          (assoc arr k (assoc-in-path existing ks value)))
        ;; Object key
        (let [existing (get current k)]
          (assoc current k (assoc-in-path existing ks value)))))))

(defn- to-json
  "Convert Clojure data to JSON string."
  [data]
  (cond
    (nil? data) "null"
    ;; If string looks like JSON object/array, pass through as-is
    (and (string? data)
         (or (and (str/starts-with? data "{") (str/ends-with? data "}"))
             (and (str/starts-with? data "[") (str/ends-with? data "]"))))
    data
    (string? data) (str "\"" (str/replace data "\"" "\\\"") "\"")
    (number? data) (str data)
    (boolean? data) (str data)
    (vector? data) (str "[" (str/join ", " (map to-json data)) "]")
    (map? data) (str "{"
                     (str/join ", "
                       (map (fn [[k v]] (str (to-json (str k)) ": " (to-json v)))
                            data))
                     "}")
    :else (str "\"" data "\"")))

(defn- build-json-body
  "Build JSON body from data items with nested path support.
   Supports: a.b=value, a.0=value, a\\.b=value (escaped dot)"
  [data-items json-items]
  (when (or (seq data-items) (seq json-items))
    (let [;; Process data items (string values)
          data (reduce (fn [acc {:keys [key value]}]
                         (let [path (parse-key-path key)]
                           (assoc-in-path acc path value)))
                       nil
                       data-items)
          ;; Process json items (raw JSON values) - parse them first
          result (reduce (fn [acc {:keys [key value]}]
                           (let [path (parse-key-path key)
                                 ;; Parse the raw JSON value
                                 parsed (cond
                                          (= value "true") true
                                          (= value "false") false
                                          (= value "null") nil
                                          (re-matches #"-?\d+" value) (Long/parseLong value)
                                          (re-matches #"-?\d+\.\d+" value) (Double/parseDouble value)
                                          :else value)]
                             (assoc-in-path acc path parsed)))
                         data
                         json-items)]
      (to-json result))))

(def ^:private colors
  {:reset   "\u001b[0m"
   :bold    "\u001b[1m"
   :green   "\u001b[32m"
   :blue    "\u001b[34m"
   :cyan    "\u001b[36m"
   :yellow  "\u001b[33m"
   :magenta "\u001b[35m"
   :red     "\u001b[31m"
   :gray    "\u001b[90m"
   :white   "\u001b[37m"})

(defn- colorize [color text]
  (str (get colors color "") text (:reset colors)))

(defn- format-status [status]
  (let [color (cond
                (< status 300) :green
                (< status 400) :yellow
                :else :red)]
    (str (colorize color (str "HTTP/1.1 " status))
         (colorize :gray (str " " (get {200 "OK" 201 "Created" 204 "No Content"
                                        301 "Moved Permanently" 302 "Found" 304 "Not Modified"
                                        400 "Bad Request" 401 "Unauthorized" 403 "Forbidden"
                                        404 "Not Found" 500 "Internal Server Error"} status ""))))))

(defn- format-header [k v]
  (str (colorize :cyan k) ": " v))

(defn- unescape-json-string
  "Unescape JSON string for display (convert \\\" to \" etc)."
  [^String s]
  (-> s
      (str/replace "\\/" "/")
      (str/replace "\\\"" "\"")
      (str/replace "\\n" "\n")
      (str/replace "\\r" "\r")
      (str/replace "\\t" "\t")
      (str/replace "\\\\" "\\")))

(defn- json-indent
  "Indent JSON string properly with syntax highlighting."
  [^String s indent]
  (let [sb (StringBuilder.)
        len (.length s)]
    (loop [i 0
           depth indent]
      (if (>= i len)
        (str sb)
        (let [c (.charAt s i)]
          (cond
            ;; String - extract entire string and colorize
            (= c \")
            (let [[str-content end-i]
                  (loop [j (inc i) acc ""]
                    (if (>= j len)
                      [acc j]
                      (let [nc (.charAt s j)]
                        (if (= nc \")
                          [acc (inc j)]
                          (if (= nc \\)
                            (recur (+ j 2) (str acc nc (.charAt s (inc j))))
                            (recur (inc j) (str acc nc)))))))
                  is-key (loop [k end-i]
                           (if (>= k len)
                             false
                             (let [nc (.charAt s k)]
                               (cond
                                 (= nc \:) true
                                 (Character/isWhitespace nc) (recur (inc k))
                                 :else false))))
                  color (if is-key :green :yellow)
                  display-str (str "\"" (unescape-json-string str-content) "\"")]
              (.append sb (colorize color display-str))
              (recur end-i depth))

            ;; Opening brace/bracket
            (or (= c \{) (= c \[))
            (let [new-depth (inc depth)
                  indent-next (apply str (repeat new-depth "    "))]
              (.append sb c)
              (.append sb "\n")
              (.append sb indent-next)
              (recur (inc i) new-depth))

            ;; Closing brace/bracket
            (or (= c \}) (= c \]))
            (let [new-depth (dec depth)
                  indent-prev (apply str (repeat new-depth "    "))]
              (.append sb "\n")
              (.append sb indent-prev)
              (.append sb c)
              (recur (inc i) new-depth))

            ;; Comma
            (= c \,)
            (let [indent-cur (apply str (repeat depth "    "))]
              (.append sb c)
              (.append sb "\n")
              (.append sb indent-cur)
              (recur (inc i) depth))

            ;; Colon (key-value separator)
            (= c \:)
            (do (.append sb ": ")
                (recur (inc i) depth))

            ;; Skip whitespace
            (Character/isWhitespace c)
            (recur (inc i) depth)

            ;; Numbers
            (or (Character/isDigit c) (and (= c \-) (< (inc i) len) (Character/isDigit (.charAt s (inc i)))))
            (let [[num rest-i] (loop [j i num-str ""]
                                 (if (and (< j len)
                                          (let [nc (.charAt s j)]
                                            (or (Character/isDigit nc) (= nc \.) (= nc \-) (= nc \e) (= nc \E) (= nc \+))))
                                   (recur (inc j) (str num-str (.charAt s j)))
                                   [num-str j]))]
              (.append sb (colorize :cyan num))
              (recur rest-i depth))

            ;; true
            (and (= c \t) (<= (+ i 4) len) (= (subs s i (+ i 4)) "true"))
            (do (.append sb (colorize :yellow "true"))
                (recur (+ i 4) depth))

            ;; false
            (and (= c \f) (<= (+ i 5) len) (= (subs s i (+ i 5)) "false"))
            (do (.append sb (colorize :yellow "false"))
                (recur (+ i 5) depth))

            ;; null
            (and (= c \n) (<= (+ i 4) len) (= (subs s i (+ i 4)) "null"))
            (do (.append sb (colorize :magenta "null"))
                (recur (+ i 4) depth))

            ;; Other chars
            :else
            (do (.append sb c)
                (recur (inc i) depth))))))))

(defn- try-format-json
  "Try to pretty-print JSON body with syntax highlighting."
  [body]
  (if (and body
           (let [trimmed (str/trim body)]
             (or (str/starts-with? trimmed "{")
                 (str/starts-with? trimmed "["))))
    (try
      (json-indent (str/trim body) 0)
      (catch Exception _ body))
    body))

(defn- ssl-error? [error]
  (let [msg (str (ex-message error))]
    (or (str/includes? msg "SSL")
        (str/includes? msg "TLS")
        (str/includes? msg "HTTPS")
        (str/includes? msg "handshake")
        (str/includes? msg "does not support")
        (str/includes? msg "Connection closed by server")
        (str/includes? msg "connection reset")
        (str/includes? msg "Worker shutting down"))))

(defn- print-response
  "Print HTTP response in HTTPie style."
  [{:keys [status headers body error]} {:keys [body-only headers-only url]}]
  (if error
    (do
      (println (colorize :red (str "Error: " (ex-message error))))
      (when (and url (ssl-error? error) (str/starts-with? url "https://"))
        (let [http-url (str "http://" (subs url 8))]
          (println (colorize :yellow (str "Hint: Server may not support HTTPS. Try: zeph " http-url)))))
      1)
    (do
      (when-not body-only
        (println (format-status status))
        (doseq [[k v] (sort-by first headers)]
          (println (format-header k v))))
      (when (and body (not headers-only))
        (println)
        (println (try-format-json body)))
      0)))

(defn- client-usage [summary]
  (str/join \newline
    [(str "Zeph " version " - HTTP client")
     ""
     "Usage: zeph client [OPTIONS] [METHOD] URL [ITEM...]"
     ""
     "Make HTTP requests with HTTPie-style syntax."
     ""
     "Items:"
     "  Header:Value     Add HTTP header"
     "  key=value        JSON string field"
     "  key:=123         Raw JSON value (number, bool, null)"
     "  a.b=value        Nested object: {\"a\": {\"b\": \"value\"}}"
     "  a.0=value        Array element: {\"a\": [\"value\"]}"
     "  a\\.b=value      Escaped dot: {\"a.b\": \"value\"}"
     ""
     "Raw Body (-d/--data):"
     "  -d '{...}'       Inline JSON/text"
     "  -d @file.json    Read from file"
     "  -d -             Read from stdin"
     ""
     "Options:"
     summary
     ""
     "Examples:"
     "  zeph c httpbin.org/get"
     "  zeph c POST httpbin.org/post name=John age:=30"
     "  zeph c httpbin.org/post user.name=John user.email=john@example.com"
     "  zeph c httpbin.org/post items.0=apple items.1=banana"
     "  zeph c -d '{\"jsonrpc\":\"2.0\",\"method\":\"init\"}' http://localhost:9100/mcp"
     "  zeph c -d @request.json POST httpbin.org/post"
     "  echo '{\"key\":\"value\"}' | zeph c -d - POST httpbin.org/post"
     "  zeph c -X httpbin.org/get                  # trace with body"
     "  zeph c -k https://self-signed.example.com  # skip SSL verify"]))

(defn- run-client-cmd
  "Run the client command (HTTPie-style)."
  [args]
  (let [{:keys [options arguments summary errors]} (parse-opts args client-cli-opts)
        pos-opts (parse-client-positional arguments)]

    (cond
      errors
      (do (doseq [e errors] (println e))
          1)

      (:help options)
      (do (println (client-usage summary))
          0)

      (not (:url pos-opts))
      (do (println "Error: URL required")
          (println "Usage: zeph client [METHOD] URL [ITEM...]")
          (println "Try: zeph client --help")
          1)

      :else
      (let [{:keys [url method headers data-items json-items]} pos-opts
            {:keys [verbose http-trace http-trace-detail trace-limit body headers-only insecure follow http1 timeout data]} options
            ;; --data takes priority over key=value items
            [raw-body content-type-hint] (when data (read-raw-body data))
            has-items (or (seq data-items) (seq json-items))
            has-data (or raw-body has-items)
            method (or method (if has-data :post :get))
            body-str (or raw-body (when has-items (build-json-body data-items json-items)))
            ;; Set Content-Type: user header > auto-detect > default for items
            headers (cond-> headers
                      (and has-data (not (get headers "Content-Type")))
                      (assoc "Content-Type" (or content-type-hint "application/json")))
            request-opts (cond-> {:url url :method method}
                           (seq headers) (assoc :headers headers)
                           body-str (assoc :body body-str)
                           timeout (assoc :timeout timeout)
                           insecure (assoc :insecure? true)
                           follow (assoc :follow-redirects true)
                           http-trace (assoc :trace true)
                           http-trace-detail (assoc :trace-detail true)
                           trace-limit (assoc :trace-limit trace-limit))
            _ (when verbose
                (println (colorize :bold (str (str/upper-case (name method)) " " url)))
                (doseq [[k v] headers]
                  (println (format-header k v)))
                (when body-str
                  (println)
                  (println (try-format-json body-str)))
                (println)
                (println (colorize :gray "---"))
                (println))
            response (try
                       (binding [client/*force-http1* http1]
                         @(client/request request-opts))
                       (catch Exception e
                         {:error e}))]
        (if (or http-trace http-trace-detail)
          (if (:error response) 1 0)
          (print-response response {:body-only body :headers-only headers-only :url url}))))))

;; ============================================================
;; Help
;; ============================================================

(defn- print-help []
  (println (str "Zeph " version " - Cross-platform HTTP server & client"))
  (println)
  (println "Usage:")
  (println "  zeph [METHOD] URL [ITEM...]       HTTP client (default)")
  (println "  zeph server [options]             Start HTTP server")
  (println)
  (println "Commands:")
  (println "  server, s    Start HTTP/HTTPS server")
  (println "  client, c    Make HTTP requests (explicit)")
  (println "  --version    Show version")
  (println)
  (println "Client is the default command - just specify URL directly.")
  (println "Run 'zeph --help' or 'zeph c --help' for client options.")
  (println)
  (println "Examples:")
  (println "  zeph httpbin.org/get")
  (println "  zeph POST httpbin.org/post name=John")
  (println "  zeph httpbin.org/post user.name=John user.age:=30")
  (println "  zeph -d '{\"jsonrpc\":\"2.0\"}' POST localhost:9100/mcp")
  (println "  zeph -X httpbin.org/get              # trace with body")
  (println "  zeph s -p 8080                       # start server"))

;; ============================================================
;; Main
;; ============================================================

(defn -main [& args]
  (if (empty? args)
    (do
      (print-help)
      (System/exit 0))
    (let [[cmd & rest-args] args]
      (case cmd
        ("server" "s") (System/exit (run-server-cmd rest-args))
        ("client" "c") (System/exit (run-client-cmd rest-args))
        "help" (print-help)
        "--help" (print-help)
        "-h" (print-help)
        "--version" (println (str "zeph " version))
        ;; Default: treat as client command (pass all args)
        (System/exit (run-client-cmd args))))))
