/*
 * Decompiled with CFR 0.152.
 */
package zeph.http2;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import zeph.http2.Http2Frame;
import zeph.http2.Http2FrameReader;
import zeph.http2.Http2FrameType;
import zeph.http2.Http2FrameWriter;
import zeph.http2.Http2Settings;
import zeph.http2.Http2Stream;
import zeph.http2.hpack.HpackDecoder;

public class Http2Connection {
    private State state;
    private final Http2Settings localSettings;
    private final Http2Settings remoteSettings;
    private final Map<Integer, Http2Stream> streams;
    private int lastClientStreamId;
    private int nextServerStreamId;
    private final AtomicInteger connectionSendWindow = new AtomicInteger();
    private final AtomicInteger connectionRecvWindow = new AtomicInteger();
    private final Http2FrameReader frameReader;
    private final Http2FrameWriter frameWriter;
    private final HpackDecoder hpackDecoder;
    private int headerStreamId;
    private ByteBuffer headerBuffer;
    private StreamHandler streamHandler;
    private static final float WINDOW_UPDATE_THRESHOLD = 0.5f;
    private final AtomicInteger connectionRecvWindowConsumed = new AtomicInteger(0);

    public Http2Connection() {
        this(new Http2Settings());
    }

    public Http2Connection(Http2Settings localSettings) {
        this.state = State.AWAITING_PREFACE;
        this.localSettings = localSettings;
        this.remoteSettings = new Http2Settings();
        this.streams = new ConcurrentHashMap<Integer, Http2Stream>();
        this.lastClientStreamId = 0;
        this.nextServerStreamId = 2;
        this.connectionSendWindow.set(65535);
        this.connectionRecvWindow.set(localSettings.getInitialWindowSize());
        this.frameReader = new Http2FrameReader(localSettings.getMaxFrameSize());
        this.frameWriter = new Http2FrameWriter(this.remoteSettings.getMaxFrameSize());
        this.hpackDecoder = new HpackDecoder(localSettings.getHeaderTableSize(), localSettings.getMaxHeaderListSize());
    }

    public void setStreamHandler(StreamHandler handler) {
        this.streamHandler = handler;
    }

    public State getState() {
        return this.state;
    }

    public boolean isOpen() {
        return this.state == State.OPEN || this.state == State.AWAITING_SETTINGS;
    }

    public Http2Stream getStream(int streamId) {
        return this.streams.get(streamId);
    }

    public List<Http2Stream> getActiveStreams() {
        ArrayList<Http2Stream> active = new ArrayList<Http2Stream>();
        for (Http2Stream stream : this.streams.values()) {
            if (!stream.isOpen()) continue;
            active.add(stream);
        }
        return active;
    }

    public void writeServerPreface(ByteBuffer buf) {
        this.frameWriter.writeSettings(buf, this.localSettings);
        int windowIncrement = this.localSettings.getInitialWindowSize() - 65535;
        if (windowIncrement > 0) {
            this.frameWriter.writeWindowUpdate(buf, 0, windowIncrement);
        }
    }

    public void writeSettingsAck(ByteBuffer buf) {
        this.frameWriter.writeSettingsAck(buf);
    }

    public void writePingAck(ByteBuffer buf, byte[] opaqueData) {
        this.frameWriter.writePingAck(buf, opaqueData);
    }

    public void writeGoaway(ByteBuffer buf, int errorCode, String debugData) {
        this.frameWriter.writeGoaway(buf, this.lastClientStreamId, errorCode, debugData);
        this.state = State.GOAWAY_SENT;
    }

    public void writeRstStream(ByteBuffer buf, int streamId, int errorCode) {
        this.frameWriter.writeRstStream(buf, streamId, errorCode);
        Http2Stream stream = this.streams.get(streamId);
        if (stream != null) {
            stream.reset(errorCode);
        }
    }

    public void writeConnectionWindowUpdate(ByteBuffer buf, int increment) {
        this.frameWriter.writeWindowUpdate(buf, 0, increment);
        this.connectionRecvWindow.addAndGet(increment);
    }

    public void writeStreamWindowUpdate(ByteBuffer buf, int streamId, int increment) {
        this.frameWriter.writeWindowUpdate(buf, streamId, increment);
        Http2Stream stream = this.streams.get(streamId);
        if (stream != null) {
            stream.updateRecvWindow(increment);
        }
    }

    public void writeResponseHeaders(ByteBuffer buf, int streamId, List<String[]> headers, boolean endStream) throws Http2FrameReader.Http2Exception {
        Http2Stream stream = this.streams.get(streamId);
        if (stream == null) {
            throw new Http2FrameReader.Http2Exception(1, streamId, "Unknown stream " + streamId);
        }
        this.frameWriter.writeHeaders(buf, streamId, headers, endStream);
        stream.sendHeaders(endStream);
    }

    public int writeResponseData(ByteBuffer buf, int streamId, byte[] data, boolean endStream) throws Http2FrameReader.Http2Exception {
        boolean actualEndStream;
        Http2Stream stream = this.streams.get(streamId);
        if (stream == null) {
            throw new Http2FrameReader.Http2Exception(1, streamId, "Unknown stream " + streamId);
        }
        int available = Math.min(this.connectionSendWindow.get(), stream.getSendWindow());
        if (available <= 0 && data.length > 0) {
            return 0;
        }
        int toSend = Math.min(data.length, available);
        boolean bl = actualEndStream = endStream && toSend == data.length;
        if (toSend > 0) {
            byte[] chunk;
            if (toSend == data.length) {
                chunk = data;
            } else {
                chunk = new byte[toSend];
                System.arraycopy(data, 0, chunk, 0, toSend);
            }
            this.frameWriter.writeData(buf, streamId, chunk, actualEndStream);
            this.connectionSendWindow.addAndGet(-toSend);
            stream.consumeSendWindow(toSend);
        } else if (data.length == 0 && endStream) {
            this.frameWriter.writeData(buf, streamId, data, true);
        }
        if (actualEndStream) {
            stream.sendData(true);
        } else if (toSend > 0) {
            stream.sendData(false);
        }
        return toSend;
    }

    public int getAvailableSendWindow(int streamId) {
        Http2Stream stream = this.streams.get(streamId);
        if (stream == null) {
            return 0;
        }
        return Math.min(this.connectionSendWindow.get(), stream.getSendWindow());
    }

    public boolean canSendData(int streamId) {
        return this.getAvailableSendWindow(streamId) > 0;
    }

    public boolean processPreface(ByteBuffer buf) throws Http2FrameReader.Http2Exception {
        if (this.state != State.AWAITING_PREFACE) {
            return true;
        }
        if (!Http2FrameReader.startsWithConnectionPreface(buf)) {
            if (buf.remaining() >= Http2FrameType.CONNECTION_PREFACE.length) {
                throw new Http2FrameReader.Http2Exception(1, "Invalid connection preface");
            }
            return false;
        }
        Http2FrameReader.consumeConnectionPreface(buf);
        this.state = State.AWAITING_SETTINGS;
        return true;
    }

    public List<Http2Frame> processFrame(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        ArrayList<Http2Frame> responses = new ArrayList<Http2Frame>();
        switch (frame.getType()) {
            case 4: {
                this.processSettings(frame, responses);
                break;
            }
            case 1: {
                this.processHeaders(frame);
                break;
            }
            case 9: {
                this.processContinuation(frame);
                break;
            }
            case 0: {
                this.processData(frame, responses);
                break;
            }
            case 6: {
                this.processPing(frame, responses);
                break;
            }
            case 7: {
                this.processGoaway(frame);
                break;
            }
            case 3: {
                this.processRstStream(frame);
                break;
            }
            case 8: {
                this.processWindowUpdate(frame);
                break;
            }
            case 2: {
                break;
            }
        }
        return responses;
    }

    private void processSettings(Http2Frame frame, List<Http2Frame> responses) throws Http2FrameReader.Http2Exception {
        if (frame.getStreamId() != 0) {
            throw new Http2FrameReader.Http2Exception(1, "SETTINGS frame on non-zero stream");
        }
        if (frame.isAck()) {
            if (frame.getLength() != 0) {
                throw new Http2FrameReader.Http2Exception(6, "SETTINGS ACK with payload");
            }
            return;
        }
        if (frame.getLength() % 6 != 0) {
            throw new Http2FrameReader.Http2Exception(6, "SETTINGS payload not multiple of 6");
        }
        Http2Settings newSettings = Http2Settings.decode(frame.getPayload());
        if (newSettings.getInitialWindowSize() < 0) {
            throw new Http2FrameReader.Http2Exception(3, "Invalid initial window size");
        }
        int windowDelta = newSettings.getInitialWindowSize() - this.remoteSettings.getInitialWindowSize();
        if (windowDelta != 0) {
            for (Http2Stream stream : this.streams.values()) {
                stream.updateSendWindow(windowDelta);
            }
        }
        this.remoteSettings.setHeaderTableSize(newSettings.getHeaderTableSize());
        this.remoteSettings.setEnablePush(newSettings.isEnablePush());
        this.remoteSettings.setMaxConcurrentStreams(newSettings.getMaxConcurrentStreams());
        this.remoteSettings.setInitialWindowSize(newSettings.getInitialWindowSize());
        this.remoteSettings.setMaxFrameSize(newSettings.getMaxFrameSize());
        this.remoteSettings.setMaxHeaderListSize(newSettings.getMaxHeaderListSize());
        responses.add(Http2Frame.settingsAck());
        if (this.state == State.AWAITING_SETTINGS) {
            this.state = State.OPEN;
        }
    }

    private void processHeaders(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        int streamId = frame.getStreamId();
        if (streamId == 0) {
            throw new Http2FrameReader.Http2Exception(1, "HEADERS on stream 0");
        }
        if (streamId % 2 == 0) {
            throw new Http2FrameReader.Http2Exception(1, "Client initiated even stream ID " + streamId);
        }
        if (streamId <= this.lastClientStreamId) {
            throw new Http2FrameReader.Http2Exception(1, streamId, "Stream ID not greater than previous");
        }
        if (this.getActiveStreams().size() >= this.localSettings.getMaxConcurrentStreams()) {
            throw new Http2FrameReader.Http2Exception(7, streamId, "Max concurrent streams exceeded");
        }
        this.lastClientStreamId = streamId;
        Http2Stream stream = new Http2Stream(streamId, this.localSettings.getInitialWindowSize());
        this.streams.put(streamId, stream);
        byte[] payload = frame.getPayload();
        int offset = 0;
        int length = payload.length;
        if (frame.isPadded()) {
            if (length < 1) {
                throw new Http2FrameReader.Http2Exception(1, streamId, "PADDED flag but no pad length");
            }
            int padLength = payload[0] & 0xFF;
            offset = 1;
            if ((length = length - 1 - padLength) < 0) {
                throw new Http2FrameReader.Http2Exception(1, streamId, "Pad length exceeds payload");
            }
        }
        if (frame.hasPriority()) {
            if (length < 5) {
                throw new Http2FrameReader.Http2Exception(6, streamId, "PRIORITY data missing");
            }
            offset += 5;
            length -= 5;
        }
        if (frame.isEndHeaders()) {
            this.processHeaderBlock(stream, payload, offset, length, frame.isEndStream());
        } else {
            this.headerStreamId = streamId;
            this.headerBuffer = ByteBuffer.allocate(this.localSettings.getMaxHeaderListSize());
            this.headerBuffer.put(payload, offset, length);
        }
    }

    private void processContinuation(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        int streamId = frame.getStreamId();
        if (streamId != this.headerStreamId) {
            throw new Http2FrameReader.Http2Exception(1, "CONTINUATION for unexpected stream");
        }
        Http2Stream stream = this.streams.get(streamId);
        if (stream == null) {
            throw new Http2FrameReader.Http2Exception(1, streamId, "CONTINUATION for unknown stream");
        }
        byte[] payload = frame.getPayload();
        if (this.headerBuffer.remaining() < payload.length) {
            throw new Http2FrameReader.Http2Exception(11, streamId, "Header block too large");
        }
        this.headerBuffer.put(payload);
        if (frame.isEndHeaders()) {
            this.headerBuffer.flip();
            byte[] headerBlock = new byte[this.headerBuffer.remaining()];
            this.headerBuffer.get(headerBlock);
            this.processHeaderBlock(stream, headerBlock, 0, headerBlock.length, false);
            this.headerStreamId = 0;
            this.headerBuffer = null;
        }
    }

    private void processHeaderBlock(Http2Stream stream, byte[] data, int offset, int length, boolean endStream) throws Http2FrameReader.Http2Exception {
        try {
            List<String[]> headers = this.hpackDecoder.decode(data, offset, length);
            stream.addRequestHeaders(headers);
            stream.recvHeaders(endStream);
            if (endStream) {
                stream.setRequestComplete(true);
                if (this.streamHandler != null) {
                    this.streamHandler.onRequest(stream);
                }
            } else if (this.streamHandler != null) {
                this.streamHandler.onHeadersComplete(stream);
            }
        }
        catch (HpackDecoder.HpackException e) {
            throw new Http2FrameReader.Http2Exception(9, stream.getStreamId(), "HPACK decoding error: " + e.getMessage());
        }
    }

    private void processData(Http2Frame frame, List<Http2Frame> responses) throws Http2FrameReader.Http2Exception {
        int streamConsumed;
        int streamId = frame.getStreamId();
        if (streamId == 0) {
            throw new Http2FrameReader.Http2Exception(1, "DATA on stream 0");
        }
        Http2Stream stream = this.streams.get(streamId);
        if (stream == null) {
            throw new Http2FrameReader.Http2Exception(5, streamId, "DATA for unknown stream");
        }
        byte[] payload = frame.getPayload();
        int dataLength = payload.length;
        int offset = 0;
        if (frame.isPadded()) {
            if (dataLength < 1) {
                throw new Http2FrameReader.Http2Exception(1, streamId, "PADDED flag but no pad length");
            }
            int padLength = payload[0] & 0xFF;
            offset = 1;
            if ((dataLength = dataLength - 1 - padLength) < 0) {
                throw new Http2FrameReader.Http2Exception(1, streamId, "Pad length exceeds payload");
            }
        }
        int consumed = frame.getLength();
        this.connectionRecvWindow.addAndGet(-consumed);
        stream.consumeRecvWindow(consumed);
        this.connectionRecvWindowConsumed.addAndGet(consumed);
        stream.addConsumedRecvWindow(consumed);
        int initialWindow = this.localSettings.getInitialWindowSize();
        int threshold = (int)((float)initialWindow * 0.5f);
        int connConsumed = this.connectionRecvWindowConsumed.get();
        if (connConsumed >= threshold) {
            this.connectionRecvWindow.addAndGet(connConsumed);
            this.connectionRecvWindowConsumed.set(0);
            responses.add(Http2Frame.windowUpdate(0, connConsumed));
        }
        if ((streamConsumed = stream.getConsumedRecvWindow()) >= threshold) {
            stream.updateRecvWindow(streamConsumed);
            stream.resetConsumedRecvWindow();
            responses.add(Http2Frame.windowUpdate(streamId, streamConsumed));
        }
        stream.appendRequestBody(payload, offset, dataLength);
        stream.recvData(frame.isEndStream());
        if (frame.isEndStream()) {
            stream.setRequestComplete(true);
            stream.markBodyComplete();
            if (this.streamHandler != null) {
                if (stream.isStreamingMode()) {
                    this.streamHandler.onBodyComplete(stream);
                } else {
                    this.streamHandler.onRequest(stream);
                }
            }
        }
    }

    private void processPing(Http2Frame frame, List<Http2Frame> responses) throws Http2FrameReader.Http2Exception {
        if (frame.getStreamId() != 0) {
            throw new Http2FrameReader.Http2Exception(1, "PING on non-zero stream");
        }
        if (frame.getLength() != 8) {
            throw new Http2FrameReader.Http2Exception(6, "PING payload must be 8 bytes");
        }
        if (!frame.isAck()) {
            responses.add(Http2Frame.pingAck(frame.getPayload()));
        }
    }

    private void processGoaway(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        if (frame.getStreamId() != 0) {
            throw new Http2FrameReader.Http2Exception(1, "GOAWAY on non-zero stream");
        }
        byte[] payload = frame.getPayload();
        if (payload.length < 8) {
            throw new Http2FrameReader.Http2Exception(6, "GOAWAY payload too short");
        }
        int lastStreamId = (payload[0] & 0x7F) << 24 | (payload[1] & 0xFF) << 16 | (payload[2] & 0xFF) << 8 | payload[3] & 0xFF;
        int errorCode = (payload[4] & 0xFF) << 24 | (payload[5] & 0xFF) << 16 | (payload[6] & 0xFF) << 8 | payload[7] & 0xFF;
        this.state = State.CLOSED;
    }

    private void processRstStream(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        int streamId = frame.getStreamId();
        if (streamId == 0) {
            throw new Http2FrameReader.Http2Exception(1, "RST_STREAM on stream 0");
        }
        if (frame.getLength() != 4) {
            throw new Http2FrameReader.Http2Exception(6, "RST_STREAM payload must be 4 bytes");
        }
        Http2Stream stream = this.streams.get(streamId);
        if (stream == null) {
            return;
        }
        byte[] payload = frame.getPayload();
        int errorCode = (payload[0] & 0xFF) << 24 | (payload[1] & 0xFF) << 16 | (payload[2] & 0xFF) << 8 | payload[3] & 0xFF;
        stream.reset(errorCode);
    }

    private void processWindowUpdate(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        boolean wasConnectionBlocked;
        if (frame.getLength() != 4) {
            throw new Http2FrameReader.Http2Exception(6, "WINDOW_UPDATE payload must be 4 bytes");
        }
        byte[] payload = frame.getPayload();
        int increment = (payload[0] & 0x7F) << 24 | (payload[1] & 0xFF) << 16 | (payload[2] & 0xFF) << 8 | payload[3] & 0xFF;
        if (increment == 0) {
            if (frame.getStreamId() == 0) {
                throw new Http2FrameReader.Http2Exception(1, "WINDOW_UPDATE increment 0 on connection");
            }
            throw new Http2FrameReader.Http2Exception(1, frame.getStreamId(), "WINDOW_UPDATE increment 0 on stream");
        }
        boolean bl = wasConnectionBlocked = this.connectionSendWindow.get() <= 0;
        if (frame.getStreamId() == 0) {
            long newWindow = (long)this.connectionSendWindow.get() + (long)increment;
            if (newWindow > Integer.MAX_VALUE) {
                throw new Http2FrameReader.Http2Exception(3, "Connection window overflow");
            }
            this.connectionSendWindow.set((int)newWindow);
            if (wasConnectionBlocked && this.streamHandler != null) {
                for (Http2Stream s : this.streams.values()) {
                    if (!s.isOpen() || s.getSendWindow() <= 0) continue;
                    this.streamHandler.onWindowAvailable(s.getStreamId());
                }
            }
        } else {
            Http2Stream stream = this.streams.get(frame.getStreamId());
            if (stream != null) {
                boolean wasStreamBlocked = stream.getSendWindow() <= 0;
                long newWindow = (long)stream.getSendWindow() + (long)increment;
                if (newWindow > Integer.MAX_VALUE) {
                    throw new Http2FrameReader.Http2Exception(3, frame.getStreamId(), "Stream window overflow");
                }
                stream.updateSendWindow(increment);
                if (wasStreamBlocked && this.connectionSendWindow.get() > 0 && this.streamHandler != null) {
                    this.streamHandler.onWindowAvailable(stream.getStreamId());
                }
            }
        }
    }

    public void cleanup() {
        this.streams.entrySet().removeIf(e -> ((Http2Stream)e.getValue()).isClosed());
    }

    public Http2Stream createUpgradeStream(List<String[]> headers, boolean hasBody) {
        Http2Stream stream = new Http2Stream(1, this.localSettings.getInitialWindowSize());
        stream.addRequestHeaders(headers);
        stream.setRequestComplete(!hasBody);
        this.streams.put(1, stream);
        this.lastClientStreamId = 1;
        return stream;
    }

    public Http2Stream getUpgradeStream() {
        return this.streams.get(1);
    }

    public Http2FrameReader getFrameReader() {
        return this.frameReader;
    }

    public Http2FrameWriter getFrameWriter() {
        return this.frameWriter;
    }

    public Http2Settings getLocalSettings() {
        return this.localSettings;
    }

    public Http2Settings getRemoteSettings() {
        return this.remoteSettings;
    }

    public List<Integer> getStreamsWithAvailableWindow() {
        ArrayList<Integer> ready = new ArrayList<Integer>();
        int connWindow = this.connectionSendWindow.get();
        if (connWindow > 0) {
            for (Http2Stream stream : this.streams.values()) {
                if (!stream.isOpen() || stream.getSendWindow() <= 0) continue;
                ready.add(stream.getStreamId());
            }
        }
        return ready;
    }

    public boolean hasStreamsWaitingForWindow() {
        if (this.connectionSendWindow.get() <= 0) {
            return true;
        }
        for (Http2Stream stream : this.streams.values()) {
            if (!stream.isOpen() || stream.getSendWindow() > 0) continue;
            return true;
        }
        return false;
    }

    public static enum State {
        AWAITING_PREFACE,
        AWAITING_SETTINGS,
        OPEN,
        GOAWAY_SENT,
        CLOSED;

    }

    public static interface StreamHandler {
        public void onRequest(Http2Stream var1);

        default public boolean onHeadersComplete(Http2Stream stream) {
            return false;
        }

        default public void onBodyComplete(Http2Stream stream) {
        }

        default public void onWindowAvailable(int streamId) {
        }
    }
}

