/*
 * Decompiled with CFR 0.152.
 */
package zeph.http2;

import java.io.ByteArrayOutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import zeph.client.HttpClientRequest;
import zeph.client.HttpClientResponse;
import zeph.http2.Http2Frame;
import zeph.http2.Http2FrameReader;
import zeph.http2.Http2FrameType;
import zeph.http2.Http2FrameWriter;
import zeph.http2.Http2Settings;
import zeph.http2.Http2Stream;
import zeph.http2.hpack.HpackDecoder;

public class Http2ClientHandler {
    private State state = State.INIT;
    private final Http2Settings localSettings;
    private final Http2Settings remoteSettings;
    private int nextStreamId = 1;
    private final Map<Integer, ClientStream> streams = new ConcurrentHashMap<Integer, ClientStream>();
    private int connectionSendWindow = 65535;
    private int connectionRecvWindow = 65535;
    private final Http2FrameReader frameReader;
    private final Http2FrameWriter frameWriter;
    private final HpackDecoder hpackDecoder;
    private int headerStreamId;
    private ByteBuffer headerBuffer;
    private ByteBuffer pendingFrameData;
    private ByteBuffer outputBuffer;
    private StreamCompleteListener streamCompleteListener;

    public Http2ClientHandler() {
        this(new Http2Settings());
    }

    public Http2ClientHandler(Http2Settings localSettings) {
        this.localSettings = localSettings;
        this.remoteSettings = new Http2Settings();
        this.frameReader = new Http2FrameReader(localSettings.getMaxFrameSize());
        this.frameWriter = new Http2FrameWriter(this.remoteSettings.getMaxFrameSize());
        this.hpackDecoder = new HpackDecoder(localSettings.getHeaderTableSize(), localSettings.getMaxHeaderListSize());
        this.outputBuffer = ByteBuffer.allocate(65536);
    }

    public void setStreamCompleteListener(StreamCompleteListener listener) {
        this.streamCompleteListener = listener;
    }

    public State getState() {
        return this.state;
    }

    public boolean isOpen() {
        return this.state == State.OPEN;
    }

    public boolean canAcceptRequest() {
        return (this.state == State.OPEN || this.state == State.AWAITING_SETTINGS) && this.streams.size() < this.remoteSettings.getMaxConcurrentStreams();
    }

    public int getActiveStreamCount() {
        return this.streams.size();
    }

    public byte[] getConnectionPreface() {
        this.outputBuffer.clear();
        this.outputBuffer.put(Http2FrameType.CONNECTION_PREFACE);
        this.frameWriter.writeSettings(this.outputBuffer, this.localSettings);
        int desiredWindow = this.localSettings.getInitialWindowSize();
        if (desiredWindow > 65535) {
            int increment = desiredWindow - 65535;
            this.frameWriter.writeWindowUpdate(this.outputBuffer, 0, increment);
            this.connectionRecvWindow = desiredWindow;
        }
        this.state = State.AWAITING_SETTINGS;
        this.outputBuffer.flip();
        byte[] result = new byte[this.outputBuffer.remaining()];
        this.outputBuffer.get(result);
        return result;
    }

    public int createStream(CompletableFuture<HttpClientResponse> future, HttpClientRequest request) {
        return this.createStream(future, request, 0);
    }

    public int createStream(CompletableFuture<HttpClientResponse> future, HttpClientRequest request, int redirectCount) {
        if (!this.canAcceptRequest()) {
            return -1;
        }
        int streamId = this.nextStreamId;
        this.nextStreamId += 2;
        ClientStream stream = new ClientStream(streamId, future, request, this.remoteSettings.getInitialWindowSize(), this.localSettings.getInitialWindowSize(), redirectCount);
        this.streams.put(streamId, stream);
        return streamId;
    }

    public List<String[]> buildHttp2Headers(HttpClientRequest request, boolean includeContentLength, long contentLength) {
        boolean hasUserAgent;
        ArrayList<String[]> headers = new ArrayList<String[]>();
        headers.add(new String[]{":method", request.getMethod()});
        headers.add(new String[]{":path", request.getRequestUri()});
        headers.add(new String[]{":scheme", request.isSecure() ? "https" : "http"});
        headers.add(new String[]{":authority", request.getHost() + (String)(this.needsPortInAuthority(request) ? ":" + request.getPort() : "")});
        Map<String, String> reqHeaders = request.getHeaders();
        if (reqHeaders != null) {
            for (Map.Entry<String, String> entry : reqHeaders.entrySet()) {
                String name = entry.getKey().toLowerCase();
                if (name.equals("connection") || name.equals("transfer-encoding") || name.equals("host") || name.equals("upgrade")) continue;
                headers.add(new String[]{name, entry.getValue()});
            }
        }
        boolean bl = hasUserAgent = reqHeaders != null && reqHeaders.keySet().stream().anyMatch(k -> k.equalsIgnoreCase("user-agent"));
        if (!hasUserAgent) {
            headers.add(new String[]{"user-agent", "zeph/1.0"});
        }
        if (includeContentLength) {
            boolean hasContentLength;
            boolean bl2 = hasContentLength = reqHeaders != null && reqHeaders.keySet().stream().anyMatch(k -> k.equalsIgnoreCase("content-length"));
            if (!hasContentLength) {
                headers.add(new String[]{"content-length", String.valueOf(contentLength)});
            }
        }
        return headers;
    }

    public byte[] buildRequestFrames(int streamId) {
        ClientStream stream = this.streams.get(streamId);
        if (stream == null) {
            return null;
        }
        HttpClientRequest request = stream.getRequest();
        byte[] body = request.getBody();
        boolean hasBody = body != null && body.length > 0;
        int maxFrameSize = this.remoteSettings.getMaxFrameSize();
        int frameCount = hasBody ? body.length / maxFrameSize + 1 : 0;
        int estimatedSize = 8192 + (hasBody ? body.length + frameCount * 9 : 0);
        ByteBuffer buf = estimatedSize > this.outputBuffer.capacity() ? ByteBuffer.allocate(estimatedSize) : this.outputBuffer;
        buf.clear();
        List<String[]> headers = this.buildHttp2Headers(request, hasBody, hasBody ? (long)body.length : 0L);
        this.frameWriter.writeHeaders(buf, streamId, headers, !hasBody);
        stream.setState(hasBody ? Http2Stream.State.OPEN : Http2Stream.State.HALF_CLOSED_LOCAL);
        if (hasBody) {
            this.frameWriter.writeData(buf, streamId, body, true);
            stream.setState(Http2Stream.State.HALF_CLOSED_LOCAL);
            this.connectionSendWindow -= body.length;
            stream.consumeSendWindow(body.length);
        }
        buf.flip();
        byte[] result = new byte[buf.remaining()];
        buf.get(result);
        return result;
    }

    public byte[] buildHeadersOnlyFrames(int streamId, long contentLength) {
        ClientStream stream = this.streams.get(streamId);
        if (stream == null) {
            return null;
        }
        HttpClientRequest request = stream.getRequest();
        ByteBuffer buf = ByteBuffer.allocate(8192);
        buf.clear();
        List<String[]> headers = this.buildHttp2Headers(request, contentLength > 0L, contentLength);
        this.frameWriter.writeHeaders(buf, streamId, headers, false);
        stream.setState(Http2Stream.State.OPEN);
        buf.flip();
        byte[] result = new byte[buf.remaining()];
        buf.get(result);
        return result;
    }

    public int getAvailableSendWindow(int streamId) {
        ClientStream stream = this.streams.get(streamId);
        if (stream == null) {
            return 0;
        }
        return Math.min(this.connectionSendWindow, stream.getSendWindow());
    }

    public int getConnectionSendWindow() {
        return this.connectionSendWindow;
    }

    public byte[] buildDataFrame(int streamId, byte[] data, boolean endStream) {
        ClientStream stream = this.streams.get(streamId);
        if (stream == null) {
            return null;
        }
        int maxFrameSize = this.remoteSettings.getMaxFrameSize();
        int frameCount = data.length > 0 ? data.length / maxFrameSize + 1 : 1;
        ByteBuffer buf = ByteBuffer.allocate(data.length + frameCount * 9 + 9);
        buf.clear();
        this.frameWriter.writeData(buf, streamId, data, endStream);
        if (endStream) {
            stream.setState(Http2Stream.State.HALF_CLOSED_LOCAL);
        }
        this.connectionSendWindow -= data.length;
        stream.consumeSendWindow(data.length);
        buf.flip();
        byte[] result = new byte[buf.remaining()];
        buf.get(result);
        return result;
    }

    private boolean needsPortInAuthority(HttpClientRequest request) {
        int port = request.getPort();
        if (request.isSecure()) {
            return port != 443;
        }
        return port != 80;
    }

    public byte[] processData(byte[] data) throws Http2FrameReader.Http2Exception {
        ByteBuffer input;
        if (this.pendingFrameData != null && this.pendingFrameData.hasRemaining()) {
            int totalSize = this.pendingFrameData.remaining() + data.length;
            input = ByteBuffer.allocate(totalSize);
            input.put(this.pendingFrameData);
            input.put(data);
            input.flip();
            this.pendingFrameData = null;
        } else {
            input = ByteBuffer.wrap(data);
        }
        this.outputBuffer.clear();
        while (input.hasRemaining()) {
            Http2Frame frame = this.frameReader.readFrame(input);
            if (frame == null) {
                if (!input.hasRemaining()) break;
                this.pendingFrameData = ByteBuffer.allocate(input.remaining());
                this.pendingFrameData.put(input);
                this.pendingFrameData.flip();
                break;
            }
            this.processFrame(frame);
        }
        this.outputBuffer.flip();
        if (this.outputBuffer.remaining() == 0) {
            return new byte[0];
        }
        byte[] result = new byte[this.outputBuffer.remaining()];
        this.outputBuffer.get(result);
        return result;
    }

    private void processFrame(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        switch (frame.getType()) {
            case 4: {
                this.processSettings(frame);
                break;
            }
            case 1: {
                this.processHeaders(frame);
                break;
            }
            case 9: {
                this.processContinuation(frame);
                break;
            }
            case 0: {
                this.processData(frame);
                break;
            }
            case 6: {
                this.processPing(frame);
                break;
            }
            case 7: {
                this.processGoaway(frame);
                break;
            }
            case 3: {
                this.processRstStream(frame);
                break;
            }
            case 8: {
                this.processWindowUpdate(frame);
                break;
            }
            case 2: {
                break;
            }
        }
    }

    private void processSettings(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        if (frame.getStreamId() != 0) {
            throw new Http2FrameReader.Http2Exception(1, "SETTINGS on non-zero stream");
        }
        if (frame.isAck()) {
            if (frame.getLength() != 0) {
                throw new Http2FrameReader.Http2Exception(6, "SETTINGS ACK with payload");
            }
            return;
        }
        if (frame.getLength() % 6 != 0) {
            throw new Http2FrameReader.Http2Exception(6, "SETTINGS payload not multiple of 6");
        }
        Http2Settings newSettings = Http2Settings.decode(frame.getPayload());
        int windowDelta = newSettings.getInitialWindowSize() - this.remoteSettings.getInitialWindowSize();
        if (windowDelta != 0) {
            for (ClientStream stream : this.streams.values()) {
                stream.updateSendWindow(windowDelta);
            }
        }
        this.remoteSettings.setHeaderTableSize(newSettings.getHeaderTableSize());
        this.remoteSettings.setEnablePush(newSettings.isEnablePush());
        this.remoteSettings.setMaxConcurrentStreams(newSettings.getMaxConcurrentStreams());
        this.remoteSettings.setInitialWindowSize(newSettings.getInitialWindowSize());
        this.remoteSettings.setMaxFrameSize(newSettings.getMaxFrameSize());
        this.remoteSettings.setMaxHeaderListSize(newSettings.getMaxHeaderListSize());
        this.frameWriter.setMaxFrameSize(newSettings.getMaxFrameSize());
        this.frameWriter.writeSettingsAck(this.outputBuffer);
        if (this.state == State.AWAITING_SETTINGS) {
            this.state = State.OPEN;
        }
    }

    private void processHeaders(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        int streamId = frame.getStreamId();
        if (streamId == 0) {
            throw new Http2FrameReader.Http2Exception(1, "HEADERS on stream 0");
        }
        ClientStream stream = this.streams.get(streamId);
        if (stream == null) {
            throw new Http2FrameReader.Http2Exception(1, streamId, "HEADERS for unknown stream");
        }
        byte[] payload = frame.getPayload();
        int offset = 0;
        int length = payload.length;
        if (frame.isPadded()) {
            if (length < 1) {
                throw new Http2FrameReader.Http2Exception(1, streamId, "PADDED flag but no pad length");
            }
            int padLength = payload[0] & 0xFF;
            offset = 1;
            if ((length = length - 1 - padLength) < 0) {
                throw new Http2FrameReader.Http2Exception(1, streamId, "Pad length exceeds payload");
            }
        }
        if (frame.hasPriority()) {
            if (length < 5) {
                throw new Http2FrameReader.Http2Exception(6, streamId, "PRIORITY data missing");
            }
            offset += 5;
            length -= 5;
        }
        if (frame.isEndHeaders()) {
            this.processHeaderBlock(stream, payload, offset, length, frame.isEndStream());
        } else {
            this.headerStreamId = streamId;
            this.headerBuffer = ByteBuffer.allocate(this.localSettings.getMaxHeaderListSize());
            this.headerBuffer.put(payload, offset, length);
        }
    }

    private void processContinuation(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        int streamId = frame.getStreamId();
        if (streamId != this.headerStreamId) {
            throw new Http2FrameReader.Http2Exception(1, "CONTINUATION for unexpected stream");
        }
        ClientStream stream = this.streams.get(streamId);
        if (stream == null) {
            throw new Http2FrameReader.Http2Exception(1, streamId, "CONTINUATION for unknown stream");
        }
        byte[] payload = frame.getPayload();
        if (this.headerBuffer.remaining() < payload.length) {
            throw new Http2FrameReader.Http2Exception(11, streamId, "Header block too large");
        }
        this.headerBuffer.put(payload);
        if (frame.isEndHeaders()) {
            this.headerBuffer.flip();
            byte[] headerBlock = new byte[this.headerBuffer.remaining()];
            this.headerBuffer.get(headerBlock);
            this.processHeaderBlock(stream, headerBlock, 0, headerBlock.length, false);
            this.headerStreamId = 0;
            this.headerBuffer = null;
        }
    }

    private void processHeaderBlock(ClientStream stream, byte[] data, int offset, int length, boolean endStream) throws Http2FrameReader.Http2Exception {
        try {
            List<String[]> headers = this.hpackDecoder.decode(data, offset, length);
            for (String[] header : headers) {
                String name = header[0];
                String value = header[1];
                if (name.equals(":status")) {
                    stream.setStatusCode(Integer.parseInt(value));
                    continue;
                }
                if (name.startsWith(":")) continue;
                stream.getResponseHeaders().put(name, value);
            }
            if (stream.getState() == Http2Stream.State.HALF_CLOSED_LOCAL) {
                if (endStream) {
                    this.completeStream(stream);
                }
            } else if (stream.getState() == Http2Stream.State.OPEN && endStream) {
                stream.setState(Http2Stream.State.HALF_CLOSED_REMOTE);
            }
        }
        catch (HpackDecoder.HpackException e) {
            throw new Http2FrameReader.Http2Exception(9, stream.getStreamId(), "HPACK decoding error: " + e.getMessage());
        }
    }

    private void processData(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        int increment;
        int streamId = frame.getStreamId();
        if (streamId == 0) {
            throw new Http2FrameReader.Http2Exception(1, "DATA on stream 0");
        }
        ClientStream stream = this.streams.get(streamId);
        if (stream == null) {
            throw new Http2FrameReader.Http2Exception(5, streamId, "DATA for unknown stream");
        }
        byte[] payload = frame.getPayload();
        int offset = 0;
        int dataLength = payload.length;
        if (frame.isPadded()) {
            if (dataLength < 1) {
                throw new Http2FrameReader.Http2Exception(1, streamId, "PADDED flag but no pad length");
            }
            int padLength = payload[0] & 0xFF;
            offset = 1;
            if ((dataLength = dataLength - 1 - padLength) < 0) {
                throw new Http2FrameReader.Http2Exception(1, streamId, "Pad length exceeds payload");
            }
        }
        this.connectionRecvWindow -= frame.getLength();
        stream.consumeRecvWindow(frame.getLength());
        int windowSize = this.localSettings.getInitialWindowSize();
        if (this.connectionRecvWindow < windowSize / 2) {
            increment = windowSize - this.connectionRecvWindow;
            this.frameWriter.writeWindowUpdate(this.outputBuffer, 0, increment);
            this.connectionRecvWindow += increment;
        }
        if (stream.getRecvWindow() < windowSize / 2) {
            increment = windowSize - stream.getRecvWindow();
            this.frameWriter.writeWindowUpdate(this.outputBuffer, streamId, increment);
            stream.updateRecvWindow(increment);
        }
        stream.appendBody(payload, offset, dataLength);
        if (frame.isEndStream()) {
            this.completeStream(stream);
        }
    }

    private void processPing(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        if (frame.getStreamId() != 0) {
            throw new Http2FrameReader.Http2Exception(1, "PING on non-zero stream");
        }
        if (frame.getLength() != 8) {
            throw new Http2FrameReader.Http2Exception(6, "PING payload must be 8 bytes");
        }
        if (!frame.isAck()) {
            this.frameWriter.writePingAck(this.outputBuffer, frame.getPayload());
        }
    }

    private void processGoaway(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        if (frame.getStreamId() != 0) {
            throw new Http2FrameReader.Http2Exception(1, "GOAWAY on non-zero stream");
        }
        byte[] payload = frame.getPayload();
        if (payload.length < 8) {
            throw new Http2FrameReader.Http2Exception(6, "GOAWAY payload too short");
        }
        int lastStreamId = (payload[0] & 0x7F) << 24 | (payload[1] & 0xFF) << 16 | (payload[2] & 0xFF) << 8 | payload[3] & 0xFF;
        int errorCode = (payload[4] & 0xFF) << 24 | (payload[5] & 0xFF) << 16 | (payload[6] & 0xFF) << 8 | payload[7] & 0xFF;
        this.state = State.GOAWAY_RECEIVED;
        for (ClientStream stream : this.streams.values()) {
            if (stream.getStreamId() <= lastStreamId) continue;
            stream.getFuture().completeExceptionally(new Exception("Stream rejected by GOAWAY, error=" + errorCode));
        }
    }

    private void processRstStream(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        int streamId = frame.getStreamId();
        if (streamId == 0) {
            throw new Http2FrameReader.Http2Exception(1, "RST_STREAM on stream 0");
        }
        if (frame.getLength() != 4) {
            throw new Http2FrameReader.Http2Exception(6, "RST_STREAM payload must be 4 bytes");
        }
        ClientStream stream = this.streams.get(streamId);
        if (stream == null) {
            return;
        }
        byte[] payload = frame.getPayload();
        int errorCode = (payload[0] & 0xFF) << 24 | (payload[1] & 0xFF) << 16 | (payload[2] & 0xFF) << 8 | payload[3] & 0xFF;
        stream.setState(Http2Stream.State.CLOSED);
        this.streams.remove(streamId);
        stream.getFuture().completeExceptionally(new Exception("Stream reset by server, error=" + errorCode));
    }

    private void processWindowUpdate(Http2Frame frame) throws Http2FrameReader.Http2Exception {
        if (frame.getLength() != 4) {
            throw new Http2FrameReader.Http2Exception(6, "WINDOW_UPDATE payload must be 4 bytes");
        }
        byte[] payload = frame.getPayload();
        int increment = (payload[0] & 0x7F) << 24 | (payload[1] & 0xFF) << 16 | (payload[2] & 0xFF) << 8 | payload[3] & 0xFF;
        if (increment == 0) {
            if (frame.getStreamId() == 0) {
                throw new Http2FrameReader.Http2Exception(1, "WINDOW_UPDATE increment 0 on connection");
            }
            throw new Http2FrameReader.Http2Exception(1, frame.getStreamId(), "WINDOW_UPDATE increment 0 on stream");
        }
        if (frame.getStreamId() == 0) {
            long newWindow = (long)this.connectionSendWindow + (long)increment;
            if (newWindow > Integer.MAX_VALUE) {
                throw new Http2FrameReader.Http2Exception(3, "Connection window overflow");
            }
            this.connectionSendWindow = (int)newWindow;
        } else {
            ClientStream stream = this.streams.get(frame.getStreamId());
            if (stream != null) {
                long newWindow = (long)stream.getSendWindow() + (long)increment;
                if (newWindow > Integer.MAX_VALUE) {
                    throw new Http2FrameReader.Http2Exception(3, frame.getStreamId(), "Stream window overflow");
                }
                stream.updateSendWindow(increment);
            }
        }
    }

    private void completeStream(ClientStream stream) {
        stream.setState(Http2Stream.State.CLOSED);
        this.streams.remove(stream.getStreamId());
        HttpClientResponse response = new HttpClientResponse(stream.getStatusCode(), "", stream.getResponseHeaders(), stream.getBody(), stream.getOpts(), "HTTP/2");
        if (this.streamCompleteListener != null) {
            this.streamCompleteListener.onStreamComplete(stream, response);
        } else {
            stream.getFuture().complete(response);
        }
    }

    public byte[] cancelStream(int streamId) {
        ClientStream stream = this.streams.remove(streamId);
        if (stream == null) {
            return null;
        }
        this.outputBuffer.clear();
        this.frameWriter.writeRstStream(this.outputBuffer, streamId, 8);
        this.outputBuffer.flip();
        byte[] result = new byte[this.outputBuffer.remaining()];
        this.outputBuffer.get(result);
        return result;
    }

    public byte[] close() {
        this.outputBuffer.clear();
        this.frameWriter.writeGoaway(this.outputBuffer, 0, 0, null);
        this.state = State.CLOSED;
        this.outputBuffer.flip();
        byte[] result = new byte[this.outputBuffer.remaining()];
        this.outputBuffer.get(result);
        return result;
    }

    public void failAllStreams(Exception e) {
        for (ClientStream stream : this.streams.values()) {
            stream.getFuture().completeExceptionally(e);
        }
        this.streams.clear();
    }

    public static enum State {
        INIT,
        AWAITING_SETTINGS,
        OPEN,
        GOAWAY_RECEIVED,
        CLOSED;

    }

    public static interface StreamCompleteListener {
        public void onStreamComplete(ClientStream var1, HttpClientResponse var2);
    }

    public static class ClientStream {
        private final int streamId;
        private final CompletableFuture<HttpClientResponse> future;
        private final HttpClientRequest request;
        private final Map<String, Object> opts;
        private final long startTime;
        private Http2Stream.State state = Http2Stream.State.IDLE;
        private int sendWindow;
        private int recvWindow;
        private int redirectCount;
        private int statusCode;
        private final Map<String, String> responseHeaders = new HashMap<String, String>();
        private final ByteArrayOutputStream bodyBuffer = new ByteArrayOutputStream();

        public ClientStream(int streamId, CompletableFuture<HttpClientResponse> future, HttpClientRequest request, int sendWindow, int recvWindow) {
            this(streamId, future, request, sendWindow, recvWindow, 0);
        }

        public ClientStream(int streamId, CompletableFuture<HttpClientResponse> future, HttpClientRequest request, int sendWindow, int recvWindow, int redirectCount) {
            this.streamId = streamId;
            this.future = future;
            this.request = request;
            this.opts = request.getOpts();
            this.startTime = System.currentTimeMillis();
            this.sendWindow = sendWindow;
            this.recvWindow = recvWindow;
            this.redirectCount = redirectCount;
        }

        public int getStreamId() {
            return this.streamId;
        }

        public CompletableFuture<HttpClientResponse> getFuture() {
            return this.future;
        }

        public HttpClientRequest getRequest() {
            return this.request;
        }

        public long getStartTime() {
            return this.startTime;
        }

        public Http2Stream.State getState() {
            return this.state;
        }

        public void setState(Http2Stream.State state) {
            this.state = state;
        }

        public int getStatusCode() {
            return this.statusCode;
        }

        public void setStatusCode(int statusCode) {
            this.statusCode = statusCode;
        }

        public Map<String, String> getResponseHeaders() {
            return this.responseHeaders;
        }

        public int getRedirectCount() {
            return this.redirectCount;
        }

        public void incrementRedirectCount() {
            ++this.redirectCount;
        }

        public void appendBody(byte[] data, int offset, int length) {
            this.bodyBuffer.write(data, offset, length);
        }

        public byte[] getBody() {
            return this.bodyBuffer.toByteArray();
        }

        public void consumeSendWindow(int amount) {
            this.sendWindow -= amount;
        }

        public void updateSendWindow(int delta) {
            this.sendWindow += delta;
        }

        public int getSendWindow() {
            return this.sendWindow;
        }

        public void consumeRecvWindow(int amount) {
            this.recvWindow -= amount;
        }

        public void updateRecvWindow(int delta) {
            this.recvWindow += delta;
        }

        public int getRecvWindow() {
            return this.recvWindow;
        }

        public Map<String, Object> getOpts() {
            return this.opts;
        }
    }
}

