/*
 * Decompiled with CFR 0.152.
 */
package zeph.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.InetSocketAddress;
import java.net.SocketOption;
import java.net.StandardSocketOptions;
import java.nio.ByteBuffer;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import zeph.http.H2cUpgradeHelper;
import zeph.http.Http1Handler;
import zeph.http.HttpRequest;
import zeph.http.HttpResponse;
import zeph.http.HttpServer;
import zeph.http.ProtocolDetector;
import zeph.http2.Http2FrameReader;
import zeph.http2.Http2ServerHandler;
import zeph.logging.Logger;

public class HttpServerNio
implements HttpServer {
    private static final int READ_BUFFER_SIZE = 262144;
    private static final int WRITE_BUFFER_SIZE = 262144;
    private static final int SELECT_TIMEOUT_MS = 100;
    private static final int DEFAULT_IDLE_TIMEOUT_MS = 30000;
    private static final Logger log = new Logger(HttpServerNio.class);
    private final String host;
    private final int port;
    private final int numWorkers;
    private final Function<HttpRequest, HttpResponse> handler;
    private final boolean streamingMode;
    private final long maxBodySize;
    private final int idleTimeoutMs;
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final AtomicBoolean draining = new AtomicBoolean(false);
    private final AtomicInteger nextWorker = new AtomicInteger(0);
    private final AtomicLong connectionIdGenerator = new AtomicLong(0L);
    private ServerSocketChannel serverChannel;
    private Selector acceptorSelector;
    private Thread acceptorThread;
    private NioWorker[] workers;
    private final ExecutorService handlerExecutor;

    public HttpServerNio(String host, int port, int numWorkers, Function<HttpRequest, HttpResponse> handler) throws IOException {
        this(host, port, numWorkers, handler, false, false, -1L, 30000);
    }

    public HttpServerNio(String host, int port, int numWorkers, Function<HttpRequest, HttpResponse> handler, boolean streamingMode) throws IOException {
        this(host, port, numWorkers, handler, streamingMode, false, -1L, 30000);
    }

    public HttpServerNio(String host, int port, int numWorkers, Function<HttpRequest, HttpResponse> handler, boolean streamingMode, boolean reusePort) throws IOException {
        this(host, port, numWorkers, handler, streamingMode, reusePort, -1L, 30000);
    }

    public HttpServerNio(String host, int port, int numWorkers, Function<HttpRequest, HttpResponse> handler, boolean streamingMode, boolean reusePort, long maxBodySize) throws IOException {
        this(host, port, numWorkers, handler, streamingMode, reusePort, maxBodySize, 30000);
    }

    public HttpServerNio(String host, int port, int numWorkers, Function<HttpRequest, HttpResponse> handler, boolean streamingMode, boolean reusePort, long maxBodySize, int idleTimeoutMs) throws IOException {
        this.host = host;
        this.port = port;
        this.numWorkers = numWorkers;
        this.handler = handler;
        this.streamingMode = streamingMode;
        this.maxBodySize = maxBodySize;
        this.idleTimeoutMs = idleTimeoutMs;
        this.handlerExecutor = streamingMode ? Executors.newFixedThreadPool(numWorkers * 2, r -> {
            Thread t = new Thread(r, "zeph-nio-handler");
            t.setDaemon(true);
            return t;
        }) : null;
        this.workers = new NioWorker[numWorkers];
        for (int i = 0; i < numWorkers; ++i) {
            this.workers[i] = new NioWorker(i);
        }
        this.serverChannel = ServerSocketChannel.open();
        this.serverChannel.configureBlocking(false);
        this.serverChannel.setOption((SocketOption)StandardSocketOptions.SO_REUSEADDR, (Object)true);
        if (reusePort) {
            try {
                this.serverChannel.setOption((SocketOption)StandardSocketOptions.SO_REUSEPORT, (Object)true);
            }
            catch (UnsupportedOperationException unsupportedOperationException) {
                // empty catch block
            }
        }
        this.serverChannel.bind(new InetSocketAddress(host, port));
        this.acceptorSelector = Selector.open();
        this.serverChannel.register(this.acceptorSelector, 16);
        this.acceptorThread = new Thread(this::acceptLoop, "zeph-nio-acceptor");
        log.info("HTTP Server (NIO) listening on " + host + ":" + port + " with " + numWorkers + " workers" + (streamingMode ? " (streaming mode)" : ""));
    }

    private void acceptLoop() {
        while (this.running.get()) {
            try {
                int ready = this.acceptorSelector.select(100L);
                if (ready <= 0) continue;
                Set<SelectionKey> selectedKeys = this.acceptorSelector.selectedKeys();
                Iterator<SelectionKey> iter = selectedKeys.iterator();
                while (iter.hasNext()) {
                    SocketChannel clientChannel;
                    SelectionKey key = iter.next();
                    iter.remove();
                    if (!key.isAcceptable() || (clientChannel = this.serverChannel.accept()) == null) continue;
                    long connId = this.connectionIdGenerator.incrementAndGet();
                    int workerIdx = Math.abs(this.nextWorker.getAndIncrement() % this.numWorkers);
                    this.workers[workerIdx].addConnection(clientChannel, connId);
                }
            }
            catch (ClosedSelectorException e) {
                break;
            }
            catch (Exception e) {
                if (!this.running.get()) continue;
                log.error("NIO Acceptor error: " + e.getMessage(), e);
            }
        }
    }

    @Override
    public void run() {
        this.running.set(true);
        for (NioWorker worker : this.workers) {
            worker.start();
        }
        this.acceptorThread.start();
        log.info("HTTP Server (NIO) started with " + this.numWorkers + " workers");
        try {
            this.acceptorThread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    @Override
    public void stop() {
        this.running.set(false);
        this.acceptorSelector.wakeup();
        for (NioWorker worker : this.workers) {
            worker.stop();
        }
        try {
            this.serverChannel.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Override
    public boolean gracefulShutdown(long timeoutMs) {
        log.info("Initiating graceful shutdown (timeout: " + timeoutMs + "ms)");
        this.draining.set(true);
        try {
            this.serverChannel.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        long deadline = System.currentTimeMillis() + timeoutMs;
        int lastCount = -1;
        while (System.currentTimeMillis() < deadline) {
            int activeCount = this.getActiveConnectionCount();
            if (activeCount == 0) {
                log.info("Graceful shutdown complete - all connections drained");
                this.stop();
                return true;
            }
            if (activeCount != lastCount) {
                log.info("Graceful shutdown: " + activeCount + " connections remaining");
                lastCount = activeCount;
            }
            for (NioWorker worker : this.workers) {
                worker.selector.wakeup();
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
        int remaining = this.getActiveConnectionCount();
        log.warn("Graceful shutdown timeout - " + remaining + " connections forcibly closed");
        this.stop();
        return false;
    }

    @Override
    public int getActiveConnectionCount() {
        int count = 0;
        for (NioWorker worker : this.workers) {
            count += worker.connections.size();
        }
        return count;
    }

    @Override
    public boolean isDraining() {
        return this.draining.get();
    }

    @Override
    public void close() {
        this.stop();
        try {
            this.acceptorThread.join(2000L);
            for (NioWorker worker : this.workers) {
                worker.thread.interrupt();
                worker.thread.join(1000L);
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        for (NioWorker worker : this.workers) {
            worker.close();
        }
        try {
            this.acceptorSelector.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        if (this.handlerExecutor != null) {
            this.handlerExecutor.shutdown();
            try {
                if (!this.handlerExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.handlerExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.handlerExecutor.shutdownNow();
            }
        }
    }

    @Override
    public int getPort() {
        return this.port;
    }

    @Override
    public String getHost() {
        return this.host;
    }

    @Override
    public boolean isRunning() {
        return this.running.get();
    }

    public static void main(String[] args) {
        int port = 8080;
        int workers = Runtime.getRuntime().availableProcessors();
        if (args.length > 0) {
            port = Integer.parseInt(args[0]);
        }
        if (args.length > 1) {
            workers = Integer.parseInt(args[1]);
        }
        log.info("Starting NIO HTTP Server");
        log.info("Port: " + port + ", Workers: " + workers);
        try {
            HttpServerNio server = new HttpServerNio("0.0.0.0", port, workers, request -> {
                String path;
                return switch (path = request.getPath()) {
                    case "/", "/hello" -> HttpResponse.ok("Hello from Zeph NIO!");
                    case "/json" -> HttpResponse.json("{\"message\": \"Hello!\"}");
                    default -> HttpResponse.notFound();
                };
            });
            Runtime.getRuntime().addShutdownHook(new Thread(() -> {
                log.info("Shutting down...");
                server.stop();
                server.close();
            }));
            server.run();
        }
        catch (Exception e) {
            log.error("Error: " + e.getMessage(), e);
        }
    }

    private class NioWorker
    implements Runnable {
        final int id;
        final Selector selector;
        final ConcurrentHashMap<SelectableChannel, NioConnection> connections = new ConcurrentHashMap();
        final Queue<NewConnection> pendingConnections = new ConcurrentLinkedQueue<NewConnection>();
        final ByteBuffer readBuffer;
        final ByteBuffer writeBuffer;
        final Thread thread;

        NioWorker(int id) throws IOException {
            this.id = id;
            this.selector = Selector.open();
            this.readBuffer = ByteBuffer.allocateDirect(262144);
            this.writeBuffer = ByteBuffer.allocateDirect(262144);
            this.thread = new Thread((Runnable)this, "zeph-nio-worker-" + id);
        }

        void start() {
            this.thread.start();
        }

        void addConnection(SocketChannel channel, long connId) {
            this.pendingConnections.offer(new NewConnection(channel, connId));
            this.selector.wakeup();
        }

        private void processPendingConnections() {
            NewConnection nc;
            while ((nc = this.pendingConnections.poll()) != null) {
                try {
                    nc.channel.configureBlocking(false);
                    nc.channel.setOption((SocketOption)StandardSocketOptions.TCP_NODELAY, (Object)true);
                    SelectionKey key = nc.channel.register(this.selector, 1);
                    NioConnection conn = new NioConnection(nc.channel, nc.id, key);
                    key.attach(conn);
                    this.connections.put(nc.channel, conn);
                }
                catch (IOException e) {
                    try {
                        nc.channel.close();
                    }
                    catch (IOException iOException) {}
                }
            }
        }

        @Override
        public void run() {
            while (HttpServerNio.this.running.get()) {
                try {
                    this.processPendingConnections();
                    int ready = this.selector.select(100L);
                    if (ready > 0) {
                        Set<SelectionKey> selectedKeys = this.selector.selectedKeys();
                        Iterator<SelectionKey> iter = selectedKeys.iterator();
                        while (iter.hasNext()) {
                            NioConnection conn;
                            SelectionKey key = iter.next();
                            iter.remove();
                            if (!key.isValid() || (conn = (NioConnection)key.attachment()) == null) continue;
                            try {
                                if (key.isReadable()) {
                                    this.handleRead(conn);
                                }
                                if (!key.isValid() || !key.isWritable()) continue;
                                this.handleWrite(conn);
                            }
                            catch (IOException e) {
                                this.closeConnection(conn);
                            }
                        }
                    }
                    if (HttpServerNio.this.streamingMode) {
                        this.checkCompletedStreamingHandlers();
                    }
                    this.checkIdleTimeouts();
                }
                catch (ClosedSelectorException e) {
                    break;
                }
                catch (Exception e) {
                    if (!HttpServerNio.this.running.get()) continue;
                    log.error("NIO Worker " + this.id + " error: " + e.getMessage(), e);
                }
            }
        }

        private void handleRead(NioConnection conn) throws IOException {
            this.readBuffer.clear();
            int bytesRead = conn.channel.read(this.readBuffer);
            if (bytesRead < 0) {
                this.closeConnection(conn);
                return;
            }
            if (bytesRead == 0) {
                return;
            }
            conn.updateActivity();
            this.readBuffer.flip();
            byte[] data = new byte[bytesRead];
            this.readBuffer.get(data);
            if (conn.isHttp2) {
                this.handleHttp2Read(conn, data);
                return;
            }
            if (ProtocolDetector.isHttp2Preface(data)) {
                this.handleHttp2PriorKnowledge(conn, data);
                return;
            }
            if (ProtocolDetector.isTlsRecord(data)) {
                this.sendTlsErrorResponse(conn);
                return;
            }
            Http1Handler.ParseResult result = conn.http1Handler.parse(data, 0, bytesRead);
            if (result == Http1Handler.ParseResult.REQUEST_COMPLETE) {
                if (conn.streamingRequest) {
                    conn.requestBodyComplete = true;
                    conn.streamingRequest = false;
                    if (conn.h2cUpgradePending) {
                        conn.isHttp2 = true;
                        conn.h2cUpgradePending = false;
                    }
                    if (conn.handlerFuture != null && conn.handlerFuture.isDone()) {
                        this.processHandlerResult(conn);
                    } else if (conn.handlerFuture == null) {
                        conn.http1Handler.reset();
                    } else {
                        try {
                            conn.key.interestOps(conn.key.interestOps() & 0xFFFFFFFE);
                        }
                        catch (CancelledKeyException cancelledKeyException) {
                            // empty catch block
                        }
                    }
                    return;
                }
                HttpRequest request = conn.http1Handler.getRequest();
                if (conn.http1Handler.isH2cUpgradeRequest(request)) {
                    if (request.getBody() != null || request.getContentLength() > 0L) {
                        this.handleH2cUpgradeWithBodyComplete(conn, request);
                    } else {
                        this.handleH2cUpgrade(conn, request);
                    }
                    return;
                }
                HttpResponse response = conn.http1Handler.handleRequest(request);
                if (!conn.http1Handler.isKeepAlive()) {
                    conn.closeAfterWrite = true;
                }
                this.prepareResponse(conn, response);
                conn.http1Handler.reset();
            } else if (result == Http1Handler.ParseResult.HEADERS_COMPLETE && HttpServerNio.this.streamingMode) {
                HttpRequest request = conn.http1Handler.getRequest();
                if (conn.http1Handler.isH2cUpgradeRequest(request)) {
                    this.handleH2cUpgradeWithBody(conn, request);
                }
                conn.streamingRequest = true;
                conn.requestBodyComplete = false;
                NioConnection finalConn = conn;
                NioWorker worker = this;
                conn.handlerFuture = HttpServerNio.this.handlerExecutor.submit(() -> {
                    try {
                        HttpResponse resp = HttpServerNio.this.handler.apply(request);
                        worker.selector.wakeup();
                        return resp;
                    }
                    catch (Exception e) {
                        worker.selector.wakeup();
                        return HttpResponse.serverError();
                    }
                });
                Http1Handler.ParseResult bodyResult = conn.http1Handler.parse(new byte[0], 0, 0);
                if (bodyResult == Http1Handler.ParseResult.REQUEST_COMPLETE) {
                    conn.requestBodyComplete = true;
                    conn.streamingRequest = false;
                    if (conn.h2cUpgradePending) {
                        conn.isHttp2 = true;
                        conn.h2cUpgradePending = false;
                    }
                    try {
                        conn.key.interestOps(conn.key.interestOps() & 0xFFFFFFFE);
                    }
                    catch (CancelledKeyException cancelledKeyException) {}
                }
            } else if (result == Http1Handler.ParseResult.ERROR) {
                log.error("Parse error: " + conn.http1Handler.getParser().getErrorMessage());
                HttpResponse response = HttpResponse.badRequest();
                response.setHeader("Connection", "close");
                conn.closeAfterWrite = true;
                this.prepareResponse(conn, response);
            }
        }

        private void prepareResponse(NioConnection conn, HttpResponse response) {
            if (response.hasBodyWriter()) {
                try {
                    PipedOutputStream pos = new PipedOutputStream();
                    PipedInputStream pis = new PipedInputStream(pos, 65536);
                    HttpServerNio.this.handlerExecutor.submit(() -> {
                        try {
                            response.getBodyWriter().accept(pos);
                        }
                        catch (Exception exception) {
                        }
                        finally {
                            try {
                                pos.close();
                            }
                            catch (Exception exception) {}
                        }
                    });
                    conn.streamingBody = pis;
                    conn.useChunkedEncoding = true;
                    conn.streamingBuffer = new byte[262144];
                    conn.pendingWrite = ByteBuffer.wrap(response.encodeStreamingHeaders());
                }
                catch (IOException e) {
                    HttpResponse errorResponse = HttpResponse.serverError();
                    conn.pendingWrite = ByteBuffer.wrap(errorResponse.encode());
                }
            } else if (response.isStreaming()) {
                conn.streamingBody = response.getBodyStream();
                conn.useChunkedEncoding = response.getContentLength() < 0L;
                conn.streamingBuffer = new byte[262144];
                conn.pendingWrite = ByteBuffer.wrap(response.encodeStreamingHeaders());
            } else {
                conn.pendingWrite = ByteBuffer.wrap(response.encode());
            }
            try {
                if (conn.streamingRequest) {
                    conn.key.interestOps(5);
                } else {
                    conn.key.interestOps(4);
                }
            }
            catch (CancelledKeyException cancelledKeyException) {
                // empty catch block
            }
        }

        private void handleWrite(NioConnection conn) throws IOException {
            if (conn.pendingWrite == null) {
                if (conn.waitingForStreamingData) {
                    return;
                }
                if (conn.isHttp2 && conn.http2Handler != null && conn.http2Handler.hasAccumulatedData()) {
                    try {
                        byte[] response = conn.http2Handler.processData(new byte[0]);
                        if (response != null && response.length > 0) {
                            conn.pendingWrite = ByteBuffer.wrap(response);
                        }
                    }
                    catch (Exception e) {
                        this.closeConnection(conn);
                        return;
                    }
                }
                if (conn.streamingBody != null) {
                    if (this.continueStreamingResponse(conn)) {
                        return;
                    }
                    conn.streamingBody = null;
                    conn.streamingBuffer = null;
                    conn.streamingQueue = null;
                }
                if (conn.streamingRequest) {
                    conn.key.interestOps(5);
                } else {
                    conn.key.interestOps(1);
                }
                if (conn.streamingBody == null && conn.pendingWrite == null) {
                    if (conn.closeAfterWrite) {
                        this.closeConnection(conn);
                    } else if (!conn.isKeepAlive()) {
                        this.closeConnection(conn);
                    }
                }
                return;
            }
            int written = conn.channel.write(conn.pendingWrite);
            if (written > 0) {
                conn.updateActivity();
            }
            if (!conn.pendingWrite.hasRemaining()) {
                conn.pendingWrite = null;
                if (conn.isHttp2 && conn.http2Handler != null && conn.http2Handler.hasPendingStreaming()) {
                    byte[] nextChunk = conn.http2Handler.continueStreaming();
                    if (nextChunk != null && nextChunk.length > 0) {
                        conn.pendingWrite = ByteBuffer.wrap(nextChunk);
                        return;
                    }
                    if (conn.http2Handler.isWaitingForWindow()) {
                        conn.key.interestOps(5);
                        return;
                    }
                    if (conn.http2Handler.isClosed()) {
                        this.closeConnection(conn);
                        return;
                    }
                }
                if (conn.isHttp2 && conn.http2Handler != null && (conn.http2Handler.hasPendingResponses() || conn.http2Handler.hasAccumulatedData())) {
                    try {
                        byte[] response = conn.http2Handler.processData(new byte[0]);
                        if (response != null && response.length > 0) {
                            conn.pendingWrite = ByteBuffer.wrap(response);
                            return;
                        }
                    }
                    catch (Exception e) {
                        this.closeConnection(conn);
                        return;
                    }
                }
                if (conn.isHttp2) {
                    if (conn.http2Handler != null && conn.http2Handler.isClosed()) {
                        this.closeConnection(conn);
                    } else {
                        conn.key.interestOps(1);
                    }
                    return;
                }
                if (conn.streamingBody != null) {
                    if (this.continueStreamingResponse(conn)) {
                        return;
                    }
                    conn.streamingBody = null;
                    conn.streamingBuffer = null;
                }
                if (conn.closeAfterWrite) {
                    this.closeConnection(conn);
                } else if (conn.isKeepAlive()) {
                    conn.key.interestOps(1);
                } else {
                    this.closeConnection(conn);
                }
            }
        }

        private boolean continueStreamingResponse(NioConnection conn) {
            byte[] data;
            if (conn.streamingBody == null) {
                return false;
            }
            if (conn.streamingQueue == null) {
                conn.streamingQueue = new LinkedBlockingQueue<byte[]>(64);
                InputStream stream = conn.streamingBody;
                BlockingQueue<byte[]> queue = conn.streamingQueue;
                byte[] EMPTY = new byte[]{};
                Selector workerSelector = this.selector;
                NioConnection connRef = conn;
                HttpServerNio.this.handlerExecutor.submit(() -> {
                    byte[] buffer = new byte[0x100000];
                    try {
                        while (true) {
                            int n;
                            if ((n = stream.read(buffer)) <= 0) {
                                while (!queue.offer(EMPTY, 100L, TimeUnit.MILLISECONDS) && connRef.key != null && connRef.key.isValid()) {
                                }
                                workerSelector.wakeup();
                                break;
                            }
                            byte[] chunk = Arrays.copyOf(buffer, n);
                            while (!queue.offer(chunk, 100L, TimeUnit.MILLISECONDS) && connRef.key != null && connRef.key.isValid()) {
                            }
                            if (!connRef.waitingForStreamingData) continue;
                            workerSelector.wakeup();
                        }
                    }
                    catch (Exception e) {
                        log.error("Zeph reader error: " + e.getMessage());
                        queue.offer(EMPTY);
                    }
                    finally {
                        conn.streamingReaderDone = true;
                        try {
                            stream.close();
                        }
                        catch (Exception exception) {}
                    }
                });
            }
            if ((data = (byte[])conn.streamingQueue.poll()) == null) {
                conn.updateActivity();
                conn.waitingForStreamingData = true;
                data = (byte[])conn.streamingQueue.poll();
                if (data != null) {
                    conn.waitingForStreamingData = false;
                } else {
                    try {
                        if (conn.streamingRequest) {
                            conn.key.interestOps(1);
                        } else {
                            conn.key.interestOps(0);
                        }
                    }
                    catch (CancelledKeyException queue) {
                        // empty catch block
                    }
                    return true;
                }
            }
            conn.waitingForStreamingData = false;
            if (data.length == 0) {
                if (conn.useChunkedEncoding) {
                    conn.pendingWrite = ByteBuffer.wrap("0\r\n\r\n".getBytes());
                    return true;
                }
                conn.streamingBody = null;
                conn.streamingQueue = null;
                return false;
            }
            if (conn.useChunkedEncoding) {
                String header = Integer.toHexString(data.length) + "\r\n";
                byte[] headerBytes = header.getBytes();
                byte[] footer = "\r\n".getBytes();
                ByteBuffer chunk = ByteBuffer.allocate(headerBytes.length + data.length + footer.length);
                chunk.put(headerBytes);
                chunk.put(data);
                chunk.put(footer);
                chunk.flip();
                conn.pendingWrite = chunk;
            } else {
                conn.pendingWrite = ByteBuffer.wrap(data);
            }
            try {
                int ops = conn.streamingRequest ? 5 : 4;
                conn.key.interestOps(ops);
            }
            catch (CancelledKeyException cancelledKeyException) {
                // empty catch block
            }
            return true;
        }

        private void checkCompletedStreamingHandlers() {
            for (NioConnection conn : this.connections.values()) {
                if (!conn.isHttp2 || conn.http2Handler == null || !conn.http2Handler.hasPendingResponses()) continue;
                try {
                    byte[] response = conn.http2Handler.processData(new byte[0]);
                    if (response == null || response.length <= 0) continue;
                    conn.pendingWrite = ByteBuffer.wrap(response);
                    try {
                        conn.key.interestOps(4);
                    }
                    catch (CancelledKeyException cancelledKeyException) {
                    }
                }
                catch (Exception e) {
                    this.closeConnection(conn);
                }
            }
            for (NioConnection conn : this.connections.values()) {
                if (conn.handlerFuture == null || !conn.handlerFuture.isDone()) continue;
                this.processHandlerResult(conn);
            }
            for (NioConnection conn : this.connections.values()) {
                if (!conn.waitingForStreamingData || conn.streamingQueue == null || conn.streamingQueue.isEmpty()) continue;
                conn.waitingForStreamingData = false;
                try {
                    if (conn.streamingRequest) {
                        conn.key.interestOps(5);
                        continue;
                    }
                    conn.key.interestOps(4);
                }
                catch (CancelledKeyException cancelledKeyException) {}
            }
        }

        private void processHandlerResult(NioConnection conn) {
            try {
                HttpResponse response = conn.handlerFuture.get();
                conn.handlerFuture = null;
                if (conn.requestBodyComplete || !response.isStreaming()) {
                    conn.streamingRequest = false;
                }
                if (conn.http2Handler != null) {
                    byte[] http2Response;
                    conn.isHttp2 = true;
                    conn.h2cUpgradePending = false;
                    conn.http2Handler.queueUpgradeResponse(response);
                    try {
                        http2Response = conn.http2Handler.processData(new byte[0]);
                        if (http2Response == null) {
                            http2Response = new byte[]{};
                        }
                    }
                    catch (Exception e) {
                        http2Response = new byte[]{};
                    }
                    byte[] combined = H2cUpgradeHelper.buildUpgradeResponseWithData(conn.http2Handler, http2Response);
                    conn.pendingWrite = ByteBuffer.wrap(combined);
                    conn.http1Handler.reset();
                    try {
                        conn.key.interestOps(4);
                    }
                    catch (CancelledKeyException cancelledKeyException) {
                        // empty catch block
                    }
                    return;
                }
                if (conn.isKeepAlive()) {
                    response.setHeader("Connection", "keep-alive");
                } else {
                    response.setHeader("Connection", "close");
                    conn.closeAfterWrite = true;
                }
                this.prepareResponse(conn, response);
                if (conn.requestBodyComplete) {
                    conn.http1Handler.reset();
                }
            }
            catch (Exception e) {
                log.error("Error processing handler result: " + e.getMessage(), e);
                conn.handlerFuture = null;
                conn.streamingRequest = false;
                HttpResponse response = HttpResponse.serverError();
                response.setHeader("Connection", "close");
                conn.closeAfterWrite = true;
                this.prepareResponse(conn, response);
                conn.http1Handler.reset();
            }
        }

        private void closeConnection(NioConnection conn) {
            this.connections.remove(conn.channel);
            conn.close();
        }

        private void checkIdleTimeouts() {
            if (HttpServerNio.this.idleTimeoutMs <= 0) {
                return;
            }
            long now = System.currentTimeMillis();
            for (NioConnection conn : this.connections.values()) {
                if (now - conn.lastActivity <= (long)HttpServerNio.this.idleTimeoutMs || conn.handlerFuture != null && !conn.handlerFuture.isDone() || conn.streamingBody != null || conn.waitingForStreamingData || conn.streamingRequest || conn.isHttp2 && conn.http2Handler != null && (conn.http2Handler.hasPendingStreaming() || conn.http2Handler.hasPendingStreamingHandlers())) continue;
                log.debug("Closing idle connection " + conn.id + " after timeout");
                this.closeConnection(conn);
            }
        }

        private void sendTlsErrorResponse(NioConnection conn) {
            String body = "This server does not support HTTPS on this port. Use HTTP instead.\n";
            String response = "HTTP/1.1 400 Bad Request\r\nContent-Type: text/plain\r\nContent-Length: " + body.length() + "\r\nConnection: close\r\n\r\n" + body;
            conn.pendingWrite = ByteBuffer.wrap(response.getBytes(StandardCharsets.UTF_8));
            conn.closeAfterWrite = true;
            try {
                conn.key.interestOps(4);
            }
            catch (CancelledKeyException cancelledKeyException) {
                // empty catch block
            }
        }

        private void handleHttp2PriorKnowledge(NioConnection conn, byte[] data) {
            conn.http2Handler = new Http2ServerHandler(HttpServerNio.this.handler, HttpServerNio.this.port, false, HttpServerNio.this.streamingMode);
            conn.http2Handler.setHandlerCompletionCallback(() -> this.selector.wakeup());
            conn.isHttp2 = true;
            conn.http1Handler.reset();
            try {
                byte[] response = conn.http2Handler.processData(data);
                if (response != null && response.length > 0) {
                    conn.pendingWrite = ByteBuffer.wrap(response);
                    try {
                        conn.key.interestOps(4);
                    }
                    catch (CancelledKeyException cancelledKeyException) {}
                }
            }
            catch (Http2FrameReader.Http2Exception e) {
                this.closeConnection(conn);
            }
        }

        private void handleH2cUpgrade(NioConnection conn, HttpRequest request) {
            conn.http2Handler = new Http2ServerHandler(HttpServerNio.this.handler, HttpServerNio.this.port, false, HttpServerNio.this.streamingMode);
            conn.http2Handler.setHandlerCompletionCallback(() -> this.selector.wakeup());
            conn.isHttp2 = true;
            conn.http2Handler.setupUpgradeStream(request);
            conn.http1Handler.reset();
            conn.pendingWrite = ByteBuffer.wrap(H2cUpgradeHelper.buildUpgradeResponse(conn.http2Handler));
            try {
                conn.key.interestOps(4);
            }
            catch (CancelledKeyException cancelledKeyException) {
                // empty catch block
            }
        }

        private void handleH2cUpgradeWithBodyComplete(NioConnection conn, HttpRequest request) {
            byte[] http2Response;
            HttpResponse response;
            conn.http2Handler = new Http2ServerHandler(HttpServerNio.this.handler, HttpServerNio.this.port, false, HttpServerNio.this.streamingMode);
            conn.http2Handler.setHandlerCompletionCallback(() -> this.selector.wakeup());
            conn.http2Handler.setUpgradeBodyAlreadyHandled();
            conn.isHttp2 = true;
            conn.http2Handler.setupUpgradeStream(request);
            try {
                response = HttpServerNio.this.handler.apply(request);
            }
            catch (Exception e) {
                response = HttpResponse.serverError();
            }
            conn.http2Handler.queueUpgradeResponse(response);
            try {
                http2Response = conn.http2Handler.processData(new byte[0]);
            }
            catch (Exception e) {
                this.closeConnection(conn);
                return;
            }
            conn.http1Handler.reset();
            conn.pendingWrite = ByteBuffer.wrap(H2cUpgradeHelper.buildUpgradeResponseWithData(conn.http2Handler, http2Response != null ? http2Response : new byte[]{}));
            try {
                conn.key.interestOps(4);
            }
            catch (CancelledKeyException cancelledKeyException) {
                // empty catch block
            }
        }

        private void handleH2cUpgradeWithBody(NioConnection conn, HttpRequest request) {
            conn.http2Handler = new Http2ServerHandler(HttpServerNio.this.handler, HttpServerNio.this.port, false, HttpServerNio.this.streamingMode);
            conn.http2Handler.setHandlerCompletionCallback(() -> this.selector.wakeup());
            conn.http2Handler.setUpgradeBodyAlreadyHandled();
            conn.h2cUpgradePending = true;
            conn.http2Handler.setupUpgradeStream(request);
        }

        private void handleHttp2Read(NioConnection conn, byte[] data) {
            try {
                byte[] response = conn.http2Handler.processData(data);
                if (response != null && response.length > 0) {
                    conn.pendingWrite = ByteBuffer.wrap(response);
                    try {
                        conn.key.interestOps(4);
                    }
                    catch (CancelledKeyException cancelledKeyException) {}
                } else if (conn.http2Handler.isClosed()) {
                    this.closeConnection(conn);
                    return;
                }
                if (conn.http2Handler.hasPendingStreaming()) {
                    byte[] streamingData = conn.http2Handler.continueStreaming();
                    if (streamingData != null && streamingData.length > 0) {
                        if (conn.pendingWrite != null && conn.pendingWrite.hasRemaining()) {
                            ByteBuffer combined = ByteBuffer.allocate(conn.pendingWrite.remaining() + streamingData.length);
                            combined.put(conn.pendingWrite);
                            combined.put(streamingData);
                            combined.flip();
                            conn.pendingWrite = combined;
                        } else {
                            conn.pendingWrite = ByteBuffer.wrap(streamingData);
                        }
                        try {
                            conn.key.interestOps(4);
                        }
                        catch (CancelledKeyException cancelledKeyException) {}
                    } else if (conn.http2Handler.isWaitingForWindow()) {
                        try {
                            conn.key.interestOps(5);
                        }
                        catch (CancelledKeyException cancelledKeyException) {}
                    }
                }
            }
            catch (Http2FrameReader.Http2Exception e) {
                this.closeConnection(conn);
            }
        }

        void stop() {
            this.selector.wakeup();
        }

        void close() {
            for (NioConnection conn : this.connections.values()) {
                conn.close();
            }
            this.connections.clear();
            try {
                this.selector.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }

        record NewConnection(SocketChannel channel, long id) {
        }
    }

    private class NioConnection {
        final SocketChannel channel;
        final long id;
        final SelectionKey key;
        final Http1Handler http1Handler;
        ByteBuffer pendingWrite;
        boolean closeAfterWrite = false;
        volatile long lastActivity = System.currentTimeMillis();
        InputStream streamingBody;
        boolean useChunkedEncoding;
        byte[] streamingBuffer;
        Future<byte[]> streamingReadFuture;
        BlockingQueue<byte[]> streamingQueue;
        volatile boolean streamingReaderDone = false;
        volatile boolean waitingForStreamingData = false;
        Future<HttpResponse> handlerFuture;
        boolean streamingRequest = false;
        boolean requestBodyComplete = false;
        boolean isHttp2 = false;
        boolean h2cUpgradePending = false;
        Http2ServerHandler http2Handler;
        byte[] pendingHttp2Data;

        NioConnection(SocketChannel channel, long id, SelectionKey key) {
            this.channel = channel;
            this.id = id;
            this.key = key;
            this.http1Handler = new Http1Handler(HttpServerNio.this.handler, HttpServerNio.this.port, HttpServerNio.this.streamingMode, HttpServerNio.this.maxBodySize);
        }

        void updateActivity() {
            this.lastActivity = System.currentTimeMillis();
        }

        boolean isTimedOut() {
            return HttpServerNio.this.idleTimeoutMs > 0 && System.currentTimeMillis() - this.lastActivity > (long)HttpServerNio.this.idleTimeoutMs;
        }

        boolean isKeepAlive() {
            return this.http1Handler.isKeepAlive();
        }

        void close() {
            try {
                this.key.cancel();
                this.channel.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            if (this.streamingBody != null) {
                try {
                    this.streamingBody.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            this.http1Handler.close();
        }
    }
}

