/*
 * Decompiled with CFR 0.152.
 */
package zeph.http;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import zeph.http.ByteArrayPool;
import zeph.http.HttpMethod;
import zeph.http.HttpRequest;
import zeph.http.StreamingBodyInputStream;

public class HttpParser {
    private static final int STATE_REQUEST_LINE = 0;
    private static final int STATE_HEADERS = 1;
    private static final int STATE_BODY = 2;
    private static final int STATE_BODY_CHUNKED = 3;
    private static final int STATE_COMPLETE = 4;
    private static final int STATE_ERROR = -1;
    private static final int MAX_REQUEST_LINE = 8192;
    private static final int MAX_HEADER_SIZE = 8192;
    private static final int MAX_HEADERS = 100;
    private static final long DEFAULT_MAX_BODY_SIZE = 0xA00000L;
    private long maxBodySize = 0xA00000L;
    private int state = 0;
    private final HttpRequest request = new HttpRequest();
    private byte[] buffer = new byte[8192];
    private int bufferPos = 0;
    private long contentLength = -1L;
    private long bodyBytesRead = 0L;
    private byte[] bodyBuffer;
    private boolean isChunked = false;
    private int chunkState = 0;
    private int currentChunkSize = 0;
    private int chunkBytesRead = 0;
    private ByteArrayOutputStream chunkedBodyBuffer;
    private String errorMessage;
    private boolean streamingMode = false;
    private StreamingBodyInputStream streamingBody;

    public Result parse(byte[] data, int offset, int length) {
        this.ensureCapacity(this.bufferPos + length);
        System.arraycopy(data, offset, this.buffer, this.bufferPos, length);
        this.bufferPos += length;
        while (true) {
            switch (this.state) {
                case 0: {
                    Result r = this.parseRequestLine();
                    if (r != Result.COMPLETE) {
                        return r;
                    }
                    this.state = 1;
                    break;
                }
                case 1: {
                    Result r = this.parseHeaders();
                    if (r != Result.COMPLETE) {
                        return r;
                    }
                    String transferEncoding = this.request.getHeader("Transfer-Encoding");
                    this.isChunked = transferEncoding != null && transferEncoding.toLowerCase().contains("chunked");
                    String contentLengthHeader = this.request.getHeader("Content-Length");
                    if (this.isChunked && contentLengthHeader != null) {
                        this.errorMessage = "Request contains both Content-Length and Transfer-Encoding";
                        this.state = -1;
                        return Result.ERROR;
                    }
                    if (this.isChunked) {
                        if (this.streamingMode) {
                            this.streamingBody = new StreamingBodyInputStream(-1L);
                            this.request.setBodyStream(this.streamingBody);
                            this.state = 3;
                            return Result.HEADERS_COMPLETE;
                        }
                        this.chunkedBodyBuffer = new ByteArrayOutputStream();
                        this.state = 3;
                        break;
                    }
                    this.contentLength = this.request.getContentLength();
                    if (this.contentLength > 0L) {
                        if (this.maxBodySize > 0L && this.contentLength > this.maxBodySize) {
                            this.errorMessage = "Request body too large: " + this.contentLength + " > " + this.maxBodySize;
                            this.state = -1;
                            return Result.ERROR;
                        }
                        if (this.streamingMode) {
                            this.streamingBody = new StreamingBodyInputStream(this.contentLength);
                            this.request.setBodyStream(this.streamingBody);
                            this.state = 2;
                            return Result.HEADERS_COMPLETE;
                        }
                        this.bodyBuffer = new byte[(int)this.contentLength];
                        this.state = 2;
                        break;
                    }
                    this.state = 4;
                    return Result.COMPLETE;
                }
                case 2: {
                    Result r = this.streamingMode ? this.parseBodyStreaming() : this.parseBody();
                    if (r != Result.COMPLETE) {
                        return r;
                    }
                    this.state = 4;
                    return Result.COMPLETE;
                }
                case 3: {
                    Result r = this.streamingMode ? this.parseBodyChunkedStreaming() : this.parseBodyChunked();
                    if (r != Result.COMPLETE) {
                        return r;
                    }
                    this.state = 4;
                    return Result.COMPLETE;
                }
                case 4: {
                    return Result.COMPLETE;
                }
                case -1: {
                    return Result.ERROR;
                }
            }
        }
    }

    private Result parseRequestLine() {
        int lineEnd = this.findCRLF(0);
        if (lineEnd < 0) {
            if (this.bufferPos > 8192) {
                this.errorMessage = "Request line too long";
                this.state = -1;
                return Result.ERROR;
            }
            return Result.NEED_MORE_DATA;
        }
        String line = new String(this.buffer, 0, lineEnd, StandardCharsets.US_ASCII);
        int firstSpace = line.indexOf(32);
        if (firstSpace < 0) {
            this.errorMessage = "Invalid request line";
            this.state = -1;
            return Result.ERROR;
        }
        int secondSpace = line.indexOf(32, firstSpace + 1);
        if (secondSpace < 0) {
            this.errorMessage = "Invalid request line";
            this.state = -1;
            return Result.ERROR;
        }
        String methodStr = line.substring(0, firstSpace);
        String uri = line.substring(firstSpace + 1, secondSpace);
        String version = line.substring(secondSpace + 1);
        try {
            this.request.setMethod(HttpMethod.parse(methodStr));
        }
        catch (IllegalArgumentException e) {
            this.errorMessage = "Unknown HTTP method: " + methodStr;
            this.state = -1;
            return Result.ERROR;
        }
        this.request.setUri(uri);
        this.request.setVersion(version);
        this.compact(lineEnd + 2);
        return Result.COMPLETE;
    }

    private Result parseHeaders() {
        int headerCount = 0;
        while (true) {
            int lineEnd;
            if ((lineEnd = this.findCRLF(0)) < 0) {
                if (this.bufferPos > 8192) {
                    this.errorMessage = "Headers too large";
                    this.state = -1;
                    return Result.ERROR;
                }
                return Result.NEED_MORE_DATA;
            }
            if (lineEnd == 0) {
                this.compact(2);
                return Result.COMPLETE;
            }
            String line = new String(this.buffer, 0, lineEnd, StandardCharsets.US_ASCII);
            int colonIdx = line.indexOf(58);
            if (colonIdx < 0) {
                this.errorMessage = "Invalid header line";
                this.state = -1;
                return Result.ERROR;
            }
            String name = line.substring(0, colonIdx).trim();
            String value = line.substring(colonIdx + 1).trim();
            this.request.setHeader(name, value);
            if (++headerCount > 100) {
                this.errorMessage = "Too many headers";
                this.state = -1;
                return Result.ERROR;
            }
            this.compact(lineEnd + 2);
        }
    }

    private Result parseBody() {
        long remaining = this.contentLength - this.bodyBytesRead;
        int available = this.bufferPos;
        int toCopy = (int)Math.min(remaining, (long)available);
        if (toCopy > 0) {
            System.arraycopy(this.buffer, 0, this.bodyBuffer, (int)this.bodyBytesRead, toCopy);
            this.bodyBytesRead += (long)toCopy;
            this.compact(toCopy);
        }
        if (this.bodyBytesRead >= this.contentLength) {
            this.request.setBody(this.bodyBuffer);
            return Result.COMPLETE;
        }
        return Result.NEED_MORE_DATA;
    }

    private Result parseBodyStreaming() {
        long remaining = this.contentLength - this.bodyBytesRead;
        int available = this.bufferPos;
        int toCopy = (int)Math.min(remaining, (long)available);
        if (toCopy > 0) {
            byte[] chunk = ByteArrayPool.getInstance().acquire(toCopy);
            System.arraycopy(this.buffer, 0, chunk, 0, toCopy);
            this.streamingBody.feedData(chunk, toCopy);
            this.bodyBytesRead += (long)toCopy;
            this.compact(toCopy);
        }
        if (this.bodyBytesRead >= this.contentLength) {
            this.streamingBody.complete();
            return Result.COMPLETE;
        }
        return Result.NEED_MORE_DATA;
    }

    private Result parseBodyChunked() {
        while (true) {
            int lineEnd;
            if (this.chunkState == 0) {
                lineEnd = this.findCRLF(0);
                if (lineEnd < 0) {
                    return Result.NEED_MORE_DATA;
                }
                String sizeLine = new String(this.buffer, 0, lineEnd, StandardCharsets.US_ASCII);
                int semicolon = sizeLine.indexOf(59);
                String hexSize = semicolon >= 0 ? sizeLine.substring(0, semicolon) : sizeLine;
                try {
                    this.currentChunkSize = Integer.parseInt(hexSize.trim(), 16);
                }
                catch (NumberFormatException e) {
                    this.errorMessage = "Invalid chunk size: " + hexSize;
                    this.state = -1;
                    return Result.ERROR;
                }
                this.compact(lineEnd + 2);
                if (this.currentChunkSize == 0) {
                    this.chunkState = 2;
                    continue;
                }
                this.chunkBytesRead = 0;
                this.chunkState = 1;
                continue;
            }
            if (this.chunkState == 1) {
                int remaining = this.currentChunkSize - this.chunkBytesRead;
                int available = this.bufferPos;
                int toCopy = Math.min(remaining, available);
                if (toCopy > 0) {
                    if (this.maxBodySize > 0L && this.bodyBytesRead + (long)toCopy > this.maxBodySize) {
                        this.errorMessage = "Chunked request body too large: exceeds " + this.maxBodySize;
                        this.state = -1;
                        return Result.ERROR;
                    }
                    this.chunkedBodyBuffer.write(this.buffer, 0, toCopy);
                    this.chunkBytesRead += toCopy;
                    this.bodyBytesRead += (long)toCopy;
                    this.compact(toCopy);
                }
                if (this.chunkBytesRead >= this.currentChunkSize) {
                    if (this.bufferPos < 2) {
                        return Result.NEED_MORE_DATA;
                    }
                    if (this.buffer[0] != 13 || this.buffer[1] != 10) {
                        this.errorMessage = "Missing CRLF after chunk data";
                        this.state = -1;
                        return Result.ERROR;
                    }
                    this.compact(2);
                    this.chunkState = 0;
                    continue;
                }
                return Result.NEED_MORE_DATA;
            }
            if (this.chunkState != 2) continue;
            lineEnd = this.findCRLF(0);
            if (lineEnd < 0) {
                return Result.NEED_MORE_DATA;
            }
            if (lineEnd == 0) {
                this.compact(2);
                this.request.setBody(this.chunkedBodyBuffer.toByteArray());
                return Result.COMPLETE;
            }
            this.compact(lineEnd + 2);
        }
    }

    private Result parseBodyChunkedStreaming() {
        while (true) {
            int lineEnd;
            if (this.chunkState == 0) {
                lineEnd = this.findCRLF(0);
                if (lineEnd < 0) {
                    return Result.NEED_MORE_DATA;
                }
                String sizeLine = new String(this.buffer, 0, lineEnd, StandardCharsets.US_ASCII);
                int semicolon = sizeLine.indexOf(59);
                String hexSize = semicolon >= 0 ? sizeLine.substring(0, semicolon) : sizeLine;
                try {
                    this.currentChunkSize = Integer.parseInt(hexSize.trim(), 16);
                }
                catch (NumberFormatException e) {
                    this.errorMessage = "Invalid chunk size: " + hexSize;
                    this.state = -1;
                    return Result.ERROR;
                }
                this.compact(lineEnd + 2);
                if (this.currentChunkSize == 0) {
                    this.chunkState = 2;
                    continue;
                }
                this.chunkBytesRead = 0;
                this.chunkState = 1;
                continue;
            }
            if (this.chunkState == 1) {
                int remaining = this.currentChunkSize - this.chunkBytesRead;
                int available = this.bufferPos;
                int toCopy = Math.min(remaining, available);
                if (toCopy > 0) {
                    if (this.maxBodySize > 0L && this.bodyBytesRead + (long)toCopy > this.maxBodySize) {
                        this.errorMessage = "Chunked request body too large: exceeds " + this.maxBodySize;
                        this.state = -1;
                        this.streamingBody.signalError(new IOException(this.errorMessage));
                        return Result.ERROR;
                    }
                    byte[] chunk = ByteArrayPool.getInstance().acquire(toCopy);
                    System.arraycopy(this.buffer, 0, chunk, 0, toCopy);
                    this.streamingBody.feedData(chunk, toCopy);
                    this.chunkBytesRead += toCopy;
                    this.bodyBytesRead += (long)toCopy;
                    this.compact(toCopy);
                }
                if (this.chunkBytesRead >= this.currentChunkSize) {
                    if (this.bufferPos < 2) {
                        return Result.NEED_MORE_DATA;
                    }
                    if (this.buffer[0] != 13 || this.buffer[1] != 10) {
                        this.errorMessage = "Missing CRLF after chunk data";
                        this.state = -1;
                        return Result.ERROR;
                    }
                    this.compact(2);
                    this.chunkState = 0;
                    continue;
                }
                return Result.NEED_MORE_DATA;
            }
            if (this.chunkState != 2) continue;
            lineEnd = this.findCRLF(0);
            if (lineEnd < 0) {
                return Result.NEED_MORE_DATA;
            }
            if (lineEnd == 0) {
                this.compact(2);
                this.streamingBody.complete();
                return Result.COMPLETE;
            }
            this.compact(lineEnd + 2);
        }
    }

    private int findCRLF(int start) {
        for (int i = start; i < this.bufferPos - 1; ++i) {
            if (this.buffer[i] != 13 || this.buffer[i + 1] != 10) continue;
            return i;
        }
        return -1;
    }

    private void compact(int consumed) {
        if (consumed > 0 && consumed <= this.bufferPos) {
            System.arraycopy(this.buffer, consumed, this.buffer, 0, this.bufferPos - consumed);
            this.bufferPos -= consumed;
        }
    }

    private void ensureCapacity(int required) {
        if (required > this.buffer.length) {
            int newSize = Math.max(this.buffer.length * 2, required);
            this.buffer = Arrays.copyOf(this.buffer, newSize);
        }
    }

    public HttpRequest getRequest() {
        return this.request;
    }

    public String getErrorMessage() {
        return this.errorMessage;
    }

    public boolean isComplete() {
        return this.state == 4;
    }

    public boolean isError() {
        return this.state == -1;
    }

    public void setStreamingMode(boolean enabled) {
        this.streamingMode = enabled;
    }

    public boolean isStreamingMode() {
        return this.streamingMode;
    }

    public StreamingBodyInputStream getStreamingBody() {
        return this.streamingBody;
    }

    public boolean isParsingBody() {
        return this.state == 2;
    }

    public long getExpectedContentLength() {
        return this.contentLength;
    }

    public long getBodyBytesRead() {
        return this.bodyBytesRead;
    }

    public void reset() {
        this.state = 0;
        this.bufferPos = 0;
        this.contentLength = -1L;
        this.bodyBytesRead = 0L;
        this.bodyBuffer = null;
        this.errorMessage = null;
        this.streamingBody = null;
        this.isChunked = false;
        this.chunkState = 0;
        this.currentChunkSize = 0;
        this.chunkBytesRead = 0;
        this.chunkedBodyBuffer = null;
        this.request.reset();
    }

    public boolean isChunked() {
        return this.isChunked;
    }

    public void setMaxBodySize(long maxSize) {
        this.maxBodySize = maxSize;
    }

    public long getMaxBodySize() {
        return this.maxBodySize;
    }

    public static enum Result {
        NEED_MORE_DATA,
        HEADERS_COMPLETE,
        COMPLETE,
        ERROR;

    }
}

