/*
 * Decompiled with CFR 0.152.
 */
package zeph.client;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import zeph.http.StreamingBodyInputStream;

public class HttpResponseParser {
    private static final int STATE_STATUS_LINE = 0;
    private static final int STATE_HEADERS = 1;
    private static final int STATE_BODY = 2;
    private static final int STATE_CHUNKED = 3;
    private static final int STATE_COMPLETE = 4;
    private static final int STATE_STREAMING_BODY = 5;
    private static final int STATE_STREAMING_CHUNKED = 6;
    private static final int STATE_ERROR = -1;
    private static final int MAX_STATUS_LINE = 8192;
    private static final int MAX_HEADER_SIZE = 8192;
    private static final int MAX_HEADERS = 100;
    private static final int MAX_BODY_SIZE = 0x6400000;
    private int state = 0;
    private String httpVersion;
    private int statusCode;
    private String statusMessage;
    private Map<String, String> headers = new HashMap<String, String>();
    private byte[] body;
    private byte[] buffer = new byte[8192];
    private int bufferPos = 0;
    private int contentLength = -1;
    private int bodyBytesRead = 0;
    private byte[] bodyBuffer;
    private boolean chunked = false;
    private int currentChunkSize = -1;
    private ByteArrayOutputStream chunkedBody;
    private boolean streamingMode = false;
    private StreamingBodyInputStream bodyStream;
    private long streamContentLength = -1L;
    private long streamBytesRead = 0L;
    private String errorMessage;

    public Result parse(byte[] data, int offset, int length) {
        this.ensureCapacity(this.bufferPos + length);
        System.arraycopy(data, offset, this.buffer, this.bufferPos, length);
        this.bufferPos += length;
        while (true) {
            switch (this.state) {
                case 0: {
                    Result r = this.parseStatusLine();
                    if (r != Result.COMPLETE) {
                        return r;
                    }
                    this.state = 1;
                    break;
                }
                case 1: {
                    Result r = this.parseHeaders();
                    if (r != Result.COMPLETE) {
                        return r;
                    }
                    String transferEncoding = this.headers.get("transfer-encoding");
                    this.chunked = transferEncoding != null && transferEncoding.toLowerCase().contains("chunked");
                    String clHeader = this.headers.get("content-length");
                    if (clHeader != null) {
                        try {
                            this.contentLength = Integer.parseInt(clHeader.trim());
                        }
                        catch (NumberFormatException e) {
                            this.errorMessage = "Invalid Content-Length";
                            this.state = -1;
                            return Result.ERROR;
                        }
                    }
                    if (this.streamingMode && (this.chunked || this.contentLength > 0 || this.contentLength == -1)) {
                        this.streamContentLength = this.contentLength;
                        this.bodyStream = new StreamingBodyInputStream(this.streamContentLength);
                        this.state = this.chunked ? 6 : (this.contentLength > 0 ? 5 : 5);
                        if (this.bufferPos > 0) {
                            byte[] initialData = Arrays.copyOf(this.buffer, this.bufferPos);
                            this.feedStreamData(initialData);
                        }
                        return Result.HEADERS_COMPLETE;
                    }
                    if (this.chunked) {
                        this.chunkedBody = new ByteArrayOutputStream();
                        this.state = 3;
                        break;
                    }
                    if (this.contentLength > 0) {
                        if (this.contentLength > 0x6400000) {
                            this.errorMessage = "Response body too large";
                            this.state = -1;
                            return Result.ERROR;
                        }
                        this.bodyBuffer = new byte[this.contentLength];
                        this.state = 2;
                        break;
                    }
                    if (this.contentLength == 0) {
                        this.body = new byte[0];
                        this.state = 4;
                        return Result.COMPLETE;
                    }
                    this.state = 4;
                    return Result.COMPLETE;
                }
                case 2: {
                    Result r = this.parseBody();
                    if (r != Result.COMPLETE) {
                        return r;
                    }
                    this.state = 4;
                    return Result.COMPLETE;
                }
                case 3: {
                    Result r = this.parseChunked();
                    if (r != Result.COMPLETE) {
                        return r;
                    }
                    this.body = this.chunkedBody.toByteArray();
                    this.state = 4;
                    return Result.COMPLETE;
                }
                case 5: {
                    return Result.NEED_MORE_DATA;
                }
                case 6: {
                    return Result.NEED_MORE_DATA;
                }
                case 4: {
                    return Result.COMPLETE;
                }
                case -1: {
                    return Result.ERROR;
                }
            }
        }
    }

    private Result parseStatusLine() {
        String statusCodeStr;
        int lineEnd = this.findCRLF(0);
        if (lineEnd < 0) {
            if (this.bufferPos > 8192) {
                this.errorMessage = "Status line too long";
                this.state = -1;
                return Result.ERROR;
            }
            return Result.NEED_MORE_DATA;
        }
        String line = new String(this.buffer, 0, lineEnd, StandardCharsets.US_ASCII);
        int firstSpace = line.indexOf(32);
        if (firstSpace < 0) {
            this.errorMessage = "Invalid status line";
            this.state = -1;
            return Result.ERROR;
        }
        this.httpVersion = line.substring(0, firstSpace);
        int secondSpace = line.indexOf(32, firstSpace + 1);
        if (secondSpace < 0) {
            statusCodeStr = line.substring(firstSpace + 1);
            this.statusMessage = "";
        } else {
            statusCodeStr = line.substring(firstSpace + 1, secondSpace);
            this.statusMessage = line.substring(secondSpace + 1);
        }
        try {
            this.statusCode = Integer.parseInt(statusCodeStr);
        }
        catch (NumberFormatException e) {
            this.errorMessage = "Invalid status code: " + statusCodeStr;
            this.state = -1;
            return Result.ERROR;
        }
        this.compact(lineEnd + 2);
        return Result.COMPLETE;
    }

    private Result parseHeaders() {
        int headerCount = 0;
        while (true) {
            int lineEnd;
            if ((lineEnd = this.findCRLF(0)) < 0) {
                if (this.bufferPos > 8192) {
                    this.errorMessage = "Headers too large";
                    this.state = -1;
                    return Result.ERROR;
                }
                return Result.NEED_MORE_DATA;
            }
            if (lineEnd == 0) {
                this.compact(2);
                return Result.COMPLETE;
            }
            String line = new String(this.buffer, 0, lineEnd, StandardCharsets.US_ASCII);
            int colonIdx = line.indexOf(58);
            if (colonIdx < 0) {
                this.errorMessage = "Invalid header line";
                this.state = -1;
                return Result.ERROR;
            }
            String name = line.substring(0, colonIdx).trim().toLowerCase();
            String value = line.substring(colonIdx + 1).trim();
            this.headers.put(name, value);
            if (++headerCount > 100) {
                this.errorMessage = "Too many headers";
                this.state = -1;
                return Result.ERROR;
            }
            this.compact(lineEnd + 2);
        }
    }

    private Result parseBody() {
        int remaining = this.contentLength - this.bodyBytesRead;
        int available = this.bufferPos;
        int toCopy = Math.min(remaining, available);
        if (toCopy > 0) {
            System.arraycopy(this.buffer, 0, this.bodyBuffer, this.bodyBytesRead, toCopy);
            this.bodyBytesRead += toCopy;
            this.compact(toCopy);
        }
        if (this.bodyBytesRead >= this.contentLength) {
            this.body = this.bodyBuffer;
            return Result.COMPLETE;
        }
        return Result.NEED_MORE_DATA;
    }

    private Result parseChunked() {
        while (true) {
            int available;
            int toRead;
            if (this.currentChunkSize < 0) {
                int lineEnd = this.findCRLF(0);
                if (lineEnd < 0) {
                    return Result.NEED_MORE_DATA;
                }
                String sizeLine = new String(this.buffer, 0, lineEnd, StandardCharsets.US_ASCII);
                int semicolon = sizeLine.indexOf(59);
                if (semicolon >= 0) {
                    sizeLine = sizeLine.substring(0, semicolon);
                }
                try {
                    this.currentChunkSize = Integer.parseInt(sizeLine.trim(), 16);
                }
                catch (NumberFormatException e) {
                    this.errorMessage = "Invalid chunk size: " + sizeLine;
                    this.state = -1;
                    return Result.ERROR;
                }
                this.compact(lineEnd + 2);
                if (this.currentChunkSize == 0) {
                    if (this.bufferPos >= 2) {
                        this.compact(2);
                    }
                    return Result.COMPLETE;
                }
            }
            if ((toRead = Math.min(this.currentChunkSize, available = this.bufferPos)) > 0) {
                this.chunkedBody.write(this.buffer, 0, toRead);
                this.currentChunkSize -= toRead;
                this.compact(toRead);
            }
            if (this.currentChunkSize != 0) break;
            if (this.bufferPos < 2) {
                return Result.NEED_MORE_DATA;
            }
            this.compact(2);
            this.currentChunkSize = -1;
        }
        return Result.NEED_MORE_DATA;
    }

    private int findCRLF(int start) {
        for (int i = start; i < this.bufferPos - 1; ++i) {
            if (this.buffer[i] != 13 || this.buffer[i + 1] != 10) continue;
            return i;
        }
        return -1;
    }

    private void compact(int consumed) {
        if (consumed > 0 && consumed <= this.bufferPos) {
            System.arraycopy(this.buffer, consumed, this.buffer, 0, this.bufferPos - consumed);
            this.bufferPos -= consumed;
        }
    }

    private void ensureCapacity(int required) {
        if (required > this.buffer.length) {
            int newSize = Math.max(this.buffer.length * 2, required);
            this.buffer = Arrays.copyOf(this.buffer, newSize);
        }
    }

    public void setStreamingMode(boolean streaming) {
        this.streamingMode = streaming;
    }

    public boolean isStreamingMode() {
        return this.streamingMode;
    }

    public InputStream getBodyStream() {
        return this.bodyStream;
    }

    public boolean isStreamingBody() {
        return this.state == 5 || this.state == 6;
    }

    public Result feedStreamData(byte[] data) {
        return this.feedStreamData(data, 0, data.length);
    }

    public Result feedStreamData(byte[] data, int offset, int length) {
        if (this.bodyStream == null) {
            return Result.ERROR;
        }
        if (this.state == 5) {
            return this.feedStreamBodyData(data, offset, length);
        }
        if (this.state == 6) {
            return this.feedStreamChunkedData(data, offset, length);
        }
        return Result.ERROR;
    }

    private Result feedStreamBodyData(byte[] data, int offset, int length) {
        if (length > 0) {
            byte[] copy = new byte[length];
            System.arraycopy(data, offset, copy, 0, length);
            this.bodyStream.feedData(copy);
            this.streamBytesRead += (long)length;
        }
        if (this.streamContentLength > 0L && this.streamBytesRead >= this.streamContentLength) {
            this.bodyStream.complete();
            this.state = 4;
            return Result.COMPLETE;
        }
        return Result.NEED_MORE_DATA;
    }

    private Result feedStreamChunkedData(byte[] data, int offset, int length) {
        this.ensureCapacity(this.bufferPos + length);
        System.arraycopy(data, offset, this.buffer, this.bufferPos, length);
        this.bufferPos += length;
        return this.parseStreamChunked();
    }

    private Result parseStreamChunked() {
        while (true) {
            int available;
            int toRead;
            if (this.currentChunkSize < 0) {
                int lineEnd = this.findCRLF(0);
                if (lineEnd < 0) {
                    return Result.NEED_MORE_DATA;
                }
                String sizeLine = new String(this.buffer, 0, lineEnd, StandardCharsets.US_ASCII);
                int semicolon = sizeLine.indexOf(59);
                if (semicolon >= 0) {
                    sizeLine = sizeLine.substring(0, semicolon);
                }
                try {
                    this.currentChunkSize = Integer.parseInt(sizeLine.trim(), 16);
                }
                catch (NumberFormatException e) {
                    this.errorMessage = "Invalid chunk size: " + sizeLine;
                    this.state = -1;
                    if (this.bodyStream != null) {
                        this.bodyStream.signalError(new IOException(this.errorMessage));
                    }
                    return Result.ERROR;
                }
                this.compact(lineEnd + 2);
                if (this.currentChunkSize == 0) {
                    if (this.bufferPos >= 2) {
                        this.compact(2);
                    }
                    this.bodyStream.complete();
                    this.state = 4;
                    return Result.COMPLETE;
                }
            }
            if ((toRead = Math.min(this.currentChunkSize, available = this.bufferPos)) > 0) {
                byte[] chunkData = new byte[toRead];
                System.arraycopy(this.buffer, 0, chunkData, 0, toRead);
                this.bodyStream.feedData(chunkData);
                this.currentChunkSize -= toRead;
                this.compact(toRead);
            }
            if (this.currentChunkSize != 0) break;
            if (this.bufferPos < 2) {
                return Result.NEED_MORE_DATA;
            }
            this.compact(2);
            this.currentChunkSize = -1;
        }
        return Result.NEED_MORE_DATA;
    }

    public void markStreamComplete() {
        if (this.bodyStream != null) {
            this.bodyStream.complete();
        }
        this.state = 4;
    }

    public void markStreamError(IOException e) {
        if (this.bodyStream != null) {
            this.bodyStream.signalError(e);
        }
        this.state = -1;
        this.errorMessage = e.getMessage();
    }

    public String getHttpVersion() {
        return this.httpVersion;
    }

    public int getStatusCode() {
        return this.statusCode;
    }

    public String getStatusMessage() {
        return this.statusMessage;
    }

    public Map<String, String> getHeaders() {
        return this.headers;
    }

    public String getHeader(String name) {
        return this.headers.get(name.toLowerCase());
    }

    public byte[] getBody() {
        return this.body;
    }

    public String getBodyAsString() {
        if (this.body == null) {
            return null;
        }
        return new String(this.body, StandardCharsets.UTF_8);
    }

    public String getErrorMessage() {
        return this.errorMessage;
    }

    public boolean isComplete() {
        return this.state == 4;
    }

    public boolean isError() {
        return this.state == -1;
    }

    public boolean isKeepAlive() {
        String connection = this.headers.get("connection");
        if (connection != null) {
            if (connection.toLowerCase().contains("close")) {
                return false;
            }
            if (connection.toLowerCase().contains("keep-alive")) {
                return true;
            }
        }
        return this.httpVersion != null && this.httpVersion.contains("1.1");
    }

    public boolean isRedirect() {
        return this.statusCode == 301 || this.statusCode == 302 || this.statusCode == 303 || this.statusCode == 307 || this.statusCode == 308;
    }

    public String getRedirectLocation() {
        return this.headers.get("location");
    }

    public void reset() {
        this.state = 0;
        this.bufferPos = 0;
        this.httpVersion = null;
        this.statusCode = 0;
        this.statusMessage = null;
        this.headers.clear();
        this.body = null;
        this.contentLength = -1;
        this.bodyBytesRead = 0;
        this.bodyBuffer = null;
        this.chunked = false;
        this.currentChunkSize = -1;
        this.chunkedBody = null;
        this.errorMessage = null;
        this.streamingMode = false;
        this.bodyStream = null;
        this.streamContentLength = -1L;
        this.streamBytesRead = 0L;
    }

    public static enum Result {
        NEED_MORE_DATA,
        HEADERS_COMPLETE,
        COMPLETE,
        ERROR;

    }
}

