/*
 * Decompiled with CFR 0.152.
 */
package zeph.client;

import clojure.lang.Keyword;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLSession;
import zeph.client.Http1ClientHandler;
import zeph.client.HttpClientConnectionPool;
import zeph.client.HttpClientRequest;
import zeph.client.HttpClientResponse;
import zeph.ssl.SslConfig;

public class HttpClientNio
implements AutoCloseable {
    private static final int READ_BUFFER_SIZE = 65536;
    private static final int WRITE_BUFFER_SIZE = 65536;
    private static final int SELECT_TIMEOUT_MS = 100;
    private static final int HANDSHAKE_TIMEOUT_MS = 3000;
    private final HttpClientConnectionPool pool;
    private final SslConfig sslConfig;
    private final SslConfig sslConfigInsecure;
    private final NioWorker[] workers;
    private final Thread[] workerThreads;
    private final AtomicInteger workerIndex = new AtomicInteger(0);
    private final AtomicBoolean running = new AtomicBoolean(true);
    private static final int DEFAULT_WORKERS = Math.max(2, Runtime.getRuntime().availableProcessors() / 2);

    public HttpClientNio() throws Exception {
        this(new HttpClientOptions());
    }

    public HttpClientNio(HttpClientOptions options) throws Exception {
        this.pool = new HttpClientConnectionPool(options.getMaxConnectionsPerHost(), options.getMaxTotalConnections(), options.getIdleTimeout(), options.getConnectionTimeout());
        this.sslConfig = SslConfig.forClient();
        this.sslConfigInsecure = SslConfig.forClientInsecure();
        int numWorkers = options.getWorkers();
        this.workers = new NioWorker[numWorkers];
        this.workerThreads = new Thread[numWorkers];
        for (int i = 0; i < numWorkers; ++i) {
            this.workers[i] = new NioWorker(i);
            this.workerThreads[i] = new Thread((Runnable)this.workers[i], "zeph-nio-client-" + i);
            this.workerThreads[i].setDaemon(true);
            this.workerThreads[i].start();
        }
    }

    public CompletableFuture<HttpClientResponse> request(HttpClientRequest request) {
        if (!this.running.get()) {
            CompletableFuture<HttpClientResponse> future = new CompletableFuture<HttpClientResponse>();
            future.completeExceptionally(new IllegalStateException("Client is closed"));
            return future;
        }
        CompletableFuture<HttpClientResponse> future = new CompletableFuture<HttpClientResponse>();
        InFlightRequest inFlight = new InFlightRequest(request, future);
        int idx = Math.abs(this.workerIndex.getAndIncrement() % this.workers.length);
        this.workers[idx].submit(inFlight);
        if (request.getTimeout() > 0) {
            future = future.orTimeout(request.getTimeout(), TimeUnit.MILLISECONDS);
        }
        return future;
    }

    public CompletableFuture<HttpClientResponse> get(String url) {
        return this.request(HttpClientRequest.get(url));
    }

    public CompletableFuture<HttpClientResponse> post(String url, byte[] body) {
        return this.request(HttpClientRequest.post(url).body(body));
    }

    public CompletableFuture<HttpClientResponse> post(String url, String body) {
        return this.request(HttpClientRequest.post(url).body(body));
    }

    public CompletableFuture<HttpClientResponse> put(String url, byte[] body) {
        return this.request(HttpClientRequest.put(url).body(body));
    }

    public CompletableFuture<HttpClientResponse> delete(String url) {
        return this.request(HttpClientRequest.delete(url));
    }

    public CompletableFuture<HttpClientResponse> head(String url) {
        return this.request(HttpClientRequest.head(url));
    }

    public HttpClientConnectionPool.Stats getPoolStats() {
        return this.pool.getStats();
    }

    @Override
    public void close() {
        if (!this.running.compareAndSet(true, false)) {
            return;
        }
        for (NioWorker nioWorker : this.workers) {
            if (nioWorker == null) continue;
            nioWorker.stop();
        }
        for (Runnable runnable : this.workerThreads) {
            if (runnable == null) continue;
            try {
                ((Thread)runnable).join(5000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        this.pool.close();
    }

    public static void main(String[] args) throws Exception {
        System.out.println("Testing NIO HTTP Client...");
        try (HttpClientNio client = new HttpClientNio();){
            System.out.println("\n=== HTTP GET ===");
            HttpClientResponse resp = client.get("http://httpbin.org/get").get(10L, TimeUnit.SECONDS);
            System.out.println("Status: " + resp.getStatus());
            System.out.println("Body length: " + (resp.getBody() != null ? resp.getBody().length : 0));
            System.out.println("\n=== HTTPS GET ===");
            resp = client.get("https://httpbin.org/get").get(10L, TimeUnit.SECONDS);
            System.out.println("Status: " + resp.getStatus());
            System.out.println("Body length: " + (resp.getBody() != null ? resp.getBody().length : 0));
            System.out.println("\n=== Redirect ===");
            resp = client.get("http://httpbin.org/redirect/2").get(10L, TimeUnit.SECONDS);
            System.out.println("Status: " + resp.getStatus());
            System.out.println("\nAll tests passed!");
        }
    }

    public static class HttpClientOptions {
        private int workers = DEFAULT_WORKERS;
        private int maxConnectionsPerHost = 20;
        private int maxTotalConnections = 200;
        private long idleTimeout = 60000L;
        private long connectionTimeout = 30000L;

        public int getWorkers() {
            return this.workers;
        }

        public HttpClientOptions workers(int workers) {
            this.workers = workers;
            return this;
        }

        public int getMaxConnectionsPerHost() {
            return this.maxConnectionsPerHost;
        }

        public HttpClientOptions maxConnectionsPerHost(int max) {
            this.maxConnectionsPerHost = max;
            return this;
        }

        public int getMaxTotalConnections() {
            return this.maxTotalConnections;
        }

        public HttpClientOptions maxTotalConnections(int max) {
            this.maxTotalConnections = max;
            return this;
        }

        public long getIdleTimeout() {
            return this.idleTimeout;
        }

        public HttpClientOptions idleTimeout(long timeout) {
            this.idleTimeout = timeout;
            return this;
        }

        public long getConnectionTimeout() {
            return this.connectionTimeout;
        }

        public HttpClientOptions connectionTimeout(long timeout) {
            this.connectionTimeout = timeout;
            return this;
        }
    }

    private class NioWorker
    implements Runnable {
        final int id;
        final Selector selector;
        final ConcurrentLinkedQueue<InFlightRequest> pendingRequests = new ConcurrentLinkedQueue();
        final Map<Long, InFlightRequest> inFlight = new ConcurrentHashMap<Long, InFlightRequest>();
        final Map<Long, NioConnection> connections = new ConcurrentHashMap<Long, NioConnection>();
        volatile boolean running = true;

        NioWorker(int id) throws IOException {
            this.id = id;
            this.selector = Selector.open();
        }

        void submit(InFlightRequest request) {
            this.pendingRequests.offer(request);
            this.selector.wakeup();
        }

        void stop() {
            this.running = false;
            this.selector.wakeup();
        }

        @Override
        public void run() {
            while (this.running) {
                try {
                    this.processPendingRequests();
                    int ready = this.selector.select(100L);
                    if (ready > 0) {
                        Set<SelectionKey> selectedKeys = this.selector.selectedKeys();
                        Iterator<SelectionKey> iter = selectedKeys.iterator();
                        while (iter.hasNext()) {
                            InFlightRequest req;
                            NioConnection conn;
                            SelectionKey key = iter.next();
                            iter.remove();
                            if (!key.isValid() || (conn = (NioConnection)key.attachment()) == null || (req = this.inFlight.get(conn.id)) == null) continue;
                            try {
                                if (key.isConnectable()) {
                                    this.handleConnect(conn, req);
                                    continue;
                                }
                                if (key.isWritable()) {
                                    this.handleWrite(conn, req);
                                    continue;
                                }
                                if (!key.isReadable()) continue;
                                this.handleRead(conn, req);
                            }
                            catch (Exception e) {
                                this.completeWithError(req, e);
                            }
                        }
                    }
                    this.checkHandshakeTimeouts();
                }
                catch (ClosedSelectorException e) {
                    break;
                }
                catch (Exception e) {
                    if (!this.running) continue;
                    System.err.println("NIO Client Worker " + this.id + " error: " + e.getMessage());
                }
            }
            this.cleanup();
        }

        private void checkHandshakeTimeouts() {
            long now = System.currentTimeMillis();
            for (NioConnection conn : this.connections.values()) {
                InFlightRequest req;
                long elapsed;
                if (!conn.secure || conn.handshakeComplete || conn.handshakeStartTime <= 0L || (elapsed = now - conn.handshakeStartTime) <= 3000L || (req = this.inFlight.get(conn.id)) == null) continue;
                this.completeWithError(req, new IOException("TLS handshake timeout - server may not support HTTPS"));
            }
        }

        private void processPendingRequests() {
            InFlightRequest req;
            while ((req = this.pendingRequests.poll()) != null) {
                try {
                    this.startRequest(req);
                }
                catch (Exception e) {
                    this.completeWithError(req, e);
                }
            }
        }

        private void startRequest(InFlightRequest req) throws Exception {
            NioConnection conn;
            HttpClientRequest request = req.request;
            req.connection = conn = new NioConnection(request.getHost(), request.getPort(), request.isSecure());
            this.connections.put(conn.id, conn);
            this.inFlight.put(conn.id, req);
            Map<String, Object> opts = request.getOpts();
            if (opts != null) {
                Object asValue = opts.get(Keyword.intern((String)"as"));
                if (Keyword.intern((String)"stream").equals(asValue)) {
                    conn.http1Handler.setStreamingMode(true);
                }
            }
            conn.channel = SocketChannel.open();
            conn.channel.configureBlocking(false);
            conn.channel.connect(new InetSocketAddress(request.getHost(), request.getPort()));
            conn.key = conn.channel.register(this.selector, 8);
            conn.key.attach(conn);
        }

        private void handleConnect(NioConnection conn, InFlightRequest req) throws Exception {
            if (!conn.channel.finishConnect()) {
                return;
            }
            conn.channel.socket().setTcpNoDelay(true);
            if (conn.secure) {
                SslConfig config = req.request.isInsecure() ? HttpClientNio.this.sslConfigInsecure : HttpClientNio.this.sslConfig;
                SSLEngine engine = config.createClientEngine(conn.host, conn.port);
                conn.initSsl(engine);
                engine.beginHandshake();
                conn.handshakeStartTime = System.currentTimeMillis();
                this.doHandshake(conn, req);
            } else {
                this.sendRequest(conn, req);
            }
        }

        private void doHandshake(NioConnection conn, InFlightRequest req) throws Exception {
            SSLEngine engine = conn.sslEngine;
            SSLEngineResult.HandshakeStatus status = engine.getHandshakeStatus();
            block5: while (status != SSLEngineResult.HandshakeStatus.FINISHED && status != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING) {
                if (System.currentTimeMillis() - conn.handshakeStartTime > 3000L) {
                    throw new IOException("TLS handshake timeout - server may not support HTTPS");
                }
                switch (status) {
                    case NEED_WRAP: {
                        conn.sslNetBuffer.clear();
                        SSLEngineResult result = engine.wrap(conn.sslAppBuffer, conn.sslNetBuffer);
                        status = result.getHandshakeStatus();
                        conn.sslNetBuffer.flip();
                        while (conn.sslNetBuffer.hasRemaining()) {
                            int written = conn.channel.write(conn.sslNetBuffer);
                            if (written != 0) continue;
                            conn.key.interestOps(5);
                            return;
                        }
                        continue block5;
                    }
                    case NEED_UNWRAP: {
                        if (conn.readBuffer.position() == 0 || !conn.readBuffer.hasRemaining()) {
                            int firstByte;
                            int read = conn.channel.read(conn.readBuffer);
                            if (read < 0) {
                                throw new IOException("Connection closed during handshake");
                            }
                            if (read == 0 && conn.readBuffer.position() == 0) {
                                conn.key.interestOps(1);
                                return;
                            }
                            if (conn.readBuffer.position() > 0 && ((firstByte = conn.readBuffer.get(0) & 0xFF) < 20 || firstByte > 23)) {
                                byte[] preview = new byte[Math.min(conn.readBuffer.position(), 10)];
                                conn.readBuffer.flip();
                                conn.readBuffer.get(preview);
                                conn.readBuffer.compact();
                                String previewStr = new String(preview, StandardCharsets.ISO_8859_1);
                                if (previewStr.startsWith("HTTP/")) {
                                    throw new IOException("Server responded with HTTP instead of HTTPS");
                                }
                                throw new IOException("Server does not support TLS");
                            }
                        }
                        conn.readBuffer.flip();
                        conn.sslAppBuffer.clear();
                        SSLEngineResult result = engine.unwrap(conn.readBuffer, conn.sslAppBuffer);
                        SSLEngineResult.Status unwrapStatus = result.getStatus();
                        conn.readBuffer.compact();
                        if (unwrapStatus == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                            int read = conn.channel.read(conn.readBuffer);
                            if (read < 0) {
                                throw new IOException("Connection closed during handshake");
                            }
                            if (read == 0) {
                                conn.key.interestOps(1);
                                return;
                            }
                            status = SSLEngineResult.HandshakeStatus.NEED_UNWRAP;
                            continue block5;
                        }
                        if (unwrapStatus == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                            ByteBuffer newBuffer = ByteBuffer.allocate(conn.sslAppBuffer.capacity() * 2);
                            conn.sslAppBuffer.flip();
                            newBuffer.put(conn.sslAppBuffer);
                            conn.sslAppBuffer = newBuffer;
                            status = SSLEngineResult.HandshakeStatus.NEED_UNWRAP;
                            continue block5;
                        }
                        status = result.getHandshakeStatus();
                        continue block5;
                    }
                    case NEED_TASK: {
                        Runnable task;
                        while ((task = engine.getDelegatedTask()) != null) {
                            task.run();
                        }
                        status = engine.getHandshakeStatus();
                        continue block5;
                    }
                }
                throw new IllegalStateException("Invalid handshake status: " + String.valueOf((Object)status));
            }
            conn.handshakeComplete = true;
            this.sendRequest(conn, req);
        }

        private void sendRequest(NioConnection conn, InFlightRequest req) throws Exception {
            List<String[]> actualHeaders = req.request.buildHttp1Headers();
            req.request.printTrace("HTTP/1.1", actualHeaders);
            InputStream bodyStream = req.request.getBodyStream();
            if (bodyStream != null) {
                conn.bodyStream = bodyStream;
                conn.useChunkedEncoding = !req.request.hasKnownBodyLength();
                conn.streamBuffer = new byte[65536];
                conn.headersSent = false;
                conn.bodyComplete = false;
                conn.totalWriteSize = req.request.getBodyFile() != null ? req.request.getBodyFile().length() : 0L;
                conn.totalBytesWritten = 0L;
            }
            byte[] requestData = req.request.encode();
            if (conn.secure) {
                ByteBuffer plainBuffer = ByteBuffer.wrap(requestData);
                conn.sslNetBuffer.clear();
                while (plainBuffer.hasRemaining()) {
                    SSLEngineResult result = conn.sslEngine.wrap(plainBuffer, conn.sslNetBuffer);
                    if (result.getStatus() != SSLEngineResult.Status.BUFFER_OVERFLOW) continue;
                    ByteBuffer newBuffer = ByteBuffer.allocate(conn.sslNetBuffer.capacity() * 2);
                    conn.sslNetBuffer.flip();
                    newBuffer.put(conn.sslNetBuffer);
                    conn.sslNetBuffer = newBuffer;
                }
                conn.sslNetBuffer.flip();
                conn.pendingWriteData = new byte[conn.sslNetBuffer.remaining()];
                conn.sslNetBuffer.get(conn.pendingWriteData);
                conn.pendingWriteOffset = 0;
            } else {
                conn.pendingWriteData = requestData;
                conn.pendingWriteOffset = 0;
            }
            if (conn.bodyStream == null) {
                conn.totalWriteSize = conn.pendingWriteData.length;
            }
            conn.lastProgressTime = 0L;
            conn.lastProgressPercent = -1;
            conn.key.interestOps(4);
        }

        private void handleWrite(NioConnection conn, InFlightRequest req) throws Exception {
            if (conn.secure && !conn.handshakeComplete) {
                this.doHandshake(conn, req);
                return;
            }
            if (conn.pendingWriteData == null) {
                if (conn.bodyStream != null && !conn.bodyComplete) {
                    this.prepareNextBodyChunk(conn);
                    if (conn.pendingWriteData == null) {
                        conn.bodyComplete = true;
                        if (conn.bodyStream != null) {
                            try {
                                conn.bodyStream.close();
                            }
                            catch (Exception exception) {
                                // empty catch block
                            }
                            conn.bodyStream = null;
                        }
                        if (req.request.isProgress() && conn.totalWriteSize > 0L) {
                            System.err.println();
                        }
                        conn.key.interestOps(1);
                        return;
                    }
                } else {
                    conn.key.interestOps(1);
                    return;
                }
            }
            int remaining = conn.pendingWriteData.length - conn.pendingWriteOffset;
            conn.writeBuffer.clear();
            conn.writeBuffer.put(conn.pendingWriteData, conn.pendingWriteOffset, Math.min(remaining, conn.writeBuffer.capacity()));
            conn.writeBuffer.flip();
            int written = conn.channel.write(conn.writeBuffer);
            conn.pendingWriteOffset += written;
            if (req.request.isProgress() && conn.totalWriteSize > 0L && conn.bodyStream != null) {
                int percent = (int)(conn.totalBytesWritten * 100L / conn.totalWriteSize);
                long now = System.currentTimeMillis();
                if (percent != conn.lastProgressPercent || now - conn.lastProgressTime > 500L) {
                    int barWidth = 30;
                    int filled = percent * barWidth / 100;
                    StringBuilder bar = new StringBuilder("\r[");
                    for (int i = 0; i < barWidth; ++i) {
                        bar.append(i < filled ? "=" : (i == filled ? ">" : " "));
                    }
                    bar.append(String.format("] %3d%% (%s / %s)", percent, this.formatBytes(conn.totalBytesWritten), this.formatBytes(conn.totalWriteSize)));
                    System.err.print(bar);
                    conn.lastProgressPercent = percent;
                    conn.lastProgressTime = now;
                }
            }
            if (conn.pendingWriteOffset >= conn.pendingWriteData.length) {
                conn.pendingWriteData = null;
                conn.pendingWriteOffset = 0;
                if (conn.bodyStream == null) {
                    if (req.request.isProgress() && conn.totalWriteSize > 0L) {
                        System.err.println();
                    }
                    conn.key.interestOps(1);
                }
            }
        }

        private void prepareNextBodyChunk(NioConnection conn) throws IOException {
            int bytesRead = conn.bodyStream.read(conn.streamBuffer);
            if (bytesRead <= 0) {
                if (conn.useChunkedEncoding) {
                    conn.pendingWriteData = "0\r\n\r\n".getBytes(StandardCharsets.US_ASCII);
                    conn.pendingWriteOffset = 0;
                } else {
                    conn.pendingWriteData = null;
                }
                return;
            }
            conn.totalBytesWritten += (long)bytesRead;
            if (conn.useChunkedEncoding) {
                String chunkHeader = Integer.toHexString(bytesRead) + "\r\n";
                byte[] headerBytes = chunkHeader.getBytes(StandardCharsets.US_ASCII);
                byte[] trailerBytes = "\r\n".getBytes(StandardCharsets.US_ASCII);
                conn.pendingWriteData = new byte[headerBytes.length + bytesRead + trailerBytes.length];
                System.arraycopy(headerBytes, 0, conn.pendingWriteData, 0, headerBytes.length);
                System.arraycopy(conn.streamBuffer, 0, conn.pendingWriteData, headerBytes.length, bytesRead);
                System.arraycopy(trailerBytes, 0, conn.pendingWriteData, headerBytes.length + bytesRead, trailerBytes.length);
            } else {
                conn.pendingWriteData = new byte[bytesRead];
                System.arraycopy(conn.streamBuffer, 0, conn.pendingWriteData, 0, bytesRead);
            }
            conn.pendingWriteOffset = 0;
        }

        private String formatBytes(long bytes) {
            if (bytes < 1024L) {
                return bytes + " B";
            }
            if (bytes < 0x100000L) {
                return String.format("%.1f KB", (double)bytes / 1024.0);
            }
            return String.format("%.1f MB", (double)bytes / 1048576.0);
        }

        private void handleRead(NioConnection conn, InFlightRequest req) throws Exception {
            byte[] data;
            int read;
            if (conn.secure && !conn.handshakeComplete) {
                this.doHandshake(conn, req);
                return;
            }
            if (!conn.secure) {
                conn.readBuffer.clear();
            }
            if ((read = conn.channel.read(conn.readBuffer)) < 0) {
                throw new IOException("Connection closed by server");
            }
            if (read == 0 && conn.readBuffer.position() == 0) {
                return;
            }
            conn.readBuffer.flip();
            if (conn.secure) {
                conn.sslAppBuffer.clear();
                boolean needMoreData = false;
                while (conn.readBuffer.hasRemaining()) {
                    SSLEngineResult result = conn.sslEngine.unwrap(conn.readBuffer, conn.sslAppBuffer);
                    if (result.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                        ByteBuffer newBuffer = ByteBuffer.allocate(conn.sslAppBuffer.capacity() * 2);
                        conn.sslAppBuffer.flip();
                        newBuffer.put(conn.sslAppBuffer);
                        conn.sslAppBuffer = newBuffer;
                        continue;
                    }
                    if (result.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                        needMoreData = true;
                        break;
                    }
                    if (result.getStatus() != SSLEngineResult.Status.CLOSED) continue;
                    throw new IOException("SSL connection closed");
                }
                conn.readBuffer.compact();
                conn.sslAppBuffer.flip();
                data = new byte[conn.sslAppBuffer.remaining()];
                conn.sslAppBuffer.get(data);
                if (data.length == 0 && needMoreData) {
                    return;
                }
            } else {
                data = new byte[conn.readBuffer.remaining()];
                conn.readBuffer.get(data);
                conn.readBuffer.clear();
            }
            if (data.length == 0) {
                return;
            }
            if (conn.http1Handler.isStreamingBody()) {
                Http1ClientHandler.ParseResult result = conn.http1Handler.feedBodyData(data, 0, data.length);
                if (result == Http1ClientHandler.ParseResult.RESPONSE_COMPLETE) {
                    this.closeConnection(conn);
                } else if (result == Http1ClientHandler.ParseResult.ERROR) {
                    throw new Exception("Stream parse error: " + conn.http1Handler.getErrorMessage());
                }
                return;
            }
            Http1ClientHandler.ParseResult result = conn.http1Handler.parse(data, 0, data.length);
            if (result == Http1ClientHandler.ParseResult.HEADERS_COMPLETE) {
                this.handleHeadersComplete(conn, req);
            } else if (result == Http1ClientHandler.ParseResult.RESPONSE_COMPLETE) {
                this.handleResponseComplete(conn, req);
            } else if (result == Http1ClientHandler.ParseResult.ERROR) {
                throw new Exception("Response parse error: " + conn.http1Handler.getErrorMessage());
            }
        }

        private void handleHeadersComplete(NioConnection conn, InFlightRequest req) throws Exception {
            String location;
            Http1ClientHandler handler = conn.http1Handler;
            Http1ClientHandler.ResponseResult responseResult = handler.buildStreamingResponse(req.request.getOpts());
            HttpClientResponse response = responseResult.getResponse();
            if (!responseResult.isRedirect() || !req.request.isFollowRedirects() || req.redirectCount >= req.request.getMaxRedirects() || (location = responseResult.getRedirectLocation()) != null) {
                // empty if block
            }
            if (req.request.isTrace()) {
                long elapsedMs = System.currentTimeMillis() - req.startTime;
                response.printTrace(req.request.isTraceDetail(), req.request.getTraceRequestId(), elapsedMs, req.request.getTraceLimit());
            }
            req.future.complete(response);
        }

        private void handleResponseComplete(NioConnection conn, InFlightRequest req) throws Exception {
            String location;
            Http1ClientHandler handler = conn.http1Handler;
            Http1ClientHandler.ResponseResult responseResult = handler.buildResponse(req.request.getOpts());
            HttpClientResponse response = responseResult.getResponse();
            if (responseResult.isRedirect() && req.request.isFollowRedirects() && req.redirectCount < req.request.getMaxRedirects() && (location = responseResult.getRedirectLocation()) != null) {
                this.handleRedirect(conn, req, response, location);
                return;
            }
            this.completeRequest(conn, req, response);
        }

        private void handleRedirect(NioConnection conn, InFlightRequest req, HttpClientResponse response, String location) throws Exception {
            Object newUrl;
            this.closeConnection(conn);
            ++req.redirectCount;
            if (location.startsWith("http://") || location.startsWith("https://")) {
                newUrl = location;
            } else if (location.startsWith("/")) {
                String scheme = req.request.isSecure() ? "https://" : "http://";
                String host = req.request.getHost();
                int port = req.request.getPort();
                String portStr = req.request.isSecure() && port == 443 || !req.request.isSecure() && port == 80 ? "" : ":" + port;
                newUrl = scheme + host + portStr + location;
            } else {
                throw new Exception("Relative redirect not supported: " + location);
            }
            req.request.url((String)newUrl);
            req.connection = null;
            this.pendingRequests.offer(req);
            this.selector.wakeup();
        }

        private void completeRequest(NioConnection conn, InFlightRequest req, HttpClientResponse response) {
            if (req.request.isTrace()) {
                long elapsedMs = System.currentTimeMillis() - req.startTime;
                response.printTrace(req.request.isTraceDetail(), req.request.getTraceRequestId(), elapsedMs, req.request.getTraceLimit());
            }
            this.closeConnection(conn);
            req.future.complete(response);
        }

        private void completeWithError(InFlightRequest req, Exception e) {
            NioConnection conn = req.connection;
            if (conn != null) {
                if (conn.http1Handler.isStreamingBody()) {
                    conn.http1Handler.markStreamError(new IOException(e.getMessage()));
                }
                this.closeConnection(conn);
            }
            if (!req.future.isDone()) {
                HttpClientResponse response = new HttpClientResponse(e, req.request.getOpts());
                req.future.complete(response);
            }
        }

        private void closeConnection(NioConnection conn) {
            if (conn != null) {
                this.connections.remove(conn.id);
                this.inFlight.remove(conn.id);
                if (conn.http1Handler.isStreamingBody()) {
                    conn.http1Handler.markStreamComplete();
                }
                conn.close();
            }
        }

        private void cleanup() {
            InFlightRequest req;
            while ((req = this.pendingRequests.poll()) != null) {
                req.future.completeExceptionally(new Exception("Worker shutting down"));
            }
            for (InFlightRequest r : this.inFlight.values()) {
                r.future.completeExceptionally(new Exception("Worker shutting down"));
            }
            this.inFlight.clear();
            for (NioConnection conn : this.connections.values()) {
                conn.close();
            }
            this.connections.clear();
            try {
                this.selector.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    static class InFlightRequest {
        final HttpClientRequest request;
        final CompletableFuture<HttpClientResponse> future;
        NioConnection connection;
        long startTime;
        int redirectCount;

        InFlightRequest(HttpClientRequest request, CompletableFuture<HttpClientResponse> future) {
            this.request = request;
            this.future = future;
            this.startTime = System.currentTimeMillis();
            this.redirectCount = 0;
        }
    }

    static class NioConnection {
        final long id = idGenerator.incrementAndGet();
        final String host;
        final int port;
        final boolean secure;
        SocketChannel channel;
        SelectionKey key;
        SSLEngine sslEngine;
        ByteBuffer readBuffer;
        ByteBuffer writeBuffer;
        ByteBuffer sslAppBuffer;
        ByteBuffer sslNetBuffer;
        Http1ClientHandler http1Handler;
        boolean handshakeComplete;
        long handshakeStartTime;
        byte[] pendingWriteData;
        int pendingWriteOffset;
        long totalWriteSize;
        long totalBytesWritten;
        long lastProgressTime;
        int lastProgressPercent;
        InputStream bodyStream;
        boolean useChunkedEncoding;
        byte[] streamBuffer;
        boolean headersSent;
        boolean bodyComplete;
        private static final AtomicInteger idGenerator = new AtomicInteger(0);

        NioConnection(String host, int port, boolean secure) {
            this.host = host;
            this.port = port;
            this.secure = secure;
            this.readBuffer = ByteBuffer.allocateDirect(65536);
            this.writeBuffer = ByteBuffer.allocateDirect(65536);
            this.http1Handler = new Http1ClientHandler();
            this.handshakeComplete = !secure;
        }

        void initSsl(SSLEngine engine) {
            this.sslEngine = engine;
            SSLSession session = engine.getSession();
            this.sslAppBuffer = ByteBuffer.allocate(session.getApplicationBufferSize());
            this.sslNetBuffer = ByteBuffer.allocate(session.getPacketBufferSize());
            this.handshakeComplete = false;
        }

        void reset() {
            this.http1Handler.reset();
            this.pendingWriteData = null;
            this.pendingWriteOffset = 0;
        }

        void close() {
            try {
                if (this.key != null) {
                    this.key.cancel();
                }
                if (this.channel != null) {
                    this.channel.close();
                }
                if (this.sslEngine != null) {
                    this.sslEngine.closeOutbound();
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }
}

