(ns zeph.server
  "Blazing fast HTTP server with io_uring.
   API compatible with http-kit."
  (:require [zeph.ring :as ring]
            [clojure.java.io :as io]
            [clojure.string :as str])
  (:import [zeph.http HttpServer HttpServerMultiRing HttpsServerMultiRing]
           [zeph.ssl SslConfig]
           [java.io File]))

(defn- create-ssl-config
  "Create SslConfig from options map."
  [{:keys [keystore keystore-password keystore-type
           cert key
           truststore truststore-password
           protocols cipher-suites client-auth]}]
  (cond
    ;; PEM files (cert + key)
    (and cert key)
    (SslConfig/fromPem cert key (or keystore-password "zeph"))

    ;; Keystore file
    keystore
    (let [builder (-> (SslConfig/builder)
                      (.keystore keystore keystore-password))]
      (when keystore-type
        (.keystoreType builder keystore-type))
      (when truststore
        (.truststore builder truststore truststore-password))
      (when client-auth
        (.needClientAuth builder true))
      (when protocols
        (.protocols builder (into-array String protocols)))
      (when cipher-suites
        (.cipherSuites builder (into-array String cipher-suites)))
      (.build builder))

    ;; Use built-in localhost certificate
    :else
    (SslConfig/localhost)))

;; MIME types for static files
(def ^:private mime-types
  {"html" "text/html; charset=utf-8"
   "htm"  "text/html; charset=utf-8"
   "css"  "text/css; charset=utf-8"
   "js"   "application/javascript; charset=utf-8"
   "json" "application/json; charset=utf-8"
   "xml"  "application/xml; charset=utf-8"
   "txt"  "text/plain; charset=utf-8"
   "md"   "text/markdown; charset=utf-8"
   "csv"  "text/csv; charset=utf-8"
   "png"  "image/png"
   "jpg"  "image/jpeg"
   "jpeg" "image/jpeg"
   "gif"  "image/gif"
   "svg"  "image/svg+xml"
   "ico"  "image/x-icon"
   "webp" "image/webp"
   "woff"  "font/woff"
   "woff2" "font/woff2"
   "ttf"   "font/ttf"
   "otf"   "font/otf"
   "eot"   "application/vnd.ms-fontobject"
   "pdf"  "application/pdf"
   "zip"  "application/zip"
   "gz"   "application/gzip"
   "tar"  "application/x-tar"
   "mp3"  "audio/mpeg"
   "mp4"  "video/mp4"
   "webm" "video/webm"
   "ogg"  "audio/ogg"
   "wav"  "audio/wav"
   "wasm" "application/wasm"})

(defn- get-extension
  "Get file extension (lowercase, without dot)."
  [^String path]
  (when-let [dot-idx (str/last-index-of path ".")]
    (str/lower-case (subs path (inc dot-idx)))))

(defn- get-mime-type
  "Get MIME type for file path."
  [path]
  (get mime-types (get-extension path) "application/octet-stream"))

(defn- safe-path?
  "Check if path is safe (no directory traversal)."
  [^String path]
  (and (not (str/includes? path ".."))
       (not (str/includes? path "//"))
       (str/starts-with? path "/")))

(defn- resolve-static-file
  "Resolve request path to a file in static root.
   Returns File if exists and readable, nil otherwise."
  [^File root ^String uri]
  (when (safe-path? uri)
    (let [path (if (= uri "/") "/index.html" uri)
          file (io/file root (subs path 1))]  ; remove leading /
      (when (and (.exists file)
                 (.isFile file)
                 (.canRead file)
                 ;; Ensure file is under root (prevent symlink escape)
                 (str/starts-with? (.getCanonicalPath file)
                                   (.getCanonicalPath root)))
        file))))

(defn- wrap-static-files
  "Wrap handler with static file serving.
   Files from static-root are served for matching paths.
   If no match, delegates to the wrapped handler."
  [handler ^String static-root]
  (let [root (io/file static-root)]
    (if (and (.exists root) (.isDirectory root))
      (fn [req]
        (let [uri (:uri req)]
          (if-let [file (resolve-static-file root uri)]
            ;; Serve static file
            {:status 200
             :headers {"Content-Type" (get-mime-type (.getName file))
                       "Content-Length" (str (.length file))}
             :body file}
            ;; Try index.html for directory-like paths
            (if (and (str/ends-with? uri "/")
                     (not= uri "/"))
              (if-let [index-file (resolve-static-file root (str uri "index.html"))]
                {:status 200
                 :headers {"Content-Type" "text/html; charset=utf-8"
                           "Content-Length" (str (.length index-file))}
                 :body index-file}
                (handler req))
              (handler req)))))
      ;; Static root doesn't exist, just use handler
      (do
        (println "Warning: static-root does not exist:" static-root)
        handler))))

(defn run-server
  "Start a blazing fast HTTP server.

   handler - Ring handler function

   Options:
     :port              - Port to listen on (default: 8080)
     :ip                - IP to bind (default: \"0.0.0.0\")
     :thread            - Number of worker threads (default: CPU count)
     :worker-name-prefix - Ignored (for http-kit compatibility)
     :queue-size        - Ignored (for http-kit compatibility)
     :max-body          - Max request body size (default: 8MB)
     :max-line          - Max request line size (default: 8KB)

     ;; io_uring specific
     :ring-size         - io_uring ring size (default: 256)
     :single-ring       - Use single ring mode (default: false)

     ;; Static file serving
     :static-root       - Path to static files directory (optional)
                          When set, serves static files from this directory.
                          Falls back to handler if file not found.

     ;; SSL/TLS options (use :cert/:key for PEM or :keystore for PKCS12/JKS)
     :ssl?              - Enable SSL/TLS (default: false)
     :cert              - Path to certificate PEM file
     :key               - Path to private key PEM file
     :keystore          - Path to keystore file (PKCS12 or JKS)
     :keystore-password - Keystore password
     :keystore-type     - Keystore type (default: PKCS12)
     :truststore        - Path to truststore file (optional)
     :truststore-password - Truststore password
     :client-auth       - Require client authentication (default: false)
     :protocols         - Enabled TLS protocols (e.g. [\"TLSv1.3\" \"TLSv1.2\"])
     :cipher-suites     - Enabled cipher suites

   Returns a function that stops the server when called.
   Call (stop-fn) or (stop-fn {:timeout ms}) to stop.

   Example:
     ;; HTTP server
     (def stop (run-server handler {:port 8080}))

     ;; Serve static files from ./public
     (def stop (run-server handler {:port 8080 :static-root \"./public\"}))

     ;; HTTPS server with PEM files
     (def stop (run-server handler {:port 8443
                                    :ssl? true
                                    :cert \"cert.pem\"
                                    :key \"key.pem\"}))

     ;; HTTPS server with keystore
     (def stop (run-server handler {:port 8443
                                    :ssl? true
                                    :keystore \"keystore.p12\"
                                    :keystore-password \"secret\"}))
     ;; later
     (stop)"
  [handler & [{:keys [port ip thread ring-size single-ring static-root
                      ssl? cert key keystore keystore-password]
               :or {port 8080
                    ip "0.0.0.0"
                    thread (.availableProcessors (Runtime/getRuntime))
                    ring-size 256
                    single-ring false
                    ssl? false}
               :as opts}]]
  (let [;; Wrap with static file serving if static-root is specified
        effective-handler (if static-root
                            (wrap-static-files handler static-root)
                            handler)
        wrapped-handler (ring/wrap-ring-handler effective-handler)
        use-ssl? ssl?  ;; ssl? true だけで動く（組み込み証明書を使用）
        server (cond
                 ;; HTTPS server
                 use-ssl?
                 (let [ssl-config (create-ssl-config opts)]
                   (HttpsServerMultiRing. ip (int port) (int thread)
                                          (int ring-size) wrapped-handler ssl-config))

                 ;; HTTP single-ring mode
                 single-ring
                 (HttpServer. ip (int port) (int ring-size) wrapped-handler)

                 ;; HTTP multi-ring mode (default)
                 :else
                 (HttpServerMultiRing. ip (int port) (int thread)
                                       (int ring-size) wrapped-handler))
        server-thread (Thread.
                        (fn []
                          (try
                            (.run server)
                            (catch Throwable t
                              (when-not (Thread/interrupted)
                                (println "Server error:" (.getMessage t))))))
                        (if use-ssl? "zeph-ssl-server" "zeph-server"))]
    (.start server-thread)
    ;; Return stop function (http-kit compatible)
    (fn stop
      ([] (stop {}))
      ([{:keys [timeout] :or {timeout 5000}}]
       (.stop server)
       (.join server-thread timeout)
       (when (.isAlive server-thread)
         (.interrupt server-thread))
       (.close server)))))

(defn server-stop!
  "Stop a server. http-kit compatible alias.

   server - The stop function returned by run-server
   opts   - {:timeout ms} (default 5000ms)"
  [server & [opts]]
  (if (fn? server)
    (server (or opts {}))
    (throw (IllegalArgumentException.
             "Expected stop function from run-server"))))

;; Convenience macros for testing

(defmacro with-server
  "Start a server, execute body, then stop the server.

   Example:
     (with-server [stop (run-server handler {:port 8080})]
       (do-something))"
  [[binding start-expr] & body]
  `(let [~binding ~start-expr]
     (try
       ~@body
       (finally
         (~binding)))))
