(ns zeph.client
  "Blazing fast HTTP client with io_uring.
   API compatible with http-kit client."
  (:import [zeph.client HttpClient HttpClient$HttpClientOptions
            HttpClientRequest HttpClientResponse]
           [java.util.concurrent CompletableFuture TimeUnit TimeoutException]))

;; ============================================================
;; Default Client (lazy initialization)
;; ============================================================

(defonce ^:private default-client
  (delay
    (HttpClient.)))

(defn- get-client []
  @default-client)

;; ============================================================
;; Response Conversion
;; ============================================================

(defn- response->map
  "Convert HttpClientResponse to Clojure map."
  [^HttpClientResponse response]
  (if (.hasError response)
    {:error (.getError response)
     :opts (.getOpts response)}
    {:status (.getStatus response)
     :headers (into {} (.getHeaders response))
     :body (.getBodyAsString response)
     :opts (.getOpts response)}))

;; ============================================================
;; Promise Type (http-kit compatible)
;; ============================================================

(deftype HttpPromise [^CompletableFuture fut callback opts]
  clojure.lang.IDeref
  (deref [_]
    (let [^HttpClientResponse response (.get fut)]
      (response->map response)))

  clojure.lang.IBlockingDeref
  (deref [_ timeout-ms timeout-val]
    (try
      (let [^HttpClientResponse response (.get fut timeout-ms TimeUnit/MILLISECONDS)]
        (response->map response))
      (catch TimeoutException _
        timeout-val)))

  clojure.lang.IPending
  (isRealized [_]
    (.isDone fut))

  clojure.lang.IFn
  (invoke [this callback]
    ;; Allow setting callback after creation (http-kit style)
    (.thenAccept fut
      (reify java.util.function.Consumer
        (accept [_ response]
          (callback (response->map response)))))
    this))

;; ============================================================
;; Request Building
;; ============================================================

(defn- build-request
  "Build HttpClientRequest from options map."
  ^HttpClientRequest
  [{:keys [url method headers body query-params form-params
           timeout keepalive follow-redirects max-redirects
           insecure? basic-auth as]
    :or {method :get
         timeout 30000
         keepalive 120000
         follow-redirects true
         max-redirects 5
         insecure? false}
    :as opts}]
  (let [req (doto (HttpClientRequest.)
              (.url url)
              (.method (name method))
              (.timeout timeout)
              (.keepalive keepalive)
              (.followRedirects follow-redirects)
              (.maxRedirects max-redirects)
              (.insecure insecure?)
              (.opts opts))]
    (when headers
      (.headers req headers))
    (when body
      (if (bytes? body)
        (.body req ^bytes body)
        (.body req (str body))))
    (when query-params
      (.queryParams req query-params))
    (when form-params
      (.formParams req form-params))
    (when basic-auth
      (let [[user pass] basic-auth]
        (.basicAuth req user pass)))
    req))

;; ============================================================
;; Public API
;; ============================================================

(defn request
  "Send an HTTP request.

   Options:
     :url              - Request URL (required)
     :method           - HTTP method (:get, :post, :put, :delete, :head)
     :headers          - Request headers map
     :body             - Request body (string or bytes)
     :query-params     - Query parameters map
     :form-params      - Form parameters map (sets Content-Type)
     :timeout          - Request timeout in ms (default: 30000)
     :keepalive        - Keep-alive time in ms (default: 120000, -1 to disable)
     :follow-redirects - Follow redirects (default: true)
     :max-redirects    - Maximum redirects to follow (default: 5)
     :insecure?        - Skip SSL certificate validation (default: false)
     :basic-auth       - Basic auth as [user password]
     :as               - Response body coercion (:text, :stream, :byte-array)

   Returns a promise that can be:
     - Dereferenced with @ for synchronous access
     - Used with deref with timeout
     - Called as a function with callback

   Examples:
     ;; Synchronous
     @(request {:url \"https://example.com\" :method :get})

     ;; With timeout
     (deref (request {:url \"https://example.com\"}) 5000 :timeout)

     ;; With callback
     (request {:url \"https://example.com\"}
              (fn [{:keys [status body error]}]
                (println status body)))"
  ([opts]
   (request opts nil))
  ([opts callback]
   (let [client (get-client)
         req (build-request opts)
         ^CompletableFuture fut (.request client req)
         promise (->HttpPromise fut callback opts)]
     (when callback
       (.thenAccept fut
         (reify java.util.function.Consumer
           (accept [_ response]
             (callback (response->map response))))))
     promise)))

(defn get
  "Send a GET request.

   Usage:
     @(get \"https://example.com\")
     @(get \"https://example.com\" {:timeout 5000})
     (get \"https://example.com\" {:timeout 5000} callback)"
  ([url]
   (request {:url url :method :get}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :get} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :get))))
  ([url opts callback]
   (request (assoc opts :url url :method :get) callback)))

(defn post
  "Send a POST request.

   Usage:
     @(post \"https://example.com\" {:body \"data\"})
     (post \"https://example.com\" {:body \"data\"} callback)"
  ([url]
   (request {:url url :method :post}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :post} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :post))))
  ([url opts callback]
   (request (assoc opts :url url :method :post) callback)))

(defn put
  "Send a PUT request."
  ([url]
   (request {:url url :method :put}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :put} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :put))))
  ([url opts callback]
   (request (assoc opts :url url :method :put) callback)))

(defn delete
  "Send a DELETE request."
  ([url]
   (request {:url url :method :delete}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :delete} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :delete))))
  ([url opts callback]
   (request (assoc opts :url url :method :delete) callback)))

(defn head
  "Send a HEAD request."
  ([url]
   (request {:url url :method :head}))
  ([url opts-or-callback]
   (if (fn? opts-or-callback)
     (request {:url url :method :head} opts-or-callback)
     (request (assoc opts-or-callback :url url :method :head))))
  ([url opts callback]
   (request (assoc opts :url url :method :head) callback)))

;; ============================================================
;; Client Management
;; ============================================================

(defn create-client
  "Create a new HTTP client with custom options.

   Options:
     :workers                - Number of worker threads (default: CPU/2)
     :ring-size              - io_uring ring size (default: 256)
     :max-connections-per-host - Max connections per host (default: 20)
     :max-total-connections  - Max total connections (default: 200)
     :idle-timeout           - Idle connection timeout in ms (default: 60000)
     :connection-timeout     - Connection timeout in ms (default: 30000)

   Returns: HttpClient instance (call .close when done)"
  ([]
   (HttpClient.))
  ([opts]
   (let [options (HttpClient$HttpClientOptions.)]
     (when-let [v (:workers opts)]
       (.workers options v))
     (when-let [v (:ring-size opts)]
       (.ringSize options v))
     (when-let [v (:max-connections-per-host opts)]
       (.maxConnectionsPerHost options v))
     (when-let [v (:max-total-connections opts)]
       (.maxTotalConnections options v))
     (when-let [v (:idle-timeout opts)]
       (.idleTimeout options v))
     (when-let [v (:connection-timeout opts)]
       (.connectionTimeout options v))
     (HttpClient. options))))

(defn close-client
  "Close an HTTP client."
  [^HttpClient client]
  (.close client))

(defn pool-stats
  "Get connection pool statistics for the default client."
  []
  (let [stats (.getPoolStats (get-client))]
    {:hosts (.hosts stats)
     :idle (.idle stats)
     :in-use (.inUse stats)
     :total (.total stats)}))

;; ============================================================
;; Shutdown hook
;; ============================================================

(defonce ^:private shutdown-hook-registered
  (delay
    (.addShutdownHook
      (Runtime/getRuntime)
      (Thread.
        (fn []
          (when (realized? default-client)
            (.close ^HttpClient @default-client)))))))

;; Force registration
@shutdown-hook-registered
