(ns zeph.cli
  "Zeph CLI - HTTP server & client powered by io_uring."
  (:require [zeph.server :as server]
            [zeph.client :as client]
            [clojure.string :as str])
  (:gen-class))

;; ============================================================
;; HTTPie-style Client Argument Parsing
;; ============================================================

(defn- str->long [s]
  (try
    (Long/parseLong s)
    (catch NumberFormatException _
      nil)))

(defn- parse-request-item
  "Parse HTTPie-style request item.
   Header:Value  -> header
   key=value     -> data field (string)
   key:=value    -> data field (raw JSON)
   key@file      -> file upload (not implemented)"
  [^String item]
  (cond
    ;; Header: value
    (str/includes? item ":")
    (let [idx (.indexOf item ":")]
      (when (and (pos? idx) (not= (.charAt item (inc idx)) \=))
        {:type :header
         :key (subs item 0 idx)
         :value (str/trim (subs item (inc idx)))}))

    ;; key=value (string data)
    (str/includes? item "=")
    (let [idx (.indexOf item "=")]
      (when (and (pos? idx) (not= (.charAt item (dec idx)) \:))
        {:type :data
         :key (subs item 0 idx)
         :value (subs item (inc idx))}))

    :else nil))

(defn- parse-json-item
  "Parse key:=value as JSON."
  [^String item]
  (when (str/includes? item ":=")
    (let [idx (.indexOf item ":=")]
      (when (pos? idx)
        {:type :json
         :key (subs item 0 idx)
         :value (subs item (+ idx 2))}))))

(defn- build-json-body
  "Build JSON body from data items."
  [data-items json-items]
  (let [data (into {} (map (fn [{:keys [key value]}] [key value]) data-items))
        json (into {} (map (fn [{:keys [key value]}] [key value]) json-items))]
    (when (or (seq data) (seq json))
      (str "{"
           (str/join ", "
             (concat
               (map (fn [[k v]] (str "\"" k "\": \"" v "\"")) data)
               (map (fn [[k v]] (str "\"" k "\": " v)) json)))
           "}"))))

(defn- http-method? [s]
  (contains? #{"GET" "POST" "PUT" "DELETE" "PATCH" "HEAD" "OPTIONS"} (str/upper-case s)))

(defn- parse-client-args
  "Parse HTTPie-style arguments."
  [args]
  (loop [args args
         opts {:headers {}
               :data-items []
               :json-items []}]
    (if (empty? args)
      opts
      (let [[arg & more] args]
        (cond
          ;; Flags
          (or (= arg "-h") (= arg "--help")) (assoc opts :show-help true)
          (= arg "-v") (recur more (assoc opts :verbose true))
          (= arg "--verbose") (recur more (assoc opts :verbose true))
          (= arg "-j") (recur more (assoc opts :json true))
          (= arg "--json") (recur more (assoc opts :json true))
          (= arg "-f") (recur more (assoc opts :form true))
          (= arg "--form") (recur more (assoc opts :form true))
          (= arg "-k") (recur more (assoc opts :insecure? true))
          (= arg "--insecure") (recur more (assoc opts :insecure? true))
          (= arg "-F") (recur more (assoc opts :follow-redirects true))
          (= arg "--follow") (recur more (assoc opts :follow-redirects true))
          (= arg "--timeout") (recur (rest more) (assoc opts :timeout (str->long (first more))))
          (= arg "-b") (recur more (assoc opts :body-only true))
          (= arg "--body") (recur more (assoc opts :body-only true))
          (= arg "--headers") (recur more (assoc opts :headers-only true))

          ;; HTTP method
          (http-method? arg)
          (recur more (assoc opts :method (keyword (str/lower-case arg))))

          ;; URL
          (or (str/starts-with? arg "http://")
              (str/starts-with? arg "https://")
              (str/starts-with? arg "localhost")
              (re-matches #"[\w.-]+:\d+.*" arg)
              (re-matches #"[\w.-]+/.*" arg))
          (let [url (cond
                      (str/starts-with? arg "http") arg
                      :else (str "http://" arg))]
            (recur more (assoc opts :url url)))

          ;; JSON data (key:=value)
          (str/includes? arg ":=")
          (if-let [item (parse-json-item arg)]
            (recur more (update opts :json-items conj item))
            (recur more opts))

          ;; Request item (Header:Value or key=value)
          :else
          (if-let [item (parse-request-item arg)]
            (case (:type item)
              :header (recur more (update opts :headers assoc (:key item) (:value item)))
              :data (recur more (update opts :data-items conj item))
              (recur more opts))
            (recur more opts)))))))

(defn- parse-server-args
  "Parse server arguments."
  [args]
  (loop [args args
         opts {}]
    (if (empty? args)
      opts
      (let [[arg & more] args]
        (cond
          (or (= arg "-h") (= arg "--help")) (assoc opts :show-help true)
          (= arg "-p") (recur (rest more) (assoc opts :port (str->long (first more))))
          (= arg "--port") (recur (rest more) (assoc opts :port (str->long (first more))))
          (= arg "--host") (recur (rest more) (assoc opts :host (first more)))
          (= arg "--bind") (recur (rest more) (assoc opts :host (first more)))
          (= arg "-t") (recur (rest more) (assoc opts :thread (str->long (first more))))
          (= arg "--thread") (recur (rest more) (assoc opts :thread (str->long (first more))))
          (= arg "--ssl") (recur more (assoc opts :ssl? true))
          (= arg "--cert") (recur (rest more) (assoc opts :cert (first more)))
          (= arg "--key") (recur (rest more) (assoc opts :key (first more)))
          :else (recur more opts))))))

;; ============================================================
;; Server Command
;; ============================================================

(defn- demo-handler
  "Demo handler for testing."
  [req]
  (case (:uri req)
    "/" {:status 200
         :headers {"Content-Type" "text/plain"}
         :body "Hello from Zeph!"}
    "/json" {:status 200
             :headers {"Content-Type" "application/json"}
             :body "{\"message\": \"Hello from Zeph!\"}"}
    "/echo" {:status 200
             :headers {"Content-Type" (get-in req [:headers "content-type"] "text/plain")
                       "X-Method" (name (:request-method req))}
             :body (if-let [body (:body req)]
                     (slurp body)
                     "")}
    {:status 404
     :headers {"Content-Type" "text/plain"}
     :body "Not Found"}))

(defn- print-server-help []
  (println "Usage: zeph server [OPTIONS]")
  (println)
  (println "Start an HTTP/HTTPS server with demo endpoints.")
  (println)
  (println "Options:")
  (println "  -p, --port PORT    Port to listen on (default: 8080)")
  (println "      --host HOST    Host/IP to bind (default: 0.0.0.0)")
  (println "  -t, --thread N     Number of worker threads (default: CPU cores)")
  (println "      --ssl          Enable HTTPS")
  (println "      --cert FILE    SSL certificate PEM file")
  (println "      --key FILE     SSL private key PEM file")
  (println "  -h, --help         Show this help")
  (println)
  (println "Examples:")
  (println "  zeph server -p 8080")
  (println "  zeph server --ssl -p 8443")
  (println "  zeph server --ssl --cert cert.pem --key key.pem"))

(defn- run-server-cmd
  "Run the server command."
  [opts]
  (if (:show-help opts)
    (print-server-help)
    (let [port (or (:port opts) 8080)
          host (or (:host opts) "0.0.0.0")
          thread (or (:thread opts) (.availableProcessors (Runtime/getRuntime)))
          ssl? (:ssl? opts)
          server-opts (cond-> {:port port :ip host :thread thread}
                        ssl? (assoc :ssl? true)
                        (:cert opts) (assoc :cert (:cert opts))
                        (:key opts) (assoc :key (:key opts)))]
      (println (str "Starting Zeph " (if ssl? "HTTPS" "HTTP") " server..."))
      (println (str "  Listening: " (if ssl? "https://" "http://") host ":" port))
      (println (str "  Workers: " thread))
      (when ssl?
        (if (and (:cert opts) (:key opts))
          (println (str "  SSL: " (:cert opts) ", " (:key opts)))
          (println "  SSL: built-in localhost certificate")))
      (println)
      (let [stop (server/run-server demo-handler server-opts)]
        (.addShutdownHook
          (Runtime/getRuntime)
          (Thread. (fn []
                     (println "\nShutting down...")
                     (stop))))
        ;; Block forever
        @(promise)))))

;; ============================================================
;; Client Command (HTTPie-style)
;; ============================================================

(def ^:private colors
  {:reset   "\u001b[0m"
   :bold    "\u001b[1m"
   :green   "\u001b[32m"
   :blue    "\u001b[34m"
   :cyan    "\u001b[36m"
   :yellow  "\u001b[33m"
   :magenta "\u001b[35m"
   :red     "\u001b[31m"
   :gray    "\u001b[90m"
   :white   "\u001b[37m"})

(defn- colorize [color text]
  (str (get colors color "") text (:reset colors)))

(defn- format-status [status]
  (let [color (cond
                (< status 300) :green
                (< status 400) :yellow
                :else :red)]
    (str (colorize color (str "HTTP/1.1 " status))
         (colorize :gray (str " " (get {200 "OK" 201 "Created" 204 "No Content"
                                        301 "Moved Permanently" 302 "Found" 304 "Not Modified"
                                        400 "Bad Request" 401 "Unauthorized" 403 "Forbidden"
                                        404 "Not Found" 500 "Internal Server Error"} status ""))))))

(defn- format-header [k v]
  (str (colorize :cyan k) ": " v))

(defn- json-indent
  "Indent JSON string properly with syntax highlighting."
  [^String s indent]
  (let [sb (StringBuilder.)
        len (.length s)]
    (loop [i 0
           depth indent]
      (if (>= i len)
        (str sb)
        (let [c (.charAt s i)]
          (cond
            ;; String - extract entire string and colorize
            (= c \")
            (let [[str-content end-i]
                  (loop [j (inc i) acc "\""]
                    (if (>= j len)
                      [acc j]
                      (let [nc (.charAt s j)]
                        (if (= nc \")
                          [(str acc "\"") (inc j)]
                          (if (= nc \\)
                            (recur (+ j 2) (str acc nc (.charAt s (inc j))))
                            (recur (inc j) (str acc nc)))))))
                  ;; Check if this is a key (followed by colon) or value
                  is-key (loop [k end-i]
                           (if (>= k len)
                             false
                             (let [nc (.charAt s k)]
                               (cond
                                 (= nc \:) true
                                 (Character/isWhitespace nc) (recur (inc k))
                                 :else false))))
                  color (if is-key :blue :green)]
              (.append sb (colorize color str-content))
              (recur end-i depth))

            ;; Opening brace/bracket
            (or (= c \{) (= c \[))
            (let [new-depth (inc depth)
                  indent-next (apply str (repeat new-depth "    "))]
              (.append sb c)
              (.append sb "\n")
              (.append sb indent-next)
              (recur (inc i) new-depth))

            ;; Closing brace/bracket
            (or (= c \}) (= c \]))
            (let [new-depth (dec depth)
                  indent-prev (apply str (repeat new-depth "    "))]
              (.append sb "\n")
              (.append sb indent-prev)
              (.append sb c)
              (recur (inc i) new-depth))

            ;; Comma
            (= c \,)
            (let [indent-cur (apply str (repeat depth "    "))]
              (.append sb c)
              (.append sb "\n")
              (.append sb indent-cur)
              (recur (inc i) depth))

            ;; Colon (key-value separator)
            (= c \:)
            (do (.append sb ": ")
                (recur (inc i) depth))

            ;; Skip whitespace
            (Character/isWhitespace c)
            (recur (inc i) depth)

            ;; Numbers
            (or (Character/isDigit c) (and (= c \-) (< (inc i) len) (Character/isDigit (.charAt s (inc i)))))
            (let [[num rest-i] (loop [j i num-str ""]
                                 (if (and (< j len)
                                          (let [nc (.charAt s j)]
                                            (or (Character/isDigit nc) (= nc \.) (= nc \-) (= nc \e) (= nc \E) (= nc \+))))
                                   (recur (inc j) (str num-str (.charAt s j)))
                                   [num-str j]))]
              (.append sb (colorize :cyan num))
              (recur rest-i depth))

            ;; true
            (and (= c \t) (<= (+ i 4) len) (= (subs s i (+ i 4)) "true"))
            (do (.append sb (colorize :yellow "true"))
                (recur (+ i 4) depth))

            ;; false
            (and (= c \f) (<= (+ i 5) len) (= (subs s i (+ i 5)) "false"))
            (do (.append sb (colorize :yellow "false"))
                (recur (+ i 5) depth))

            ;; null
            (and (= c \n) (<= (+ i 4) len) (= (subs s i (+ i 4)) "null"))
            (do (.append sb (colorize :magenta "null"))
                (recur (+ i 4) depth))

            ;; Other chars
            :else
            (do (.append sb c)
                (recur (inc i) depth))))))))

(defn- try-format-json
  "Try to pretty-print JSON body with syntax highlighting."
  [body]
  (if (and body
           (let [trimmed (str/trim body)]
             (or (str/starts-with? trimmed "{")
                 (str/starts-with? trimmed "["))))
    (try
      (json-indent (str/trim body) 0)
      (catch Exception _ body))
    body))

(defn- print-response
  "Print HTTP response in HTTPie style."
  [{:keys [status headers body error]} {:keys [verbose body-only headers-only]}]
  (if error
    (do
      (println (colorize :red (str "Error: " (ex-message error))))
      1)
    (do
      (when-not body-only
        ;; Status line
        (println (format-status status))
        ;; Headers (always show by default, like HTTPie)
        (doseq [[k v] (sort-by first headers)]
          (println (format-header k v))))
      ;; Body
      (when (and body (not headers-only))
        (println)
        (println (try-format-json body)))
      0)))

(defn- run-client-cmd
  "Run the client command (HTTPie-style)."
  [opts]
  (when (:show-help opts)
    (println "Usage: zeph client [METHOD] URL [ITEM...]")
    (println)
    (println "Make HTTP requests with HTTPie-style syntax.")
    (println)
    (println "Items:")
    (println "  Header:Value   Add HTTP header")
    (println "  key=value      Add JSON string field")
    (println "  key:=123       Add raw JSON value (number, bool, null)")
    (println)
    (println "Options:")
    (println "  -v, --verbose   Show request headers")
    (println "  -b, --body      Only show response body")
    (println "      --headers   Only show response headers")
    (println "  -k, --insecure  Skip SSL certificate verification")
    (println "  -F, --follow    Follow redirects")
    (println "      --timeout N Request timeout in ms (default: 30000)")
    (println "  -h, --help      Show this help")
    (println)
    (println "Examples:")
    (println "  zeph client httpbin.org/get")
    (println "  zeph client POST httpbin.org/post name=John age:=30")
    (println "  zeph client -k https://self-signed.example.com")
    (println "  zeph client example.com User-Agent:Zeph")
    (System/exit 0))

  (let [url (:url opts)]
    (if-not url
      (do
        (println "Error: URL required")
        (println "Usage: zeph client [METHOD] URL [ITEM...]")
        (println "Try: zeph client --help")
        1)
      (let [;; Determine method
            has-data (or (seq (:data-items opts)) (seq (:json-items opts)))
            method (or (:method opts) (if has-data :post :get))

            ;; Build body
            body (when has-data
                   (build-json-body (:data-items opts) (:json-items opts)))

            ;; Build headers
            headers (cond-> (:headers opts)
                      (and has-data (not (:form opts)))
                      (assoc "Content-Type" "application/json"))

            ;; Build request
            request-opts (cond-> {:url url :method method}
                           (seq headers) (assoc :headers headers)
                           body (assoc :body body)
                           (:timeout opts) (assoc :timeout (:timeout opts))
                           (:insecure? opts) (assoc :insecure? true)
                           (:follow-redirects opts) (assoc :follow-redirects true))

            ;; Print request info in verbose mode
            _ (when (:verbose opts)
                (println (colorize :bold (str (str/upper-case (name method)) " " url)))
                (doseq [[k v] headers]
                  (println (format-header k v)))
                (when body
                  (println)
                  (println body))
                (println)
                (println (colorize :gray "---"))
                (println))

            response (try
                       @(client/request request-opts)
                       (catch Exception e
                         {:error e}))]
        (print-response response opts)))))

;; ============================================================
;; Help
;; ============================================================

(defn- print-help []
  (println "Zeph - High-performance HTTP server & client powered by io_uring")
  (println)
  (println "Usage:")
  (println "  zeph <command> [options]")
  (println)
  (println "Commands:")
  (println "  server    Start HTTP/HTTPS server")
  (println "  client    Make HTTP requests (HTTPie-style)")
  (println)
  (println "Run 'zeph <command> -h' for command-specific help.")
  (println)
  (println "Examples:")
  (println "  zeph server -p 8080")
  (println "  zeph server --ssl -p 8443")
  (println "  zeph client httpbin.org/get")
  (println "  zeph client POST httpbin.org/post name=John age:=30"))

;; ============================================================
;; Main
;; ============================================================

(defn -main [& args]
  (if (empty? args)
    (do
      (print-help)
      (System/exit 0))
    (let [[cmd & rest-args] args]
      (case cmd
        "server" (run-server-cmd (parse-server-args rest-args))
        "client" (System/exit (run-client-cmd (parse-client-args rest-args)))
        "help" (print-help)
        "--help" (print-help)
        "-h" (print-help)
        (do
          (println "Unknown command:" cmd)
          (println)
          (print-help)
          (System/exit 1))))))
