/*
 * Decompiled with CFR 0.152.
 */
package zeph.http;

import java.io.InputStream;
import java.lang.foreign.Arena;
import java.lang.foreign.MemorySegment;
import java.lang.foreign.ValueLayout;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import javax.net.ssl.SSLException;
import zeph.buffer.BufferPool;
import zeph.http.HttpParser;
import zeph.http.HttpRequest;
import zeph.http.HttpResponse;
import zeph.http2.Http2FrameReader;
import zeph.http2.Http2ServerHandler;
import zeph.ssl.SslConfig;
import zeph.ssl.SslHandler;
import zeph.uring.IoUring;
import zeph.uring.Socket;

public class HttpsServerMultiRing
implements AutoCloseable {
    private final int port;
    private final int ringCount;
    private final WorkerRing[] workers;
    private final AtomicBoolean running = new AtomicBoolean(true);
    private final AtomicInteger nextWorker = new AtomicInteger(0);
    private final Function<HttpRequest, HttpResponse> handler;
    private final SslConfig sslConfig;
    private final IoUring acceptorRing;
    private final Arena acceptorArena;
    private final int serverFd;
    private final Thread acceptorThread;
    private static final long OP_ACCEPT = 0x100000000000000L;
    private static final long OP_MASK = -72057594037927936L;
    private static final long FD_MASK = 0xFFFFFFFFFFFFFFL;
    private final AtomicLong connectionIdGenerator = new AtomicLong(0L);
    private volatile int acceptorPendingSubmits = 0;

    public HttpsServerMultiRing(String ip, int port, int ringCount, int ringSize, Function<HttpRequest, HttpResponse> handler, SslConfig sslConfig) throws Exception {
        this.port = port;
        this.ringCount = ringCount;
        this.handler = handler;
        this.sslConfig = sslConfig;
        this.workers = new WorkerRing[ringCount];
        for (int i = 0; i < ringCount; ++i) {
            this.workers[i] = new WorkerRing(this, i, ringSize);
        }
        this.acceptorArena = Arena.ofShared();
        this.acceptorRing = new IoUring(ringSize, 0);
        this.serverFd = Socket.createServerSocket();
        try {
            Socket.setReuseAddr(this.serverFd, this.acceptorArena);
            Socket.setReusePort(this.serverFd, this.acceptorArena);
            Socket.bind(this.serverFd, ip, port, this.acceptorArena);
            Socket.listen(this.serverFd, 4096);
        }
        catch (Exception e) {
            Socket.close(this.serverFd);
            this.acceptorRing.close();
            this.acceptorArena.close();
            throw e;
        }
        this.acceptorThread = new Thread(this::acceptLoop, "zeph-ssl-acceptor");
        System.out.println("HTTPS Server (multi-ring) listening on " + ip + ":" + port + " with " + ringCount + " workers");
    }

    private void submitAccept() {
        MemorySegment sqe = this.acceptorRing.getSqe();
        if (sqe != null) {
            this.acceptorRing.prepareAccept(sqe, this.serverFd, null, null, 526336, 0x100000000000000L | (long)this.serverFd);
            this.acceptorRing.submit();
            ++this.acceptorPendingSubmits;
        }
    }

    private void acceptLoop() {
        this.submitAccept();
        while (this.running.get()) {
            try {
                MemorySegment cqe;
                int toSubmit = this.acceptorPendingSubmits;
                this.acceptorPendingSubmits = 0;
                this.acceptorRing.enter(toSubmit, 1, 1);
                while ((cqe = this.acceptorRing.peekCqe()) != null) {
                    long userData = IoUring.getUserData(cqe);
                    int result = IoUring.getResult(cqe);
                    this.acceptorRing.advanceCq();
                    long op = userData & 0xFF00000000000000L;
                    if (op != 0x100000000000000L) continue;
                    if (result >= 0) {
                        int clientFd = result;
                        long connId = this.connectionIdGenerator.incrementAndGet();
                        int workerIdx = Math.abs(this.nextWorker.getAndIncrement() % this.ringCount);
                        this.workers[workerIdx].addConnection(clientFd, connId);
                    }
                    this.submitAccept();
                }
            }
            catch (Exception e) {
                if (!this.running.get()) continue;
                System.err.println("SSL Acceptor error: " + e.getMessage());
                e.printStackTrace();
            }
        }
    }

    public void run() {
        for (WorkerRing worker : this.workers) {
            worker.start();
        }
        this.acceptorThread.start();
        System.out.println("HTTPS Server (multi-ring) started with " + this.ringCount + " workers");
        try {
            this.acceptorThread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public void stop() {
        this.running.set(false);
        for (WorkerRing worker : this.workers) {
            worker.stop();
        }
        try {
            MemorySegment sqe = this.acceptorRing.getSqe();
            if (sqe != null) {
                this.acceptorRing.prepareNop(sqe, 0L);
                this.acceptorRing.submit();
                this.acceptorRing.enter(1, 0, 0);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Override
    public void close() {
        this.stop();
        try {
            this.acceptorThread.join(2000L);
            for (WorkerRing worker : this.workers) {
                worker.thread.interrupt();
                worker.thread.join(1000L);
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        for (WorkerRing worker : this.workers) {
            worker.close();
        }
        try {
            Socket.close(this.serverFd);
        }
        catch (Exception exception) {
            // empty catch block
        }
        this.acceptorRing.close();
        this.acceptorArena.close();
    }

    public int getPort() {
        return this.port;
    }

    public static void main(String[] args) {
        if (args.length < 2) {
            System.out.println("Usage: HttpsServerMultiRing <keystore.p12> <password> [port] [workers]");
            System.exit(1);
        }
        String keystorePath = args[0];
        String password = args[1];
        int port = args.length > 2 ? Integer.parseInt(args[2]) : 8443;
        int workers = args.length > 3 ? Integer.parseInt(args[3]) : Runtime.getRuntime().availableProcessors();
        System.out.println("Starting Multi-Ring HTTPS Server");
        System.out.println("Port: " + port + ", Workers: " + workers);
        try {
            SslConfig sslConfig = SslConfig.fromPkcs12(keystorePath, password);
            try (HttpsServerMultiRing server = new HttpsServerMultiRing("0.0.0.0", port, workers, 256, request -> {
                String path;
                return switch (path = request.getPath()) {
                    case "/", "/hello" -> HttpResponse.ok("Hello from Zeph HTTPS!");
                    case "/json" -> HttpResponse.json("{\"message\": \"Hello!\", \"secure\": true}");
                    default -> HttpResponse.notFound();
                };
            }, sslConfig);){
                Runtime.getRuntime().addShutdownHook(new Thread(() -> {
                    System.out.println("\nShutting down...");
                    server.stop();
                }));
                server.run();
            }
        }
        catch (Exception e) {
            System.err.println("Error: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private class WorkerRing
    implements Runnable {
        final int id;
        final IoUring ring;
        final Arena arena;
        final BufferPool bufferPool;
        final ConcurrentHashMap<Integer, HttpsConnection> connections;
        final ConcurrentLinkedQueue<NewConnection> pendingConnections;
        final Thread thread;
        volatile int pendingSubmits;
        final int eventFd;
        final MemorySegment eventFdBuffer;
        final MemorySegment eventFdWriteBuffer;
        private static final long OP_READ = 0x200000000000000L;
        private static final long OP_WRITE = 0x300000000000000L;
        private static final long OP_CLOSE = 0x400000000000000L;
        private static final long OP_EVENTFD = 0x500000000000000L;
        private static final int BUFFER_SIZE = 16384;
        private static final int POOL_SIZE = 256;
        final /* synthetic */ HttpsServerMultiRing this$0;

        WorkerRing(HttpsServerMultiRing httpsServerMultiRing, int id, int ringSize) throws Exception {
            HttpsServerMultiRing httpsServerMultiRing2 = httpsServerMultiRing;
            Objects.requireNonNull(httpsServerMultiRing2);
            this.this$0 = httpsServerMultiRing2;
            this.connections = new ConcurrentHashMap();
            this.pendingConnections = new ConcurrentLinkedQueue();
            this.pendingSubmits = 0;
            this.id = id;
            this.arena = Arena.ofShared();
            this.ring = new IoUring(ringSize, 0);
            this.bufferPool = new BufferPool(16384, 256);
            this.thread = new Thread((Runnable)this, "zeph-ssl-worker-" + id);
            this.eventFd = IoUring.createEventFd(0, 0);
            this.eventFdBuffer = this.arena.allocate(8L, 8L);
            this.eventFdWriteBuffer = this.arena.allocate(8L, 8L);
            this.eventFdWriteBuffer.set(ValueLayout.JAVA_LONG, 0L, 1L);
        }

        void start() {
            this.thread.start();
        }

        void addConnection(int fd, long connId) {
            this.pendingConnections.offer(new NewConnection(fd, connId));
            this.wakeup();
        }

        private void wakeup() {
            try {
                IoUring.writeEventFd(this.eventFd, this.eventFdWriteBuffer);
            }
            catch (Exception exception) {
                // empty catch block
            }
        }

        private void submitEventFdRead() {
            MemorySegment sqe = this.ring.getSqe();
            if (sqe != null) {
                this.ring.prepareRead(sqe, this.eventFd, this.eventFdBuffer, 8, 0L, 0x500000000000000L);
                this.ring.submit();
                ++this.pendingSubmits;
            }
        }

        private void processPendingConnections() {
            NewConnection nc;
            while ((nc = this.pendingConnections.poll()) != null) {
                BufferPool.PooledBuffer readBuf = this.bufferPool.acquireOrAllocate();
                BufferPool.PooledBuffer writeBuf = this.bufferPool.acquireOrAllocate();
                try {
                    SslHandler sslHandler = new SslHandler(this.this$0.sslConfig.createEngineWithAlpn());
                    sslHandler.beginHandshake();
                    HttpsConnection conn = new HttpsConnection(nc.fd, nc.id, readBuf, writeBuf, sslHandler);
                    this.connections.put(nc.fd, conn);
                    this.submitRead(conn);
                }
                catch (Exception e) {
                    readBuf.release();
                    writeBuf.release();
                    try {
                        Socket.close(nc.fd);
                    }
                    catch (Exception exception) {}
                }
            }
        }

        private void submitRead(HttpsConnection conn) {
            MemorySegment sqe = this.ring.getSqe();
            if (sqe != null) {
                this.ring.prepareRecv(sqe, conn.fd, conn.readBuffer, 16384, 0, 0x200000000000000L | (long)conn.fd);
                this.ring.submit();
                ++this.pendingSubmits;
            }
        }

        private void submitWrite(HttpsConnection conn) {
            if (conn.pendingWrite == null || conn.writeOffset >= conn.pendingWrite.length) {
                return;
            }
            int remaining = conn.pendingWrite.length - conn.writeOffset;
            int toWrite = Math.min(remaining, 16384);
            conn.writeBuffer.asSlice(0L, toWrite).asByteBuffer().put(conn.pendingWrite, conn.writeOffset, toWrite);
            MemorySegment sqe = this.ring.getSqe();
            if (sqe != null) {
                this.ring.prepareSend(sqe, conn.fd, conn.writeBuffer, toWrite, 0, 0x300000000000000L | (long)conn.fd);
                this.ring.submit();
                ++this.pendingSubmits;
            }
        }

        private void submitClose(int fd) {
            MemorySegment sqe = this.ring.getSqe();
            if (sqe != null) {
                this.ring.prepareClose(sqe, fd, 0x400000000000000L | (long)fd);
                this.ring.submit();
                ++this.pendingSubmits;
            }
        }

        @Override
        public void run() {
            this.submitEventFdRead();
            while (this.this$0.running.get()) {
                try {
                    MemorySegment cqe;
                    int toSubmit = this.pendingSubmits;
                    this.pendingSubmits = 0;
                    this.ring.enter(toSubmit, 1, 1);
                    while ((cqe = this.ring.peekCqe()) != null) {
                        long userData = IoUring.getUserData(cqe);
                        int result = IoUring.getResult(cqe);
                        this.ring.advanceCq();
                        long op = userData & 0xFF00000000000000L;
                        int fd = (int)(userData & 0xFFFFFFFFFFFFFFL);
                        if (op == 0x500000000000000L) {
                            this.submitEventFdRead();
                            this.processPendingConnections();
                            continue;
                        }
                        if (op == 0x200000000000000L) {
                            this.handleRead(fd, result);
                            continue;
                        }
                        if (op == 0x300000000000000L) {
                            this.handleWrite(fd, result);
                            continue;
                        }
                        if (op != 0x400000000000000L) continue;
                        this.handleClose(fd);
                    }
                }
                catch (Exception e) {
                    if (!this.this$0.running.get()) continue;
                    System.err.println("SSL Worker " + this.id + " error: " + e.getMessage());
                }
            }
        }

        private void handleRead(int fd, int result) {
            HttpsConnection conn = this.connections.get(fd);
            if (conn == null) {
                return;
            }
            if (result > 0) {
                byte[] data = new byte[result];
                conn.readBuffer.asSlice(0L, result).asByteBuffer().get(data);
                try {
                    if (conn.state == ConnectionState.HANDSHAKING) {
                        this.handleHandshakeRead(conn, data);
                    } else if (conn.state == ConnectionState.CONNECTED) {
                        this.handleApplicationRead(conn, data);
                    }
                }
                catch (SSLException e) {
                    this.closeConnection(fd);
                }
            } else if (result == 0) {
                this.closeConnection(fd);
            } else {
                int err = -result;
                if (err != 11 && err != 35) {
                    this.closeConnection(fd);
                } else {
                    this.submitRead(conn);
                }
            }
        }

        private void handleHandshakeRead(HttpsConnection conn, byte[] data) throws SSLException {
            byte[] handshakeResponse = conn.sslHandler.processHandshake(data);
            if (conn.sslHandler.isHandshakeComplete()) {
                byte[] remaining;
                conn.state = ConnectionState.CONNECTED;
                String protocol = conn.sslHandler.getApplicationProtocol();
                if ("h2".equals(protocol)) {
                    conn.isHttp2 = true;
                    conn.http2Handler = new Http2ServerHandler(this.this$0.handler, this.this$0.port, true);
                }
                if ((remaining = conn.sslHandler.getRemainingData()) != null && remaining.length > 0) {
                    if (handshakeResponse != null && handshakeResponse.length > 0) {
                        conn.pendingWrite = handshakeResponse;
                        conn.writeOffset = 0;
                        conn.pendingAppData = remaining;
                        conn.readAfterWrite = false;
                        this.submitWrite(conn);
                    } else {
                        this.handleApplicationRead(conn, remaining);
                    }
                } else if (handshakeResponse != null && handshakeResponse.length > 0) {
                    conn.pendingWrite = handshakeResponse;
                    conn.writeOffset = 0;
                    conn.readAfterWrite = true;
                    this.submitWrite(conn);
                } else {
                    this.submitRead(conn);
                }
            } else if (handshakeResponse != null && handshakeResponse.length > 0) {
                conn.pendingWrite = handshakeResponse;
                conn.writeOffset = 0;
                conn.readAfterWrite = true;
                this.submitWrite(conn);
            } else {
                this.submitRead(conn);
            }
        }

        private void handleApplicationRead(HttpsConnection conn, byte[] encryptedData) throws SSLException {
            byte[] decrypted = conn.sslHandler.decrypt(encryptedData);
            if (decrypted.length == 0) {
                this.submitRead(conn);
                return;
            }
            if (conn.isHttp2) {
                this.handleHttp2Read(conn, decrypted);
            } else {
                this.handleHttp1Read(conn, decrypted);
            }
        }

        private void handleHttp2Read(HttpsConnection conn, byte[] decrypted) throws SSLException {
            try {
                byte[] response = conn.http2Handler.processData(decrypted);
                if (response != null && response.length > 0) {
                    byte[] encrypted = conn.sslHandler.encrypt(response);
                    conn.pendingWrite = encrypted;
                    conn.writeOffset = 0;
                    conn.readAfterWrite = true;
                    this.submitWrite(conn);
                } else {
                    this.submitRead(conn);
                }
            }
            catch (Http2FrameReader.Http2Exception e) {
                this.closeConnection(conn.fd);
            }
        }

        private void handleHttp1Read(HttpsConnection conn, byte[] decrypted) throws SSLException {
            HttpParser.Result parseResult = conn.parser.parse(decrypted, 0, decrypted.length);
            if (parseResult == HttpParser.Result.COMPLETE) {
                HttpResponse response;
                HttpRequest request = conn.parser.getRequest();
                request.setServerPort(this.this$0.port);
                request.setSecure(true);
                conn.keepAlive = request.isKeepAlive();
                try {
                    response = this.this$0.handler.apply(request);
                }
                catch (Exception e) {
                    response = HttpResponse.serverError();
                }
                if (conn.keepAlive) {
                    response.setHeader("Connection", "keep-alive");
                } else {
                    response.setHeader("Connection", "close");
                    conn.closeAfterWrite = true;
                }
                if (response.isStreaming()) {
                    this.startStreamingResponse(conn, response);
                } else {
                    byte[] plainResponse = response.encode();
                    byte[] encrypted = conn.sslHandler.encrypt(plainResponse);
                    conn.pendingWrite = encrypted;
                    conn.writeOffset = 0;
                    conn.readAfterWrite = false;
                    this.submitWrite(conn);
                }
                conn.parser.reset();
            } else if (parseResult == HttpParser.Result.ERROR) {
                HttpResponse response = HttpResponse.badRequest();
                response.setHeader("Connection", "close");
                byte[] encrypted = conn.sslHandler.encrypt(response.encode());
                conn.pendingWrite = encrypted;
                conn.writeOffset = 0;
                conn.closeAfterWrite = true;
                conn.readAfterWrite = false;
                this.submitWrite(conn);
            } else {
                this.submitRead(conn);
            }
        }

        private void startStreamingResponse(HttpsConnection conn, HttpResponse response) throws SSLException {
            InputStream stream = response.getBodyStream();
            if (stream == null) {
                byte[] encrypted = conn.sslHandler.encrypt(response.encode());
                conn.pendingWrite = encrypted;
                conn.writeOffset = 0;
                conn.readAfterWrite = false;
                this.submitWrite(conn);
                return;
            }
            conn.streamingBody = stream;
            conn.useChunkedEncoding = response.getContentLength() < 0L;
            conn.streamingBuffer = new byte[16384];
            byte[] encrypted = conn.sslHandler.encrypt(response.encodeStreamingHeaders());
            conn.pendingWrite = encrypted;
            conn.writeOffset = 0;
            conn.readAfterWrite = false;
            this.submitWrite(conn);
        }

        private boolean continueStreamingResponse(HttpsConnection conn) throws SSLException {
            if (conn.streamingBody == null) {
                return false;
            }
            try {
                byte[] chunk;
                int n = conn.streamingBody.read(conn.streamingBuffer);
                if (n <= 0) {
                    byte[] finalChunk = null;
                    if (conn.useChunkedEncoding) {
                        finalChunk = "0\r\n\r\n".getBytes(StandardCharsets.US_ASCII);
                    }
                    conn.streamingBody.close();
                    conn.streamingBody = null;
                    conn.streamingBuffer = null;
                    if (finalChunk != null) {
                        conn.pendingWrite = conn.sslHandler.encrypt(finalChunk);
                        conn.writeOffset = 0;
                        return true;
                    }
                    return false;
                }
                if (conn.useChunkedEncoding) {
                    String sizeHex = Integer.toHexString(n);
                    byte[] header = (sizeHex + "\r\n").getBytes(StandardCharsets.US_ASCII);
                    byte[] footer = "\r\n".getBytes(StandardCharsets.US_ASCII);
                    chunk = new byte[header.length + n + footer.length];
                    System.arraycopy(header, 0, chunk, 0, header.length);
                    System.arraycopy(conn.streamingBuffer, 0, chunk, header.length, n);
                    System.arraycopy(footer, 0, chunk, header.length + n, footer.length);
                } else {
                    chunk = new byte[n];
                    System.arraycopy(conn.streamingBuffer, 0, chunk, 0, n);
                }
                conn.pendingWrite = conn.sslHandler.encrypt(chunk);
                conn.writeOffset = 0;
                return true;
            }
            catch (Exception e) {
                try {
                    conn.streamingBody.close();
                }
                catch (Exception exception) {
                    // empty catch block
                }
                conn.streamingBody = null;
                conn.streamingBuffer = null;
                return false;
            }
        }

        private void handleWrite(int fd, int result) {
            block39: {
                HttpsConnection conn = this.connections.get(fd);
                if (conn == null) {
                    return;
                }
                if (result > 0) {
                    conn.writeOffset += result;
                    if (conn.writeOffset < conn.pendingWrite.length) {
                        this.submitWrite(conn);
                    } else {
                        conn.pendingWrite = null;
                        conn.writeOffset = 0;
                        if (conn.streamingBody != null) {
                            try {
                                if (this.continueStreamingResponse(conn)) {
                                    this.submitWrite(conn);
                                } else if (conn.closeAfterWrite) {
                                    this.closeConnection(fd);
                                } else if (conn.keepAlive) {
                                    this.submitRead(conn);
                                } else {
                                    this.closeConnection(fd);
                                }
                            }
                            catch (SSLException e) {
                                this.closeConnection(fd);
                            }
                            return;
                        }
                        if (conn.isHttp2 && conn.http2Handler != null && conn.http2Handler.hasPendingStreaming()) {
                            try {
                                byte[] nextChunk = conn.http2Handler.continueStreaming();
                                if (nextChunk != null && nextChunk.length > 0) {
                                    byte[] encrypted = conn.sslHandler.encrypt(nextChunk);
                                    conn.pendingWrite = encrypted;
                                    conn.writeOffset = 0;
                                    this.submitWrite(conn);
                                } else if (conn.http2Handler.isOpen()) {
                                    this.submitRead(conn);
                                } else {
                                    this.closeConnection(fd);
                                }
                            }
                            catch (SSLException e) {
                                this.closeConnection(fd);
                            }
                            return;
                        }
                        if (conn.closeAfterWrite) {
                            try {
                                byte[] closeData = conn.sslHandler.close();
                                if (closeData != null && closeData.length > 0) {
                                    conn.pendingWrite = closeData;
                                    conn.writeOffset = 0;
                                    conn.closeAfterWrite = false;
                                    conn.state = ConnectionState.CLOSING;
                                    this.submitWrite(conn);
                                    break block39;
                                }
                                this.closeConnection(fd);
                            }
                            catch (SSLException e) {
                                this.closeConnection(fd);
                            }
                        } else if (conn.state == ConnectionState.CLOSING) {
                            this.closeConnection(fd);
                        } else if (conn.pendingAppData != null) {
                            byte[] appData = conn.pendingAppData;
                            conn.pendingAppData = null;
                            try {
                                this.handleApplicationRead(conn, appData);
                            }
                            catch (SSLException e) {
                                this.closeConnection(fd);
                            }
                        } else if (conn.readAfterWrite) {
                            this.submitRead(conn);
                        } else if (conn.keepAlive) {
                            this.submitRead(conn);
                        } else {
                            this.closeConnection(fd);
                        }
                    }
                } else if (result < 0) {
                    int err = -result;
                    if (err != 11 && err != 35) {
                        this.closeConnection(fd);
                    } else {
                        this.submitWrite(conn);
                    }
                }
            }
        }

        private void handleClose(int fd) {
            HttpsConnection conn = this.connections.remove(fd);
            if (conn != null) {
                conn.close();
            }
        }

        private void closeConnection(int fd) {
            if (this.connections.containsKey(fd)) {
                this.submitClose(fd);
            }
        }

        void stop() {
            this.wakeup();
        }

        void close() {
            for (HttpsConnection conn : this.connections.values()) {
                try {
                    Socket.close(conn.fd);
                }
                catch (Exception exception) {
                    // empty catch block
                }
                conn.close();
            }
            this.connections.clear();
            IoUring.closeEventFd(this.eventFd);
            this.ring.close();
            this.bufferPool.close();
            this.arena.close();
        }

        record NewConnection(int fd, long id) {
        }
    }

    private static class HttpsConnection {
        final int fd;
        final long id;
        final BufferPool.PooledBuffer readPooledBuffer;
        final BufferPool.PooledBuffer writePooledBuffer;
        final MemorySegment readBuffer;
        final MemorySegment writeBuffer;
        final HttpParser parser;
        final SslHandler sslHandler;
        ConnectionState state = ConnectionState.HANDSHAKING;
        byte[] pendingWrite;
        int writeOffset;
        boolean keepAlive = true;
        boolean closeAfterWrite = false;
        boolean readAfterWrite = false;
        byte[] pendingAppData;
        Http2ServerHandler http2Handler;
        boolean isHttp2 = false;
        InputStream streamingBody;
        boolean useChunkedEncoding;
        byte[] streamingBuffer;
        static final int STREAMING_BUFFER_SIZE = 16384;
        private byte[] handshakeBuffer;
        private int handshakeOffset;

        HttpsConnection(int fd, long id, BufferPool.PooledBuffer readBuf, BufferPool.PooledBuffer writeBuf, SslHandler sslHandler) {
            this.fd = fd;
            this.id = id;
            this.readPooledBuffer = readBuf;
            this.writePooledBuffer = writeBuf;
            this.readBuffer = readBuf.segment();
            this.writeBuffer = writeBuf.segment();
            this.parser = new HttpParser();
            this.sslHandler = sslHandler;
            this.handshakeBuffer = new byte[16384];
            this.handshakeOffset = 0;
        }

        void appendHandshakeData(byte[] data) {
            if (this.handshakeOffset + data.length > this.handshakeBuffer.length) {
                byte[] newBuf = new byte[this.handshakeBuffer.length * 2];
                System.arraycopy(this.handshakeBuffer, 0, newBuf, 0, this.handshakeOffset);
                this.handshakeBuffer = newBuf;
            }
            System.arraycopy(data, 0, this.handshakeBuffer, this.handshakeOffset, data.length);
            this.handshakeOffset += data.length;
        }

        byte[] getHandshakeData() {
            if (this.handshakeOffset == 0) {
                return null;
            }
            byte[] data = new byte[this.handshakeOffset];
            System.arraycopy(this.handshakeBuffer, 0, data, 0, this.handshakeOffset);
            return data;
        }

        void clearHandshakeData() {
            this.handshakeOffset = 0;
        }

        void close() {
            this.readPooledBuffer.release();
            this.writePooledBuffer.release();
            if (this.streamingBody != null) {
                try {
                    this.streamingBody.close();
                }
                catch (Exception exception) {
                    // empty catch block
                }
                this.streamingBody = null;
            }
        }
    }

    private static enum ConnectionState {
        HANDSHAKING,
        CONNECTED,
        CLOSING;

    }
}

