/*
 * Decompiled with CFR 0.152.
 */
package zeph.http;

import java.lang.foreign.Arena;
import java.lang.foreign.MemorySegment;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import zeph.http.HttpParser;
import zeph.http.HttpRequest;
import zeph.http.HttpResponse;
import zeph.uring.IoUring;
import zeph.uring.Socket;

public class HttpServer
implements AutoCloseable {
    private static final long OP_ACCEPT = 0x100000000000000L;
    private static final long OP_READ = 0x200000000000000L;
    private static final long OP_WRITE = 0x300000000000000L;
    private static final long OP_CLOSE = 0x400000000000000L;
    private static final long OP_MASK = -72057594037927936L;
    private static final long FD_MASK = 0xFFFFFFFFFFFFFFL;
    private final IoUring ring;
    private final Arena arena;
    private final int serverFd;
    private final int port;
    private final AtomicBoolean running = new AtomicBoolean(true);
    private final AtomicLong connectionIdGenerator = new AtomicLong(0L);
    private final ConcurrentHashMap<Integer, HttpConnection> connections = new ConcurrentHashMap();
    private final Function<HttpRequest, HttpResponse> handler;
    private static final int BUFFER_SIZE = 16384;
    private volatile int pendingSubmits = 0;

    public HttpServer(String ip, int port, int ringSize, Function<HttpRequest, HttpResponse> handler) throws Exception {
        this.port = port;
        this.arena = Arena.ofShared();
        this.handler = handler;
        this.ring = new IoUring(ringSize, 0);
        this.serverFd = Socket.createServerSocket();
        try {
            Socket.setReuseAddr(this.serverFd, this.arena);
            Socket.setReusePort(this.serverFd, this.arena);
            Socket.bind(this.serverFd, ip, port, this.arena);
            Socket.listen(this.serverFd, 4096);
            System.out.println("HTTP Server listening on " + ip + ":" + port);
            this.submitAccept();
        }
        catch (Exception e) {
            Socket.close(this.serverFd);
            this.ring.close();
            this.arena.close();
            throw e;
        }
    }

    private void submitAccept() {
        MemorySegment sqe = this.ring.getSqe();
        if (sqe != null) {
            this.ring.prepareAccept(sqe, this.serverFd, null, null, 526336, 0x100000000000000L | (long)this.serverFd);
            this.ring.submit();
            ++this.pendingSubmits;
        }
    }

    private void submitRead(HttpConnection conn) {
        MemorySegment sqe = this.ring.getSqe();
        if (sqe != null) {
            this.ring.prepareRecv(sqe, conn.fd, conn.readBuffer, 16384, 0, 0x200000000000000L | (long)conn.fd);
            this.ring.submit();
            ++this.pendingSubmits;
        }
    }

    private void submitWrite(HttpConnection conn) {
        if (conn.pendingWrite == null || conn.writeOffset >= conn.pendingWrite.length) {
            return;
        }
        int remaining = conn.pendingWrite.length - conn.writeOffset;
        int toWrite = Math.min(remaining, 16384);
        conn.writeBuffer.asSlice(0L, toWrite).asByteBuffer().put(conn.pendingWrite, conn.writeOffset, toWrite);
        MemorySegment sqe = this.ring.getSqe();
        if (sqe != null) {
            this.ring.prepareSend(sqe, conn.fd, conn.writeBuffer, toWrite, 0, 0x300000000000000L | (long)conn.fd);
            this.ring.submit();
            ++this.pendingSubmits;
        }
    }

    private void submitClose(int fd) {
        MemorySegment sqe = this.ring.getSqe();
        if (sqe != null) {
            this.ring.prepareClose(sqe, fd, 0x400000000000000L | (long)fd);
            this.ring.submit();
            ++this.pendingSubmits;
        }
    }

    public void run() {
        System.out.println("HTTP Server started");
        while (this.running.get()) {
            try {
                MemorySegment cqe;
                int toSubmit = this.pendingSubmits;
                this.pendingSubmits = 0;
                this.ring.enter(toSubmit, 1, 1);
                while ((cqe = this.ring.peekCqe()) != null) {
                    long userData = IoUring.getUserData(cqe);
                    int result = IoUring.getResult(cqe);
                    this.ring.advanceCq();
                    long op = userData & 0xFF00000000000000L;
                    int fd = (int)(userData & 0xFFFFFFFFFFFFFFL);
                    if (op == 0x100000000000000L) {
                        this.handleAccept(result);
                        continue;
                    }
                    if (op == 0x200000000000000L) {
                        this.handleRead(fd, result);
                        continue;
                    }
                    if (op == 0x300000000000000L) {
                        this.handleWrite(fd, result);
                        continue;
                    }
                    if (op != 0x400000000000000L) continue;
                    this.handleClose(fd, result);
                }
            }
            catch (Exception e) {
                if (!this.running.get()) continue;
                System.err.println("Server error: " + e.getMessage());
                e.printStackTrace();
            }
        }
        System.out.println("HTTP Server stopped");
    }

    private void handleAccept(int result) {
        if (result >= 0) {
            int clientFd = result;
            HttpConnection conn = new HttpConnection(clientFd, this.connectionIdGenerator.incrementAndGet());
            this.connections.put(clientFd, conn);
            this.submitRead(conn);
            this.submitAccept();
        } else {
            System.err.println("Accept failed: " + -result);
            this.submitAccept();
        }
    }

    private void handleRead(int fd, int result) {
        HttpConnection conn = this.connections.get(fd);
        if (conn == null) {
            return;
        }
        if (result > 0) {
            byte[] data = new byte[result];
            conn.readBuffer.asSlice(0L, result).asByteBuffer().get(data);
            HttpParser.Result parseResult = conn.parser.parse(data, 0, result);
            if (parseResult == HttpParser.Result.COMPLETE) {
                HttpResponse response;
                HttpRequest request = conn.parser.getRequest();
                request.setServerPort(this.port);
                conn.keepAlive = request.isKeepAlive();
                try {
                    response = this.handler.apply(request);
                }
                catch (Exception e) {
                    System.err.println("Handler error: " + e.getMessage());
                    response = HttpResponse.serverError();
                }
                if (conn.keepAlive) {
                    response.setHeader("Connection", "keep-alive");
                } else {
                    response.setHeader("Connection", "close");
                    conn.closeAfterWrite = true;
                }
                conn.pendingWrite = response.encode();
                conn.writeOffset = 0;
                this.submitWrite(conn);
                conn.parser.reset();
            } else if (parseResult == HttpParser.Result.ERROR) {
                HttpResponse response = HttpResponse.badRequest();
                response.setHeader("Connection", "close");
                conn.pendingWrite = response.encode();
                conn.writeOffset = 0;
                conn.closeAfterWrite = true;
                this.submitWrite(conn);
            } else {
                this.submitRead(conn);
            }
        } else if (result == 0) {
            this.closeConnection(fd);
        } else {
            int err = -result;
            if (err != 11 && err != 35) {
                this.closeConnection(fd);
            } else {
                this.submitRead(conn);
            }
        }
    }

    private void handleWrite(int fd, int result) {
        HttpConnection conn = this.connections.get(fd);
        if (conn == null) {
            return;
        }
        if (result > 0) {
            conn.writeOffset += result;
            if (conn.writeOffset < conn.pendingWrite.length) {
                this.submitWrite(conn);
            } else {
                conn.pendingWrite = null;
                conn.writeOffset = 0;
                if (conn.closeAfterWrite) {
                    this.closeConnection(fd);
                } else if (conn.keepAlive) {
                    this.submitRead(conn);
                } else {
                    this.closeConnection(fd);
                }
            }
        } else if (result < 0) {
            int err = -result;
            if (err != 11 && err != 35) {
                this.closeConnection(fd);
            } else {
                this.submitWrite(conn);
            }
        }
    }

    private void handleClose(int fd, int result) {
        HttpConnection conn = this.connections.remove(fd);
        if (conn != null) {
            conn.close();
        }
    }

    private void closeConnection(int fd) {
        HttpConnection conn = this.connections.get(fd);
        if (conn != null) {
            this.submitClose(fd);
        }
    }

    public void stop() {
        this.running.set(false);
        try {
            MemorySegment sqe = this.ring.getSqe();
            if (sqe != null) {
                this.ring.prepareNop(sqe, 0L);
                this.ring.submit();
                this.ring.enter(1, 0, 0);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Override
    public void close() {
        this.stop();
        for (HttpConnection conn : this.connections.values()) {
            try {
                Socket.close(conn.fd);
            }
            catch (Exception exception) {
                // empty catch block
            }
            conn.close();
        }
        this.connections.clear();
        try {
            Socket.close(this.serverFd);
        }
        catch (Exception exception) {
            // empty catch block
        }
        this.ring.close();
        this.arena.close();
    }

    public int getPort() {
        return this.port;
    }

    public int getConnectionCount() {
        return this.connections.size();
    }

    public static void main(String[] args) {
        int port = 8080;
        if (args.length > 0) {
            port = Integer.parseInt(args[0]);
        }
        System.out.println("Starting HTTP Server on port " + port);
        try (HttpServer server = new HttpServer("0.0.0.0", port, 256, request -> {
            String path;
            return switch (path = request.getPath()) {
                case "/", "/hello" -> HttpResponse.ok("Hello from Zeph!");
                case "/json" -> HttpResponse.json("{\"message\": \"Hello from Zeph!\", \"status\": \"ok\"}");
                case "/echo" -> {
                    String body = request.getBody() != null ? new String(request.getBody()) : "";
                    yield HttpResponse.ok(body);
                }
                default -> HttpResponse.notFound();
            };
        });){
            Runtime.getRuntime().addShutdownHook(new Thread(() -> {
                System.out.println("\nShutting down...");
                server.stop();
            }));
            server.run();
        }
        catch (Exception e) {
            System.err.println("Error: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private static class HttpConnection {
        final int fd;
        final long id;
        final MemorySegment readBuffer;
        final MemorySegment writeBuffer;
        final Arena arena;
        final HttpParser parser;
        byte[] pendingWrite;
        int writeOffset;
        boolean keepAlive = true;
        boolean closeAfterWrite = false;

        HttpConnection(int fd, long id) {
            this.fd = fd;
            this.id = id;
            this.arena = Arena.ofShared();
            this.readBuffer = this.arena.allocate(16384L);
            this.writeBuffer = this.arena.allocate(16384L);
            this.parser = new HttpParser();
        }

        void close() {
            this.arena.close();
        }
    }
}

