/*
 * Decompiled with CFR 0.152.
 */
package zeph.client;

import java.lang.foreign.Arena;
import java.lang.foreign.MemorySegment;
import java.lang.foreign.ValueLayout;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import javax.net.ssl.SSLEngine;
import zeph.client.HttpClientConnection;
import zeph.client.HttpClientConnectionPool;
import zeph.client.HttpClientRequest;
import zeph.client.HttpClientResponse;
import zeph.client.HttpResponseParser;
import zeph.http2.Http2ClientHandler;
import zeph.http2.Http2FrameReader;
import zeph.ssl.SslConfig;
import zeph.ssl.SslHandler;
import zeph.uring.IoUring;
import zeph.uring.Socket;

public class HttpClientWorker
implements Runnable {
    private static final long OP_CONNECT = 0x100000000000000L;
    private static final long OP_READ = 0x200000000000000L;
    private static final long OP_WRITE = 0x300000000000000L;
    private static final long OP_CLOSE = 0x400000000000000L;
    private static final long OP_EVENTFD = 0x500000000000000L;
    private static final long OP_MASK = -72057594037927936L;
    private static final long ID_MASK = 0xFFFFFFFFFFFFFFL;
    private final int workerId;
    private final HttpClientConnectionPool pool;
    private final SslConfig sslConfig;
    private final SslConfig sslConfigInsecure;
    private final IoUring ring;
    private final Arena arena;
    private final int eventFd;
    private final MemorySegment eventFdBuffer;
    private final MemorySegment eventFdWriteBuffer;
    private final ConcurrentLinkedQueue<InFlightRequest> pendingRequests = new ConcurrentLinkedQueue();
    private final ConcurrentHashMap<Long, InFlightRequest> inFlight = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, HttpClientConnection> connections = new ConcurrentHashMap();
    private volatile boolean running = true;
    private int pendingSubmits = 0;
    private PendingRedirect pendingRedirect = null;
    private static final ValueLayout.OfByte ValueLayout_JAVA_BYTE = java.lang.foreign.ValueLayout.JAVA_BYTE;

    public HttpClientWorker(int workerId, HttpClientConnectionPool pool, SslConfig sslConfig, int ringSize) throws Exception {
        this.workerId = workerId;
        this.pool = pool;
        this.sslConfig = sslConfig;
        this.sslConfigInsecure = SslConfig.forClientInsecure();
        this.arena = Arena.ofShared();
        this.ring = new IoUring(ringSize, 0);
        this.eventFd = IoUring.createEventFd(0, 0);
        this.eventFdBuffer = this.arena.allocate(8L, 8L);
        this.eventFdWriteBuffer = this.arena.allocate(8L, 8L);
        this.eventFdWriteBuffer.set(java.lang.foreign.ValueLayout.JAVA_LONG, 0L, 1L);
    }

    public void submit(InFlightRequest request) {
        this.pendingRequests.offer(request);
        this.wakeup();
    }

    public void wakeup() {
        try {
            IoUring.writeEventFd(this.eventFd, this.eventFdWriteBuffer);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void submitEventFdRead() {
        MemorySegment sqe = this.ring.getSqe();
        if (sqe != null) {
            this.ring.prepareRead(sqe, this.eventFd, this.eventFdBuffer, 8, 0L, 0x500000000000000L);
            this.ring.submit();
            ++this.pendingSubmits;
        }
    }

    public void stop() {
        this.running = false;
        this.wakeup();
    }

    @Override
    public void run() {
        this.submitEventFdRead();
        try {
            while (this.running) {
                int toSubmit = this.pendingSubmits;
                this.pendingSubmits = 0;
                try {
                    this.ring.enter(toSubmit, 1, 1);
                }
                catch (Exception e) {
                    if (!this.running) continue;
                    System.err.println("Worker " + this.workerId + " enter error: " + e.getMessage());
                    continue;
                }
                this.processCompletions();
            }
        }
        finally {
            this.cleanup();
        }
    }

    private void processPendingRequests() {
        InFlightRequest req;
        while ((req = this.pendingRequests.poll()) != null) {
            try {
                this.startRequest(req);
            }
            catch (Exception e) {
                this.completeWithError(req, e);
            }
        }
    }

    private void startRequest(InFlightRequest req) throws Exception {
        HttpClientRequest request = req.request;
        HttpClientConnection conn = this.pool.tryAcquire(request.getHost(), request.getPort(), request.isSecure());
        if (conn != null) {
            conn.setWorkerIndex(this.workerId);
            req.connection = conn;
            this.connections.put(conn.getId(), conn);
            this.inFlight.put(conn.getId(), req);
            this.sendRequest(req);
        } else {
            conn = this.pool.createConnection(request.getHost(), request.getPort(), request.isSecure());
            if (conn == null) {
                throw new Exception("Connection pool exhausted");
            }
            conn.setWorkerIndex(this.workerId);
            conn.initBuffers(this.arena);
            req.connection = conn;
            this.connections.put(conn.getId(), conn);
            this.inFlight.put(conn.getId(), req);
            this.startConnect(req);
        }
    }

    private void startConnect(InFlightRequest req) throws Exception {
        HttpClientConnection conn = req.connection;
        HttpClientRequest request = req.request;
        String ip = Socket.resolveHost(request.getHost());
        int fd = Socket.createClientSocket();
        conn.setFd(fd);
        Socket.setTcpNoDelay(fd, this.arena);
        Socket.setKeepAlive(fd, this.arena);
        MemorySegment sockAddr = Socket.createSockAddr(this.arena, ip, request.getPort());
        conn.setSockAddr(sockAddr);
        MemorySegment sqe = this.ring.getSqe();
        if (sqe == null) {
            throw new Exception("SQE queue full");
        }
        this.ring.prepareConnect(sqe, fd, sockAddr, Socket.getSockAddrInSize(), 0x100000000000000L | conn.getId());
        this.ring.submit();
        ++this.pendingSubmits;
    }

    private void processCompletions() {
        MemorySegment cqe;
        while ((cqe = this.ring.peekCqe()) != null) {
            long userData = IoUring.getUserData(cqe);
            int result = IoUring.getResult(cqe);
            this.ring.advanceCq();
            long op = userData & 0xFF00000000000000L;
            long connId = userData & 0xFFFFFFFFFFFFFFL;
            if (op == 0x500000000000000L) {
                this.submitEventFdRead();
                this.processPendingRequests();
                continue;
            }
            HttpClientConnection conn = this.connections.get(connId);
            InFlightRequest req = this.inFlight.get(connId);
            if (conn == null || req == null) continue;
            try {
                if (op == 0x100000000000000L) {
                    this.handleConnectComplete(conn, req, result);
                    continue;
                }
                if (op == 0x300000000000000L) {
                    this.handleWriteComplete(conn, req, result);
                    continue;
                }
                if (op == 0x200000000000000L) {
                    this.handleReadComplete(conn, req, result);
                    continue;
                }
                if (op != 0x400000000000000L) continue;
                this.handleCloseComplete(conn, req);
            }
            catch (Exception e) {
                this.completeWithError(req, e);
                this.closeConnection(conn);
            }
        }
    }

    private void handleConnectComplete(HttpClientConnection conn, InFlightRequest req, int result) throws Exception {
        if (result < 0) {
            throw new Exception("Connect failed with error: " + -result);
        }
        conn.touch();
        if (conn.isSecure()) {
            conn.setState(HttpClientConnection.State.HANDSHAKING);
            this.startSslHandshake(conn, req);
        } else {
            conn.setState(HttpClientConnection.State.IN_USE);
            this.sendRequest(req);
        }
    }

    private void startSslHandshake(HttpClientConnection conn, InFlightRequest req) throws Exception {
        SslConfig config = req.request.isInsecure() ? this.sslConfigInsecure : this.sslConfig;
        SSLEngine engine = config.createClientEngineWithAlpn(req.request.getHost(), req.request.getPort());
        SslHandler sslHandler = new SslHandler(engine);
        conn.initSsl(sslHandler);
        sslHandler.beginHandshake();
        byte[] handshakeData = sslHandler.processHandshake(null);
        if (handshakeData != null && handshakeData.length > 0) {
            this.sendData(conn, handshakeData);
        } else {
            this.submitRead(conn);
        }
    }

    private void sendRequest(InFlightRequest req) throws Exception {
        HttpClientConnection conn = req.connection;
        byte[] requestData = req.request.encode();
        if (conn.isSecure()) {
            SslHandler ssl = conn.getSslHandler();
            byte[] encrypted = ssl.encrypt(requestData);
            this.sendData(conn, encrypted);
        } else {
            this.sendData(conn, requestData);
        }
    }

    private void sendData(HttpClientConnection conn, byte[] data) throws Exception {
        MemorySegment writeBuffer = conn.getWriteBuffer();
        int len = Math.min(data.length, (int)writeBuffer.byteSize());
        MemorySegment.copy(data, 0, writeBuffer, ValueLayout.JAVA_BYTE, 0L, len);
        MemorySegment sqe = this.ring.getSqe();
        if (sqe == null) {
            throw new Exception("SQE queue full");
        }
        this.ring.prepareSend(sqe, conn.getFd(), writeBuffer, len, 0, 0x300000000000000L | conn.getId());
        this.ring.submit();
        ++this.pendingSubmits;
    }

    private void submitRead(HttpClientConnection conn) throws Exception {
        MemorySegment readBuffer = conn.getReadBuffer();
        MemorySegment sqe = this.ring.getSqe();
        if (sqe == null) {
            throw new Exception("SQE queue full");
        }
        this.ring.prepareRecv(sqe, conn.getFd(), readBuffer, (int)readBuffer.byteSize(), 0, 0x200000000000000L | conn.getId());
        this.ring.submit();
        ++this.pendingSubmits;
    }

    private void handleWriteComplete(HttpClientConnection conn, InFlightRequest req, int result) throws Exception {
        if (result < 0) {
            throw new Exception("Write failed with error: " + -result);
        }
        conn.touch();
        if (conn.getState() == HttpClientConnection.State.HANDSHAKING && Boolean.TRUE.equals(conn.getCurrentRequest())) {
            this.handleHandshakeContinue(conn, req);
        } else if ("SEND_H2_PREFACE".equals(conn.getCurrentRequest())) {
            conn.setCurrentRequest(null);
            this.sendHttp2PrefaceAndRequest(conn, req);
        } else if ("SEND_H1_REQUEST".equals(conn.getCurrentRequest())) {
            conn.setCurrentRequest(null);
            this.sendRequest(req);
        } else if (conn.getCurrentRequest() instanceof byte[]) {
            byte[] pendingData = (byte[])conn.getCurrentRequest();
            conn.setCurrentRequest(null);
            if (pendingData.length > 5 && (pendingData[0] & 0xFF) == 23) {
                this.sendData(conn, pendingData);
            } else if (conn.isHttp2()) {
                this.processHttp2Data(conn, req, pendingData);
            } else {
                this.processResponseData(conn, req, pendingData);
            }
        } else if (conn.getCurrentRequest() instanceof PendingRedirect) {
            PendingRedirect redirect = (PendingRedirect)conn.getCurrentRequest();
            conn.setCurrentRequest(null);
            try {
                this.handleHttp2Redirect(redirect.conn, redirect.stream, redirect.response, redirect.location);
            }
            catch (Exception e) {
                redirect.stream.getFuture().completeExceptionally(e);
                this.closeConnection(conn);
            }
        } else {
            this.submitRead(conn);
        }
    }

    private void sendDataThenProcessDecrypted(HttpClientConnection conn, InFlightRequest req, byte[] dataToSend) throws Exception {
        this.sendData(conn, dataToSend);
    }

    private void handleReadComplete(HttpClientConnection conn, InFlightRequest req, int result) throws Exception {
        if (result < 0) {
            throw new Exception("Read failed with error: " + -result);
        }
        if (result == 0) {
            throw new Exception("Connection closed by server");
        }
        conn.touch();
        MemorySegment readBuffer = conn.getReadBuffer();
        byte[] data = new byte[result];
        MemorySegment.copy(readBuffer, ValueLayout.JAVA_BYTE, 0L, data, 0, result);
        if (conn.getState() == HttpClientConnection.State.HANDSHAKING) {
            this.handleSslHandshake(conn, req, data);
        } else if (conn.isSecure()) {
            byte[] wrapData;
            SslHandler ssl = conn.getSslHandler();
            byte[] decrypted = ssl.decrypt(data);
            if (ssl.hasPendingWrap() && (wrapData = ssl.processPendingWrap()) != null && wrapData.length > 0) {
                if (decrypted != null && decrypted.length > 0) {
                    conn.setCurrentRequest(decrypted);
                }
                this.sendDataThenProcessDecrypted(conn, req, wrapData);
                return;
            }
            if (decrypted != null && decrypted.length > 0) {
                if (conn.isHttp2()) {
                    this.processHttp2Data(conn, req, decrypted);
                } else {
                    this.processResponseData(conn, req, decrypted);
                }
            } else {
                this.submitRead(conn);
            }
        } else {
            this.processResponseData(conn, req, data);
        }
    }

    private void processHttp2Data(HttpClientConnection conn, InFlightRequest req, byte[] data) throws Exception {
        Http2ClientHandler h2 = conn.getHttp2Handler();
        try {
            this.pendingRedirect = null;
            byte[] responseFrames = h2.processData(data);
            PendingRedirect redirect = this.pendingRedirect;
            this.pendingRedirect = null;
            if (responseFrames != null && responseFrames.length > 0) {
                SslHandler ssl = conn.getSslHandler();
                byte[] encrypted = ssl.encrypt(responseFrames);
                if (redirect != null) {
                    conn.setCurrentRequest(redirect);
                }
                this.sendData(conn, encrypted);
            } else if (redirect != null) {
                try {
                    this.handleHttp2Redirect(redirect.conn, redirect.stream, redirect.response, redirect.location);
                }
                catch (Exception e) {
                    redirect.stream.getFuture().completeExceptionally(e);
                }
            } else if (h2.getActiveStreamCount() > 0) {
                this.submitRead(conn);
            } else if (h2.isOpen()) {
                this.releaseHttp2Connection(conn);
            }
        }
        catch (Http2FrameReader.Http2Exception e) {
            h2.failAllStreams(e);
            this.closeConnection(conn);
        }
    }

    private void releaseHttp2Connection(HttpClientConnection conn) {
        InFlightRequest req = this.inFlight.remove(conn.getId());
        if (req != null) {
            this.connections.remove(conn.getId());
        }
        this.closeConnection(conn);
    }

    private void handleSslHandshake(HttpClientConnection conn, InFlightRequest req, byte[] data) throws Exception {
        SslHandler ssl = conn.getSslHandler();
        byte[] response = ssl.processHandshake(data);
        if (ssl.isHandshakeComplete()) {
            String protocol = ssl.getApplicationProtocol();
            boolean isHttp2 = "h2".equals(protocol);
            if (isHttp2) {
                conn.initHttp2(this.createHttp2StreamListener(conn));
            }
            conn.setState(HttpClientConnection.State.IN_USE);
            byte[] remaining = ssl.getRemainingData();
            if (remaining != null && remaining.length > 0) {
                ssl.setPendingCiphertext(remaining);
            }
            if (response != null && response.length > 0) {
                if (isHttp2) {
                    this.sendHandshakeAndHttp2Request(conn, req, response);
                } else {
                    this.sendHandshakeThenRequest(conn, req, response);
                }
            } else if (remaining != null && remaining.length > 0) {
                if (isHttp2) {
                    this.sendHttp2PrefaceAndRequest(conn, req);
                } else {
                    this.sendRequest(req);
                }
            } else if (isHttp2) {
                this.sendHttp2PrefaceAndRequest(conn, req);
            } else {
                this.sendRequest(req);
            }
        } else if (response != null && response.length > 0) {
            this.sendDataThenContinueHandshake(conn, req, response);
        } else {
            this.submitRead(conn);
        }
    }

    private void sendDataThenContinueHandshake(HttpClientConnection conn, InFlightRequest req, byte[] dataToSend) throws Exception {
        this.sendData(conn, dataToSend);
        conn.setCurrentRequest(Boolean.TRUE);
    }

    private void handleHandshakeContinue(HttpClientConnection conn, InFlightRequest req) throws Exception {
        SslHandler ssl = conn.getSslHandler();
        byte[] response = ssl.processHandshake(null);
        if (ssl.isHandshakeComplete()) {
            String protocol = ssl.getApplicationProtocol();
            boolean isHttp2 = "h2".equals(protocol);
            if (isHttp2) {
                conn.initHttp2(this.createHttp2StreamListener(conn));
            }
            conn.setState(HttpClientConnection.State.IN_USE);
            conn.setCurrentRequest(null);
            byte[] remaining = ssl.getRemainingData();
            if (remaining != null && remaining.length > 0) {
                ssl.setPendingCiphertext(remaining);
            }
            if (response != null && response.length > 0) {
                if (isHttp2) {
                    this.sendHandshakeAndHttp2Request(conn, req, response);
                } else {
                    this.sendHandshakeThenRequest(conn, req, response);
                }
            } else if (isHttp2) {
                this.sendHttp2PrefaceAndRequest(conn, req);
            } else {
                this.sendRequest(req);
            }
        } else if (response != null && response.length > 0) {
            this.sendDataThenContinueHandshake(conn, req, response);
        } else {
            conn.setCurrentRequest(null);
            this.submitRead(conn);
        }
    }

    private void sendDataAndContinue(HttpClientConnection conn, byte[] dataToSend, byte[] remainingData) throws Exception {
        conn.setCurrentRequest(remainingData);
        this.sendData(conn, dataToSend);
    }

    private void sendHandshakeAndHttp2Request(HttpClientConnection conn, InFlightRequest req, byte[] handshakeData) throws Exception {
        conn.setCurrentRequest("SEND_H2_PREFACE");
        this.sendData(conn, handshakeData);
    }

    private void sendHandshakeThenRequest(HttpClientConnection conn, InFlightRequest req, byte[] handshakeData) throws Exception {
        conn.setCurrentRequest("SEND_H1_REQUEST");
        this.sendData(conn, handshakeData);
    }

    private void sendHttp2PrefaceAndRequest(HttpClientConnection conn, InFlightRequest req) throws Exception {
        Http2ClientHandler h2 = conn.getHttp2Handler();
        byte[] preface = h2.getConnectionPreface();
        int streamId = h2.createStream(req.future, req.request);
        if (streamId < 0) {
            throw new Exception("Cannot create HTTP/2 stream");
        }
        byte[] requestFrames = h2.buildRequestFrames(streamId);
        byte[] combined = new byte[preface.length + requestFrames.length];
        System.arraycopy(preface, 0, combined, 0, preface.length);
        System.arraycopy(requestFrames, 0, combined, preface.length, requestFrames.length);
        SslHandler ssl = conn.getSslHandler();
        byte[] encrypted = ssl.encrypt(combined);
        this.sendData(conn, encrypted);
    }

    private void processResponseData(HttpClientConnection conn, InFlightRequest req, byte[] data) throws Exception {
        HttpResponseParser parser = conn.getResponseParser();
        HttpResponseParser.Result result = parser.parse(data, 0, data.length);
        if (result == HttpResponseParser.Result.COMPLETE) {
            this.handleResponseComplete(conn, req, parser);
        } else {
            if (result == HttpResponseParser.Result.ERROR) {
                throw new Exception("Response parse error: " + parser.getErrorMessage());
            }
            this.submitRead(conn);
        }
    }

    private void handleResponseComplete(HttpClientConnection conn, InFlightRequest req, HttpResponseParser parser) throws Exception {
        String location;
        HttpClientResponse response = new HttpClientResponse(parser.getStatusCode(), parser.getStatusMessage(), parser.getHeaders(), parser.getBody(), req.request.getOpts());
        if (parser.isRedirect() && req.request.isFollowRedirects() && req.redirectCount < req.request.getMaxRedirects() && (location = parser.getRedirectLocation()) != null) {
            this.handleRedirect(req, response, location);
            return;
        }
        this.completeRequest(conn, req, response);
    }

    private void handleRedirect(InFlightRequest req, HttpClientResponse response, String location) throws Exception {
        Object newUrl;
        HttpClientConnection conn = req.connection;
        if (response.isKeepAlive()) {
            this.pool.release(conn);
        } else {
            this.closeConnection(conn);
        }
        this.connections.remove(conn.getId());
        this.inFlight.remove(conn.getId());
        ++req.redirectCount;
        if (location.startsWith("http://") || location.startsWith("https://")) {
            newUrl = location;
        } else if (location.startsWith("/")) {
            String scheme = req.request.isSecure() ? "https://" : "http://";
            String host = req.request.getHost();
            int port = req.request.getPort();
            String portStr = req.request.isSecure() && port == 443 || !req.request.isSecure() && port == 80 ? "" : ":" + port;
            newUrl = scheme + host + portStr + location;
        } else {
            throw new Exception("Relative redirect not supported: " + location);
        }
        req.request.url((String)newUrl);
        req.connection = null;
        this.pendingRequests.offer(req);
        this.wakeup();
    }

    private void completeRequest(HttpClientConnection conn, InFlightRequest req, HttpClientResponse response) {
        if (response.isKeepAlive() && req.request.getKeepalive() > 0) {
            this.pool.release(conn);
        } else {
            this.closeConnection(conn);
        }
        this.connections.remove(conn.getId());
        this.inFlight.remove(conn.getId());
        req.future.complete(response);
    }

    private void completeWithError(InFlightRequest req, Exception e) {
        HttpClientConnection conn = req.connection;
        if (conn != null) {
            this.connections.remove(conn.getId());
            this.inFlight.remove(conn.getId());
            this.closeConnection(conn);
        }
        HttpClientResponse response = new HttpClientResponse(e, req.request.getOpts());
        req.future.complete(response);
    }

    private void closeConnection(HttpClientConnection conn) {
        if (conn != null && conn.getFd() >= 0) {
            this.pool.remove(conn);
            try {
                Socket.close(conn.getFd());
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    private Http2ClientHandler.StreamCompleteListener createHttp2StreamListener(HttpClientConnection conn) {
        return (stream, response) -> {
            String location;
            HttpClientRequest request = stream.getRequest();
            if (response.isRedirect() && request.isFollowRedirects() && stream.getRedirectCount() < request.getMaxRedirects() && (location = response.getHeaders().get("location")) != null) {
                this.pendingRedirect = new PendingRedirect(conn, stream, response, location);
                return;
            }
            stream.getFuture().complete(response);
        };
    }

    private void handleHttp2Redirect(HttpClientConnection conn, Http2ClientHandler.ClientStream stream, HttpClientResponse response, String location) throws Exception {
        Http2ClientHandler h2;
        int streamId;
        boolean sameHost;
        Object newUrl;
        HttpClientRequest request = stream.getRequest();
        int redirectCount = stream.getRedirectCount() + 1;
        if (location.startsWith("http://") || location.startsWith("https://")) {
            newUrl = location;
        } else if (location.startsWith("/")) {
            String scheme = request.isSecure() ? "https://" : "http://";
            String host = request.getHost();
            int port = request.getPort();
            Object portStr = request.isSecure() && port == 443 || !request.isSecure() && port == 80 ? "" : ":" + port;
            newUrl = scheme + host + (String)portStr + location;
        } else {
            throw new Exception("Relative redirect not supported: " + location);
        }
        request.url((String)newUrl);
        boolean bl = sameHost = request.getHost().equals(conn.getHost()) && request.getPort() == conn.getPort() && request.isSecure() == conn.isSecure();
        if (sameHost && conn.getHttp2Handler() != null && conn.getHttp2Handler().canAcceptRequest() && (streamId = (h2 = conn.getHttp2Handler()).createStream(stream.getFuture(), request, redirectCount)) > 0) {
            byte[] requestFrames = h2.buildRequestFrames(streamId);
            SslHandler ssl = conn.getSslHandler();
            byte[] encrypted = ssl.encrypt(requestFrames);
            this.sendData(conn, encrypted);
            return;
        }
        InFlightRequest newReq = new InFlightRequest(request, stream.getFuture());
        newReq.redirectCount = redirectCount;
        this.pendingRequests.offer(newReq);
        this.wakeup();
    }

    private void handleCloseComplete(HttpClientConnection conn, InFlightRequest req) {
        this.connections.remove(conn.getId());
    }

    private void cleanup() {
        InFlightRequest req;
        this.running = false;
        while ((req = this.pendingRequests.poll()) != null) {
            req.future.completeExceptionally(new Exception("Worker shutting down"));
        }
        for (InFlightRequest r : this.inFlight.values()) {
            r.future.completeExceptionally(new Exception("Worker shutting down"));
        }
        this.inFlight.clear();
        for (HttpClientConnection conn : this.connections.values()) {
            this.closeConnection(conn);
        }
        this.connections.clear();
        IoUring.closeEventFd(this.eventFd);
        this.ring.close();
        this.arena.close();
    }

    private static class PendingRedirect {
        final HttpClientConnection conn;
        final Http2ClientHandler.ClientStream stream;
        final HttpClientResponse response;
        final String location;

        PendingRedirect(HttpClientConnection conn, Http2ClientHandler.ClientStream stream, HttpClientResponse response, String location) {
            this.conn = conn;
            this.stream = stream;
            this.response = response;
            this.location = location;
        }
    }

    public static class InFlightRequest {
        public final HttpClientRequest request;
        public final CompletableFuture<HttpClientResponse> future;
        public HttpClientConnection connection;
        public long startTime;
        public int redirectCount;

        public InFlightRequest(HttpClientRequest request, CompletableFuture<HttpClientResponse> future) {
            this.request = request;
            this.future = future;
            this.startTime = System.currentTimeMillis();
            this.redirectCount = 0;
        }
    }

    private static class ValueLayout {
        static final ValueLayout.OfByte JAVA_BYTE = java.lang.foreign.ValueLayout.JAVA_BYTE;

        private ValueLayout() {
        }
    }
}

