/*
 * Decompiled with CFR 0.152.
 */
package zeph.client;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicLong;
import zeph.client.HttpClientConnection;

public class HttpClientConnectionPool
implements AutoCloseable {
    private final int maxConnectionsPerHost;
    private final int maxTotalConnections;
    private final long idleTimeout;
    private final long connectionTimeout;
    private final ConcurrentHashMap<String, HostPool> pools = new ConcurrentHashMap();
    private final AtomicLong connectionIdGenerator = new AtomicLong(0L);
    private final AtomicLong totalConnections = new AtomicLong(0L);
    private final Thread evictionThread;
    private volatile boolean running = true;

    public HttpClientConnectionPool() {
        this(20, 200, 60000L, 30000L);
    }

    public HttpClientConnectionPool(int maxConnectionsPerHost, int maxTotalConnections, long idleTimeout, long connectionTimeout) {
        this.maxConnectionsPerHost = maxConnectionsPerHost;
        this.maxTotalConnections = maxTotalConnections;
        this.idleTimeout = idleTimeout;
        this.connectionTimeout = connectionTimeout;
        this.evictionThread = new Thread(this::evictionLoop, "zeph-pool-evictor");
        this.evictionThread.setDaemon(true);
        this.evictionThread.start();
    }

    public HttpClientConnection tryAcquire(String host, int port, boolean secure) {
        HttpClientConnection conn;
        String key = HttpClientConnectionPool.makeKey(host, port, secure);
        HostPool pool = this.pools.get(key);
        if (pool == null) {
            return null;
        }
        while ((conn = pool.idle.pollFirst()) != null) {
            if (conn.isAlive() && !conn.isIdleTooLong(this.idleTimeout)) {
                conn.setState(HttpClientConnection.State.IN_USE);
                pool.inUse.put(conn.getId(), conn);
                return conn;
            }
            this.closeConnection(conn);
        }
        return null;
    }

    public HttpClientConnection createConnection(String host, int port, boolean secure) {
        String key = HttpClientConnectionPool.makeKey(host, port, secure);
        if (this.totalConnections.get() >= (long)this.maxTotalConnections) {
            return null;
        }
        HostPool pool = this.pools.computeIfAbsent(key, k -> new HostPool((String)k, host, port, secure));
        if (pool.totalSize() >= this.maxConnectionsPerHost) {
            return null;
        }
        long id = this.connectionIdGenerator.incrementAndGet();
        HttpClientConnection conn = new HttpClientConnection(id, host, port, secure);
        pool.inUse.put(id, conn);
        this.totalConnections.incrementAndGet();
        return conn;
    }

    public void release(HttpClientConnection conn) {
        String key = conn.getPoolKey();
        HostPool pool = this.pools.get(key);
        if (pool == null) {
            this.closeConnection(conn);
            return;
        }
        pool.inUse.remove(conn.getId());
        if (conn.isAlive() && conn.getState() != HttpClientConnection.State.CLOSING) {
            conn.reset();
            pool.idle.addFirst(conn);
        } else {
            this.closeConnection(conn);
        }
    }

    public void remove(HttpClientConnection conn) {
        String key = conn.getPoolKey();
        HostPool pool = this.pools.get(key);
        if (pool != null) {
            pool.inUse.remove(conn.getId());
            pool.idle.remove(conn);
        }
        this.closeConnection(conn);
    }

    private void closeConnection(HttpClientConnection conn) {
        if (conn.getState() != HttpClientConnection.State.CLOSED) {
            conn.close();
            this.totalConnections.decrementAndGet();
        }
    }

    public void evictIdle() {
        for (HostPool pool : this.pools.values()) {
            pool.idle.removeIf(conn -> {
                if (conn.isIdleTooLong(this.idleTimeout) || !conn.isAlive()) {
                    this.closeConnection((HttpClientConnection)conn);
                    return true;
                }
                return false;
            });
            if (pool.totalSize() != 0) continue;
            this.pools.remove(pool.key);
        }
    }

    private void evictionLoop() {
        while (this.running) {
            try {
                Thread.sleep(this.idleTimeout / 2L);
                this.evictIdle();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
    }

    public Stats getStats() {
        int idle = 0;
        int inUse = 0;
        for (HostPool pool : this.pools.values()) {
            idle += pool.idle.size();
            inUse += pool.inUse.size();
        }
        return new Stats(this.pools.size(), idle, inUse, this.totalConnections.get());
    }

    private static String makeKey(String host, int port, boolean secure) {
        return host + ":" + port + (secure ? ":s" : "");
    }

    public int getMaxConnectionsPerHost() {
        return this.maxConnectionsPerHost;
    }

    public int getMaxTotalConnections() {
        return this.maxTotalConnections;
    }

    public long getIdleTimeout() {
        return this.idleTimeout;
    }

    public long getConnectionTimeout() {
        return this.connectionTimeout;
    }

    @Override
    public void close() {
        this.running = false;
        this.evictionThread.interrupt();
        for (HostPool pool : this.pools.values()) {
            for (HttpClientConnection conn : pool.idle) {
                conn.close();
            }
            for (HttpClientConnection conn : pool.inUse.values()) {
                conn.close();
            }
        }
        this.pools.clear();
        this.totalConnections.set(0L);
    }

    private static class HostPool {
        final String key;
        final String host;
        final int port;
        final boolean secure;
        final ConcurrentLinkedDeque<HttpClientConnection> idle = new ConcurrentLinkedDeque();
        final ConcurrentHashMap<Long, HttpClientConnection> inUse = new ConcurrentHashMap();

        HostPool(String key, String host, int port, boolean secure) {
            this.key = key;
            this.host = host;
            this.port = port;
            this.secure = secure;
        }

        int totalSize() {
            return this.idle.size() + this.inUse.size();
        }
    }

    public record Stats(int hosts, int idle, int inUse, long total) {
        @Override
        public String toString() {
            return String.format("HttpClientConnectionPool[hosts=%d, idle=%d, inUse=%d, total=%d]", this.hosts, this.idle, this.inUse, this.total);
        }
    }
}

