(ns silvur.oauth2
  (:gen-class)
  (:require [silvur.oauth2.core :refer [run-oauth2-authorization-code config* server*]]
            [clojure.tools.cli :refer [parse-opts]]
            [taoensso.timbre :as log]
            [clojure.string :as str]
            [silvur.nio :as nio]
            [silvur.oauth2.x]
            [silvur.oauth2.idp]
            [silvur.oauth2.azure]
            [mount.core :as mount :refer [defstate ]]))

(def cli-options
  [["-H" "--host <host>" "Listen host"
    :default-desc "0.0.0.0"]
   ["-p" "--port <PORT>" "Listen port"
    :default-desc "9180"
    :parse-fn parse-long]
   ["-f" "--config-file <FILE>" "Configuration file"
    :validate [nio/exists? "No file"]]
   ["-a" "--auth-uri <URI>" "Authentication URI"]
   ["-t" "--token-uri <URI>" "Token URI"]
   ["-u" "--redirect-uri <URI>" "Redirect URI"]
   ["-i" "--client-id <ID>" "Client ID"]
   ["-e" "--client-secret <SECRET>" "Client secret"]
   ["-S" "--scope <SCOPE>" "Scopes as openid,email,profile"
    :parse-fn #(str/split % #",")]
   ["-r" "--root-dir <Root directory>"
    :default "."]
   ;; SSL/TLS options
   ["-s" "--ssl" "Enable SSL/TLS (uses built-in localhost cert if no cert specified)"
    :default false]
   ["-c" "--ssl-cert <path>" "PEM certificate file"]
   ["-k" "--ssl-key <path>" "PEM private key file"]
   ["-K" "--ssl-keystore <path>" "Path to JKS keystore file"]
   ["-P" "--ssl-keystore-password <password>" "Keystore password"]
   ["-N" "--nrepl" "Boot nREPL"
    :default false]
   [nil "--nrepl-port <PORT>" "nREPL Listen port"
    :default 7999
    :parse-fn parse-long]
   ["-d" "--debug <LEVEL>" "Debug Level"
    :default :info
    :parse-fn keyword
    :validate [#{:trace :debug :info :warn :error :fatal :report} "Should be report/fatal/error/warn/info/debug/trace"]]
   ["-h" "--help" "This help"
    :default false]])



(defn usage [summary]
  (->> ["Usage: slv oauth2 <command> [options]"
        ""
        "OAuth2 authorization server and client"
        ""
        "commands:"
        ""
        "  listen              Start OAuth2 authorization server"
        "  sample <x|azure>    Show sample configuration for providers"
        "  client              Run as OAuth2 client (authorization code flow)"
        ""
        "options:"
        summary
        ""
        "examples:"
        "  slv oauth2 listen -f config.edn"
        "  slv oauth2 listen -p 8080 -f idp.edn"
        "  slv oauth2 listen -s -p 8443                    # HTTPS with built-in localhost cert"
        "  slv oauth2 listen -s -c cert.pem -k key.pem     # HTTPS with PEM files"
        "  slv oauth2 sample x"
        "  slv oauth2 sample azure"
        "  slv oauth2 client -i client_id -e client_secret -a https://auth.example.com/authorize"
        ""]
      (str/join \newline)))



(defn internal-cli* [{:keys [options arguments summary errors]}]
  (let [opts (-> (dissoc options :help)
                 (update :options update-keys {:host :ip}))
        [op saas] arguments]

    (cond 
      (or (:help options) (nil? op)) (do (println (usage summary))
                                         (System/exit 0))
      (not-empty errors) (println (str/join "\n" errors))
      (= "sample" op) (println (with-out-str
                                 (println)
                                 (clojure.pprint/pprint
                                  (try
                                    (deref (resolve (symbol (str "silvur.oauth2." saas "/sample-config"))))
                                    (catch Exception e (log/warn "No specified config. The sample for X or Azure are available."))))))
      (#{"cli" "client"} op) (run-oauth2-authorization-code opts)
      :else (do
              (-> (mount/with-args opts)
                  (mount/only [#'config* #'server*])
                  (mount/start))
              (println "== Context: " )
              (println (with-out-str (clojure.pprint/pprint config*)))))))

(defn main [& args]
  (let [{:keys [options arguments summary errors] :as parsed-args} (parse-opts args cli-options)]
    (log/set-level! (:debug options))
    (log/debug options)
    (internal-cli* parsed-args)))

(defn dev []
  (main "oauth2" "listen" "-c" "idp.edn"))




