;; *   Silvur
;; *
;; *   Copyright (c) Tsutomu Miyashita. All rights reserved.
;; *
;; *   The use and distribution terms for this software are covered by the
;; *   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; *   which can be found in the file epl-v10.html at the root of this distribution.
;; *   By using this software in any fashion, you are agreeing to be bound by
;; * 	 the terms of this license.
;; *   You must not remove this notice, or any other, from this software.

(ns silvur.util
  (:gen-class)
  (:require [clojure.string :as str :refer [upper-case lower-case]]
            [clojure.core.async :refer (chan put! alts! go-loop timeout close!)]
            [zeph.client :as http]
            [zeph.server :as hs]
            [clojure.data.xml :as xml]
            [clojure.java.io :as io]
            [postal.core :as postal]
            [taoensso.timbre :as log]
            [jsonista.core :as json]
            [camel-snake-kebab.core :as csk]
            [camel-snake-kebab.extras :as cske]
            [cognitect.transit :as transit]
            [clojure.java.shell :refer [sh]])
  (:import [java.security MessageDigest]
           [java.util.concurrent BlockingQueue LinkedBlockingQueue SynchronousQueue TimeUnit]
           [java.io File EOFException PushbackInputStream]
           [java.util.stream Collectors]
           [java.util.concurrent LinkedBlockingQueue TimeUnit]
           [java.net Socket]))

(defmacro uuid
  "Generate or parse a UUID.
   (uuid) - generates a random UUID
   (uuid \"550e8400-...\") - parses a UUID string"
  ([]
   `(random-uuid))
  ([s]
   `(java.util.UUID/fromString ~s)))

(defprotocol NumberConvert
  (str->int [x])
  (str->long [x])
  (str->double [x])
  (hex->long [x])
  (chars->hex [xs])
  (chars->long [xs]))

(extend-protocol NumberConvert
  java.lang.String
  (str->int [x] (Integer/valueOf x))
  (str->long [x] (Long/valueOf x))
  (str->double [x] (Double/valueOf x))
  (hex->long [x] (Long/valueOf x 16))
  (chars->hex [xs] (str/join (map #(Long/toHexString (long %)) xs)))
  (chars->long [xs] (-> xs chars->hex hex->long)))


(defn sha1sum
  "Compute SHA1 hash of one or more values.
   Returns a hex string representation."
  [& vals]
  (let [sha1 (MessageDigest/getInstance "SHA1")]
    (doseq [v vals]
      (.update sha1 (.getBytes (str v) "UTF-8")))
    (str/join (map #(format "%02x" %) (.digest sha1)))))

(defmacro uk [s]
  `(try (keyword (upper-case (name ~s))) (catch Exception e# nil)))

(defmacro us [s]
  `(try (upper-case (name ~s)) (catch Exception e# nil)))

(defmacro lk [s]
  `(try (keyword (lower-case (name ~s))) (catch Exception e# nil)))

(defmacro ls [s]
  `(try (lower-case (name ~s)) (catch Exception e# nil)))

(defn json->edn
  "Parse JSON string to EDN with key transformation.
   type: :kebab (default), :camel, or nil (keyword only)
   Example: {\"userName\":\"foo\"} -> {:user-name \"foo\"} with :kebab"
  ([x]
   (json->edn :kebab x))
  ([type x]
   (try
     (cond->> (json/read-value x)
       (= type :kebab) (cske/transform-keys csk/->kebab-case-keyword)
       (= type :camel) (cske/transform-keys csk/->camelCase)
       :else (cske/transform-keys keyword))
     (catch Exception _e x))))

(defn edn->json
  "Convert EDN to JSON string with key transformation.
   type: :camel (default), :pascal, :kebab, :snake, :header
   Example: {:user-name \"foo\"} -> {\"userName\":\"foo\"} with :camel"
  ([x]
   (edn->json :camel x))
  ([type x]
   (->> x (cske/transform-keys
           (case type
             :pascal csk/->PascalCaseString
             :kebab csk/->kebab-case-string
             :camel csk/->camelCaseString
             :snake csk/->snake_case_string
             :header csk/->HTTP-Header-Case-String
             name))
        (json/write-value-as-string))))

(defn edn->xml
  "Convert EDN to XML string.
   If raw? is true, returns the raw XML data structure instead of a formatted string."
  [edn & [raw?]]
  (letfn [(parse [edn]
            (loop [m edn
                   result {}]
              (cond
                (nil? m) result
                ;; pickup 1 element
                ;; handle element
                (map? m)  (if-let [[k v] (first m)]
                            (cond
                              (.matches (name k) "^-.*") (recur (into {} (rest m)) (update result :attrs assoc (keyword (subs (name k) 1)) v))
                              :else (recur (into {} (rest m))
                                           (let [z (parse v)]
                                             
                                             (if (vector? z)
                                               {:content (mapv #(assoc % :tag k) z)}
                                               (update result :content
                                                       (comp vec conj)
                                                       (conj {:tag k} (parse v)))))))
                            result)
                (vector? m) (mapv parse m)
                :else {:content (vec (flatten [m]))})))]
    (let [xml (:content (parse edn))]
      (cond-> (if (and (vector? xml) (< 1 (count xml)))
                {:tag :root :content xml}
                xml)
        (not raw?) (xml/indent-str)))))

;; NIO



;; SMTP
(defn mail
  "Send an email via SMTP.
   Options: :from, :to (vector), :subject, :host (localhost), :port (25), :contents"
  [& {:keys [from to subject host port contents] :or {host "localhost" port 25}}]
  (postal/send-message {:host host}
                       {:from from :to to :subject subject
                        :body (reduce conj [] (map (fn [[type content]]
                                                     (hash-map :type (condp = type
                                                                       :attachment type
                                                                       :inline type
                                                                       (clojure.string/replace (str type "; charset=utf-8") #":" ""))
                                                               :content content))
                                                   contents))}))


(defn parse-binary-message
  "Parse binary data according to a format specification.
   format-coll: vector of field specs, each can be:
     - number: field size in bytes
     - [size converter-fn]: field size and conversion function
     - :* take remaining bytes
   Example: (parse-binary-message [1 3 [4 chars->hex]] data)"
  [format-coll data-coll]
  (loop [remaining-format format-coll
         result []
         remaining-data data-coll]
    (let [format-entry (first remaining-format)
          [field-size converter-fn] (cond
                                      (number? format-entry) [format-entry]
                                      (sequential? format-entry) format-entry
                                      (= :* format-entry) [(count remaining-data)]
                                      :else [])]
      (if-not (seq remaining-format)
        result
        (recur (rest remaining-format)
               (conj result ((or converter-fn identity) (take field-size remaining-data)))
               (drop field-size remaining-data))))))




(defonce control-ports (atom {}))

(defn stop-routine
  "Stop a running background routine by its ID.
   If no ID is provided, stops the most recently started routine."
  ([]
   (put! (last (first @control-ports)) :quit))
  ([id]
   (put! (@control-ports id) :quit)))


(defn start-routine
  "Start a background routine that repeatedly calls function f.
   Options:
     :id           - unique identifier for the routine
     :interval     - ms between calls (or fn returning ms)
     :queue-length - control channel buffer size
   Returns a control channel. Use stop-routine to stop."
  [f & [{:keys [id queue-length interval]
         :as opts}]]
  (let [ctrl-port (chan (or queue-length 1))
        id (or id (gensym))]
    (swap! control-ports assoc id ctrl-port)
    (go-loop [id id]
      (let [[v ch] (alts! [ctrl-port (timeout (cond
                                                (fn? interval) (interval)
                                                (number? interval) interval
                                                :else 2000))])]
        (if (and (= :quit v) (= ch ctrl-port))
          (do (log/info "Go routine" id "quit")
              (close! (@control-ports id))
              (swap! control-ports dissoc id))
          (do (f opts)
              (recur id)))))
    ctrl-port))

(defn merge-options
  "Parse CLI arguments into a map of options.
   Handles both key=value style arguments and regular positional arguments.
   Returns {:args [positional-args] :opts {keyword-options}}"
  [arguments options]
  (let [eq-opts (->> (keep #(re-find #"([\w-.*]+)=([\w-.*]+)" %) arguments)
                     (mapv (comp vec rest))
                     (mapv #(update-in % [0] keyword))
                     (into {}))
        reg-opts (filterv #(not (re-find #"([\w-.*]+)=([\w-.*]+)" %)) arguments)]
    {:args reg-opts
     :opts (merge eq-opts options)}))
