(ns silvur.jwt
  (:require [clojure.java.shell :refer [sh with-sh-dir with-sh-env]]
            [clojure.tools.cli :refer (parse-opts)]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [clojure.java.process :as proc]
            [silvur.util :as util]
            [silvur.datetime :refer [datetime datetime*] :as sd]
            [buddy.core.keys :as keys]
            [buddy.sign.jwt :as jwt]
            [babashka.fs :as fs]))


(def specs [["-h" "--help" "This help"]
            ["-P" "--private-key <path>" "A path to the private key file to use"]])

(defn usage [summary]
  (println (->> ["Usage: slv jwt <command> [options]"
                 ""
                 "Sign JWT tokens with RS256 algorithm"
                 ""
                 "commands:"
                 ""
                 "  sign    Sign a JWT token with claims"
                 ""
                 "options:"
                 summary
                 ""
                 "claims (key=value format):"
                 "  iss=<issuer>       Token issuer"
                 "  sub=<subject>      Token subject"
                 "  aud=<audience>     Token audience"
                 "  exp=<timestamp>    Expiration time (default: +20 hours)"
                 ""
                 "examples:"
                 "  slv jwt sign -P private.pem iss=myapp sub=user123"
                 "  slv jwt sign --private-key ./keys/rsa.pem iss=api aud=client1"
                 ""]
                (str/join \newline))))


(defn sign* [{:keys [iss sub exp aud] :as claims} & {:keys [private-key-path alg]}]
  (let []
    (-> {:exp (or exp (* 1000 (datetime* (sd/+time (datetime) (sd/hour 20) ))))
         :sub sub}
        (merge claims)
        (jwt/sign  (keys/private-key private-key-path) {:alg (or alg :rs256)}))))

(defn main [& args]
  (let [{:keys [arguments options summary]} (parse-opts args specs)
        [op & _cmd-args] args
        merged-opts (util/merge-options arguments options)]
    (if (:help options)
      (usage summary)
      (condp = (keyword op)
        :sign (let [{:keys [opts]} merged-opts
                    {:keys [private-key]} options]
                (if private-key
                  (println (sign* opts :private-key-path private-key))
                  (do (println "Error: --private-key is required")
                      (usage summary))))
        (usage summary)))))





