(ns silvur.http
  (:gen-class)  
  (:refer-clojure :exclude [get])
  (:require [silvur.util :refer [json->edn edn->json]]
            [silvur.log :as slog]
            [silvur.nio :as nio]
            [clojure.tools.cli :refer (parse-opts)]
            [silvur.datetime :refer [datetime* chrono-unit *precision*]]
            [mount.core :as mount :refer [defstate]]
            [zeph.client :as http]
            [zeph.server :as srv]
            [clojure.java.io :as io]
            [taoensso.timbre :as log]
            [clojure.string :as str]
            [reitit.core :as r]
            [reitit.http :as rh]
            [reitit.interceptor.sieppari :as sieppari]
            [reitit.http.interceptors.parameters :as params]
            [clojure.pprint :as pp]
            [reitit.ring :as ring]
            [clojure.data.json :as json])
  (:import [java.net URI]
           [java.io FileInputStream ByteArrayInputStream]
           [java.nio.file Files Paths]
           [java.security KeyStore KeyFactory]
           [java.security.cert CertificateFactory]
           [java.security.spec PKCS8EncodedKeySpec]
           [java.util Base64]
           [javax.net.ssl SNIHostName SNIServerName SSLEngine SSLParameters
            SSLContext KeyManagerFactory TrustManagerFactory X509TrustManager]))

;; SSL - PEM file support for GraalVM native-image compatibility

(defn- read-pem-file
  "Read a PEM file and return its content as a string"
  ^String [^String path]
  (String. (Files/readAllBytes (Paths/get path (into-array String [])))))

(defn- extract-pem-section
  "Extract base64 content between BEGIN/END markers"
  [^String content ^String type]
  (let [begin-marker (str "-----BEGIN " type "-----")
        end-marker (str "-----END " type "-----")
        begin-idx (.indexOf content begin-marker)
        end-idx (.indexOf content end-marker)]
    (when (and (>= begin-idx 0) (>= end-idx 0))
      (-> content
          (subs (+ begin-idx (count begin-marker)) end-idx)
          (str/replace #"\s+" "")))))

(defn- decode-base64 ^bytes [^String s]
  (.decode (Base64/getDecoder) s))

(defn- pem-content?
  "Check if string is PEM content (vs a file path)"
  [^String s]
  (or (str/includes? s "-----BEGIN")
      (str/includes? s "\n")))

(defn- load-pem-certs
  "Load X.509 certificates from a PEM file path or PEM string content.
   Supports certificate chains (multiple certs in one file/string)."
  [^String path-or-content]
  (let [content (if (pem-content? path-or-content)
                  path-or-content
                  (read-pem-file path-or-content))
        cf (CertificateFactory/getInstance "X.509")]
    (loop [remaining content
           result []]
      (let [begin-idx (.indexOf remaining "-----BEGIN CERTIFICATE-----")]
        (if (< begin-idx 0)
          result
          (let [end-idx (.indexOf remaining "-----END CERTIFICATE-----")
                cert-pem (subs remaining begin-idx (+ end-idx (count "-----END CERTIFICATE-----")))
                cert-b64 (extract-pem-section cert-pem "CERTIFICATE")
                cert-bytes (decode-base64 cert-b64)
                cert (.generateCertificate cf (ByteArrayInputStream. cert-bytes))]
            (recur (subs remaining (+ end-idx (count "-----END CERTIFICATE-----")))
                   (conj result cert))))))))

(defn- load-pem-private-key
  "Load a private key from a PEM file path or PEM string content.
   Supports both PKCS8 and RSA PRIVATE KEY formats."
  [^String path-or-content]
  (let [content (if (pem-content? path-or-content)
                  path-or-content
                  (read-pem-file path-or-content))
        key-b64 (or (extract-pem-section content "PRIVATE KEY")
                    (extract-pem-section content "RSA PRIVATE KEY"))
        _ (when-not key-b64
            (throw (ex-info "No private key found in PEM" {:input path-or-content})))
        key-bytes (decode-base64 key-b64)
        kf (KeyFactory/getInstance "RSA")]
    (.generatePrivate kf (PKCS8EncodedKeySpec. key-bytes))))

(defn make-ssl-context-from-pem
  "Create SSLContext from PEM certificate and key files or content.
   Arguments can be file paths or PEM string content.
   Supports certificate chains."
  [{:keys [cert key]}]
  (let [certs (load-pem-certs cert)
        private-key (load-pem-private-key key)
        cert-array (into-array java.security.cert.Certificate certs)
        ks (doto (KeyStore/getInstance "PKCS12")
             (.load nil nil)
             (.setKeyEntry "server" private-key (char-array "") cert-array))
        kmf (doto (KeyManagerFactory/getInstance (KeyManagerFactory/getDefaultAlgorithm))
              (.init ks (char-array "")))
        ctx (SSLContext/getInstance "TLS")]
    (.init ctx (.getKeyManagers kmf) nil nil)
    ctx))

;; Built-in self-signed certificate for localhost (100 years validity)
(def ^:private localhost-cert-pem
  "LS0tLS1CRUdJTiBDRVJUSUZJQ0FURS0tLS0tCk1JSURSekNDQWkrZ0F3SUJBZ0lVUjlENy8rdlFpMHRiOWVSdHc4R1h6RkJiQlJnd0RRWUpLb1pJaHZjTkFRRUwKQlFBd01qRVNNQkFHQTFVRUF3d0piRzlqWVd4b2IzTjBNUTh3RFFZRFZRUUtEQVpUYVd4MmRYSXhDekFKQmdOVgpCQVlUQWtwUU1DQVhEVEkxTVRJeE16QXpNVE14TmxvWUR6SXhNalV4TVRFNU1ETXhNekUyV2pBeU1SSXdFQVlEClZRUUREQWxzYjJOaGJHaHZjM1F4RHpBTkJnTlZCQW9NQmxOcGJIWjFjakVMTUFrR0ExVUVCaE1DU2xBd2dnRWkKTUEwR0NTcUdTSWIzRFFFQkFRVUFBNElCRHdBd2dnRUtBb0lCQVFEU3RTd1pFMktGUEw1UmxwdFNOSDZMUThSRApzVVQzYjNwMllUZ1BpNk1vK3FsMVFpeC9FalU4anNVRmFQcVZyNXBXOUNINFlaclB0VWh4SEdrdFprQk1VT1Q0CjV0OGdGSGpEZXgzNDNhc2xFSVMxMm1IdG91WDhYOU8zeTZSdnp4ZmN1YWRCMGQvTDdFQ2FtK3FHcmpvbUF4ZmcKNTNWTEJlZEYxMDY4dE80WjJrTzR0WUw2em9ObGlwNGtSb25lbytENWJ5eGJUbVRpZXR0VUdyaUlSMVArUFRiMgo5R0xZUDVVaE5WVVR2ZU1RQTVzR3d0ZUtPdmNEb25HZmlGeHNhc2k2OTBVeEk2SnMwMHVTTlZVUWNvUUpUZVRHCnpiNWpLdEduZ3RvSGdBRHFGd1FNYkFoMks2aHJoSTRKSnlRcFgxK0g3U3RGU00rbnVoWlZlY0o2YUpKWEFnTUIKQUFHalV6QlJNQjBHQTFVZERnUVdCQlRENVR4bGxWajc5cWhqSEtiaHdUczZQR0M5c2pBZkJnTlZIU01FR0RBVwpnQlRENVR4bGxWajc5cWhqSEtiaHdUczZQR0M5c2pBUEJnTlZIUk1CQWY4RUJUQURBUUgvTUEwR0NTcUdTSWIzCkRRRUJDd1VBQTRJQkFRQVc2WFhJUnhPOGFBcFAyVjF1QUgrUTFra3haZ0pvK21XTTBSN1p0MGp2WURlME1RUzUKTDJIMVkwa3NsQVdIQnN5ZG9VQTJyWDg3aDVvQzJBZFp1bWhKdWJ2d2s5eFMyaXNTYWU5ZHNRTXBXR3BFOW9LWgo5WWovekNpMml3eWltbjZVcFJFVWJ2a0kyRGhYN2txcUwxdEVSQ3IzcHZwQjBoTCs1dlFTN0UxcE1IZkkzc054Ckx0TjkzQU8zVDNVWmxjSWp3NURsT21qUVdPV1UvRVlYNm1KM3Fxb2N4dnRVek1sWUk2Z09SOElpU29JU0V4RDEKTDZoY2N6VnJPNXpoUGhYTUt3YzYyNVlSWkxQYU0relJmWWwrc3FCcHQvT2dFcjBzQktrMld6aXMxZmpwQU1JSwpwc0JTaUo1VXhlZ1RuRFBxM25wK0YzelhVMWF4L0VZOURnb3UKLS0tLS1FTkQgQ0VSVElGSUNBVEUtLS0tLQo=")

(def ^:private localhost-key-pem
  "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")

(defonce ^:private localhost-ssl-context
  (delay
    (let [cert-pem (String. (decode-base64 localhost-cert-pem) "UTF-8")
          key-pem (String. (decode-base64 localhost-key-pem) "UTF-8")
          certs (load-pem-certs cert-pem)
          private-key (load-pem-private-key key-pem)
          cert-array (into-array java.security.cert.Certificate certs)
          ks (doto (KeyStore/getInstance "PKCS12")
               (.load nil nil)
               (.setKeyEntry "localhost" private-key (char-array "")
                             cert-array))
          kmf (doto (KeyManagerFactory/getInstance (KeyManagerFactory/getDefaultAlgorithm))
                (.init ks (char-array "")))
          ctx (SSLContext/getInstance "TLS")]
      (.init ctx (.getKeyManagers kmf) nil nil)
      ctx)))

(defn sni-configure [^SSLEngine ssl-engine ^URI uri]
  (let [^SSLParameters ssl-params (.getSSLParameters ssl-engine)]
    (.setServerNames ssl-params [(SNIHostName. (.getHost uri))])
    (.setUseClientMode ssl-engine true)
    (.setSSLParameters ssl-engine ssl-params)))

;; When using SNI, it should make a client as below for httpkit
;; (defonce sni-client
;;   ;; SNI (Server Name Indication) is a TLS extension to multiplex multiple SSL termination on a single IP  
;;   (http/make-client {:ssl-configurer ssl/sni-configure}))

(defn engine
  "Create an SSLEngine that trusts all certificates.
   WARNING: FOR DEVELOPMENT/TESTING ONLY - DO NOT USE IN PRODUCTION!
   This bypasses all certificate validation, making connections
   vulnerable to MITM attacks. Use only for local testing or
   when connecting to servers with self-signed certificates in
   controlled environments."
  []
  (let [tm (reify javax.net.ssl.X509TrustManager
             (getAcceptedIssuers [_this] (make-array java.security.cert.X509Certificate 0))
             (checkClientTrusted [_this _chain _auth-type])
             (checkServerTrusted [_this _chain _auth-type]))
        client-context (javax.net.ssl.SSLContext/getInstance "TLSv1.2")]
    (.init client-context nil
           (into-array javax.net.ssl.TrustManager [tm])
           nil)
    (doto (.createSSLEngine client-context)
      (.setUseClientMode true))))

;; Basic Authentication

(defn encode-basic-auth
  "Encode user:pass to Base64 for Basic Auth header"
  [user pass]
  (.encodeToString (java.util.Base64/getEncoder) (.getBytes (str user ":" pass) "UTF-8")))

(defn parse-basic-auth
  "Parse Authorization header and return [user pass] or nil"
  [auth-header]
  (when (and auth-header (str/starts-with? auth-header "Basic "))
    (try
      (let [decoded (String. (.decode (java.util.Base64/getDecoder)
                                       (subs auth-header 6))
                             "UTF-8")]
        (str/split decoded #":" 2))
      (catch Exception _ nil))))

(defn wrap-basic-auth
  "Ring middleware for Basic Authentication.
   credentials is a string in format 'user:pass'"
  [handler credentials]
  (let [[expected-user expected-pass] (str/split credentials #":" 2)]
    (fn [request]
      (let [auth-header (get-in request [:headers "authorization"])
            [user pass] (parse-basic-auth auth-header)]
        (if (and (= user expected-user) (= pass expected-pass))
          (handler request)
          {:status 401
           :headers {"WWW-Authenticate" "Basic realm=\"Restricted\""
                     "Content-Type" "text/plain"}
           :body "Unauthorized"})))))

(def STATUS
  {100 "Continue",
   101 "Switching Protocols",
   102 "Processing",
   103 "Early Hints",
   200 "OK",
   201 "Created",
   202 "Accepted",
   203 "Non-Authoritative Information",
   204 "No Content",
   205 "Reset Content",
   206 "Partial Content",
   207 "Multi-Status",
   208 "Already Reported",
   226 "IM Used",
   300 "Multiple Choices",
   301 "Moved Permanently",
   302 "Found",
   303 "See Other",
   304 "Not Modified",
   305 "Use Proxy",
   306 "Unused",
   307 "Temporary Redirect",
   308 "Permanent Redirect",
   400 "Bad Request",
   401 "Unauthorized",
   402 "Payment Required",
   403 "Forbidden",
   404 "Not Found",
   405 "Method Not Allowed",
   406 "Not Acceptable",
   407 "Proxy Authentication Required",
   408 "Request Timeout",
   409 "Conflict",
   410 "Gone",
   411 "Length Required",
   412 "Precondition Failed",
   413 "Payload Too Large",
   414 "URI Too Long",
   415 "Unsupported Media Type",
   416 "Range Not Satisfiable",
   417 "Expectation Failed",
   418 "I'm a Teapot",
   421 "Misdirected Request",
   422 "Unprocessable Entity",
   423 "Locked",
   424 "Failed Dependency",
   425 "Too Early",
   426 "Upgrade Required",
   428 "Precondition Required",
   429 "Too Many Requests",
   431 "Request Header Fields Too Large",
   451 "Unavailable For Legal Reasons",
   500 "Internal Server Error",
   501 "Not Implemented",
   502 "Bad Gateway",
   503 "Service Unavailable",
   504 "Gateway Timeout",
   505 "HTTP Version Not Supported",
   506 "Variant Also Negotiates",
   507 "Insufficient Storage",
   508 "Loop Detected",
   510 "Not Extended",
   511 "Network Authentication Required"})


(defmacro with-tracing [opts & body]
  `(binding [*precision* (chrono-unit :millis)]
     (let [s# (datetime*)
           tran-id# (symbol (format "%04d" (rand-int 9999)))]
       (log/trace :http-trace-appender ~opts 0 tran-id#)
       (let [result# (deref (do ~@body))
             e# (datetime*)]
         (log/trace :http-trace-appender result# (- e# s#) tran-id#)
         result#))))

;; zeph has built-in connection pooling, no need for make-client

(defn -request [{:keys [url method] :as opts} & [callback]]
  (with-tracing opts
    (http/request opts callback)))


(defn get [uri & [opts callback]]
  (-request (assoc opts :method :get :url uri) callback))

(defn post [uri & [opts callback]]
  (-request (assoc opts :method :post :url uri) callback))

(defn put [uri & [opts callback]]
  (-request (assoc opts :method :put :url uri) callback))

(defn patch [uri & [opts callback]]
  (-request (assoc opts :method :patch :url uri) callback))

(defn delete [uri & [opts callback]]
  (-request (assoc opts :method :delete :url uri) callback))

(defn head [uri & [opts callback]]
  (-request (assoc opts :method :head :url uri) callback))

(defn move [uri & [opts callback]]
  (-request (assoc opts :method :move :url uri) callback)  )

(defn http-trace-appender
  "Disable default println as (taoensso.timbre/merge-config! {:appenders {:println {:enabled? false} :http-tracer (log/http-trace-appender)}})
  And (set-min-level! :trace)
  "
  [{:keys [level]}]
  {:enabled? true
   :ns-filter #{"silvur.http"}
   :fn (let []
         (fn [{:keys [vargs]}]
           (when (= :http-trace-appender (first vargs))
             (let [[id-key {:keys [headers body multipart method url status] :as http-data} duration & [tran-id]] vargs]
               (print (if status
                        (str "<=== " [tran-id] " "  status " " (STATUS status) (when duration (str " " duration "ms") ) "\n")
                        (str "===> " [tran-id] " "  (str/upper-case (name method)) " " url "\n")))
               (flush)
               (when (< 0 level)
                 (doseq [[k v] headers]
                   (print (format  "%s: %s\n" (name k) v))
                   (flush))
                 (println)
                 (println)
                 (cond
                   ;;No body
                   (and (empty? body) (empty? multipart))
                   (do (print "--- No body ---\n")
                       (flush))
                   
                   ;; JSON
                   (and (string? body)
                        (->> headers
                             (filter (fn [[k v]]
                                       (and (= (str/lower-case (name k)) "content-type")
                                            (= (re-find #"json" v)))))
                             (first)))
                   (json/pprint-json (json/read-str body))
                   ;; Multipart
                   (some? multipart) (do (print "----- Multipart -----\n")
                                         (flush)
                                         (clojure.pprint/pprint multipart)
                                         (print "---------------------\n")
                                         (flush))
                   ;; Map
                   (map? body) (clojure.pprint/pprint body )
                   ;; Seq
                   (seq? body) (clojure.pprint/pprint body)
                   ;; Null
                   (and (nil? body) (nil? multipart)) (print "--- No body ---\n")
                   ;; else
                   :else (do (print body) (flush))) 
                 (println)
                 (println))))))})


(defn gen-app [{:keys [root-dir]}]
  (fn [{:keys [uri]}]
    (let [f (io/file root-dir (str/replace uri #"^/*" ""))]
      (if (nio/exists? f)
        {:status 200 :body f}
        {:status 404}))))

(defn app [{:keys [root-dir base-path auth] :as opts}]
  (cond-> (rh/ring-handler
           (rh/router [["/challenge" (fn [req]
                                       {:status 200 :body (slurp (:body req))})]

                       ["/dump" (fn [{:keys [body] :as req}]
                                  (prn req)
                                  (pp/pprint (-> (select-keys req (cond-> [:request-method :headers]
                                                                    body (conj :body)))
                                                 (update :body slurp)))

                                  {:status 200})]
                       ["/update" (fn [{:keys [body]}]
                                    {:body (edn->json (update (json->edn (slurp body)) :a inc))
                                     :status 200})]
                       ["/users" (fn [{:keys [headers]}]
                                   (println (edn->json headers))
                                   {:status 500})]
                       ["/hello" (fn [{:keys [headers body] :as req}]
                                   (pp/pprint headers)
                                   {:body (edn->json {:message "Yes, I can operate now."})
                                    :headers {"Content-Type" "application/json"}
                                    :status 200})]])
           (ring/routes
            (ring/create-file-handler {:root (or root-dir ".") ;; resource directory to be mounted
                                       :path (or base-path "/")}) ;; base path to access by URL
            (ring/create-default-handler))
           {:executor sieppari/executor
            :interceptors [(params/parameters-interceptor)]})
    auth (wrap-basic-auth auth)))

(defstate server
  :start (let [{:keys [host port ssl ssl-keystore ssl-keystore-password ssl-cert ssl-key]
                :or {host "0.0.0.0" port 9180}} (mount/args)
               protocol (if ssl "https" "http")]
           (println (str "HTTP server listening on " protocol "://" host ":" port "/"))
           (srv/run-server (app (mount/args))
                           (cond-> {:ip host :port port}
                             ;; PEM format (cert + key files)
                             (and ssl ssl-cert ssl-key)
                             (assoc :ssl? true
                                    :ssl-context (make-ssl-context-from-pem
                                                  {:cert ssl-cert :key ssl-key}))
                             ;; JKS keystore format
                             (and ssl ssl-keystore (not ssl-cert))
                             (assoc :ssl? true
                                    :ssl-keystore ssl-keystore
                                    :ssl-keystore-password ssl-keystore-password)
                             ;; Built-in self-signed certificate for localhost
                             (and ssl (not ssl-cert) (not ssl-keystore))
                             (assoc :ssl? true
                                    :ssl-context @localhost-ssl-context))))
  :stop (server))

(def specs [[nil "--help" "This help"]
            ["-p" "--port <port>" "listen port"
             :default 9180
             :parse-fn parse-long]
            ["-x" "--base-path <path>"
             :default "/"]
            ["-h" "--host <host>" "listen host"
             :default "0.0.0.0"]
            ["-a" "--auth <user:pass>" "Basic auth credentials (user:pass)"]
            ["-s" "--ssl" "Enable SSL/TLS (uses built-in localhost cert if no cert specified)"
             :default false]
            ["-c" "--ssl-cert <path>" "PEM certificate file"]
            ["-k" "--ssl-key <path>" "PEM private key file"]
            ["-K" "--ssl-keystore <path>" "Path to JKS keystore file"]
            ["-P" "--ssl-keystore-password <password>" "Keystore password"]])

(defn usage [summary]
  (->> ["Usage: slv http <command> [options]"
        ""
        "Simple HTTP file server with optional Basic Authentication and SSL/TLS"
        ""
        "commands:"
        ""
        "  listen [dir]    Start HTTP server serving files from directory"
        ""
        "options:"
        summary
        ""
        "examples:"
        "  slv http listen"
        "  slv http listen ./public"
        "  slv http listen ./public -p 8080"
        "  slv http listen . -a admin:secret"
        "  slv http listen /var/www -p 80 -h 0.0.0.0 -a user:pass"
        ""
        "  # SSL with built-in self-signed cert (for localhost)"
        "  slv http listen . -s"
        ""
        "  # SSL with PEM files"
        "  slv http listen . -s -c server.crt -k server.key"
        ""
        "  # SSL with Let's Encrypt"
        "  slv http listen . -s -c /etc/letsencrypt/live/example.com/fullchain.pem \\"
        "                       -k /etc/letsencrypt/live/example.com/privkey.pem"
        ""
        "  # SSL with JKS keystore"
        "  slv http listen . -s -K keystore.jks -P changeit"
        ""]
      (str/join \newline)))

(defn main [& args]
  (let [{:keys [options arguments summary]} (parse-opts args specs)]
    (if (:help options)
      (println (usage summary))
      (condp = (first arguments)
        "listen" (-> (mount/only #{#'server})
                     (mount/with-args (assoc options :root-dir (or (second arguments) ".")))
                     (mount/start))
        nil (println (usage summary))
        (println "Not support" (first arguments))))))




