(ns silvur.oauth2
  (:gen-class)
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.pprint :as pp]
            [clojure.data.json :as json]
            [org.httpkit.server :refer [run-server
                                        with-channel
                                        on-close
                                        on-receive
                                        websocket?
                                        send!]]
            [org.httpkit.client :as http]
            [mount.core :as mount :refer [defstate]]
            [reitit.core :as r]
            [reitit.http :as rh]
            [reitit.http.interceptors.parameters :as params]
            [reitit.http.interceptors.muuntaja :as mu]
            [reitit.interceptor.sieppari]
            [muuntaja.core :as m]
            [reitit.ring :as ring]
            [clojure.core.async :refer [go]]
            [clojure.java.shell :refer [with-sh-env sh]]
            [clojure.tools.cli :refer [parse-opts]]
            [hiccup.core :refer :all]
            ;; [hiccup.form :refer [submit-button label email-field password-field]]
            [hiccup.page :refer [html5 include-css include-js]]
            [silvur.datetime :refer :all]
            [silvur.util :refer (uuid nrepl-start edn->json json->edn nrepl-stop)]
            [silvur.nio :as nio]
            [taoensso.timbre :as log]
            ;; [moncee.core :refer :all]
            [camel-snake-kebab.core :as csk]
            [camel-snake-kebab.extras :as cske]
            [buddy.core.codecs :as codecs]
            [buddy.sign.jwt :as jwt]))

;; {"access_token":"eyJhbGciOiJIUzUxMiIsInR5cCI6IkpXVCJ9.eyJhdWQiOiJvYXV0aDJjbGkiLCJleHAiOjE2MjAzNjAwNDksInN1YiI6IjVlYTc3MWQzMzU3MDFlMDAwMTc3NmFlMSJ9.4p5Ez2mxlm1vpO4vBir7E9nF07mFGDSqeucRokd9nssKgVOIB-pbBf-3N4JNzyw4jDrW5B5iC3iPiAebkxJqtw","expires_in":7200,"refresh_token":"XZCIRPEXWQAI5DVLOUJD_G","token_type":"Bearer"}

;; Set default
;; (def mule-context {:auth-uri "https://anypoint.mulesoft.com/accounts/api/v2/oauth2/authorize"
;;                    :token-endpoint "https://anypoint.mulesoft.com/accounts/api/v2/oauth2/token"
;;                    :client-id "myclient"
;;                    :clinet-secret "mysecret"
;;                    :redirect-uri "https://anypoint.mulesoft.com/accounts/login/mulesoft-06863/providers/f41de43d-7d25-484d-9219-31f7236b9775/redirect"
;;                    :port 9180
;;                    })



(def context {}
  ;; {:port          9180
  ;;  :auth-endpoint "https://anypoint.mulesoft.com/accounts/api/v2/oauth2/authorize"
  ;;  :auth-options  {:client_id     "ac158d2a638946c3818135ee18a451e5"
  ;;                  :response_type "code"
  ;;                  :nonce     (str (rand-int 100000))
  ;;                  :scope ["openid"]
  ;;                  :redirect_uri  "http://localhost:9180/oauth2/callback"}
   
  ;;  :token-endpoint "https://anypoint.mulesoft.com/accounts/api/v2/oauth2/token"
  ;;  :token-options  {:grant_type    "authorization_code"
  ;;                   :client_id     "ac158d2a638946c3818135ee18a451e5"
  ;;                   :client_secret "3e11e730d254479eb75f9B5e79877F1D"
  ;;                   :redirect_uri  "http://localhost:9180/oauth2/callback"
  ;;                   :code          "need-code"}}
  )





;; (def claim {:iss "theorems"
;;             :exp (plus (now) (days 1))
;;             :lat (now)})

;; In memory database when mongodb  is not specified
(def users (atom [{:index 0 :user "dx" :password "dx"}]))

(def idp-clients (atom {}))

(defonce channel-hub (atom {}))

(defn find-account [u p]
  (if (:database context)
    ;; (dissoc (fetch (restrict :user u :password p :accounts)) :_id)
    (do (println "No support")
        (System/exit 0))
    (first (filter #(and (= (:password %) p)
                         (= (:user %) u)) @users))))


(defn find-token [k v]
  (if (:database context) 
    (do (println "No support")
        (System/exit 0)
        ;;(dissoc (fetch (restrict k v :tokens)) :_id)
        )
    (reduce (fn [_ [i item]]
                  (when (= v (k item)) (reduced (assoc item :index i))))
                []
                (map vector (range) @users))))


(defn store-item! [{:keys [index user password] :as item} & kvs]
  (log/debug "Store:" (seq (map vec (partition-all 2 kvs))))
  (let [m (into {} (map vec (partition-all 2 kvs)))
        z (dissoc (merge item m) :password)]
    (if (:database context) 
      (do (println "No support")
          (System/exit 0)
          ;;(insert! :tokens z)
          )
      (swap! users update-in [index] merge z))))



(defn page-content
  ([m]
   (page-content "" m))
  ([message {:keys [state response-type redirect-uri]}]
   [:form {:novalidate "" :role "form" :method "POST" :action "/oauth2/auth"}
    [:div {:class "field"}
     [:label {:class "label"} "User ID"]
     [:div {:class "control"}
      [:input {:class "input" :name "user" :type "user" :placeholder "you@example.com, or your ID"}]]]
    [:div {:class "field"}
     [:label {:class "label"} "Password"]
     [:div {:class "control"}
      [:input {:class "input" :name "password" :type "password" :placeholder "Password"}]]]
    [:input {:type "hidden" :name "state" :value state}]
    [:input {:type "hidden" :name "response_type" :value response-type}]
    [:input {:type "hidden" :name "redirect_uri" :value redirect-uri}]
    [:div {:class "field is-grouped"}
     [:div {:class "control"}
      [:input {:class "button is-link" :type "submit"}]]]
    [:div {:styles {:color "red"}}message]]))

(defn login-page [title & content]
  (log/debug "Login Page")
  (html5 {:ng-app "OAuth2" :lang "en"}
         [:head
          [:title title]
          [:meta {:name "viewport" :content "width=device-width,initial-scale=1"}]
          (include-css "https://cdn.jsdelivr.net/npm/bulma@0.9.1/css/bulma.min.css")
          [:body
           [:br]
           [:div.container.is-fluid
            content ]]]))

(defn auth-handle [{:keys [params] :as req}]
  (prn req)
  (log/debug "Auth" params)  
  (let [{:strs [client_id response_type state scope redirect_uri nonce]} params
        client-id (-> context :auth-options :client-id)]

    (log/debug client-id)

    (if (= client_id client-id)
      (do
        (swap! idp-clients assoc state params)
        {:status 200 :body (login-page "Authorization" (page-content {:response-type response_type
                                                                     :state         state
                                                                     :scope         scope
                                                                     :redirect-uri  redirect_uri}))})
      {:status 401 :body "Not authorized"})))

(defn login-handle [{:keys [params] :as req}]
  (log/debug "Login" params)
  (let [{:strs [client_id response_type state scope redirect_uri user password]} params
        code (str (uuid))
        {idx :index correct-passwd :password :as item} (find-account user password)]
    (log/debug user password correct-passwd)
    (if (and idx (= password correct-passwd))
      (do (store-item! item :code code)
          (swap! idp-clients assoc code params)
          {:status 302
           :headers {"Location" (str redirect_uri "?code=" code "&" "state=" state)}})
      {:status 200 :body (->> {:response-type response_type
                               :state         state
                               :scope         scope
                               :redirect-uri  redirect_uri}
                              (page-content "User and/or Password incorrect")
                              (login-page "Authorization"))})))

(defn token-handle [{:keys [params]}]
  (log/info "Token" params)
  (let [{:strs [grant_type code client_id redirect_uri refresh_token]} params
        {:keys [index] :as item} (or (find-token :code code)
                                     (and (= grant_type "refresh_token")
                                          (find-token :refresh-token refresh_token)))]

    (jwt/sign {:iss "https://theorems.io" :aud "myclient" :exp (+ 3600 (datetime*)) } "secret")
    (log/debug "Index" index)
    (if index
      (let [token (str (uuid))
            refresh-token (str (uuid))
            state (get-in @idp-clients [code "state"])
            nonce (get-in @idp-clients [state "nonce"])
            
            id-token-payload {:iss "https://theorems.io"
                              :aud "myclient"
                              :exp (+ 3600 (datetime*))
                              :iat (datetime*)
                              :nonce nonce}
            id-token (jwt/sign id-token-payload "secret")]
        
        (log/debug @idp-clients [code "nonce"])
        (log/debug "id-token: " id-token-payload )
        (store-item! item :token token :refresh-token refresh-token :id_token id-token :code nil )
        
        {:status 200
         :headers {"Content-Type" "application/json"}
         :body (m/encode "application/json" {:access_token  token
                                             :token_type    "Bearer"
                                             :expires_in    3600
                                             ;; :scope         "general"
                                             :refresh_token refresh-token
                                             :id_token id-token
                                             ;;:t (datetime)
                                             })})
      
      
      {:status 401 :body "Invalid code"})))




;; Behave oauth2 client


(defn gen-auth-uri []
  (format "%s?%s"
          (:auth-endpoint context)
          (->> (:auth-options context) (reduce (fn [r [k v]] (str r "&" (csk/->snake_case_string k) "=" (if (string? v) v (str/join "," v)))) ""))
          ;; (str/join "&" (map #(str/join "=" (map name %)) (->> (dissoc (:auth-options context) :scope)
          ;;                                                (cske/transform-keys csk/->snake_case ))))
          ;; (str/join "%20" (:scope (:auth-options context)))
          )

  
  ;; (str (:auth-endpoint context)
  ;;   "?"
  ;;   (str/join "&" (map #(str/join "=" (map name %)) (->> (dissoc (:auth-options context) :scope)
  ;;                                                        (cske/transform-keys csk/->snake_case ))))
  ;;   "&"
  ;;   "scope="
  ;;   (str/join "%20" (:scope (:auth-options context))))
  )

;; %20read:api_policies%20manage:api_proxies%20read:servers%20read:exchange%20read:secrets%20read:secrets_metadata

; https://anypoint.mulesoft.com/accounts/api/v2/oauth2/authorize?client_id=ac158d2a638946c3818135ee18a451e5&scope=manage:api_configuration%20manage:api_policies&response_type=code&redirect_uri=http://localhost:9180/oauth2/callback&nonce=123456



(defn callback-handle [{:keys [params status body] :as req}]
  (let [{:strs [code state]} params
        {:keys [token-endpoint token-options]} context
        token-opts (->> (assoc token-options :code code)
                        (cske/transform-keys csk/->snake_case ))]
    (log/debug "ENDPOINT" token-endpoint)
    (log/debug  "BODY:" token-opts)
    (if code
      (let [{:keys [redirect-uri]} context
            {cbody :body :as resp} @(http/post (:token-endpoint context) 
                                               { :headers {"Content-Type" "application/x-www-form-urlencoded"}
                                                ;;:basic-auth [(:client_id token-opts) (:client_secret token-opts)]
                                                ;; :body (edn->json :snake token-opts)
                                                ;;:body (edn->json token-opts)
                                                :form-params token-opts})]

        (log/debug cbody)
        {:status 200 :body cbody :headers {"Content-Type" "application/json"}})
      {:status 400 :body body :headers {"Content-Type" "application/json"}})))






;; Make sure to use ring-handler in reitit.http , otherwise , 406 returned

;; Your resource directory is on classpath
;; root-dir is the relative path of resource directory

;; Server:
;;  $ java -cp target/app.jar:. app.core -r tmp
;;  README.md should be in ./tmp 
;; Client:
;;  $ wget http://localhost:8090/README.md

(defn debug-body-intercepter []
  {:name ::debug-body
   :enter (fn [ctx]
            (let [req (:request ctx)
                  {:keys [headers]} req]
              (cond
                (= (headers "content-type") "application/json")
                (clojure.pprint/pprint  (m/decode "application/json" (str/join (map char (.bytes (:body req))))))
                :else
                (clojure.pprint/pprint  req))
              ctx))})

(defn no-handle [req]
  (clojure.pprint/pprint  (m/decode "application/json" (str/join (map char (.bytes (:body req))))))
  {:status 200})

(defn oidc-dynamic-registration-handler [req]
  {:status 201
   :body (edn->json {:client_id "myclient"
                     :client_secret "mysecret"
                     :client_secret_expires_at (+ 3600 (datetime*))})})

;;  https://anypoint.mulesoft.com/login/domain/mulesoft-06863

(defn oidc-userinfo [{:keys [params] :as req }]
  (log/debug req)
  ;; Actually request is HTTP Long polling. with-channel can be used.
  
  {:status 200
   :body (edn->json {:sub "tm00001"
                     :name "Tsutomu Miyashita"
                     :given_name "Tsutomu"
                     :family_name "Miyashita"
                     :email "myst3m@gmail.com"})}
  )


(defn oidc-register [req]
  (log/debug "register:" req)
  (log/info "instrospect:" (:params req))
  {:status 201
   :body (edn->json :snake {:client_id (str (datetime*))
                            :client_secret (str (datetime*))
                            :client_secret_expires_at 300
                            :registration_client_url "http://mule-dev.com/oauth2/connect/register"
                            :redirect_urls ["http://mule-dev.com/oauth2/callback1"
                                            "http://mule-dev.com/oauth2/callback2"]})})

(defn oidc-introspect [req]
  (log/debug "register:" req)
  (log/info "instrospect:" (:params req))
  (doto {:status (or (parse-long (get-in req [:params "token"])) 200)
         :body (edn->json {"token_type" "Bearer",
                           "mail" "richard@example.com",
                           "uid" "richard",
                           "exp" (+ (datetime*) 10)})}
    (log/info)))



(def app
  (rh/ring-handler
   (rh/router
    ["" {:muuntaja m/instance}
     ["/timeout/:wait-time" {:get (fn [req]
                                    (log/debug (:headers req))
                                    (log/debug (str "Requested: " (parse-long (-> req :path-params :wait-time))))
                                    (Thread/sleep (parse-long (-> req :path-params :wait-time)))
                                    {:status 200
                                     :body (str "waited: " (-> req :path-params :wait-time) " ms")
                                     }

                                    )}]
     ["/default" {:get (fn [req] (clojure.pprint/pprint req) {:status 200 :body "ok"})
                  :post (fn [req] (clojure.pprint/pprint req))}]
     ["/oauth2/connect"
      ["/login" {:post oidc-dynamic-registration-handler}]
      ["/issue" {:get no-handle}]
      ["/userinfo" {:get oidc-userinfo}]
      ["/register" {:post oidc-register
                    :get oidc-register
                    :put (fn [req] (clojure.pprint/pprint req) {:status 200})}]
      ["/register/:client-id" {:put (fn [req]
                                      (clojure.pprint/pprint (:body req))
                                      {:status 200
                                       :body (edn->json :snake {:client-id "1696163842852"
                                                                :client-secret (str (datetime*))})})
                               :post (fn [req] (clojure.pprint/pprint req) {:status 200})
                               :delete (fn [req]
                                         (clojure.pprint/pprint req)
                                         {:status 200})}]
      ["/introspect" {:get oidc-introspect
                      :post oidc-introspect}]]
     ["/oauth2/auth" {:get auth-handle
                      :post login-handle}]
     ["/oauth2/token" {:post token-handle}]
     ["/oauth2/callback" {:get callback-handle}]])
   (ring/routes
    (ring/create-resource-handler {:root (:root-dir context)
                                   :path "/"})
    (ring/create-default-handler))
   {:interceptors [(mu/format-interceptor)
                   (params/parameters-interceptor)
                   (debug-body-intercepter)]
    :executor     reitit.interceptor.sieppari/executor}))

;; (defonce server (atom nil))

;; (defn start []
;;   (run-server #'app (merge {:port 9180} (mount/args))))

;; (defn stop [srv]
;;   (srv))

(defn load-config [& [config-file]]
  (read-string (slurp config-file)))




(defstate config
  :start
  (let [{:keys [auth-uri token-uri client-id client-secret redirect-uri port
                context config-file nrepl nrepl-port
                scope] :as opts} (mount/args)]
    (->> (cond-> {} 
           config-file (conj (read-string (slurp config-file)))
           port (assoc-in [:port] port)
           auth-uri (assoc-in [:auth-endpoint] auth-uri)
           token-uri (assoc-in [:token-endpoint] token-uri)
           client-id (assoc-in [:auth-options :client-id] client-id)
           redirect-uri (assoc-in [:auth-options :redirect-uri] redirect-uri)
           client-id (assoc-in [:token-options :client-id] client-id)
           client-secret (assoc-in [:token-options :client-secret] client-secret)
           redirect-uri (assoc-in [:token-options :redirect-uri] redirect-uri)
           scope (assoc-in [:auth-options :scope] scope)
           nrepl (assoc :nrepl nrepl)
           nrepl-port (assoc :nrepl-port nrepl-port))
         (constantly)
         (alter-var-root #'context)))
  :stop {})

(defstate server
  :start  (run-server #'app (merge {:port 9180} (select-keys (mount/args) [:port]))) ;;(start)
  :stop (server))


(defstate nrepl
  :start (when (:nrepl config) (nrepl-start :ip "localhost" :port (or (:nrepl-port (mount/args)) 7888)))
  :stop (nrepl-stop))

(def options
  [["-p" "--port <PORT>" "Listen port"
    :default-desc "9180"
    :parse-fn parse-long]
   ["-c" "--config-file <FILE>" "Configuration file"
    :missing "A config-file is required to specify"
    :validate [nio/exists? "No file"]]
   ["-a" "--auth-uri <URI>" "Authentication URI"]
   ["-t" "--token-uri <URI>" "Token URI"]
   ["-u" "--redirect-uri <URI>" "Redirect URI"]
   ["-i" "--client-id <ID>" "Client ID"]
   ["-s" "--client-secret <SECRET>" "Client secret"]
   ["-S" "--scope <SCOPE>" "Scopes as openid,email,profile"
    :default [:openid]
    :parse-fn #(str/split % #",")]
   ["-N" "--nrepl" "Boot nREPL"
    :default false]
   [nil "--nrepl-port <PORT>" "nREPL Listen port"
    :default 7999
    :parse-fn parse-long]
   ["-d" "--debug <LEVEL>" "Debug Level"
    :default :info
    :parse-fn keyword
    :validate [#{:trace :debug :info :warn :error :fatal :report} "Should be report/fatal/error/warn/info/debug/trace"]]
   ["-h" "--help" "This help"
    :default false]])


(defn dev []
  (mount/start (mount/with-args {:config-file "azure-dev.edn"})))

(def sample
  {:port 9180,
   :auth-endpoint "https://login.microsoftonline.com/17bccb98-ce0c-411d-a8ae-76f0d6920643/oauth2/v2.0/authorize",
   :auth-options {:client-id "ac18d9d9e-b9e0-4cb3-b805-192d860914b3",
                  :response-type "code",
                  :scope ["openid"],
                  :redirect-uri "http://localhost:9180/oauth2/callback"}
   ,
   :token-endpoint "https://login.microsoftonline.com/17bccb98-ce0c-411d-a8ae-76f0d6920643/oauth2/v2.0/token",
   :token-options {:grant-type "authorization_code",
                   :client-id "ac18d9d9e-b9e0-4cb3-b805-192d860914b3",
                   :client-secret "z3.8Q~GgBrX-l0WctVgq60rZt.tc1lOCjwCYNbN6",
                   :redirect-uri "http://localhost:9180/oauth2/callback"}})

(defn usage [summary]
  (->> ["Usage: slv oauth2 <listen|client> [options]" 
        ""
        "sub command:"
        " * sample: Show sample config"
        ""
        "options:"
        ""
        summary
        ""]
      (str/join \newline)))

(defn main [& args]
  (let [{:keys [options arguments summary errors]} (parse-opts args options)
        opts (dissoc options :help)
        op (first arguments)]
    (log/set-level! (:debug opts))
    (log/debug opts)
    (cond 
      (or (:help options) (nil? op)) (do (println (usage summary))
                                         (System/exit 0))
      (not-empty errors) (println (str/join "\n" errors))
      (= "sample" (first arguments)) (println (with-out-str (println) (clojure.pprint/pprint sample)))
      (#{"cli" "client"} op) (do
                               (println)
                               (println "== Context: " )
                               (mount/start #'config (mount/with-args opts))
                               (println (with-out-str (clojure.pprint/pprint config)))
                               (println "== Auth URI:\n" (gen-auth-uri))
                               (when-let [cfg (:config opts)] 
                                 (reset! users (read-string (slurp (io/resource cfg)))))
                               (mount/start #'server))
      :else (do
              
              (-> (mount/with-args opts)
                  (mount/only [#'config #'server])
                  (mount/start))
              (println "== Context: " )
              (println (with-out-str (clojure.pprint/pprint config)))))))




;; @(http/post "https://proxy-jsonplaceholder-api-49wz6g.fmxi3r-2.jpn-e1.cloudhub.io/todos/1"
;;   {:headers {"Authorization" "Bearer eyJhbGciOiJIUzI1NiJ9.eyJpc3MiOiJodHRwczovL3RoZW9yZW1zLmlvIiwiYXVkIjoibXljbGllbnQiLCJleHAiOjE2OTYxNzc1OTB9.l66Tvyi4e1-YEIRYgC6WUAIMWY19UOLUoGlIigghmjc"}})



;;123456

;; http://theorems.io:9180/oauth2/connect/register
;; http://theorems.io:9180/oauth2/auth
;; http://theorems.io:9180/oauth2/token
;; http://theorems.io:9180/oauth2/connect/introspect



