;; *   Silvur
;; *
;; *   Copyright (c) Tsutomu Miyashita. All rights reserved.
;; *
;; *   The use and distribution terms for this software are covered by the
;; *   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; *   which can be found in the file epl-v10.html at the root of this distribution.
;; *   By using this software in any fashion, you are agreeing to be bound by
;; * 	 the terms of this license.
;; *   You must not remove this notice, or any other, from this software.

(ns silvur.util
  (:gen-class)
  (:require [clojure.string :as str :refer [upper-case lower-case]]
            [clojure.core.async :refer (chan put! alts! go-loop timeout close!)]
            [org.httpkit.client :as http]
            [clojure.data.xml :as xml]
            [clojure.java.io :as io]
            [postal.core :as postal]
            [nrepl.server :as n]
            [nrepl.core :refer (message client)]
            [nrepl.transport :as transport]
            [taoensso.timbre :as log]
            [jsonista.core :as json])
  (:import [java.security MessageDigest]
           [java.io File]
           [java.util.stream Collectors]
           [java.util.concurrent LinkedBlockingQueue TimeUnit]))

(defmacro uuid
  ([]
   `(random-uuid))
  ([s]
   `(java.util.UUID/fromString ~s)))

(defprotocol NumberConvert
  (str->int [x])
  (str->long [x])
  (str->double [x])
  (hex->long [x])
  (chars->hex [xs])
  (chars->long [xs]))

(extend-protocol NumberConvert
  java.lang.String
  (str->int [x] (Integer/valueOf x))
  (str->long [x] (Long/valueOf x))
  (str->double [x] (Double/valueOf x))
  (hex->long [x] (Long/valueOf x 16))
  (chars->hex [xs] (str/join (map #(Long/toHexString (long %)) xs)))
  (chars->long [xs] (-> xs chars->hex hex->long)))


(defn sha1sum [& vals]
  (let [sha1 (MessageDigest/getInstance "SHA1")]
    (doseq [v vals]
      (.update sha1 (.getBytes (str v) "UTF-8")))
    (str/join (map #(format "%02x" %) (.digest sha1)))))

(defmacro uk [s]
  `(try (keyword (upper-case (name ~s))) (catch Exception e# nil)))

(defmacro us [s]
  `(try (upper-case (name ~s)) (catch Exception e# nil)))

(defmacro lk [s]
  `(try (keyword (lower-case (name ~s))) (catch Exception e# nil)))

(defmacro ls [s]
  `(try (lower-case (name ~s)) (catch Exception e# nil)))

(defn json->edn [x]
  ;; JSON String "\"string\"" can be handled, but cannot for string likes "string".
  (try
    (json/read-value x)
    (catch Exception e x)))

(defn edn->json [x]
  (json/write-value-as-string x))

(defn edn->xml [edn & [raw?]]
  (letfn [(parse [edn]
            (loop [m edn
                   result {}]
              (cond
                (nil? m) result
                ;; pickup 1 element
                ;; handle element
                (map? m)  (if-let [[k v] (first m)]
                            (cond
                              (.matches (name k) "^-.*") (recur (into {} (rest m)) (update result :attrs assoc (keyword (subs (name k) 1)) v))
                              :else (recur (into {} (rest m))
                                           (let [z (parse v)]
                                             
                                             (if (vector? z)
                                               {:content (mapv #(assoc % :tag k) z)}
                                               (update result :content
                                                       (comp vec conj)
                                                       (conj {:tag k} (parse v)))))))
                            result)
                (vector? m) (mapv parse m)
                :else {:content (vec (flatten [m]))})))]
    (let [xml (:content (parse edn))]
      (cond-> (if (and (vector? xml) (< 1 (count xml)))
                {:tag :root :content xml}
                xml)
        (not raw?) (xml/indent-str)))))

;; NIO



;; SMTP
(defn mail [& {:keys [from to subject host port contents] :or {host "localhost" port 25}}]
  ;; Ex. (mail :host "lebesgue" :from "analysis@theorems.co" :to ["myst3m@gmail.com"] :subject "Test" :contents {:text/plain "Hello"})
  (postal/send-message {:host host}
                       {:from from :to to :subject subject
                        :body (reduce conj [] (map (fn [[type content]]
                                                     (hash-map :type (condp = type
                                                                       :attachment type
                                                                       :inline type
                                                                       (clojure.string/replace (str type "; charset=utf-8") #":" ""))
                                                               :content content))
                                                   contents))}))


(defn parse-binary-message [format-coll data-coll]
  ;; format-coll: [1 3 3 4 4 4 4 4 4]
  ;;              [[1 chars->hex] [3 chars->hex] ....]
  (loop [sf format-coll
         pdata []
         zs data-coll]
    (let [v (first sf)
          [s cf] (cond
                   (number? v) [v]
                   (sequential? v) v
                   (= :* v) [(count zs)]
                   :else [])]
      (if-not (seq sf)
        pdata
        (recur (rest sf) (conj pdata ((or cf identity) (take s zs))) (drop s zs))))))


(def nrepl-server (atom nil))
(def lock (Object.))
;; nREPL

(defn nrepl-start [& {:keys [ip port cider] :or {ip "0.0.0.0" port 7888}}]
  (locking lock
   (when-not @nrepl-server
     (letfn [(nrepl-handler []
               (require 'cider.nrepl)
               (ns-resolve 'cider.nrepl 'cider-nrepl-handler))]

       (println (str "Boot nREPL server on " ip  ":" port (when cider (str " with cider option"))))
       (reset! nrepl-server
               (if cider
                 (n/start-server :port port :bind ip :handler (nrepl-handler))
                 (n/start-server :port port :bind ip
                                 :transport-fn transport/tty
                                 :greeting-fn (fn [transport]
                                                (transport/send transport
                                                                {:out (str "\nWelcome to  nREPL !\n\n"
                                                                           "user=> ")})))))))))

(defn nrepl-stop [& [srv]]
  (locking lock
   (when @nrepl-server
     (n/stop-server (or srv @nrepl-server))
     (reset! nrepl-server nil))))

;; (comment
;;   (def c (client (client-transport "http://localhost:12345") 1000))
;;   (message c {:op "eval" :code "(+ 1 2 3)"}))

(defn nrepl-http-transport
  "Returns an nREPL client-side transport to connect to HTTP nREPL
   endpoints implemented by `ring-handler`.

   This fn is implicitly registered as the implementation of
   `nrepl.core/url-connect` for `http` and `https` schemes;
   so, once this namespace is loaded, any tool that uses `url-connect`
   will use this implementation for connecting to HTTP and HTTPS
   nREPL endpoints."
  [url]
  (let [incoming (LinkedBlockingQueue.)
        fill #(when-let [responses (->> (io/reader %)
                                        line-seq
                                        rest
                                        drop-last
                                        (map json->edn)
                                        (remove nil?)
                                        seq)]
                (.addAll incoming responses))
        session-cookies (atom nil)
        http (fn [& [msg]]
               (let [{:keys [cookies body] :as resp} @((if msg http/post http/get)
                                                       url
                                                       (merge {:as :stream
                                                               :cookies @session-cookies}
                                                              (when msg {:form-params msg})
                                                              ;;(when http-headers {:headers http-headers})
                                                              ))]
                 (swap! session-cookies merge cookies)
                 (fill body)))]
    (transport/->FnTransport
     (fn read [timeout]
       (let [t (System/currentTimeMillis)]
         (or (.poll incoming 0 TimeUnit/MILLISECONDS)
             (when (pos? timeout)
               (http)
               (recur (- timeout (- (System/currentTimeMillis) t)))))))
     http
     (fn close []))))


;; To be used from Java
(defmulti nrepl (fn [type & _] (keyword (name type))))
(defmethod nrepl :start [type & args]
  (apply nrepl-start args))
(defmethod nrepl :stop [type & args]
  (nrepl-stop))


(defonce control-ports (atom {}))

(defn stop-routine
  ([]
   (put! (last (first @control-ports)) :quit))
  ([id]
   (put! (@control-ports id) :quit)))


(defn start-routine [f & [{:keys [id queue-length interval]
                           :as opts}]]
  (let [ctrl-port (chan (or queue-length 1))
        id (or id (gensym))]
    (swap! control-ports assoc id ctrl-port)
    (go-loop [id id]
      (let [[v ch] (alts! [ctrl-port (timeout (cond
                                                (fn? interval) (interval)
                                                (number? interval) interval
                                                :else 2000))])]
        (if (and (= :quit v) (= ch ctrl-port))
          (do (log/info "Go routine" id "quit")
              (close! (@control-ports id))
              (swap! control-ports dissoc id))
          (do (f opts)
              (recur id)))))
    ctrl-port))
