(ns mulify.apikit
  (:require [clojure.data.xml :as dx]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [mulify.ee]
            [mulify.http]
            [mulify.core :refer (defprocess global-process-map >camel)]))

(defprocess config {:name                      "apikit-config"
                    :outbound-headers-map-name "outboundHeaders"
                    :http-status-var-name      "httpStatus"}
  :xml-real-ns "mulify.mule-apikit")

(defprocess router {:config-ref "apikit-router-config"} :xml-real-ns "mulify.mule-apikit")

(defprocess console {:config-ref "apikit-config"} :xml-real-ns "mulify.mule-apikit")

(defprocess flow-mappings {} :xml-real-ns "mulify.mule-apikit")

(defprocess flow-mapping {:resource-path "/" :action "get" :content-type "application/json"}
  :xml-real-ns "mulify.mule-apikit")

(defmacro main-flow* [{:keys [name path listener-config-ref router-config-ref cors?]
                       :or {name "apkit-flow-main"
                            path "*"
                            listener-config-ref "HTTPS_Listener_config"
                            router-config-ref "apikit-router-config"}} & body]
  `(mulify.core/flow
    {:name ~name}
    (mulify.http/listener* {:path ~path, :config-ref ~listener-config-ref :cors? ~cors?})

    (router {:config-ref ~router-config-ref})

    ~@body
    
    (mulify.core/error-handler
     {:name "error-handler"}
     (mulify.core/on-error-propagate
      {:type "APIKIT:BAD_REQUEST"}
      (mulify.ee/transform
       (mulify.ee/message
        (mulify.ee/set-payload
         "%dw 2.0\noutput application/json\n---\n{message: \"Bad request\"}"))
       (mulify.ee/variables
        (mulify.ee/set-variable {:variable-name "httpStatus"} "400"))))
     (mulify.core/on-error-propagate
      {:type "APIKIT:NOT_FOUND"}
      (mulify.ee/transform
       (mulify.ee/message
        (mulify.ee/set-payload
         "%dw 2.0\noutput application/json\n---\n{message: \"Resource not found\"}"))
       (mulify.ee/variables
        (mulify.ee/set-variable {:variable-name "httpStatus"} "404"))))
     (mulify.core/on-error-propagate
      {:type "APIKIT:METHOD_NOT_ALLOWED"}
      (mulify.ee/transform
       (mulify.ee/message
        (mulify.ee/set-payload
         "%dw 2.0\noutput application/json\n---\n{message: \"Method not allowed\"}"))
       (mulify.ee/variables
        (mulify.ee/set-variable {:variable-name "httpStatus"} "405"))))
     (mulify.core/on-error-propagate
      {:type "APIKIT:NOT_ACCEPTABLE"}
      (mulify.ee/transform
       (mulify.ee/message
        (mulify.ee/set-payload
         "%dw 2.0\noutput application/json\n---\n{message: \"Not acceptable\"}"))
       (mulify.ee/variables
        (mulify.ee/set-variable {:variable-name "httpStatus"} "406"))))
     (mulify.core/on-error-propagate
      {:type "APIKIT:UNSUPPORTED_MEDIA_TYPE"}
      (mulify.ee/transform
       (mulify.ee/message
        (mulify.ee/set-payload
         "%dw 2.0\noutput application/json\n---\n{message: \"Unsupported media type\"}"))
       (mulify.ee/variables
        (mulify.ee/set-variable {:variable-name "httpStatus"} "415"))))
     (mulify.core/on-error-propagate
      {:type "APIKIT:NOT_IMPLEMENTED"}
      (mulify.ee/transform
       (mulify.ee/message
        (mulify.ee/set-payload
         "%dw 2.0\noutput application/json\n---\n{message: \"Not Implemented\"}"))
       (mulify.ee/variables
        (mulify.ee/set-variable
         {:variableName "httpStatus"}
         "501")))))))


(defmacro with-apikit [{:keys [name api-spec port
                               key-type key-path key-password
                               listener-config-name
                               cors?]
                        :or {name "apikit-main"
                             api-spec "api.raml"
                             port "${https.port}"
                             key-type "pkcs12"
                             key-path "server.p12"
                             key-password "wbx2"
                             cors? true
                             listener-config-name "https_listener_config"}} & body]
  `(let [router-config-ref# ~(str name "-config")]
     (list  (config {:name router-config-ref#
                          :outbound-headers-map-name "outboundHeaders"
                          :http-status-var-name "httpStatus"
                          :api ~api-spec})

                 (mulify.http/https-listener-config* {:name ~listener-config-name
                                                      :port ~port
                                                      :key-type ~key-type
                                                      :key-path ~key-path
                                                      :key-password ~key-password})

                 (main-flow* {:name ~name
                              :cors? ~cors?
                              :listener-config-ref ~listener-config-name
                              :router-config-ref router-config-ref#})
                 ~@body)))
