(ns moncee.shell
  (:gen-class)
  (:require [moncee.core :as mongo]
            [moncee.sql :as sql]
            [moncee.completer :as completer]
            [silvur.datetime :refer (datetime datetime*)]
            [taoensso.timbre :as log]
            [clojure.tools.cli :refer [parse-opts]]
            [clojure.string :as str]
            [clojure.pprint :refer [pprint print-table]])
  (:import [org.jline.reader LineReaderBuilder LineReader LineReader$Option]
           [org.jline.reader.impl DefaultParser]
           [org.jline.terminal TerminalBuilder]
           [org.jline.reader EndOfFileException UserInterruptException]))

(def options-schema
  [["-t" "--type TYPE" "Database type: mongo, postgres, oracle"
    :default "postgres"
    :validate [#(contains? #{"mongo" "mongodb" "postgres" "postgresql" "pg" "oracle" "ora"} %)
               "Must be mongo, postgres, or oracle"]]
   ["-h" "--host HOST" "Database host"
    :default "localhost"]
   ["-p" "--port PORT" "Database port (default: auto by type)"
    :parse-fn #(Integer/parseInt %)]
   ["-d" "--database DATABASE" "Database name / service name"]
   ["-u" "--user USER" "Username"]
   ["-P" "--password PASSWORD" "Password"]
   ["-e" "--execute SQL" "Execute SQL and exit"]
   ["-i" "--interactive" "Interactive SQL mode (works with native-image)"]
   ["-T" "--tables" "List tables and exit"]
   ["-D" "--describe TABLE" "Describe table and exit"]
   [nil "--repl" "Force Clojure REPL mode (requires JVM)"]
   [nil "--help" "Show this help"]])

(defn- normalize-dbtype [t]
  (case t
    ("mongo" "mongodb") :mongo
    ("postgres" "postgresql" "pg") :postgresql
    ("oracle" "ora") :oracle
    :postgresql))

(defn help
  "Show available functions"
  ([]
   (println "=== moncee shell ===")
   (println)
   (println "Database functions:")
   (println "  (tables)              - List tables")
   (println "  (describe :tablename) - Describe table structure")
   (println "  (sql! \"SELECT ...\")   - Execute raw SQL")
   (println "  (exec! \"INSERT ...\")  - Execute DML/DDL")
   (println)
   (println "Query DSL:")
   (println "  (->> (sql-table :users)")
   (println "       (restrict :id 1)")
   (println "       (project [:name :email])")
   (println "       (order :name :asc)")
   (println "       (limit 10)")
   (println "       seq)")
   (println)
   (println "Shortcuts:")
   (println "  (t :tablename)        - Quick table query")
   (println "  (t :tablename 10)     - With limit")
   (println))
  ([ns-name]
   (let [nss (ns-publics (the-ns ns-name))
         length (apply max (map #(count (str %)) (keys nss)))]
     (dorun
      (map println
           (keep identity (map (fn [[k v]]
                                 (when-let [args (:arglists (meta v))]
                                   (format (str "* %-" (+ 4 length) "s" "%s")
                                           k
                                           (or (:doc (meta v)) "-"))))
                               nss)))))))

;; Shortcuts for interactive use
(defn t
  "Quick table query.
   (t :users)      - all rows
   (t :users 10)   - first 10 rows"
  ([table-name]
   (seq (sql/sql-table table-name)))
  ([table-name n]
   (seq (sql/limit n (sql/sql-table table-name)))))

(defn pp
  "Pretty print query results as table"
  [rows]
  (if (seq rows)
    (print-table rows)
    (println "No results")))

(defn run-sql-repl [{:keys [host port database user password dbtype]}]
  (let [ds (sql/datasource {:dbtype dbtype
                            :host host
                            :port port
                            :dbname database
                            :user user
                            :password password})]
    (sql/set-datasource! ds dbtype)
    (println)
    (println (str "Connected to " (name dbtype) "://" host ":" port "/" database))
    (println "Type (help) for available commands")
    (println)
    (clojure.main/repl
     :init (fn []
             (log/set-level! :warn)
             (in-ns 'moncee.shell)
             (require '[moncee.sql :refer :all])
             (require '[silvur.datetime :refer :all])))))

(defn run-mongo-repl [{:keys [host database]}]
  (println)
  (println (str "Connecting to MongoDB: " host "/" database))
  (clojure.main/repl
   :init (fn []
           (log/set-level! :warn)
           (when host
             (mongo/set-default-client! (mongo/mongo host)))
           (when database
             (mongo/set-default-db! database))
           (in-ns 'moncee.shell)
           (require '[moncee.core :refer :all])
           (require '[silvur.datetime :refer :all])
           (println "Type (help) for available commands\n"))))

;; JLine3 utilities (shared between SQL and MongoDB)
(defn- read-line-jline
  "Read a line with JLine3, returns nil on EOF/interrupt"
  [^LineReader reader prompt]
  (try
    (let [line (.readLine reader prompt)]
      (when-not (str/blank? line)
        (str/trim line)))
    (catch EndOfFileException _ nil)
    (catch UserInterruptException _ nil)))

;; MongoDB interactive mode for native-image
(defn- connect-mongo! [{:keys [host port database]}]
  (log/set-level! :warn)
  (when host
    (mongo/set-default-client! (mongo/mongo {:host host :port port})))
  (when database
    (mongo/set-default-db! database)))

(defn- create-mongo-line-reader []
  (let [terminal (-> (TerminalBuilder/builder)
                     (.system true)
                     (.build))
        parser (DefaultParser.)
        reader (-> (LineReaderBuilder/builder)
                   (.terminal terminal)
                   (.parser parser)
                   (.build))]
    reader))

(defn- mongo-help []
  (println "=== MongoDB Shell ===")
  (println)
  (println "Collections:")
  (println "  (list-collection-names)    - List collections")
  (println "  (count* :coll)             - Count documents")
  (println)
  (println "Query DSL:")
  (println "  (->> :users")
  (println "       (restrict :status \"active\")")
  (println "       (project [:name :email])")
  (println "       (order :created_at :desc)")
  (println "       (limit 10)")
  (println "       seq)")
  (println)
  (println "Insert/Update/Delete:")
  (println "  (insert! :coll {:name \"test\"})")
  (println "  (update! :coll {:_id id} {:$set {:name \"new\"}})")
  (println "  (delete! :coll {:_id id})")
  (println)
  (println "Commands: \\c (collections), \\q (quit), help"))

(declare eval-mongo)

(defn- eval-mongo
  "Recursively evaluate MongoDB expression"
  [expr]
  (cond
    ;; Keyword - just return as collection reference
    (keyword? expr) expr

    ;; Map - return as-is (for queries/documents)
    (map? expr) expr

    ;; Vector - return as-is (for projection)
    (vector? expr) expr

    ;; Number/String - return as-is
    (or (number? expr) (string? expr)) expr

    ;; List - evaluate as function call
    (list? expr)
    (let [[op & args] expr]
      (case op
        ;; Collection listing
        list-collection-names (vec (mongo/list-collection-names))

        ;; Count
        count* (mongo/count* (eval-mongo (first args)))

        ;; Query operations - evaluate args recursively
        limit (mongo/limit (eval-mongo (first args)) (eval-mongo (second args)))
        skip (mongo/skip (eval-mongo (first args)) (eval-mongo (second args)))
        restrict (apply mongo/restrict (map eval-mongo args))
        project (apply mongo/project (map eval-mongo args))
        order (apply mongo/order (map eval-mongo args))

        ;; Mutations
        insert! (apply mongo/insert! (map eval-mongo args))
        delete! (apply mongo/delete! (map eval-mongo args))
        update! (apply mongo/update! (map eval-mongo args))

        ;; seq - realize lazy sequence
        seq (seq (eval-mongo (first args)))

        ;; Thread-last macro
        ->> (let [init (eval-mongo (first args))
                  ops (rest args)]
              (reduce (fn [acc op-expr]
                        (let [[op & op-args] op-expr]
                          (case op
                            restrict (apply mongo/restrict (concat (map eval-mongo op-args) [acc]))
                            project (apply mongo/project (concat (map eval-mongo op-args) [acc]))
                            order (apply mongo/order (concat (map eval-mongo op-args) [acc]))
                            limit (mongo/limit (eval-mongo (first op-args)) acc)
                            skip (mongo/skip (eval-mongo (first op-args)) acc)
                            seq (seq acc)
                            acc)))
                      init ops))

        ;; Unknown
        (throw (ex-info (str "Unknown function: " op) {:expr expr}))))

    :else
    (throw (ex-info "Unknown expression type" {:expr expr}))))

(defn- parse-mongo-expr
  "Parse and execute MongoDB commands"
  [expr-str]
  (let [expr (read-string expr-str)
        result (eval-mongo expr)]
    ;; Only auto-realize if just a bare keyword
    (if (keyword? result)
      (seq (mongo/limit 100 result))
      result)))

(defn- result-seq?
  "Check if result should be displayed as a table"
  [result]
  (and (seqable? result)
       (not (string? result))
       (not (map? result))
       (or (seq? result)
           (vector? result))
       (seq result)
       (map? (first result))))

(defn run-interactive-mongo [{:keys [host port database]}]
  (connect-mongo! {:host host :port port :database database})
  (println)
  (println (str "Connected to MongoDB: " host ":" port "/" database))
  (println "Type Clojure expressions. Use (help) or \\? for commands.")
  (println)
  (let [reader (create-mongo-line-reader)
        prompt "mongo> "]
    (loop []
      (when-let [input (read-line-jline reader prompt)]
        (let [trimmed (str/trim input)]
          (cond
            ;; Quit
            (or (= trimmed "\\q") (= trimmed "quit") (= trimmed "exit"))
            (println "Bye!")

            ;; Empty
            (str/blank? trimmed)
            (recur)

            ;; List collections
            (= trimmed "\\c")
            (do
              (doseq [c (mongo/list-collection-names)] (println c))
              (recur))

            ;; Help
            (or (= trimmed "\\?") (= trimmed "\\h") (= trimmed "help") (= trimmed "(help)"))
            (do
              (mongo-help)
              (recur))

            ;; Clojure expression
            :else
            (do
              (try
                (let [result (parse-mongo-expr trimmed)]
                  (cond
                    (nil? result) (println "nil")
                    (result-seq? result) (pp (take 100 result))
                    (seqable? result) (let [realized (seq result)]
                                        (if (and realized (map? (first realized)))
                                          (pp (take 100 realized))
                                          (pprint result)))
                    (number? result) (println result)
                    :else (pprint result)))
                (catch Exception e
                  (println "Error:" (.getMessage e))))
              (recur))))))))

(defn- connect-sql! [{:keys [host port database user password dbtype]}]
  (let [ds (sql/datasource {:dbtype dbtype
                            :host host
                            :port port
                            :dbname database
                            :user user
                            :password password})]
    (sql/set-datasource! ds dbtype)))

(defn- create-line-reader
  "Create JLine3 LineReader with SQL completion"
  [prompt]
  (let [terminal (-> (TerminalBuilder/builder)
                     (.system true)
                     (.build))
        parser (DefaultParser.)
        completer (completer/make-completer)
        reader (-> (LineReaderBuilder/builder)
                   (.terminal terminal)
                   (.completer completer)
                   (.parser parser)
                   (.option LineReader$Option/CASE_INSENSITIVE true)
                   (.variable LineReader/SECONDARY_PROMPT_PATTERN "%P > ")
                   (.build))]
    reader))

(defn run-interactive-sql [{:keys [host port database user password dbtype]}]
  (connect-sql! {:host host :port port :database database
                 :user user :password password :dbtype dbtype})
  ;; Clear completion cache and pre-fetch tables
  (completer/clear-cache!)
  (println)
  (println (str "Connected to " (name dbtype) "://" host ":" port "/" database))
  (println "Tab completion enabled. Type SQL and press Enter.")
  (println "Commands: \\t (tables), \\d TABLE (describe), \\q (quit)")
  (println)
  (let [reader (create-line-reader (str (name dbtype) "> "))
        prompt (str (name dbtype) "> ")]
    (loop []
      (when-let [input (read-line-jline reader prompt)]
        (let [trimmed (str/replace input #";+\s*$" "")]
          (cond
            ;; Quit
            (or (= trimmed "\\q") (= trimmed "quit") (= trimmed "exit"))
            (println "Bye!")

            ;; Empty
            (str/blank? trimmed)
            (recur)

            ;; List tables
            (= trimmed "\\t")
            (do
              (doseq [t (sql/tables)] (println t))
              (recur))

            ;; Describe table
            (str/starts-with? trimmed "\\d ")
            (do
              (let [table (str/trim (subs trimmed 3))]
                (pp (sql/describe table)))
              (recur))

            ;; Help
            (or (= trimmed "\\?") (= trimmed "\\h") (= trimmed "help"))
            (do
              (println "Commands:")
              (println "  \\t          List tables")
              (println "  \\d TABLE    Describe table")
              (println "  \\q          Quit")
              (println "  Tab         Auto-complete")
              (println "  Ctrl-R      History search")
              (println "  SQL;        Execute SQL")
              (recur))

            ;; SQL
            :else
            (do
              (try
                (let [results (sql/sql! trimmed)]
                  (if (seq results)
                    (pp results)
                    (println "OK")))
                (catch Exception e
                  (println "Error:" (.getMessage e))))
              (recur))))))))

(defn -main [& args]
  (let [{:keys [options summary errors]} (parse-opts args options-schema)]
    (cond
      errors
      (do
        (doseq [e errors] (println "Error:" e))
        (System/exit 1))

      (:help options)
      (do
        (println "moncee - Database shell for MongoDB, PostgreSQL, and Oracle")
        (println)
        (println "Usage: moncee [options]")
        (println)
        (println "Options:")
        (println summary)
        (println)
        (println "Examples:")
        (println "  # PostgreSQL / Oracle interactive shell")
        (println "  moncee -t postgres -h localhost -d mydb -u postgres -P secret")
        (println "  moncee -t oracle -h dbhost -d ORCL -u scott -P tiger")
        (println)
        (println "  # MongoDB interactive shell")
        (println "  moncee -t mongo -h localhost -d testdb")
        (println)
        (println "  # Execute SQL and exit")
        (println "  moncee -t postgres -d mydb -u postgres -P secret -e \"SELECT * FROM users\"")
        (println "  moncee -t oracle -h dbhost -d ORCL -u scott -P tiger -T")
        (println "  moncee -t postgres -d mydb -u postgres -P secret -D users")
        (println)
        (println "Shell commands (SQL):")
        (println "  \\t          List tables")
        (println "  \\d TABLE    Describe table structure")
        (println "  \\q          Quit")
        (println "  Tab         SQL keyword/table/column completion")
        (println "  Ctrl-R      History search")
        (println)
        (println "Shell commands (MongoDB):")
        (println "  \\c          List collections")
        (println "  \\q          Quit")
        (println "  Clojure expressions: (->> :coll (restrict :k v) (limit 10) seq)"))

      :else
      (let [dbtype (normalize-dbtype (:type options))
            port (or (:port options)
                     (case dbtype
                       :oracle 1521
                       :postgresql 5432
                       :mongo 27017
                       5432))
            conn-opts {:host (:host options)
                       :port port
                       :database (:database options)
                       :user (:user options)
                       :password (:password options)
                       :dbtype dbtype}]

        (cond
          ;; MongoDB: List collections and exit
          (and (= dbtype :mongo) (:tables options))
          (do
            (connect-mongo! {:host (:host options) :port port :database (:database options)})
            (doseq [c (mongo/list-collection-names)]
              (println c)))

          ;; MongoDB REPL (JVM only)
          (and (= dbtype :mongo) (:repl options))
          (run-mongo-repl {:host (:host options)
                           :database (:database options)})

          ;; MongoDB interactive mode (default)
          (= dbtype :mongo)
          (run-interactive-mongo {:host (:host options)
                                  :port port
                                  :database (:database options)})

          ;; Execute SQL and exit
          (:execute options)
          (do
            (connect-sql! conn-opts)
            (let [results (sql/sql! (:execute options))]
              (if (seq results)
                (pp results)
                (println "OK"))))

          ;; List tables and exit
          (:tables options)
          (do
            (connect-sql! conn-opts)
            (doseq [t (sql/tables)]
              (println t)))

          ;; Describe table and exit
          (:describe options)
          (do
            (connect-sql! conn-opts)
            (pp (sql/describe (:describe options))))

          ;; Force Clojure REPL mode (JVM only)
          (:repl options)
          (run-sql-repl conn-opts)

          ;; Default: Interactive SQL mode
          :else
          (run-interactive-sql conn-opts))))))
