(ns moncee.completer
  "SQL completion for JLine3"
  (:require [moncee.sql :as sql]
            [clojure.string :as str])
  (:import [org.jline.reader Completer Candidate]
           [org.jline.reader.impl DefaultParser]))

;; SQL Keywords (Oracle + PostgreSQL)
(def sql-keywords
  [;; Common SQL
   "SELECT" "FROM" "WHERE" "AND" "OR" "NOT" "IN" "EXISTS" "BETWEEN" "LIKE"
   "IS" "NULL" "TRUE" "FALSE"
   "ORDER" "BY" "ASC" "DESC" "NULLS" "FIRST" "LAST"
   "GROUP" "HAVING" "DISTINCT" "ALL" "AS"
   "JOIN" "INNER" "LEFT" "RIGHT" "FULL" "OUTER" "CROSS" "ON" "USING"
   "UNION" "INTERSECT" "EXCEPT"
   "INSERT" "INTO" "VALUES" "UPDATE" "SET" "DELETE" "MERGE"
   "CREATE" "ALTER" "DROP" "TRUNCATE" "TABLE" "INDEX" "VIEW" "SEQUENCE"
   "PRIMARY" "KEY" "FOREIGN" "REFERENCES" "UNIQUE" "CHECK" "DEFAULT" "CONSTRAINT"
   "GRANT" "REVOKE" "COMMIT" "ROLLBACK" "SAVEPOINT"
   "CASE" "WHEN" "THEN" "ELSE" "END"
   "COUNT" "SUM" "AVG" "MIN" "MAX" "COALESCE"
   "UPPER" "LOWER" "TRIM" "LTRIM" "RTRIM" "REPLACE" "LENGTH"
   "CURRENT_DATE" "CURRENT_TIMESTAMP" "CURRENT_TIME"
   "WITH" "RECURSIVE"
   "FETCH" "NEXT" "ROWS" "ONLY" "OFFSET" "LIMIT"
   "PARTITION" "OVER" "ROW_NUMBER" "RANK" "DENSE_RANK" "LAG" "LEAD" "NTILE"
   "FIRST_VALUE" "LAST_VALUE" "NTH_VALUE"
   "CURSOR" "FOR"
   "BEGIN" "END" "DECLARE" "RETURN"
   "FUNCTION" "TRIGGER" "TYPE"
   ;; Oracle specific
   "MINUS" "START" "CONNECT" "PRIOR" "LEVEL" "ROWNUM" "ROWID"
   "NVL" "NVL2" "DECODE" "LISTAGG" "WITHIN"
   "TO_CHAR" "TO_DATE" "TO_NUMBER" "TRUNC" "ROUND" "SUBSTR" "INSTR" "LPAD" "RPAD"
   "SYSDATE" "SYSTIMESTAMP"
   "PROCEDURE" "PACKAGE" "BODY" "EXCEPTION" "RAISE" "LOOP" "EXIT" "CONTINUE"
   "PERCENT"
   ;; PostgreSQL specific
   "ILIKE" "SIMILAR" "RETURNING" "CONFLICT" "DO" "NOTHING"
   "LATERAL" "TABLESAMPLE" "BERNOULLI" "SYSTEM"
   "SERIAL" "BIGSERIAL" "SMALLSERIAL" "UUID"
   "JSONB" "JSON" "ARRAY" "ARRAY_AGG" "STRING_AGG"
   "GENERATE_SERIES" "UNNEST"
   "NOW" "AGE" "DATE_PART" "DATE_TRUNC" "EXTRACT" "INTERVAL"
   "GREATEST" "LEAST" "NULLIF"
   "BOOLEAN" "TEXT" "VARCHAR" "INTEGER" "BIGINT" "SMALLINT" "NUMERIC" "REAL" "DOUBLE"
   "TIMESTAMP" "TIMESTAMPTZ" "DATE" "TIME" "TIMETZ"
   "COPY" "VACUUM" "ANALYZE" "EXPLAIN" "VERBOSE"
   "MATERIALIZED" "REFRESH" "CONCURRENTLY"
   "SCHEMA" "EXTENSION" "ROLE" "DATABASE" "TABLESPACE"
   "NOTIFY" "LISTEN" "UNLISTEN"
   "LOCK" "SHARE" "EXCLUSIVE" "ACCESS" "ROW"
   "FILTER" "WITHIN GROUP"])

;; Cache for table and column names
(defonce ^:private table-cache (atom nil))
(defonce ^:private column-cache (atom {}))

(defn clear-cache! []
  (reset! table-cache nil)
  (reset! column-cache {}))

(defn- fetch-tables []
  (when-not @table-cache
    (try
      (reset! table-cache (vec (sql/tables)))
      (catch Exception _ (reset! table-cache []))))
  @table-cache)

(defn- fetch-columns [table-name]
  (let [upper-table (str/upper-case (name table-name))]
    (when-not (contains? @column-cache upper-table)
      (try
        (let [cols (->> (sql/describe table-name)
                        (map #(or (:column-name %) (:column_name %)))
                        (filter some?)
                        vec)]
          (swap! column-cache assoc upper-table cols))
        (catch Exception _ (swap! column-cache assoc upper-table []))))
    (get @column-cache upper-table [])))

(def ^:private table-context-keywords
  #{"FROM" "JOIN" "INTO" "UPDATE" "TABLE"})

(def ^:private column-context-keywords
  #{"SELECT" "WHERE" "AND" "OR" "SET" "BY" "ON"})

(defn- get-context
  "Analyze SQL to determine completion context"
  [line cursor]
  (let [before-cursor (subs line 0 cursor)
        upper (str/upper-case before-cursor)
        ;; Check if ends with space (just typed keyword)
        ends-with-space? (str/ends-with? before-cursor " ")
        tokens (str/split (str/trim before-cursor) #"\s+")
        last-token (when (seq tokens) (str/upper-case (last tokens)))
        prev-token (when (> (count tokens) 1)
                     (str/upper-case (nth tokens (- (count tokens) 2))))]
    (cond
      ;; Ends with space after table keyword -> show tables
      (and ends-with-space? (contains? table-context-keywords last-token))
      :table

      ;; Typing after table keyword (e.g., "FROM SAL") -> table completion
      (and (not ends-with-space?) prev-token (contains? table-context-keywords prev-token))
      :table

      ;; After table. -> column names for that table
      (and last-token (str/includes? (last tokens) "."))
      (let [parts (str/split (last tokens) #"\." 2)
            table (first parts)]
        {:type :column :table table :prefix (second parts)})

      ;; Ends with space after column keyword -> show columns
      (and ends-with-space?
           (or (contains? column-context-keywords last-token)
               (= (last before-cursor) \,)))
      :column-or-keyword

      ;; Typing after column keyword -> column completion
      (and (not ends-with-space?) prev-token (contains? column-context-keywords prev-token))
      :column-or-keyword

      ;; Default -> keywords
      :else
      :keyword)))

(defn- add-candidates [^java.util.List candidates items prefix]
  (let [upper-prefix (when prefix (str/upper-case prefix))]
    (doseq [item items]
      (let [upper-item (str/upper-case (str item))]
        (when (or (nil? prefix)
                  (str/blank? prefix)
                  (str/starts-with? upper-item upper-prefix))
          (.add candidates (Candidate. (str item))))))))

(defn make-completer
  "Create a JLine Completer for SQL"
  []
  (reify Completer
    (complete [_ reader line candidates]
      (let [word (.word line)
            cursor (.cursor line)
            buffer (str (.line line))
            context (get-context buffer cursor)
            prefix (when (and word (not (str/blank? word))) word)]
        (cond
          ;; Table completion
          (= context :table)
          (add-candidates candidates (fetch-tables) prefix)

          ;; Column completion for specific table
          (and (map? context) (= (:type context) :column))
          (let [cols (fetch-columns (:table context))]
            (add-candidates candidates cols (:prefix context)))

          ;; Column or keyword
          (= context :column-or-keyword)
          (do
            ;; Add all columns from all known tables
            (doseq [table (fetch-tables)]
              (add-candidates candidates (fetch-columns table) prefix))
            (add-candidates candidates sql-keywords prefix))

          ;; Keywords only
          :else
          (add-candidates candidates sql-keywords prefix))))))

(defn make-parser
  "Create a parser that handles SQL multiline (semicolon terminated)"
  []
  (doto (DefaultParser.)
    (.setEofOnUnclosedQuote true)))
