(ns sade.core
  (:require [clojure.string :as str]))

;; ------------------------------------------------------------
;; HTML5 タグ定義テーブル
;;  - :void?          閉じタグを持たない要素かどうか
;;  - :boolean-attrs  「存在＝true」として扱う属性名のセット
;; ------------------------------------------------------------

(def html-tags
  {;; Document metadata / root
   "html"    {:void? false :boolean-attrs #{}}
   "head"    {:void? false :boolean-attrs #{}}
   "body"    {:void? false :boolean-attrs #{}}
   "title"   {:void? false :boolean-attrs #{}}
   "base"    {:void? true  :boolean-attrs #{}}
   "link"    {:void? true  :boolean-attrs #{"disabled"}}      ;; disabled
   "meta"    {:void? true  :boolean-attrs #{}}
   "style"   {:void? false :boolean-attrs #{"scoped"}}        ;; scoped は廃止方向だが一応

   ;; Scripting
   "script"  {:void? false :boolean-attrs #{"async" "defer" "nomodule"}} ;;
   "noscript"{:void? false :boolean-attrs #{}}

   ;; Sections
   "section" {:void? false :boolean-attrs #{}}
   "nav"     {:void? false :boolean-attrs #{}}
   "article" {:void? false :boolean-attrs #{}}
   "aside"   {:void? false :boolean-attrs #{}}
   "header"  {:void? false :boolean-attrs #{}}
   "footer"  {:void? false :boolean-attrs #{}}
   "main"    {:void? false :boolean-attrs #{}}
   "h1"      {:void? false :boolean-attrs #{}}
   "h2"      {:void? false :boolean-attrs #{}}
   "h3"      {:void? false :boolean-attrs #{}}
   "h4"      {:void? false :boolean-attrs #{}}
   "h5"      {:void? false :boolean-attrs #{}}
   "h6"      {:void? false :boolean-attrs #{}}

   ;; Grouping content
   "p"       {:void? false :boolean-attrs #{}}
   "hr"      {:void? true  :boolean-attrs #{}}
   "pre"     {:void? false :boolean-attrs #{}}
   "blockquote" {:void? false :boolean-attrs #{}}
   "ol"      {:void? false :boolean-attrs #{}}
   "ul"      {:void? false :boolean-attrs #{}}
   "li"      {:void? false :boolean-attrs #{}}
   "dl"      {:void? false :boolean-attrs #{}}
   "dt"      {:void? false :boolean-attrs #{}}
   "dd"      {:void? false :boolean-attrs #{}}
   "figure"  {:void? false :boolean-attrs #{}}
   "figcaption" {:void? false :boolean-attrs #{}}
   "div"     {:void? false :boolean-attrs #{}}

   ;; Text-level semantics
   "a"       {:void? false :boolean-attrs #{"download"}}      ;; download
   "em"      {:void? false :boolean-attrs #{}}
   "strong"  {:void? false :boolean-attrs #{}}
   "small"   {:void? false :boolean-attrs #{}}
   "s"       {:void? false :boolean-attrs #{}}
   "cite"    {:void? false :boolean-attrs #{}}
   "q"       {:void? false :boolean-attrs #{}}
   "dfn"     {:void? false :boolean-attrs #{}}
   "abbr"    {:void? false :boolean-attrs #{}}
   "data"    {:void? false :boolean-attrs #{}}
   "time"    {:void? false :boolean-attrs #{}}
   "code"    {:void? false :boolean-attrs #{}}
   "var"     {:void? false :boolean-attrs #{}}
   "samp"    {:void? false :boolean-attrs #{}}
   "kbd"     {:void? false :boolean-attrs #{}}
   "sub"     {:void? false :boolean-attrs #{}}
   "sup"     {:void? false :boolean-attrs #{}}
   "i"       {:void? false :boolean-attrs #{}}
   "b"       {:void? false :boolean-attrs #{}}
   "u"       {:void? false :boolean-attrs #{}}
   "mark"    {:void? false :boolean-attrs #{}}
   "ruby"    {:void? false :boolean-attrs #{}}
   "rt"      {:void? false :boolean-attrs #{}}
   "rp"      {:void? false :boolean-attrs #{}}
   "bdi"     {:void? false :boolean-attrs #{}}
   "bdo"     {:void? false :boolean-attrs #{}}
   "span"    {:void? false :boolean-attrs #{}}
   "br"      {:void? true  :boolean-attrs #{}}
   "wbr"     {:void? true  :boolean-attrs #{}}

   ;; Edits
   "ins"     {:void? false :boolean-attrs #{}}
   "del"     {:void? false :boolean-attrs #{}}

   ;; Embedded content
   "img"     {:void? true  :boolean-attrs #{"ismap"}}         ;; ismap
   "iframe"  {:void? false :boolean-attrs #{"allowfullscreen"}} ;; allowfullscreen
   "embed"   {:void? true  :boolean-attrs #{}}
   "object"  {:void? false :boolean-attrs #{}}
   "param"   {:void? true  :boolean-attrs #{}}
   "video"   {:void? false :boolean-attrs #{"autoplay" "controls" "loop" "muted" "playsinline"}} ;;
   "audio"   {:void? false :boolean-attrs #{"autoplay" "controls" "loop" "muted"}}               ;;
   "source"  {:void? true  :boolean-attrs #{}}
   "track"   {:void? true  :boolean-attrs #{"default"}}       ;; default
   "canvas"  {:void? false :boolean-attrs #{}}
   "map"     {:void? false :boolean-attrs #{}}
   "area"    {:void? true  :boolean-attrs #{"download"}}      ;; download
   "svg"     {:void? false :boolean-attrs #{}}
   "math"    {:void? false :boolean-attrs #{}}

   ;; Tabular data
   "table"   {:void? false :boolean-attrs #{}}
   "caption" {:void? false :boolean-attrs #{}}
   "colgroup"{:void? false :boolean-attrs #{}}
   "col"     {:void? true  :boolean-attrs #{}}
   "tbody"   {:void? false :boolean-attrs #{}}
   "thead"   {:void? false :boolean-attrs #{}}
   "tfoot"   {:void? false :boolean-attrs #{}}
   "tr"      {:void? false :boolean-attrs #{}}
   "td"      {:void? false :boolean-attrs #{}}
   "th"      {:void? false :boolean-attrs #{}}

   ;; Forms
   "form"    {:void? false :boolean-attrs #{"novalidate"}}    ;; novalidate
   "label"   {:void? false :boolean-attrs #{}}
   "input"   {:void? true  :boolean-attrs #{"disabled" "readonly" "required"
                                           "autofocus" "checked" "multiple"
                                           "formnovalidate"}}                 ;;
   "button"  {:void? false :boolean-attrs #{"disabled" "autofocus" "formnovalidate"}} ;;
   "select"  {:void? false :boolean-attrs #{"disabled" "multiple" "required" "autofocus"}} ;;
   "datalist"{:void? false :boolean-attrs #{}}
   "optgroup"{:void? false :boolean-attrs #{"disabled"}}      ;;
   "option"  {:void? false :boolean-attrs #{"disabled" "selected"}} ;;
   "textarea"{:void? false :boolean-attrs #{"disabled" "readonly" "required" "autofocus"}} ;;
   "output"  {:void? false :boolean-attrs #{}}
   "progress"{:void? false :boolean-attrs #{}}
   "meter"   {:void? false :boolean-attrs #{}}
   "fieldset"{:void? false :boolean-attrs #{"disabled"}}      ;;
   "legend"  {:void? false :boolean-attrs #{}}

   ;; Interactive elements
   "details" {:void? false :boolean-attrs #{"open"}}          ;; open
   "summary" {:void? false :boolean-attrs #{}}
   "dialog"  {:void? false :boolean-attrs #{"open"}}          ;; open
   "menu"    {:void? false :boolean-attrs #{}}

   ;; Global boolean attributes（要素個別ではなくグローバルだが、ここでは代表的なものを）
   ;; hidden, draggable, contenteditable などは値付きも許されるためここでは扱わない
   })

;; ------------------------------------------------------------
;; 属性レンダリング
;; ------------------------------------------------------------

(defn render-attrs [tag-name attrs]
  (let [{:keys [boolean-attrs]} (get html-tags tag-name)]
    (->> attrs
         (map (fn [[k v]]
                (let [kname (name k)]
                  (cond
                    ;; boolean 属性：値が truthy かつ空文字でない場合に属性名のみ出力
                    (and boolean-attrs
                         (boolean-attrs kname)
                         (not (str/blank? (str v))))
                    (str " " kname)

                    ;; 通常属性
                    (not (nil? v))
                    (str " " kname "=\"" v "\"")

                    :else ""))))
         (apply str))))

;; ------------------------------------------------------------
;; 再帰的レンダリング
;; ------------------------------------------------------------

(declare render-node)

(defn render-children [children]
  (apply str (map render-node children)))

(defn render-tag [[_ tag-name attrs children]]
  (let [{:keys [void?]} (get html-tags tag-name {:void? false})]
    (if void?
      ;; void タグ
      (str "<" tag-name (render-attrs tag-name attrs) ">")
      ;; 通常タグ
      (str "<" tag-name (render-attrs tag-name attrs) ">"
           (render-children children)
           "</" tag-name ">"))))

(defn render-node [node]
  (cond
    ;; [:tag ...]
    (and (vector? node)
         (= :tag (first node)))
    (render-tag node)

    ;; 文字列ノード
    (string? node)
    node

    ;; [:raw "..."]
    (and (vector? node)
         (= :raw (first node)))
    (second node)

    :else
    (str node)))

;; ------------------------------------------------------------
;; エントリ関数
;; ------------------------------------------------------------

(defn render-document [ast]
  ;; ast は [ [:tag ...] [:tag ...] ... ] の形式
  (apply str (map render-node ast)))

