(ns happygapi2.jobs
  "Cloud Talent Solution API
Cloud Talent Solution provides the capability to create, read, update, and delete job postings, as well as search jobs based on keywords and filters. 
See: https://cloud.google.com/talent-solution/job-search/docs/"
  (:require [happy.oauth2.client :as client]))

(defn projects-operations-get
  "Gets the latest state of a long-running operation. Clients can use this method to poll the operation result at intervals as recommended by the API service.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.operations/get

name <string> The name of the operation resource."
  [name]
  (client/api-request
    {:method "GET",
     :uri-template "https://jobs.googleapis.com/v4/{+name}",
     :uri-template-args {:name name},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"]}))

(defn projects-tenants-completeQuery
  "Completes the specified prefix with keyword suggestions. Intended for use by a job search auto-complete search box.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants/completeQuery

tenant <string> Required. Resource name of tenant the completion is performed within. The format is \"projects/{project_id}/tenants/{tenant_id}\", for example, \"projects/foo/tenants/bar\".

optional:
query <string> Required. The query used to generate suggestions. The maximum number of allowed characters is 255.
languageCodes <string> The list of languages of the query. This is the BCP-47 language code, such as \"en-US\" or \"sr-Latn\". For more information, see [Tags for Identifying Languages](https://tools.ietf.org/html/bcp47). The maximum number of allowed characters is 255.
pageSize <integer> Required. Completion result count. The maximum allowed page size is 10.
company <string> If provided, restricts completion to specified company. The format is \"projects/{project_id}/tenants/{tenant_id}/companies/{company_id}\", for example, \"projects/foo/tenants/bar/companies/baz\".
scope <string> The scope of the completion. The defaults is CompletionScope.PUBLIC.
type <string> The completion topic. The default is CompletionType.COMBINED."
  ([tenant] (projects-tenants-completeQuery tenant nil))
  ([tenant optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://jobs.googleapis.com/v4/{+tenant}:completeQuery",
       :uri-template-args {:tenant tenant},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/jobs"]})))

(defn projects-tenants-create
  "Creates a new tenant entity.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants/create

parent <string> Required. Resource name of the project under which the tenant is created. The format is \"projects/{project_id}\", for example, \"projects/foo\".
Tenant:
{:name string, :externalId string}"
  [parent Tenant]
  (client/api-request
    {:method "POST",
     :uri-template "https://jobs.googleapis.com/v4/{+parent}/tenants",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body Tenant}))

(defn projects-tenants-get
  "Retrieves specified tenant.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants/get

name <string> Required. The resource name of the tenant to be retrieved. The format is \"projects/{project_id}/tenants/{tenant_id}\", for example, \"projects/foo/tenants/bar\"."
  [name]
  (client/api-request
    {:method "GET",
     :uri-template "https://jobs.googleapis.com/v4/{+name}",
     :uri-template-args {:name name},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"]}))

(defn projects-tenants-patch
  "Updates specified tenant.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants/patch

name <string> Required during tenant update. The resource name for a tenant. This is generated by the service when a tenant is created. The format is \"projects/{project_id}/tenants/{tenant_id}\", for example, \"projects/foo/tenants/bar\".
Tenant:
{:name string, :externalId string}

optional:
updateMask <string> Strongly recommended for the best service experience. If update_mask is provided, only the specified fields in tenant are updated. Otherwise all the fields are updated. A field mask to specify the tenant fields to be updated. Only top level fields of Tenant are supported."
  ([name Tenant] (projects-tenants-patch name Tenant nil))
  ([name Tenant optional]
    (client/api-request
      {:method "PATCH",
       :uri-template "https://jobs.googleapis.com/v4/{+name}",
       :uri-template-args {:name name},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/jobs"],
       :body Tenant})))

(defn projects-tenants-delete
  "Deletes specified tenant.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants/delete

name <string> Required. The resource name of the tenant to be deleted. The format is \"projects/{project_id}/tenants/{tenant_id}\", for example, \"projects/foo/tenants/bar\"."
  [name]
  (client/api-request
    {:method "DELETE",
     :uri-template "https://jobs.googleapis.com/v4/{+name}",
     :uri-template-args {:name name},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"]}))

(defn projects-tenants-list
  "Lists all tenants associated with the project.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants/list

parent <string> Required. Resource name of the project under which the tenant is created. The format is \"projects/{project_id}\", for example, \"projects/foo\".

optional:
pageSize <integer> The maximum number of tenants to be returned, at most 100. Default is 100 if a non-positive number is provided."
  ([parent] (projects-tenants-list parent nil))
  ([parent optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://jobs.googleapis.com/v4/{+parent}/tenants",
       :uri-template-args {:parent parent},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/jobs"]})))

(defn projects-tenants-companies-create
  "Creates a new company entity.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.companies/create

parent <string> Required. Resource name of the tenant under which the company is created. The format is \"projects/{project_id}/tenants/{tenant_id}\", for example, \"projects/foo/tenants/bar\".
Company:
{:imageUri string,
 :keywordSearchableJobCustomAttributes [string],
 :careerSiteUri string,
 :headquartersAddress string,
 :displayName string,
 :name string,
 :eeoText string,
 :derivedInfo
 {:headquartersLocation
  {:locationType
   [LOCATION_TYPE_UNSPECIFIED
    COUNTRY
    ADMINISTRATIVE_AREA
    SUB_ADMINISTRATIVE_AREA
    LOCALITY
    POSTAL_CODE
    SUB_LOCALITY
    SUB_LOCALITY_1
    SUB_LOCALITY_2
    NEIGHBORHOOD
    STREET_ADDRESS],
   :postalAddress
   {:sortingCode string,
    :locality string,
    :revision integer,
    :administrativeArea string,
    :addressLines [string],
    :organization string,
    :recipients [string],
    :languageCode string,
    :regionCode string,
    :postalCode string,
    :sublocality string},
   :latLng {:latitude number, :longitude number},
   :radiusMiles number}},
 :suspended boolean,
 :size
 [COMPANY_SIZE_UNSPECIFIED MINI SMALL SMEDIUM MEDIUM BIG BIGGER GIANT],
 :externalId string,
 :hiringAgency boolean,
 :websiteUri string}"
  [parent Company]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://jobs.googleapis.com/v4/{+parent}/companies",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body Company}))

(defn projects-tenants-companies-get
  "Retrieves specified company.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.companies/get

name <string> Required. The resource name of the company to be retrieved. The format is \"projects/{project_id}/tenants/{tenant_id}/companies/{company_id}\", for example, \"projects/api-test-project/tenants/foo/companies/bar\"."
  [name]
  (client/api-request
    {:method "GET",
     :uri-template "https://jobs.googleapis.com/v4/{+name}",
     :uri-template-args {:name name},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"]}))

(defn projects-tenants-companies-patch
  "Updates specified company.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.companies/patch

name <string> Required during company update. The resource name for a company. This is generated by the service when a company is created. The format is \"projects/{project_id}/tenants/{tenant_id}/companies/{company_id}\", for example, \"projects/foo/tenants/bar/companies/baz\".
Company:
{:imageUri string,
 :keywordSearchableJobCustomAttributes [string],
 :careerSiteUri string,
 :headquartersAddress string,
 :displayName string,
 :name string,
 :eeoText string,
 :derivedInfo
 {:headquartersLocation
  {:locationType
   [LOCATION_TYPE_UNSPECIFIED
    COUNTRY
    ADMINISTRATIVE_AREA
    SUB_ADMINISTRATIVE_AREA
    LOCALITY
    POSTAL_CODE
    SUB_LOCALITY
    SUB_LOCALITY_1
    SUB_LOCALITY_2
    NEIGHBORHOOD
    STREET_ADDRESS],
   :postalAddress
   {:sortingCode string,
    :locality string,
    :revision integer,
    :administrativeArea string,
    :addressLines [string],
    :organization string,
    :recipients [string],
    :languageCode string,
    :regionCode string,
    :postalCode string,
    :sublocality string},
   :latLng {:latitude number, :longitude number},
   :radiusMiles number}},
 :suspended boolean,
 :size
 [COMPANY_SIZE_UNSPECIFIED MINI SMALL SMEDIUM MEDIUM BIG BIGGER GIANT],
 :externalId string,
 :hiringAgency boolean,
 :websiteUri string}

optional:
updateMask <string> Strongly recommended for the best service experience. If update_mask is provided, only the specified fields in company are updated. Otherwise all the fields are updated. A field mask to specify the company fields to be updated. Only top level fields of Company are supported."
  ([name Company] (projects-tenants-companies-patch name Company nil))
  ([name Company optional]
    (client/api-request
      {:method "PATCH",
       :uri-template "https://jobs.googleapis.com/v4/{+name}",
       :uri-template-args {:name name},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/jobs"],
       :body Company})))

(defn projects-tenants-companies-delete
  "Deletes specified company. Prerequisite: The company has no jobs associated with it.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.companies/delete

name <string> Required. The resource name of the company to be deleted. The format is \"projects/{project_id}/tenants/{tenant_id}/companies/{company_id}\", for example, \"projects/foo/tenants/bar/companies/baz\"."
  [name]
  (client/api-request
    {:method "DELETE",
     :uri-template "https://jobs.googleapis.com/v4/{+name}",
     :uri-template-args {:name name},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"]}))

(defn projects-tenants-companies-list
  "Lists all companies associated with the project.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.companies/list

parent <string> Required. Resource name of the tenant under which the company is created. The format is \"projects/{project_id}/tenants/{tenant_id}\", for example, \"projects/foo/tenants/bar\".

optional:
pageSize <integer> The maximum number of companies to be returned, at most 100. Default is 100 if a non-positive number is provided.
requireOpenJobs <boolean> Set to true if the companies requested must have open jobs. Defaults to false. If true, at most page_size of companies are fetched, among which only those with open jobs are returned."
  ([parent] (projects-tenants-companies-list parent nil))
  ([parent optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://jobs.googleapis.com/v4/{+parent}/companies",
       :uri-template-args {:parent parent},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/jobs"]})))

(defn projects-tenants-clientEvents-create
  "Report events issued when end user interacts with customer's application that uses Cloud Talent Solution. You may inspect the created events in [self service tools](https://console.cloud.google.com/talent-solution/overview). [Learn more](https://cloud.google.com/talent-solution/docs/management-tools) about self service tools.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.clientEvents/create

parent <string> Required. Resource name of the tenant under which the event is created. The format is \"projects/{project_id}/tenants/{tenant_id}\", for example, \"projects/foo/tenants/bar\".
ClientEvent:
{:requestId string,
 :eventId string,
 :createTime string,
 :jobEvent
 {:type
  [JOB_EVENT_TYPE_UNSPECIFIED
   IMPRESSION
   VIEW
   VIEW_REDIRECT
   APPLICATION_START
   APPLICATION_FINISH
   APPLICATION_QUICK_SUBMISSION
   APPLICATION_REDIRECT
   APPLICATION_START_FROM_SEARCH
   APPLICATION_REDIRECT_FROM_SEARCH
   APPLICATION_COMPANY_SUBMIT
   BOOKMARK
   NOTIFICATION
   HIRED
   SENT_CV
   INTERVIEW_GRANTED],
  :jobs [string]},
 :eventNotes string}"
  [parent ClientEvent]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://jobs.googleapis.com/v4/{+parent}/clientEvents",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body ClientEvent}))

(defn projects-tenants-jobs-get
  "Retrieves the specified job, whose status is OPEN or recently EXPIRED within the last 90 days.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/get

name <string> Required. The resource name of the job to retrieve. The format is \"projects/{project_id}/tenants/{tenant_id}/jobs/{job_id}\". For example, \"projects/foo/tenants/bar/jobs/baz\"."
  [name]
  (client/api-request
    {:method "GET",
     :uri-template "https://jobs.googleapis.com/v4/{+name}",
     :uri-template-args {:name name},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"]}))

(defn projects-tenants-jobs-patch
  "Updates specified job. Typically, updated contents become visible in search results within 10 seconds, but it may take up to 5 minutes.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/patch

name <string> Required during job update. The resource name for the job. This is generated by the service when a job is created. The format is \"projects/{project_id}/tenants/{tenant_id}/jobs/{job_id}\". For example, \"projects/foo/tenants/bar/jobs/baz\". Use of this field in job queries and API calls is preferred over the use of requisition_id since this value is unique.
Job:
{:processingOptions
 {:disableStreetAddressResolution boolean,
  :htmlSanitization
  [HTML_SANITIZATION_UNSPECIFIED
   HTML_SANITIZATION_DISABLED
   SIMPLE_FORMATTING_ONLY]},
 :description string,
 :department string,
 :addresses [string],
 :postingPublishTime string,
 :jobBenefits
 [[JOB_BENEFIT_UNSPECIFIED
   CHILD_CARE
   DENTAL
   DOMESTIC_PARTNER
   FLEXIBLE_HOURS
   MEDICAL
   LIFE_INSURANCE
   PARENTAL_LEAVE
   RETIREMENT_PLAN
   SICK_DAYS
   VACATION
   VISION]],
 :requisitionId string,
 :degreeTypes
 [[DEGREE_TYPE_UNSPECIFIED
   PRIMARY_EDUCATION
   LOWER_SECONDARY_EDUCATION
   UPPER_SECONDARY_EDUCATION
   ADULT_REMEDIAL_EDUCATION
   ASSOCIATES_OR_EQUIVALENT
   BACHELORS_OR_EQUIVALENT
   MASTERS_OR_EQUIVALENT
   DOCTORAL_OR_EQUIVALENT]],
 :postingCreateTime string,
 :name string,
 :customAttributes object,
 :jobEndTime string,
 :derivedInfo
 {:locations
  [{:locationType
    [LOCATION_TYPE_UNSPECIFIED
     COUNTRY
     ADMINISTRATIVE_AREA
     SUB_ADMINISTRATIVE_AREA
     LOCALITY
     POSTAL_CODE
     SUB_LOCALITY
     SUB_LOCALITY_1
     SUB_LOCALITY_2
     NEIGHBORHOOD
     STREET_ADDRESS],
    :postalAddress
    {:sortingCode string,
     :locality string,
     :revision integer,
     :administrativeArea string,
     :addressLines [string],
     :organization string,
     :recipients [string],
     :languageCode string,
     :regionCode string,
     :postalCode string,
     :sublocality string},
    :latLng {:latitude number, :longitude number},
    :radiusMiles number}],
  :jobCategories
  [[JOB_CATEGORY_UNSPECIFIED
    ACCOUNTING_AND_FINANCE
    ADMINISTRATIVE_AND_OFFICE
    ADVERTISING_AND_MARKETING
    ANIMAL_CARE
    ART_FASHION_AND_DESIGN
    BUSINESS_OPERATIONS
    CLEANING_AND_FACILITIES
    COMPUTER_AND_IT
    CONSTRUCTION
    CUSTOMER_SERVICE
    EDUCATION
    ENTERTAINMENT_AND_TRAVEL
    FARMING_AND_OUTDOORS
    HEALTHCARE
    HUMAN_RESOURCES
    INSTALLATION_MAINTENANCE_AND_REPAIR
    LEGAL
    MANAGEMENT
    MANUFACTURING_AND_WAREHOUSE
    MEDIA_COMMUNICATIONS_AND_WRITING
    OIL_GAS_AND_MINING
    PERSONAL_CARE_AND_SERVICES
    PROTECTIVE_SERVICES
    REAL_ESTATE
    RESTAURANT_AND_HOSPITALITY
    SALES_AND_RETAIL
    SCIENCE_AND_ENGINEERING
    SOCIAL_SERVICES_AND_NON_PROFIT
    SPORTS_FITNESS_AND_RECREATION
    TRANSPORTATION_AND_LOGISTICS]]},
 :responsibilities string,
 :companyDisplayName string,
 :title string,
 :qualifications string,
 :postingExpireTime string,
 :promotionValue integer,
 :languageCode string,
 :postingUpdateTime string,
 :postingRegion
 [POSTING_REGION_UNSPECIFIED ADMINISTRATIVE_AREA NATION TELECOMMUTE],
 :applicationInfo
 {:emails [string], :instruction string, :uris [string]},
 :jobStartTime string,
 :compensationInfo
 {:entries
  [{:type
    [COMPENSATION_TYPE_UNSPECIFIED
     BASE
     BONUS
     SIGNING_BONUS
     EQUITY
     PROFIT_SHARING
     COMMISSIONS
     TIPS
     OTHER_COMPENSATION_TYPE],
    :unit
    [COMPENSATION_UNIT_UNSPECIFIED
     HOURLY
     DAILY
     WEEKLY
     MONTHLY
     YEARLY
     ONE_TIME
     OTHER_COMPENSATION_UNIT],
    :amount {:currencyCode string, :units string, :nanos integer},
    :range
    {:maxCompensation
     {:currencyCode string, :units string, :nanos integer},
     :minCompensation
     {:currencyCode string, :units string, :nanos integer}},
    :description string,
    :expectedUnitsPerYear number}],
  :annualizedBaseCompensationRange
  {:maxCompensation
   {:currencyCode string, :units string, :nanos integer},
   :minCompensation
   {:currencyCode string, :units string, :nanos integer}},
  :annualizedTotalCompensationRange
  {:maxCompensation
   {:currencyCode string, :units string, :nanos integer},
   :minCompensation
   {:currencyCode string, :units string, :nanos integer}}},
 :employmentTypes
 [[EMPLOYMENT_TYPE_UNSPECIFIED
   FULL_TIME
   PART_TIME
   CONTRACTOR
   CONTRACT_TO_HIRE
   TEMPORARY
   INTERN
   VOLUNTEER
   PER_DIEM
   FLY_IN_FLY_OUT
   OTHER_EMPLOYMENT_TYPE]],
 :visibility
 [VISIBILITY_UNSPECIFIED
  ACCOUNT_ONLY
  SHARED_WITH_GOOGLE
  SHARED_WITH_PUBLIC],
 :company string,
 :incentives string,
 :jobLevel
 [JOB_LEVEL_UNSPECIFIED
  ENTRY_LEVEL
  EXPERIENCED
  MANAGER
  DIRECTOR
  EXECUTIVE]}

optional:
updateMask <string> Strongly recommended for the best service experience. If update_mask is provided, only the specified fields in job are updated. Otherwise all the fields are updated. A field mask to restrict the fields that are updated. Only top level fields of Job are supported."
  ([name Job] (projects-tenants-jobs-patch name Job nil))
  ([name Job optional]
    (client/api-request
      {:method "PATCH",
       :uri-template "https://jobs.googleapis.com/v4/{+name}",
       :uri-template-args {:name name},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/jobs"],
       :body Job})))

(defn projects-tenants-jobs-batchUpdate
  "Begins executing a batch update jobs operation.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/batchUpdate

parent <string> Required. The resource name of the tenant under which the job is created. The format is \"projects/{project_id}/tenants/{tenant_id}\". For example, \"projects/foo/tenants/bar\".
BatchUpdateJobsRequest:
{:jobs
 [{:processingOptions
   {:disableStreetAddressResolution boolean,
    :htmlSanitization
    [HTML_SANITIZATION_UNSPECIFIED
     HTML_SANITIZATION_DISABLED
     SIMPLE_FORMATTING_ONLY]},
   :description string,
   :department string,
   :addresses [string],
   :postingPublishTime string,
   :jobBenefits
   [[JOB_BENEFIT_UNSPECIFIED
     CHILD_CARE
     DENTAL
     DOMESTIC_PARTNER
     FLEXIBLE_HOURS
     MEDICAL
     LIFE_INSURANCE
     PARENTAL_LEAVE
     RETIREMENT_PLAN
     SICK_DAYS
     VACATION
     VISION]],
   :requisitionId string,
   :degreeTypes
   [[DEGREE_TYPE_UNSPECIFIED
     PRIMARY_EDUCATION
     LOWER_SECONDARY_EDUCATION
     UPPER_SECONDARY_EDUCATION
     ADULT_REMEDIAL_EDUCATION
     ASSOCIATES_OR_EQUIVALENT
     BACHELORS_OR_EQUIVALENT
     MASTERS_OR_EQUIVALENT
     DOCTORAL_OR_EQUIVALENT]],
   :postingCreateTime string,
   :name string,
   :customAttributes object,
   :jobEndTime string,
   :derivedInfo
   {:locations
    [{:locationType
      [LOCATION_TYPE_UNSPECIFIED
       COUNTRY
       ADMINISTRATIVE_AREA
       SUB_ADMINISTRATIVE_AREA
       LOCALITY
       POSTAL_CODE
       SUB_LOCALITY
       SUB_LOCALITY_1
       SUB_LOCALITY_2
       NEIGHBORHOOD
       STREET_ADDRESS],
      :postalAddress
      {:sortingCode string,
       :locality string,
       :revision integer,
       :administrativeArea string,
       :addressLines [string],
       :organization string,
       :recipients [string],
       :languageCode string,
       :regionCode string,
       :postalCode string,
       :sublocality string},
      :latLng {:latitude number, :longitude number},
      :radiusMiles number}],
    :jobCategories
    [[JOB_CATEGORY_UNSPECIFIED
      ACCOUNTING_AND_FINANCE
      ADMINISTRATIVE_AND_OFFICE
      ADVERTISING_AND_MARKETING
      ANIMAL_CARE
      ART_FASHION_AND_DESIGN
      BUSINESS_OPERATIONS
      CLEANING_AND_FACILITIES
      COMPUTER_AND_IT
      CONSTRUCTION
      CUSTOMER_SERVICE
      EDUCATION
      ENTERTAINMENT_AND_TRAVEL
      FARMING_AND_OUTDOORS
      HEALTHCARE
      HUMAN_RESOURCES
      INSTALLATION_MAINTENANCE_AND_REPAIR
      LEGAL
      MANAGEMENT
      MANUFACTURING_AND_WAREHOUSE
      MEDIA_COMMUNICATIONS_AND_WRITING
      OIL_GAS_AND_MINING
      PERSONAL_CARE_AND_SERVICES
      PROTECTIVE_SERVICES
      REAL_ESTATE
      RESTAURANT_AND_HOSPITALITY
      SALES_AND_RETAIL
      SCIENCE_AND_ENGINEERING
      SOCIAL_SERVICES_AND_NON_PROFIT
      SPORTS_FITNESS_AND_RECREATION
      TRANSPORTATION_AND_LOGISTICS]]},
   :responsibilities string,
   :companyDisplayName string,
   :title string,
   :qualifications string,
   :postingExpireTime string,
   :promotionValue integer,
   :languageCode string,
   :postingUpdateTime string,
   :postingRegion
   [POSTING_REGION_UNSPECIFIED ADMINISTRATIVE_AREA NATION TELECOMMUTE],
   :applicationInfo
   {:emails [string], :instruction string, :uris [string]},
   :jobStartTime string,
   :compensationInfo
   {:entries
    [{:type
      [COMPENSATION_TYPE_UNSPECIFIED
       BASE
       BONUS
       SIGNING_BONUS
       EQUITY
       PROFIT_SHARING
       COMMISSIONS
       TIPS
       OTHER_COMPENSATION_TYPE],
      :unit
      [COMPENSATION_UNIT_UNSPECIFIED
       HOURLY
       DAILY
       WEEKLY
       MONTHLY
       YEARLY
       ONE_TIME
       OTHER_COMPENSATION_UNIT],
      :amount {:currencyCode string, :units string, :nanos integer},
      :range {:maxCompensation Money, :minCompensation Money},
      :description string,
      :expectedUnitsPerYear number}],
    :annualizedBaseCompensationRange
    {:maxCompensation
     {:currencyCode string, :units string, :nanos integer},
     :minCompensation
     {:currencyCode string, :units string, :nanos integer}},
    :annualizedTotalCompensationRange
    {:maxCompensation
     {:currencyCode string, :units string, :nanos integer},
     :minCompensation
     {:currencyCode string, :units string, :nanos integer}}},
   :employmentTypes
   [[EMPLOYMENT_TYPE_UNSPECIFIED
     FULL_TIME
     PART_TIME
     CONTRACTOR
     CONTRACT_TO_HIRE
     TEMPORARY
     INTERN
     VOLUNTEER
     PER_DIEM
     FLY_IN_FLY_OUT
     OTHER_EMPLOYMENT_TYPE]],
   :visibility
   [VISIBILITY_UNSPECIFIED
    ACCOUNT_ONLY
    SHARED_WITH_GOOGLE
    SHARED_WITH_PUBLIC],
   :company string,
   :incentives string,
   :jobLevel
   [JOB_LEVEL_UNSPECIFIED
    ENTRY_LEVEL
    EXPERIENCED
    MANAGER
    DIRECTOR
    EXECUTIVE]}],
 :updateMask string}"
  [parent BatchUpdateJobsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://jobs.googleapis.com/v4/{+parent}/jobs:batchUpdate",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body BatchUpdateJobsRequest}))

(defn projects-tenants-jobs-create
  "Creates a new job. Typically, the job becomes searchable within 10 seconds, but it may take up to 5 minutes.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/create

parent <string> Required. The resource name of the tenant under which the job is created. The format is \"projects/{project_id}/tenants/{tenant_id}\". For example, \"projects/foo/tenants/bar\".
Job:
{:processingOptions
 {:disableStreetAddressResolution boolean,
  :htmlSanitization
  [HTML_SANITIZATION_UNSPECIFIED
   HTML_SANITIZATION_DISABLED
   SIMPLE_FORMATTING_ONLY]},
 :description string,
 :department string,
 :addresses [string],
 :postingPublishTime string,
 :jobBenefits
 [[JOB_BENEFIT_UNSPECIFIED
   CHILD_CARE
   DENTAL
   DOMESTIC_PARTNER
   FLEXIBLE_HOURS
   MEDICAL
   LIFE_INSURANCE
   PARENTAL_LEAVE
   RETIREMENT_PLAN
   SICK_DAYS
   VACATION
   VISION]],
 :requisitionId string,
 :degreeTypes
 [[DEGREE_TYPE_UNSPECIFIED
   PRIMARY_EDUCATION
   LOWER_SECONDARY_EDUCATION
   UPPER_SECONDARY_EDUCATION
   ADULT_REMEDIAL_EDUCATION
   ASSOCIATES_OR_EQUIVALENT
   BACHELORS_OR_EQUIVALENT
   MASTERS_OR_EQUIVALENT
   DOCTORAL_OR_EQUIVALENT]],
 :postingCreateTime string,
 :name string,
 :customAttributes object,
 :jobEndTime string,
 :derivedInfo
 {:locations
  [{:locationType
    [LOCATION_TYPE_UNSPECIFIED
     COUNTRY
     ADMINISTRATIVE_AREA
     SUB_ADMINISTRATIVE_AREA
     LOCALITY
     POSTAL_CODE
     SUB_LOCALITY
     SUB_LOCALITY_1
     SUB_LOCALITY_2
     NEIGHBORHOOD
     STREET_ADDRESS],
    :postalAddress
    {:sortingCode string,
     :locality string,
     :revision integer,
     :administrativeArea string,
     :addressLines [string],
     :organization string,
     :recipients [string],
     :languageCode string,
     :regionCode string,
     :postalCode string,
     :sublocality string},
    :latLng {:latitude number, :longitude number},
    :radiusMiles number}],
  :jobCategories
  [[JOB_CATEGORY_UNSPECIFIED
    ACCOUNTING_AND_FINANCE
    ADMINISTRATIVE_AND_OFFICE
    ADVERTISING_AND_MARKETING
    ANIMAL_CARE
    ART_FASHION_AND_DESIGN
    BUSINESS_OPERATIONS
    CLEANING_AND_FACILITIES
    COMPUTER_AND_IT
    CONSTRUCTION
    CUSTOMER_SERVICE
    EDUCATION
    ENTERTAINMENT_AND_TRAVEL
    FARMING_AND_OUTDOORS
    HEALTHCARE
    HUMAN_RESOURCES
    INSTALLATION_MAINTENANCE_AND_REPAIR
    LEGAL
    MANAGEMENT
    MANUFACTURING_AND_WAREHOUSE
    MEDIA_COMMUNICATIONS_AND_WRITING
    OIL_GAS_AND_MINING
    PERSONAL_CARE_AND_SERVICES
    PROTECTIVE_SERVICES
    REAL_ESTATE
    RESTAURANT_AND_HOSPITALITY
    SALES_AND_RETAIL
    SCIENCE_AND_ENGINEERING
    SOCIAL_SERVICES_AND_NON_PROFIT
    SPORTS_FITNESS_AND_RECREATION
    TRANSPORTATION_AND_LOGISTICS]]},
 :responsibilities string,
 :companyDisplayName string,
 :title string,
 :qualifications string,
 :postingExpireTime string,
 :promotionValue integer,
 :languageCode string,
 :postingUpdateTime string,
 :postingRegion
 [POSTING_REGION_UNSPECIFIED ADMINISTRATIVE_AREA NATION TELECOMMUTE],
 :applicationInfo
 {:emails [string], :instruction string, :uris [string]},
 :jobStartTime string,
 :compensationInfo
 {:entries
  [{:type
    [COMPENSATION_TYPE_UNSPECIFIED
     BASE
     BONUS
     SIGNING_BONUS
     EQUITY
     PROFIT_SHARING
     COMMISSIONS
     TIPS
     OTHER_COMPENSATION_TYPE],
    :unit
    [COMPENSATION_UNIT_UNSPECIFIED
     HOURLY
     DAILY
     WEEKLY
     MONTHLY
     YEARLY
     ONE_TIME
     OTHER_COMPENSATION_UNIT],
    :amount {:currencyCode string, :units string, :nanos integer},
    :range
    {:maxCompensation
     {:currencyCode string, :units string, :nanos integer},
     :minCompensation
     {:currencyCode string, :units string, :nanos integer}},
    :description string,
    :expectedUnitsPerYear number}],
  :annualizedBaseCompensationRange
  {:maxCompensation
   {:currencyCode string, :units string, :nanos integer},
   :minCompensation
   {:currencyCode string, :units string, :nanos integer}},
  :annualizedTotalCompensationRange
  {:maxCompensation
   {:currencyCode string, :units string, :nanos integer},
   :minCompensation
   {:currencyCode string, :units string, :nanos integer}}},
 :employmentTypes
 [[EMPLOYMENT_TYPE_UNSPECIFIED
   FULL_TIME
   PART_TIME
   CONTRACTOR
   CONTRACT_TO_HIRE
   TEMPORARY
   INTERN
   VOLUNTEER
   PER_DIEM
   FLY_IN_FLY_OUT
   OTHER_EMPLOYMENT_TYPE]],
 :visibility
 [VISIBILITY_UNSPECIFIED
  ACCOUNT_ONLY
  SHARED_WITH_GOOGLE
  SHARED_WITH_PUBLIC],
 :company string,
 :incentives string,
 :jobLevel
 [JOB_LEVEL_UNSPECIFIED
  ENTRY_LEVEL
  EXPERIENCED
  MANAGER
  DIRECTOR
  EXECUTIVE]}"
  [parent Job]
  (client/api-request
    {:method "POST",
     :uri-template "https://jobs.googleapis.com/v4/{+parent}/jobs",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body Job}))

(defn projects-tenants-jobs-batchCreate
  "Begins executing a batch create jobs operation.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/batchCreate

parent <string> Required. The resource name of the tenant under which the job is created. The format is \"projects/{project_id}/tenants/{tenant_id}\". For example, \"projects/foo/tenants/bar\".
BatchCreateJobsRequest:
{:jobs
 [{:processingOptions
   {:disableStreetAddressResolution boolean,
    :htmlSanitization
    [HTML_SANITIZATION_UNSPECIFIED
     HTML_SANITIZATION_DISABLED
     SIMPLE_FORMATTING_ONLY]},
   :description string,
   :department string,
   :addresses [string],
   :postingPublishTime string,
   :jobBenefits
   [[JOB_BENEFIT_UNSPECIFIED
     CHILD_CARE
     DENTAL
     DOMESTIC_PARTNER
     FLEXIBLE_HOURS
     MEDICAL
     LIFE_INSURANCE
     PARENTAL_LEAVE
     RETIREMENT_PLAN
     SICK_DAYS
     VACATION
     VISION]],
   :requisitionId string,
   :degreeTypes
   [[DEGREE_TYPE_UNSPECIFIED
     PRIMARY_EDUCATION
     LOWER_SECONDARY_EDUCATION
     UPPER_SECONDARY_EDUCATION
     ADULT_REMEDIAL_EDUCATION
     ASSOCIATES_OR_EQUIVALENT
     BACHELORS_OR_EQUIVALENT
     MASTERS_OR_EQUIVALENT
     DOCTORAL_OR_EQUIVALENT]],
   :postingCreateTime string,
   :name string,
   :customAttributes object,
   :jobEndTime string,
   :derivedInfo
   {:locations
    [{:locationType
      [LOCATION_TYPE_UNSPECIFIED
       COUNTRY
       ADMINISTRATIVE_AREA
       SUB_ADMINISTRATIVE_AREA
       LOCALITY
       POSTAL_CODE
       SUB_LOCALITY
       SUB_LOCALITY_1
       SUB_LOCALITY_2
       NEIGHBORHOOD
       STREET_ADDRESS],
      :postalAddress
      {:sortingCode string,
       :locality string,
       :revision integer,
       :administrativeArea string,
       :addressLines [string],
       :organization string,
       :recipients [string],
       :languageCode string,
       :regionCode string,
       :postalCode string,
       :sublocality string},
      :latLng {:latitude number, :longitude number},
      :radiusMiles number}],
    :jobCategories
    [[JOB_CATEGORY_UNSPECIFIED
      ACCOUNTING_AND_FINANCE
      ADMINISTRATIVE_AND_OFFICE
      ADVERTISING_AND_MARKETING
      ANIMAL_CARE
      ART_FASHION_AND_DESIGN
      BUSINESS_OPERATIONS
      CLEANING_AND_FACILITIES
      COMPUTER_AND_IT
      CONSTRUCTION
      CUSTOMER_SERVICE
      EDUCATION
      ENTERTAINMENT_AND_TRAVEL
      FARMING_AND_OUTDOORS
      HEALTHCARE
      HUMAN_RESOURCES
      INSTALLATION_MAINTENANCE_AND_REPAIR
      LEGAL
      MANAGEMENT
      MANUFACTURING_AND_WAREHOUSE
      MEDIA_COMMUNICATIONS_AND_WRITING
      OIL_GAS_AND_MINING
      PERSONAL_CARE_AND_SERVICES
      PROTECTIVE_SERVICES
      REAL_ESTATE
      RESTAURANT_AND_HOSPITALITY
      SALES_AND_RETAIL
      SCIENCE_AND_ENGINEERING
      SOCIAL_SERVICES_AND_NON_PROFIT
      SPORTS_FITNESS_AND_RECREATION
      TRANSPORTATION_AND_LOGISTICS]]},
   :responsibilities string,
   :companyDisplayName string,
   :title string,
   :qualifications string,
   :postingExpireTime string,
   :promotionValue integer,
   :languageCode string,
   :postingUpdateTime string,
   :postingRegion
   [POSTING_REGION_UNSPECIFIED ADMINISTRATIVE_AREA NATION TELECOMMUTE],
   :applicationInfo
   {:emails [string], :instruction string, :uris [string]},
   :jobStartTime string,
   :compensationInfo
   {:entries
    [{:type
      [COMPENSATION_TYPE_UNSPECIFIED
       BASE
       BONUS
       SIGNING_BONUS
       EQUITY
       PROFIT_SHARING
       COMMISSIONS
       TIPS
       OTHER_COMPENSATION_TYPE],
      :unit
      [COMPENSATION_UNIT_UNSPECIFIED
       HOURLY
       DAILY
       WEEKLY
       MONTHLY
       YEARLY
       ONE_TIME
       OTHER_COMPENSATION_UNIT],
      :amount {:currencyCode string, :units string, :nanos integer},
      :range {:maxCompensation Money, :minCompensation Money},
      :description string,
      :expectedUnitsPerYear number}],
    :annualizedBaseCompensationRange
    {:maxCompensation
     {:currencyCode string, :units string, :nanos integer},
     :minCompensation
     {:currencyCode string, :units string, :nanos integer}},
    :annualizedTotalCompensationRange
    {:maxCompensation
     {:currencyCode string, :units string, :nanos integer},
     :minCompensation
     {:currencyCode string, :units string, :nanos integer}}},
   :employmentTypes
   [[EMPLOYMENT_TYPE_UNSPECIFIED
     FULL_TIME
     PART_TIME
     CONTRACTOR
     CONTRACT_TO_HIRE
     TEMPORARY
     INTERN
     VOLUNTEER
     PER_DIEM
     FLY_IN_FLY_OUT
     OTHER_EMPLOYMENT_TYPE]],
   :visibility
   [VISIBILITY_UNSPECIFIED
    ACCOUNT_ONLY
    SHARED_WITH_GOOGLE
    SHARED_WITH_PUBLIC],
   :company string,
   :incentives string,
   :jobLevel
   [JOB_LEVEL_UNSPECIFIED
    ENTRY_LEVEL
    EXPERIENCED
    MANAGER
    DIRECTOR
    EXECUTIVE]}]}"
  [parent BatchCreateJobsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://jobs.googleapis.com/v4/{+parent}/jobs:batchCreate",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body BatchCreateJobsRequest}))

(defn projects-tenants-jobs-delete
  "Deletes the specified job. Typically, the job becomes unsearchable within 10 seconds, but it may take up to 5 minutes.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/delete

name <string> Required. The resource name of the job to be deleted. The format is \"projects/{project_id}/tenants/{tenant_id}/jobs/{job_id}\". For example, \"projects/foo/tenants/bar/jobs/baz\"."
  [name]
  (client/api-request
    {:method "DELETE",
     :uri-template "https://jobs.googleapis.com/v4/{+name}",
     :uri-template-args {:name name},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"]}))

(defn projects-tenants-jobs-search
  "Searches for jobs using the provided SearchJobsRequest. This call constrains the visibility of jobs present in the database, and only returns jobs that the caller has permission to search against.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/search

parent <string> Required. The resource name of the tenant to search within. The format is \"projects/{project_id}/tenants/{tenant_id}\". For example, \"projects/foo/tenants/bar\".
SearchJobsRequest:
{:customRankingInfo
 {:importanceLevel
  [IMPORTANCE_LEVEL_UNSPECIFIED NONE LOW MILD MEDIUM HIGH EXTREME],
  :rankingExpression string},
 :searchMode [SEARCH_MODE_UNSPECIFIED JOB_SEARCH FEATURED_JOB_SEARCH],
 :maxPageSize integer,
 :offset integer,
 :disableKeywordMatch boolean,
 :histogramQueries [{:histogramQuery string}],
 :requestMetadata
 {:domain string,
  :sessionId string,
  :userId string,
  :allowMissingIds boolean,
  :deviceInfo
  {:deviceType
   [DEVICE_TYPE_UNSPECIFIED WEB MOBILE_WEB ANDROID IOS BOT OTHER],
   :id string}},
 :enableBroadening boolean,
 :diversificationLevel
 [DIVERSIFICATION_LEVEL_UNSPECIFIED
  DISABLED
  SIMPLE
  ONE_PER_COMPANY
  TWO_PER_COMPANY
  MAX_THREE_PER_COMPANY
  DIVERSIFY_BY_LOOSER_SIMILARITY],
 :pageToken string,
 :jobView
 [JOB_VIEW_UNSPECIFIED
  JOB_VIEW_ID_ONLY
  JOB_VIEW_MINIMAL
  JOB_VIEW_SMALL
  JOB_VIEW_FULL],
 :jobQuery
 {:customAttributeFilter string,
  :publishTimeRange {:startTime string, :endTime string},
  :queryLanguageCode string,
  :companyDisplayNames [string],
  :languageCodes [string],
  :compensationFilter
  {:type
   [FILTER_TYPE_UNSPECIFIED
    UNIT_ONLY
    UNIT_AND_AMOUNT
    ANNUALIZED_BASE_AMOUNT
    ANNUALIZED_TOTAL_AMOUNT],
   :units
   [[COMPENSATION_UNIT_UNSPECIFIED
     HOURLY
     DAILY
     WEEKLY
     MONTHLY
     YEARLY
     ONE_TIME
     OTHER_COMPENSATION_UNIT]],
   :range
   {:maxCompensation
    {:currencyCode string, :units string, :nanos integer},
    :minCompensation
    {:currencyCode string, :units string, :nanos integer}},
   :includeJobsWithUnspecifiedCompensationRange boolean},
  :locationFilters
  [{:address string,
    :regionCode string,
    :latLng {:latitude number, :longitude number},
    :distanceInMiles number,
    :telecommutePreference
    [TELECOMMUTE_PREFERENCE_UNSPECIFIED
     TELECOMMUTE_EXCLUDED
     TELECOMMUTE_ALLOWED
     TELECOMMUTE_JOBS_EXCLUDED]}],
  :disableSpellCheck boolean,
  :jobCategories
  [[JOB_CATEGORY_UNSPECIFIED
    ACCOUNTING_AND_FINANCE
    ADMINISTRATIVE_AND_OFFICE
    ADVERTISING_AND_MARKETING
    ANIMAL_CARE
    ART_FASHION_AND_DESIGN
    BUSINESS_OPERATIONS
    CLEANING_AND_FACILITIES
    COMPUTER_AND_IT
    CONSTRUCTION
    CUSTOMER_SERVICE
    EDUCATION
    ENTERTAINMENT_AND_TRAVEL
    FARMING_AND_OUTDOORS
    HEALTHCARE
    HUMAN_RESOURCES
    INSTALLATION_MAINTENANCE_AND_REPAIR
    LEGAL
    MANAGEMENT
    MANUFACTURING_AND_WAREHOUSE
    MEDIA_COMMUNICATIONS_AND_WRITING
    OIL_GAS_AND_MINING
    PERSONAL_CARE_AND_SERVICES
    PROTECTIVE_SERVICES
    REAL_ESTATE
    RESTAURANT_AND_HOSPITALITY
    SALES_AND_RETAIL
    SCIENCE_AND_ENGINEERING
    SOCIAL_SERVICES_AND_NON_PROFIT
    SPORTS_FITNESS_AND_RECREATION
    TRANSPORTATION_AND_LOGISTICS]],
  :companies [string],
  :commuteFilter
  {:commuteMethod
   [COMMUTE_METHOD_UNSPECIFIED
    DRIVING
    TRANSIT
    WALKING
    CYCLING
    TRANSIT_ACCESSIBLE],
   :startCoordinates {:latitude number, :longitude number},
   :travelDuration string,
   :allowImpreciseAddresses boolean,
   :roadTraffic [ROAD_TRAFFIC_UNSPECIFIED TRAFFIC_FREE BUSY_HOUR],
   :departureTime
   {:hours integer,
    :minutes integer,
    :seconds integer,
    :nanos integer}},
  :excludedJobs [string],
  :query string,
  :employmentTypes
  [[EMPLOYMENT_TYPE_UNSPECIFIED
    FULL_TIME
    PART_TIME
    CONTRACTOR
    CONTRACT_TO_HIRE
    TEMPORARY
    INTERN
    VOLUNTEER
    PER_DIEM
    FLY_IN_FLY_OUT
    OTHER_EMPLOYMENT_TYPE]]},
 :keywordMatchMode
 [KEYWORD_MATCH_MODE_UNSPECIFIED
  KEYWORD_MATCH_DISABLED
  KEYWORD_MATCH_ALL
  KEYWORD_MATCH_TITLE_ONLY],
 :orderBy string}"
  [parent SearchJobsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://jobs.googleapis.com/v4/{+parent}/jobs:search",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body SearchJobsRequest}))

(defn projects-tenants-jobs-batchDelete
  "Begins executing a batch delete jobs operation.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/batchDelete

parent <string> Required. The resource name of the tenant under which the job is created. The format is \"projects/{project_id}/tenants/{tenant_id}\". For example, \"projects/foo/tenants/bar\". The parent of all of the jobs specified in `names` must match this field.
BatchDeleteJobsRequest:
{:names [string]}"
  [parent BatchDeleteJobsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://jobs.googleapis.com/v4/{+parent}/jobs:batchDelete",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body BatchDeleteJobsRequest}))

(defn projects-tenants-jobs-list
  "Lists jobs by filter.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/list

parent <string> Required. The resource name of the tenant under which the job is created. The format is \"projects/{project_id}/tenants/{tenant_id}\". For example, \"projects/foo/tenants/bar\".

optional:
filter <string> Required. The filter string specifies the jobs to be enumerated. Supported operator: =, AND The fields eligible for filtering are: * `companyName` * `requisitionId` * `status` Available values: OPEN, EXPIRED, ALL. Defaults to OPEN if no value is specified. At least one of `companyName` and `requisitionId` must present or an INVALID_ARGUMENT error is thrown. Sample Query: * companyName = \"projects/foo/tenants/bar/companies/baz\" * companyName = \"projects/foo/tenants/bar/companies/baz\" AND requisitionId = \"req-1\" * companyName = \"projects/foo/tenants/bar/companies/baz\" AND status = \"EXPIRED\" * requisitionId = \"req-1\" * requisitionId = \"req-1\" AND status = \"EXPIRED\"
pageSize <integer> The maximum number of jobs to be returned per page of results. If job_view is set to JobView.JOB_VIEW_ID_ONLY, the maximum allowed page size is 1000. Otherwise, the maximum allowed page size is 100. Default is 100 if empty or a number < 1 is specified.
jobView <string> The desired job attributes returned for jobs in the search response. Defaults to JobView.JOB_VIEW_FULL if no value is specified."
  ([parent] (projects-tenants-jobs-list parent nil))
  ([parent optional]
    (client/api-request
      {:method "GET",
       :uri-template "https://jobs.googleapis.com/v4/{+parent}/jobs",
       :uri-template-args {:parent parent},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/jobs"]})))

(defn projects-tenants-jobs-searchForAlert
  "Searches for jobs using the provided SearchJobsRequest. This API call is intended for the use case of targeting passive job seekers (for example, job seekers who have signed up to receive email alerts about potential job opportunities), it has different algorithmic adjustments that are designed to specifically target passive job seekers. This call constrains the visibility of jobs present in the database, and only returns jobs the caller has permission to search against.
https://cloud.google.com/talent-solution/job-search/docs/reference/rest/v4/projects.tenants.jobs/searchForAlert

parent <string> Required. The resource name of the tenant to search within. The format is \"projects/{project_id}/tenants/{tenant_id}\". For example, \"projects/foo/tenants/bar\".
SearchJobsRequest:
{:customRankingInfo
 {:importanceLevel
  [IMPORTANCE_LEVEL_UNSPECIFIED NONE LOW MILD MEDIUM HIGH EXTREME],
  :rankingExpression string},
 :searchMode [SEARCH_MODE_UNSPECIFIED JOB_SEARCH FEATURED_JOB_SEARCH],
 :maxPageSize integer,
 :offset integer,
 :disableKeywordMatch boolean,
 :histogramQueries [{:histogramQuery string}],
 :requestMetadata
 {:domain string,
  :sessionId string,
  :userId string,
  :allowMissingIds boolean,
  :deviceInfo
  {:deviceType
   [DEVICE_TYPE_UNSPECIFIED WEB MOBILE_WEB ANDROID IOS BOT OTHER],
   :id string}},
 :enableBroadening boolean,
 :diversificationLevel
 [DIVERSIFICATION_LEVEL_UNSPECIFIED
  DISABLED
  SIMPLE
  ONE_PER_COMPANY
  TWO_PER_COMPANY
  MAX_THREE_PER_COMPANY
  DIVERSIFY_BY_LOOSER_SIMILARITY],
 :pageToken string,
 :jobView
 [JOB_VIEW_UNSPECIFIED
  JOB_VIEW_ID_ONLY
  JOB_VIEW_MINIMAL
  JOB_VIEW_SMALL
  JOB_VIEW_FULL],
 :jobQuery
 {:customAttributeFilter string,
  :publishTimeRange {:startTime string, :endTime string},
  :queryLanguageCode string,
  :companyDisplayNames [string],
  :languageCodes [string],
  :compensationFilter
  {:type
   [FILTER_TYPE_UNSPECIFIED
    UNIT_ONLY
    UNIT_AND_AMOUNT
    ANNUALIZED_BASE_AMOUNT
    ANNUALIZED_TOTAL_AMOUNT],
   :units
   [[COMPENSATION_UNIT_UNSPECIFIED
     HOURLY
     DAILY
     WEEKLY
     MONTHLY
     YEARLY
     ONE_TIME
     OTHER_COMPENSATION_UNIT]],
   :range
   {:maxCompensation
    {:currencyCode string, :units string, :nanos integer},
    :minCompensation
    {:currencyCode string, :units string, :nanos integer}},
   :includeJobsWithUnspecifiedCompensationRange boolean},
  :locationFilters
  [{:address string,
    :regionCode string,
    :latLng {:latitude number, :longitude number},
    :distanceInMiles number,
    :telecommutePreference
    [TELECOMMUTE_PREFERENCE_UNSPECIFIED
     TELECOMMUTE_EXCLUDED
     TELECOMMUTE_ALLOWED
     TELECOMMUTE_JOBS_EXCLUDED]}],
  :disableSpellCheck boolean,
  :jobCategories
  [[JOB_CATEGORY_UNSPECIFIED
    ACCOUNTING_AND_FINANCE
    ADMINISTRATIVE_AND_OFFICE
    ADVERTISING_AND_MARKETING
    ANIMAL_CARE
    ART_FASHION_AND_DESIGN
    BUSINESS_OPERATIONS
    CLEANING_AND_FACILITIES
    COMPUTER_AND_IT
    CONSTRUCTION
    CUSTOMER_SERVICE
    EDUCATION
    ENTERTAINMENT_AND_TRAVEL
    FARMING_AND_OUTDOORS
    HEALTHCARE
    HUMAN_RESOURCES
    INSTALLATION_MAINTENANCE_AND_REPAIR
    LEGAL
    MANAGEMENT
    MANUFACTURING_AND_WAREHOUSE
    MEDIA_COMMUNICATIONS_AND_WRITING
    OIL_GAS_AND_MINING
    PERSONAL_CARE_AND_SERVICES
    PROTECTIVE_SERVICES
    REAL_ESTATE
    RESTAURANT_AND_HOSPITALITY
    SALES_AND_RETAIL
    SCIENCE_AND_ENGINEERING
    SOCIAL_SERVICES_AND_NON_PROFIT
    SPORTS_FITNESS_AND_RECREATION
    TRANSPORTATION_AND_LOGISTICS]],
  :companies [string],
  :commuteFilter
  {:commuteMethod
   [COMMUTE_METHOD_UNSPECIFIED
    DRIVING
    TRANSIT
    WALKING
    CYCLING
    TRANSIT_ACCESSIBLE],
   :startCoordinates {:latitude number, :longitude number},
   :travelDuration string,
   :allowImpreciseAddresses boolean,
   :roadTraffic [ROAD_TRAFFIC_UNSPECIFIED TRAFFIC_FREE BUSY_HOUR],
   :departureTime
   {:hours integer,
    :minutes integer,
    :seconds integer,
    :nanos integer}},
  :excludedJobs [string],
  :query string,
  :employmentTypes
  [[EMPLOYMENT_TYPE_UNSPECIFIED
    FULL_TIME
    PART_TIME
    CONTRACTOR
    CONTRACT_TO_HIRE
    TEMPORARY
    INTERN
    VOLUNTEER
    PER_DIEM
    FLY_IN_FLY_OUT
    OTHER_EMPLOYMENT_TYPE]]},
 :keywordMatchMode
 [KEYWORD_MATCH_MODE_UNSPECIFIED
  KEYWORD_MATCH_DISABLED
  KEYWORD_MATCH_ALL
  KEYWORD_MATCH_TITLE_ONLY],
 :orderBy string}"
  [parent SearchJobsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://jobs.googleapis.com/v4/{+parent}/jobs:searchForAlert",
     :uri-template-args {:parent parent},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/jobs"],
     :body SearchJobsRequest}))
