(ns happygapi2.compute
  "Compute Engine API
Creates and runs virtual machines on Google Cloud Platform. 
See: https://cloud.google.com/products/compute"
  (:require [happy.oauth2.client :as client]))

(defn targetGrpcProxies-list
  "Lists the TargetGrpcProxies for a project in the given scope.
https://cloud.google.com/products/compute/v1/docs/targetGrpcProxies/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (targetGrpcProxies-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetGrpcProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetGrpcProxies-get
  "Returns the specified TargetGrpcProxy resource in the given scope.
https://cloud.google.com/products/compute/v1/docs/targetGrpcProxies/get

project <string> Project ID for this request.
targetGrpcProxy <string> Name of the TargetGrpcProxy resource to return."
  [project targetGrpcProxy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetGrpcProxies/{targetGrpcProxy}",
     :uri-template-args
     {:project project, :targetGrpcProxy targetGrpcProxy},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn targetGrpcProxies-insert
  "Creates a TargetGrpcProxy in the specified project in the given scope using the parameters that are included in the request.
https://cloud.google.com/products/compute/v1/docs/targetGrpcProxies/insert

project <string> Project ID for this request.
TargetGrpcProxy:
{:description string,
 :creationTimestamp string,
 :urlMap string,
 :name string,
 :selfLink string,
 :validateForProxyless boolean,
 :selfLinkWithId string,
 :id string,
 :kind string,
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project TargetGrpcProxy]
    (targetGrpcProxies-insert project TargetGrpcProxy nil))
  ([project TargetGrpcProxy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetGrpcProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetGrpcProxy})))

(defn targetGrpcProxies-delete
  "Deletes the specified TargetGrpcProxy in the given scope
https://cloud.google.com/products/compute/v1/docs/targetGrpcProxies/delete

project <string> Project ID for this request.
targetGrpcProxy <string> Name of the TargetGrpcProxy resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetGrpcProxy]
    (targetGrpcProxies-delete project targetGrpcProxy nil))
  ([project targetGrpcProxy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetGrpcProxies/{targetGrpcProxy}",
       :uri-template-args
       {:project project, :targetGrpcProxy targetGrpcProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn targetGrpcProxies-patch
  "Patches the specified TargetGrpcProxy resource with the data included in the request. This method supports PATCH semantics and uses JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/targetGrpcProxies/patch

project <string> Project ID for this request.
targetGrpcProxy <string> Name of the TargetGrpcProxy resource to patch.
TargetGrpcProxy:
{:description string,
 :creationTimestamp string,
 :urlMap string,
 :name string,
 :selfLink string,
 :validateForProxyless boolean,
 :selfLinkWithId string,
 :id string,
 :kind string,
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetGrpcProxy TargetGrpcProxy]
    (targetGrpcProxies-patch
      project
      targetGrpcProxy
      TargetGrpcProxy
      nil))
  ([project targetGrpcProxy TargetGrpcProxy optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetGrpcProxies/{targetGrpcProxy}",
       :uri-template-args
       {:project project, :targetGrpcProxy targetGrpcProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetGrpcProxy})))

(defn instantSnapshots-get
  "Returns the specified InstantSnapshot resource in the specified zone.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/get

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
instantSnapshot <string> Name of the InstantSnapshot resource to return."
  [project zone instantSnapshot]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instantSnapshots/{instantSnapshot}",
     :uri-template-args
     {:instantSnapshot instantSnapshot, :project project, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn instantSnapshots-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/setIamPolicy

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.
ZoneSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project zone resource ZoneSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instantSnapshots/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :resource resource, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body ZoneSetPolicyRequest}))

(defn instantSnapshots-insert
  "Creates an instant snapshot in the specified zone.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/insert

project <string> Project ID for this request.
zone <string> Name of the zone for this request.
InstantSnapshot:
{:description string,
 :labels object,
 :sourceDiskId string,
 :architecture [ARCHITECTURE_UNSPECIFIED ARM64 X86_64],
 :creationTimestamp string,
 :resourceStatus {:storageSizeBytes string},
 :satisfiesPzi boolean,
 :zone string,
 :name string,
 :sourceDisk string,
 :selfLink string,
 :region string,
 :selfLinkWithId string,
 :satisfiesPzs boolean,
 :status [CREATING DELETING FAILED READY UNAVAILABLE],
 :id string,
 :kind string,
 :diskSizeGb string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone InstantSnapshot]
    (instantSnapshots-insert project zone InstantSnapshot nil))
  ([project zone InstantSnapshot optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instantSnapshots",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstantSnapshot})))

(defn instantSnapshots-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/testIamPermissions

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project zone resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instantSnapshots/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :resource resource, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn instantSnapshots-setLabels
  "Sets the labels on a instantSnapshot in the given zone. To learn more about labels, read the Labeling Resources documentation.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/setLabels

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.
ZoneSetLabelsRequest:
{:labels object, :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone resource ZoneSetLabelsRequest]
    (instantSnapshots-setLabels
      project
      zone
      resource
      ZoneSetLabelsRequest
      nil))
  ([project zone resource ZoneSetLabelsRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instantSnapshots/{resource}/setLabels",
       :uri-template-args
       {:project project, :resource resource, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body ZoneSetLabelsRequest})))

(defn instantSnapshots-aggregatedList
  "Retrieves an aggregated list of instantSnapshots. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (instantSnapshots-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/instantSnapshots",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instantSnapshots-delete
  "Deletes the specified InstantSnapshot resource. Keep in mind that deleting a single instantSnapshot might not necessarily delete all the data on that instantSnapshot. If any data on the instantSnapshot that is marked for deletion is needed for subsequent instantSnapshots, the data will be moved to the next corresponding instantSnapshot. For more information, see Deleting instantSnapshots.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/delete

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
instantSnapshot <string> Name of the InstantSnapshot resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone instantSnapshot]
    (instantSnapshots-delete project zone instantSnapshot nil))
  ([project zone instantSnapshot optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instantSnapshots/{instantSnapshot}",
       :uri-template-args
       {:instantSnapshot instantSnapshot,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn instantSnapshots-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/getIamPolicy

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project zone resource]
    (instantSnapshots-getIamPolicy project zone resource nil))
  ([project zone resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instantSnapshots/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :resource resource, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instantSnapshots-list
  "Retrieves the list of InstantSnapshot resources contained within the specified zone.
https://cloud.google.com/products/compute/v1/docs/instantSnapshots/list

project <string> Project ID for this request.
zone <string> The name of the zone for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone] (instantSnapshots-list project zone nil))
  ([project zone optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instantSnapshots",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetVpnGateways-list
  "Retrieves a list of target VPN gateways available to the specified project and region.
https://cloud.google.com/products/compute/v1/docs/targetVpnGateways/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (targetVpnGateways-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetVpnGateways",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetVpnGateways-aggregatedList
  "Retrieves an aggregated list of target VPN gateways. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/targetVpnGateways/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (targetVpnGateways-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/targetVpnGateways",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetVpnGateways-get
  "Returns the specified target VPN gateway.
https://cloud.google.com/products/compute/v1/docs/targetVpnGateways/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
targetVpnGateway <string> Name of the target VPN gateway to return."
  [project region targetVpnGateway]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetVpnGateways/{targetVpnGateway}",
     :uri-template-args
     {:project project,
      :region region,
      :targetVpnGateway targetVpnGateway},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn targetVpnGateways-insert
  "Creates a target VPN gateway in the specified project and region using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/targetVpnGateways/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
TargetVpnGateway:
{:description string,
 :labels object,
 :tunnels [string],
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :forwardingRules [string],
 :status [CREATING DELETING FAILED READY],
 :id string,
 :kind string,
 :network string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region TargetVpnGateway]
    (targetVpnGateways-insert project region TargetVpnGateway nil))
  ([project region TargetVpnGateway optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetVpnGateways",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetVpnGateway})))

(defn targetVpnGateways-delete
  "Deletes the specified target VPN gateway.
https://cloud.google.com/products/compute/v1/docs/targetVpnGateways/delete

project <string> Project ID for this request.
region <string> Name of the region for this request.
targetVpnGateway <string> Name of the target VPN gateway to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region targetVpnGateway]
    (targetVpnGateways-delete project region targetVpnGateway nil))
  ([project region targetVpnGateway optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetVpnGateways/{targetVpnGateway}",
       :uri-template-args
       {:project project,
        :region region,
        :targetVpnGateway targetVpnGateway},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn targetVpnGateways-setLabels
  "Sets the labels on a TargetVpnGateway. To learn more about labels, read the Labeling Resources documentation.
https://cloud.google.com/products/compute/v1/docs/targetVpnGateways/setLabels

project <string> Project ID for this request.
region <string> The region for this request.
resource <string> Name or id of the resource for this request.
RegionSetLabelsRequest:
{:labels object, :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region resource RegionSetLabelsRequest]
    (targetVpnGateways-setLabels
      project
      region
      resource
      RegionSetLabelsRequest
      nil))
  ([project region resource RegionSetLabelsRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetVpnGateways/{resource}/setLabels",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionSetLabelsRequest})))

(defn routers-get
  "Returns the specified Router resource.
https://cloud.google.com/products/compute/v1/docs/routers/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
router <string> Name of the Router resource to return."
  [project region router]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers/{router}",
     :uri-template-args
     {:project project, :region region, :router router},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn routers-insert
  "Creates a Router resource in the specified project and region using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/routers/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
Router:
{:description string,
 :interfaces
 [{:linkedVpnTunnel string,
   :redundantInterface string,
   :name string,
   :managementType [MANAGED_BY_ATTACHMENT MANAGED_BY_USER],
   :ipVersion [IPV4 IPV6],
   :ipRange string,
   :privateIpAddress string,
   :linkedInterconnectAttachment string,
   :subnetwork string}],
 :creationTimestamp string,
 :nats
 [{:minPortsPerVm integer,
   :name string,
   :type [PRIVATE PUBLIC],
   :sourceSubnetworkIpRangesToNat
   [ALL_SUBNETWORKS_ALL_IP_RANGES
    ALL_SUBNETWORKS_ALL_PRIMARY_IP_RANGES
    LIST_OF_SUBNETWORKS],
   :rules
   [{:ruleNumber integer,
     :description string,
     :match string,
     :action
     {:sourceNatActiveIps [string],
      :sourceNatDrainIps [string],
      :sourceNatActiveRanges [string],
      :sourceNatDrainRanges [string]}}],
   :tcpTransitoryIdleTimeoutSec integer,
   :natIpAllocateOption [AUTO_ONLY MANUAL_ONLY],
   :autoNetworkTier
   [FIXED_STANDARD PREMIUM STANDARD STANDARD_OVERRIDES_FIXED_STANDARD],
   :udpIdleTimeoutSec integer,
   :subnetworks
   [{:name string,
     :sourceIpRangesToNat
     [[ALL_IP_RANGES LIST_OF_SECONDARY_IP_RANGES PRIMARY_IP_RANGE]],
     :secondaryIpRangeNames [string]}],
   :natIps [string],
   :enableDynamicPortAllocation boolean,
   :icmpIdleTimeoutSec integer,
   :tcpEstablishedIdleTimeoutSec integer,
   :maxPortsPerVm integer,
   :drainNatIps [string],
   :endpointTypes
   [[ENDPOINT_TYPE_MANAGED_PROXY_LB
     ENDPOINT_TYPE_SWG
     ENDPOINT_TYPE_VM]],
   :enableEndpointIndependentMapping boolean,
   :tcpTimeWaitTimeoutSec integer,
   :logConfig
   {:enable boolean, :filter [ALL ERRORS_ONLY TRANSLATIONS_ONLY]}}],
 :name string,
 :bgpPeers
 [{:peerIpv6NexthopAddress string,
   :bfd
   {:sessionInitializationMode [ACTIVE DISABLED PASSIVE],
    :minTransmitInterval integer,
    :minReceiveInterval integer,
    :multiplier integer},
   :enable [FALSE TRUE],
   :md5AuthenticationKeyName string,
   :peerIpAddress string,
   :name string,
   :advertisedIpRanges [{:range string, :description string}],
   :advertisedGroups [[ALL_SUBNETS]],
   :routerApplianceInstance string,
   :managementType [MANAGED_BY_ATTACHMENT MANAGED_BY_USER],
   :ipv6NexthopAddress string,
   :exportPolicies [string],
   :advertiseMode [CUSTOM DEFAULT],
   :peerAsn integer,
   :customLearnedRoutePriority integer,
   :interfaceName string,
   :importPolicies [string],
   :enableIpv6 boolean,
   :ipAddress string,
   :peerIpv4NexthopAddress string,
   :enableIpv4 boolean,
   :advertisedRoutePriority integer,
   :ipv4NexthopAddress string,
   :customLearnedIpRanges [{:range string}]}],
 :selfLink string,
 :bgp
 {:asn integer,
  :advertiseMode [CUSTOM DEFAULT],
  :advertisedGroups [[ALL_SUBNETS]],
  :advertisedIpRanges [{:range string, :description string}],
  :keepaliveInterval integer,
  :identifierRange string},
 :md5AuthenticationKeys [{:name string, :key string}],
 :region string,
 :id string,
 :kind string,
 :encryptedInterconnectRouter boolean,
 :network string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region Router] (routers-insert project region Router nil))
  ([project region Router optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Router})))

(defn routers-patch
  "Patches the specified Router resource with the data included in the request. This method supports PATCH semantics and uses JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/routers/patch

project <string> Project ID for this request.
region <string> Name of the region for this request.
router <string> Name of the Router resource to patch.
Router:
{:description string,
 :interfaces
 [{:linkedVpnTunnel string,
   :redundantInterface string,
   :name string,
   :managementType [MANAGED_BY_ATTACHMENT MANAGED_BY_USER],
   :ipVersion [IPV4 IPV6],
   :ipRange string,
   :privateIpAddress string,
   :linkedInterconnectAttachment string,
   :subnetwork string}],
 :creationTimestamp string,
 :nats
 [{:minPortsPerVm integer,
   :name string,
   :type [PRIVATE PUBLIC],
   :sourceSubnetworkIpRangesToNat
   [ALL_SUBNETWORKS_ALL_IP_RANGES
    ALL_SUBNETWORKS_ALL_PRIMARY_IP_RANGES
    LIST_OF_SUBNETWORKS],
   :rules
   [{:ruleNumber integer,
     :description string,
     :match string,
     :action
     {:sourceNatActiveIps [string],
      :sourceNatDrainIps [string],
      :sourceNatActiveRanges [string],
      :sourceNatDrainRanges [string]}}],
   :tcpTransitoryIdleTimeoutSec integer,
   :natIpAllocateOption [AUTO_ONLY MANUAL_ONLY],
   :autoNetworkTier
   [FIXED_STANDARD PREMIUM STANDARD STANDARD_OVERRIDES_FIXED_STANDARD],
   :udpIdleTimeoutSec integer,
   :subnetworks
   [{:name string,
     :sourceIpRangesToNat
     [[ALL_IP_RANGES LIST_OF_SECONDARY_IP_RANGES PRIMARY_IP_RANGE]],
     :secondaryIpRangeNames [string]}],
   :natIps [string],
   :enableDynamicPortAllocation boolean,
   :icmpIdleTimeoutSec integer,
   :tcpEstablishedIdleTimeoutSec integer,
   :maxPortsPerVm integer,
   :drainNatIps [string],
   :endpointTypes
   [[ENDPOINT_TYPE_MANAGED_PROXY_LB
     ENDPOINT_TYPE_SWG
     ENDPOINT_TYPE_VM]],
   :enableEndpointIndependentMapping boolean,
   :tcpTimeWaitTimeoutSec integer,
   :logConfig
   {:enable boolean, :filter [ALL ERRORS_ONLY TRANSLATIONS_ONLY]}}],
 :name string,
 :bgpPeers
 [{:peerIpv6NexthopAddress string,
   :bfd
   {:sessionInitializationMode [ACTIVE DISABLED PASSIVE],
    :minTransmitInterval integer,
    :minReceiveInterval integer,
    :multiplier integer},
   :enable [FALSE TRUE],
   :md5AuthenticationKeyName string,
   :peerIpAddress string,
   :name string,
   :advertisedIpRanges [{:range string, :description string}],
   :advertisedGroups [[ALL_SUBNETS]],
   :routerApplianceInstance string,
   :managementType [MANAGED_BY_ATTACHMENT MANAGED_BY_USER],
   :ipv6NexthopAddress string,
   :exportPolicies [string],
   :advertiseMode [CUSTOM DEFAULT],
   :peerAsn integer,
   :customLearnedRoutePriority integer,
   :interfaceName string,
   :importPolicies [string],
   :enableIpv6 boolean,
   :ipAddress string,
   :peerIpv4NexthopAddress string,
   :enableIpv4 boolean,
   :advertisedRoutePriority integer,
   :ipv4NexthopAddress string,
   :customLearnedIpRanges [{:range string}]}],
 :selfLink string,
 :bgp
 {:asn integer,
  :advertiseMode [CUSTOM DEFAULT],
  :advertisedGroups [[ALL_SUBNETS]],
  :advertisedIpRanges [{:range string, :description string}],
  :keepaliveInterval integer,
  :identifierRange string},
 :md5AuthenticationKeys [{:name string, :key string}],
 :region string,
 :id string,
 :kind string,
 :encryptedInterconnectRouter boolean,
 :network string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region router Router]
    (routers-patch project region router Router nil))
  ([project region router Router optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers/{router}",
       :uri-template-args
       {:project project, :region region, :router router},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Router})))

(defn routers-getRouterStatus
  "Retrieves runtime information of the specified router.
https://cloud.google.com/products/compute/v1/docs/routers/getRouterStatus

project <string> Project ID for this request.
region <string> Name of the region for this request.
router <string> Name of the Router resource to query."
  [project region router]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers/{router}/getRouterStatus",
     :uri-template-args
     {:project project, :region region, :router router},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn routers-aggregatedList
  "Retrieves an aggregated list of routers. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/routers/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (routers-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/routers",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn routers-update
  "Updates the specified Router resource with the data included in the request. This method conforms to PUT semantics, which requests that the state of the target resource be created or replaced with the state defined by the representation enclosed in the request message payload.
https://cloud.google.com/products/compute/v1/docs/routers/update

project <string> Project ID for this request.
region <string> Name of the region for this request.
router <string> Name of the Router resource to update.
Router:
{:description string,
 :interfaces
 [{:linkedVpnTunnel string,
   :redundantInterface string,
   :name string,
   :managementType [MANAGED_BY_ATTACHMENT MANAGED_BY_USER],
   :ipVersion [IPV4 IPV6],
   :ipRange string,
   :privateIpAddress string,
   :linkedInterconnectAttachment string,
   :subnetwork string}],
 :creationTimestamp string,
 :nats
 [{:minPortsPerVm integer,
   :name string,
   :type [PRIVATE PUBLIC],
   :sourceSubnetworkIpRangesToNat
   [ALL_SUBNETWORKS_ALL_IP_RANGES
    ALL_SUBNETWORKS_ALL_PRIMARY_IP_RANGES
    LIST_OF_SUBNETWORKS],
   :rules
   [{:ruleNumber integer,
     :description string,
     :match string,
     :action
     {:sourceNatActiveIps [string],
      :sourceNatDrainIps [string],
      :sourceNatActiveRanges [string],
      :sourceNatDrainRanges [string]}}],
   :tcpTransitoryIdleTimeoutSec integer,
   :natIpAllocateOption [AUTO_ONLY MANUAL_ONLY],
   :autoNetworkTier
   [FIXED_STANDARD PREMIUM STANDARD STANDARD_OVERRIDES_FIXED_STANDARD],
   :udpIdleTimeoutSec integer,
   :subnetworks
   [{:name string,
     :sourceIpRangesToNat
     [[ALL_IP_RANGES LIST_OF_SECONDARY_IP_RANGES PRIMARY_IP_RANGE]],
     :secondaryIpRangeNames [string]}],
   :natIps [string],
   :enableDynamicPortAllocation boolean,
   :icmpIdleTimeoutSec integer,
   :tcpEstablishedIdleTimeoutSec integer,
   :maxPortsPerVm integer,
   :drainNatIps [string],
   :endpointTypes
   [[ENDPOINT_TYPE_MANAGED_PROXY_LB
     ENDPOINT_TYPE_SWG
     ENDPOINT_TYPE_VM]],
   :enableEndpointIndependentMapping boolean,
   :tcpTimeWaitTimeoutSec integer,
   :logConfig
   {:enable boolean, :filter [ALL ERRORS_ONLY TRANSLATIONS_ONLY]}}],
 :name string,
 :bgpPeers
 [{:peerIpv6NexthopAddress string,
   :bfd
   {:sessionInitializationMode [ACTIVE DISABLED PASSIVE],
    :minTransmitInterval integer,
    :minReceiveInterval integer,
    :multiplier integer},
   :enable [FALSE TRUE],
   :md5AuthenticationKeyName string,
   :peerIpAddress string,
   :name string,
   :advertisedIpRanges [{:range string, :description string}],
   :advertisedGroups [[ALL_SUBNETS]],
   :routerApplianceInstance string,
   :managementType [MANAGED_BY_ATTACHMENT MANAGED_BY_USER],
   :ipv6NexthopAddress string,
   :exportPolicies [string],
   :advertiseMode [CUSTOM DEFAULT],
   :peerAsn integer,
   :customLearnedRoutePriority integer,
   :interfaceName string,
   :importPolicies [string],
   :enableIpv6 boolean,
   :ipAddress string,
   :peerIpv4NexthopAddress string,
   :enableIpv4 boolean,
   :advertisedRoutePriority integer,
   :ipv4NexthopAddress string,
   :customLearnedIpRanges [{:range string}]}],
 :selfLink string,
 :bgp
 {:asn integer,
  :advertiseMode [CUSTOM DEFAULT],
  :advertisedGroups [[ALL_SUBNETS]],
  :advertisedIpRanges [{:range string, :description string}],
  :keepaliveInterval integer,
  :identifierRange string},
 :md5AuthenticationKeys [{:name string, :key string}],
 :region string,
 :id string,
 :kind string,
 :encryptedInterconnectRouter boolean,
 :network string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region router Router]
    (routers-update project region router Router nil))
  ([project region router Router optional]
    (client/api-request
      {:method "PUT",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers/{router}",
       :uri-template-args
       {:project project, :region region, :router router},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Router})))

(defn routers-delete
  "Deletes the specified Router resource.
https://cloud.google.com/products/compute/v1/docs/routers/delete

project <string> Project ID for this request.
region <string> Name of the region for this request.
router <string> Name of the Router resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region router] (routers-delete project region router nil))
  ([project region router optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers/{router}",
       :uri-template-args
       {:project project, :region region, :router router},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn routers-preview
  "Preview fields auto-generated during router create and update operations. Calling this method does NOT create or update the router.
https://cloud.google.com/products/compute/v1/docs/routers/preview

project <string> Project ID for this request.
region <string> Name of the region for this request.
router <string> Name of the Router resource to query.
Router:
{:description string,
 :interfaces
 [{:linkedVpnTunnel string,
   :redundantInterface string,
   :name string,
   :managementType [MANAGED_BY_ATTACHMENT MANAGED_BY_USER],
   :ipVersion [IPV4 IPV6],
   :ipRange string,
   :privateIpAddress string,
   :linkedInterconnectAttachment string,
   :subnetwork string}],
 :creationTimestamp string,
 :nats
 [{:minPortsPerVm integer,
   :name string,
   :type [PRIVATE PUBLIC],
   :sourceSubnetworkIpRangesToNat
   [ALL_SUBNETWORKS_ALL_IP_RANGES
    ALL_SUBNETWORKS_ALL_PRIMARY_IP_RANGES
    LIST_OF_SUBNETWORKS],
   :rules
   [{:ruleNumber integer,
     :description string,
     :match string,
     :action
     {:sourceNatActiveIps [string],
      :sourceNatDrainIps [string],
      :sourceNatActiveRanges [string],
      :sourceNatDrainRanges [string]}}],
   :tcpTransitoryIdleTimeoutSec integer,
   :natIpAllocateOption [AUTO_ONLY MANUAL_ONLY],
   :autoNetworkTier
   [FIXED_STANDARD PREMIUM STANDARD STANDARD_OVERRIDES_FIXED_STANDARD],
   :udpIdleTimeoutSec integer,
   :subnetworks
   [{:name string,
     :sourceIpRangesToNat
     [[ALL_IP_RANGES LIST_OF_SECONDARY_IP_RANGES PRIMARY_IP_RANGE]],
     :secondaryIpRangeNames [string]}],
   :natIps [string],
   :enableDynamicPortAllocation boolean,
   :icmpIdleTimeoutSec integer,
   :tcpEstablishedIdleTimeoutSec integer,
   :maxPortsPerVm integer,
   :drainNatIps [string],
   :endpointTypes
   [[ENDPOINT_TYPE_MANAGED_PROXY_LB
     ENDPOINT_TYPE_SWG
     ENDPOINT_TYPE_VM]],
   :enableEndpointIndependentMapping boolean,
   :tcpTimeWaitTimeoutSec integer,
   :logConfig
   {:enable boolean, :filter [ALL ERRORS_ONLY TRANSLATIONS_ONLY]}}],
 :name string,
 :bgpPeers
 [{:peerIpv6NexthopAddress string,
   :bfd
   {:sessionInitializationMode [ACTIVE DISABLED PASSIVE],
    :minTransmitInterval integer,
    :minReceiveInterval integer,
    :multiplier integer},
   :enable [FALSE TRUE],
   :md5AuthenticationKeyName string,
   :peerIpAddress string,
   :name string,
   :advertisedIpRanges [{:range string, :description string}],
   :advertisedGroups [[ALL_SUBNETS]],
   :routerApplianceInstance string,
   :managementType [MANAGED_BY_ATTACHMENT MANAGED_BY_USER],
   :ipv6NexthopAddress string,
   :exportPolicies [string],
   :advertiseMode [CUSTOM DEFAULT],
   :peerAsn integer,
   :customLearnedRoutePriority integer,
   :interfaceName string,
   :importPolicies [string],
   :enableIpv6 boolean,
   :ipAddress string,
   :peerIpv4NexthopAddress string,
   :enableIpv4 boolean,
   :advertisedRoutePriority integer,
   :ipv4NexthopAddress string,
   :customLearnedIpRanges [{:range string}]}],
 :selfLink string,
 :bgp
 {:asn integer,
  :advertiseMode [CUSTOM DEFAULT],
  :advertisedGroups [[ALL_SUBNETS]],
  :advertisedIpRanges [{:range string, :description string}],
  :keepaliveInterval integer,
  :identifierRange string},
 :md5AuthenticationKeys [{:name string, :key string}],
 :region string,
 :id string,
 :kind string,
 :encryptedInterconnectRouter boolean,
 :network string}"
  [project region router Router]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers/{router}/preview",
     :uri-template-args
     {:project project, :region region, :router router},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body Router}))

(defn routers-getNatMappingInfo
  "Retrieves runtime Nat mapping information of VM endpoints.
https://cloud.google.com/products/compute/v1/docs/routers/getNatMappingInfo

project <string> Project ID for this request.
region <string> Name of the region for this request.
router <string> Name of the Router resource to query for Nat Mapping information of VM endpoints.

optional:
natName <string> Name of the nat service to filter the Nat Mapping information. If it is omitted, all nats for this router will be returned. Name should conform to RFC1035.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported."
  ([project region router]
    (routers-getNatMappingInfo project region router nil))
  ([project region router optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers/{router}/getNatMappingInfo",
       :uri-template-args
       {:router router, :region region, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn routers-list
  "Retrieves a list of Router resources available to the specified project.
https://cloud.google.com/products/compute/v1/docs/routers/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (routers-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn routers-getNatIpInfo
  "Retrieves runtime NAT IP information.
https://cloud.google.com/products/compute/v1/docs/routers/getNatIpInfo

project <string> Project ID for this request.
region <string> Name of the region for this request.
router <string> Name of the Router resource to query for Nat IP information. The name should conform to RFC1035.

optional:
natName <string> Name of the nat service to filter the NAT IP information. If it is omitted, all nats for this router will be returned. Name should conform to RFC1035."
  ([project region router]
    (routers-getNatIpInfo project region router nil))
  ([project region router optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/routers/{router}/getNatIpInfo",
       :uri-template-args
       {:project project, :region region, :router router},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn projects-moveDisk
  "Moves a persistent disk from one zone to another.
https://cloud.google.com/products/compute/v1/docs/projects/moveDisk

project <string> Project ID for this request.
DiskMoveRequest:
{:targetDisk string, :destinationZone string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project DiskMoveRequest]
    (projects-moveDisk project DiskMoveRequest nil))
  ([project DiskMoveRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/moveDisk",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body DiskMoveRequest})))

(defn projects-setDefaultNetworkTier
  "Sets the default network tier of the project. The default network tier is used when an address/forwardingRule/instance is created without specifying the network tier field.
https://cloud.google.com/products/compute/v1/docs/projects/setDefaultNetworkTier

project <string> Project ID for this request.
ProjectsSetDefaultNetworkTierRequest:
{:networkTier
 [FIXED_STANDARD PREMIUM STANDARD STANDARD_OVERRIDES_FIXED_STANDARD]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project ProjectsSetDefaultNetworkTierRequest]
    (projects-setDefaultNetworkTier
      project
      ProjectsSetDefaultNetworkTierRequest
      nil))
  ([project ProjectsSetDefaultNetworkTierRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/setDefaultNetworkTier",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body ProjectsSetDefaultNetworkTierRequest})))

(defn projects-setUsageExportBucket
  "Enables the usage export feature and sets the usage export bucket where reports are stored. If you provide an empty request body using this method, the usage export feature will be disabled.
https://cloud.google.com/products/compute/v1/docs/projects/setUsageExportBucket

project <string> Project ID for this request.
UsageExportLocation:
{:bucketName string, :reportNamePrefix string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project UsageExportLocation]
    (projects-setUsageExportBucket project UsageExportLocation nil))
  ([project UsageExportLocation optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/setUsageExportBucket",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/devstorage.full_control"
        "https://www.googleapis.com/auth/devstorage.read_only"
        "https://www.googleapis.com/auth/devstorage.read_write"],
       :body UsageExportLocation})))

(defn projects-disableXpnHost
  "Disable this project as a shared VPC host project.
https://cloud.google.com/products/compute/v1/docs/projects/disableXpnHost

project <string> Project ID for this request.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project] (projects-disableXpnHost project nil))
  ([project optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/disableXpnHost",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn projects-get
  "Returns the specified Project resource. To decrease latency for this method, you can optionally omit any unneeded information from the response by using a field mask. This practice is especially recommended for unused quota information (the `quotas` field). To exclude one or more fields, set your request's `fields` query parameter to only include the fields you need. For example, to only include the `id` and `selfLink` fields, add the query parameter `?fields=id,selfLink` to your request.
https://cloud.google.com/products/compute/v1/docs/projects/get

project <string> Project ID for this request."
  [project]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}",
     :uri-template-args {:project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn projects-setCloudArmorTier
  "Sets the Cloud Armor tier of the project. To set ENTERPRISE or above the billing account of the project must be subscribed to Cloud Armor Enterprise. See Subscribing to Cloud Armor Enterprise for more information.
https://cloud.google.com/products/compute/v1/docs/projects/setCloudArmorTier

project <string> Project ID for this request.
ProjectsSetCloudArmorTierRequest:
{:cloudArmorTier [CA_ENTERPRISE_ANNUAL CA_ENTERPRISE_PAYGO CA_STANDARD]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project ProjectsSetCloudArmorTierRequest]
    (projects-setCloudArmorTier
      project
      ProjectsSetCloudArmorTierRequest
      nil))
  ([project ProjectsSetCloudArmorTierRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/setCloudArmorTier",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body ProjectsSetCloudArmorTierRequest})))

(defn projects-getXpnResources
  "Gets service resources (a.k.a service project) associated with this host project.
https://cloud.google.com/products/compute/v1/docs/projects/getXpnResources

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (projects-getXpnResources project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/getXpnResources",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn projects-setCommonInstanceMetadata
  "Sets metadata common to all instances within the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/projects/setCommonInstanceMetadata

project <string> Project ID for this request.
Metadata:
{:kind string,
 :fingerprint string,
 :items [{:key string, :value string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project Metadata]
    (projects-setCommonInstanceMetadata project Metadata nil))
  ([project Metadata optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/setCommonInstanceMetadata",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Metadata})))

(defn projects-moveInstance
  "Moves an instance and its attached persistent disks from one zone to another. *Note*: Moving VMs or disks by using this method might cause unexpected behavior. For more information, see the [known issue](/compute/docs/troubleshooting/known-issues#moving_vms_or_disks_using_the_moveinstance_api_or_the_causes_unexpected_behavior). [Deprecated] This method is deprecated. See [moving instance across zones](/compute/docs/instances/moving-instance-across-zones) instead.
https://cloud.google.com/products/compute/v1/docs/projects/moveInstance

project <string> Project ID for this request.
InstanceMoveRequest:
{:targetInstance string, :destinationZone string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project InstanceMoveRequest]
    (projects-moveInstance project InstanceMoveRequest nil))
  ([project InstanceMoveRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/moveInstance",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceMoveRequest})))

(defn projects-getXpnHost
  "Gets the shared VPC host project that this project links to. May be empty if no link exists.
https://cloud.google.com/products/compute/v1/docs/projects/getXpnHost

project <string> Project ID for this request."
  [project]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/getXpnHost",
     :uri-template-args {:project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"]}))

(defn projects-disableXpnResource
  "Disable a service resource (also known as service project) associated with this host project.
https://cloud.google.com/products/compute/v1/docs/projects/disableXpnResource

project <string> Project ID for this request.
ProjectsDisableXpnResourceRequest:
{:xpnResource
 {:type [PROJECT XPN_RESOURCE_TYPE_UNSPECIFIED], :id string}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project ProjectsDisableXpnResourceRequest]
    (projects-disableXpnResource
      project
      ProjectsDisableXpnResourceRequest
      nil))
  ([project ProjectsDisableXpnResourceRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/disableXpnResource",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body ProjectsDisableXpnResourceRequest})))

(defn projects-enableXpnHost
  "Enable this project as a shared VPC host project.
https://cloud.google.com/products/compute/v1/docs/projects/enableXpnHost

project <string> Project ID for this request.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project] (projects-enableXpnHost project nil))
  ([project optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/enableXpnHost",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn projects-enableXpnResource
  "Enable service resource (a.k.a service project) for a host project, so that subnets in the host project can be used by instances in the service project.
https://cloud.google.com/products/compute/v1/docs/projects/enableXpnResource

project <string> Project ID for this request.
ProjectsEnableXpnResourceRequest:
{:xpnResource
 {:type [PROJECT XPN_RESOURCE_TYPE_UNSPECIFIED], :id string}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project ProjectsEnableXpnResourceRequest]
    (projects-enableXpnResource
      project
      ProjectsEnableXpnResourceRequest
      nil))
  ([project ProjectsEnableXpnResourceRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/enableXpnResource",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body ProjectsEnableXpnResourceRequest})))

(defn projects-listXpnHosts
  "Lists all shared VPC host projects visible to the user in an organization.
https://cloud.google.com/products/compute/v1/docs/projects/listXpnHosts

project <string> Project ID for this request.
ProjectsListXpnHostsRequest:
{:organization string}

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project ProjectsListXpnHostsRequest]
    (projects-listXpnHosts project ProjectsListXpnHostsRequest nil))
  ([project ProjectsListXpnHostsRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/listXpnHosts",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body ProjectsListXpnHostsRequest})))

(defn imageFamilyViews-get
  "Returns the latest image that is part of an image family, is not deprecated and is rolled out in the specified zone.
https://cloud.google.com/products/compute/v1/docs/imageFamilyViews/get

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
family <string> Name of the image family to search for."
  [project zone family]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/imageFamilyViews/{family}",
     :uri-template-args {:family family, :project project, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionTargetHttpsProxies-list
  "Retrieves the list of TargetHttpsProxy resources available to the specified project in the specified region.
https://cloud.google.com/products/compute/v1/docs/regionTargetHttpsProxies/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionTargetHttpsProxies-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetHttpsProxies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionTargetHttpsProxies-get
  "Returns the specified TargetHttpsProxy resource in the specified region.
https://cloud.google.com/products/compute/v1/docs/regionTargetHttpsProxies/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
targetHttpsProxy <string> Name of the TargetHttpsProxy resource to return."
  [project region targetHttpsProxy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetHttpsProxies/{targetHttpsProxy}",
     :uri-template-args
     {:project project,
      :region region,
      :targetHttpsProxy targetHttpsProxy},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionTargetHttpsProxies-insert
  "Creates a TargetHttpsProxy resource in the specified project and region using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionTargetHttpsProxies/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
TargetHttpsProxy:
{:description string,
 :creationTimestamp string,
 :urlMap string,
 :name string,
 :httpKeepAliveTimeoutSec integer,
 :quicOverride [DISABLE ENABLE NONE],
 :selfLink string,
 :serverTlsPolicy string,
 :region string,
 :certificateMap string,
 :id string,
 :kind string,
 :tlsEarlyData [DISABLED PERMISSIVE STRICT],
 :sslPolicy string,
 :sslCertificates [string],
 :authorizationPolicy string,
 :fingerprint string,
 :proxyBind boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region TargetHttpsProxy]
    (regionTargetHttpsProxies-insert
      project
      region
      TargetHttpsProxy
      nil))
  ([project region TargetHttpsProxy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetHttpsProxies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetHttpsProxy})))

(defn regionTargetHttpsProxies-delete
  "Deletes the specified TargetHttpsProxy resource.
https://cloud.google.com/products/compute/v1/docs/regionTargetHttpsProxies/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
targetHttpsProxy <string> Name of the TargetHttpsProxy resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region targetHttpsProxy]
    (regionTargetHttpsProxies-delete
      project
      region
      targetHttpsProxy
      nil))
  ([project region targetHttpsProxy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetHttpsProxies/{targetHttpsProxy}",
       :uri-template-args
       {:project project,
        :region region,
        :targetHttpsProxy targetHttpsProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionTargetHttpsProxies-patch
  "Patches the specified regional TargetHttpsProxy resource with the data included in the request. This method supports PATCH semantics and uses JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/regionTargetHttpsProxies/patch

project <string> Project ID for this request.
region <string> Name of the region for this request.
targetHttpsProxy <string> Name of the TargetHttpsProxy resource to patch.
TargetHttpsProxy:
{:description string,
 :creationTimestamp string,
 :urlMap string,
 :name string,
 :httpKeepAliveTimeoutSec integer,
 :quicOverride [DISABLE ENABLE NONE],
 :selfLink string,
 :serverTlsPolicy string,
 :region string,
 :certificateMap string,
 :id string,
 :kind string,
 :tlsEarlyData [DISABLED PERMISSIVE STRICT],
 :sslPolicy string,
 :sslCertificates [string],
 :authorizationPolicy string,
 :fingerprint string,
 :proxyBind boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region targetHttpsProxy TargetHttpsProxy]
    (regionTargetHttpsProxies-patch
      project
      region
      targetHttpsProxy
      TargetHttpsProxy
      nil))
  ([project region targetHttpsProxy TargetHttpsProxy optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetHttpsProxies/{targetHttpsProxy}",
       :uri-template-args
       {:project project,
        :region region,
        :targetHttpsProxy targetHttpsProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetHttpsProxy})))

(defn regionTargetHttpsProxies-setUrlMap
  "Changes the URL map for TargetHttpsProxy.
https://cloud.google.com/products/compute/v1/docs/regionTargetHttpsProxies/setUrlMap

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
targetHttpsProxy <string> Name of the TargetHttpsProxy to set a URL map for.
UrlMapReference:
{:urlMap string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region targetHttpsProxy UrlMapReference]
    (regionTargetHttpsProxies-setUrlMap
      project
      region
      targetHttpsProxy
      UrlMapReference
      nil))
  ([project region targetHttpsProxy UrlMapReference optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetHttpsProxies/{targetHttpsProxy}/setUrlMap",
       :uri-template-args
       {:project project,
        :region region,
        :targetHttpsProxy targetHttpsProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body UrlMapReference})))

(defn regionTargetHttpsProxies-setSslCertificates
  "Replaces SslCertificates for TargetHttpsProxy.
https://cloud.google.com/products/compute/v1/docs/regionTargetHttpsProxies/setSslCertificates

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
targetHttpsProxy <string> Name of the TargetHttpsProxy resource to set an SslCertificates resource for.
RegionTargetHttpsProxiesSetSslCertificatesRequest:
{:sslCertificates [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    targetHttpsProxy
    RegionTargetHttpsProxiesSetSslCertificatesRequest]
    (regionTargetHttpsProxies-setSslCertificates
      project
      region
      targetHttpsProxy
      RegionTargetHttpsProxiesSetSslCertificatesRequest
      nil))
  ([project
    region
    targetHttpsProxy
    RegionTargetHttpsProxiesSetSslCertificatesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetHttpsProxies/{targetHttpsProxy}/setSslCertificates",
       :uri-template-args
       {:project project,
        :region region,
        :targetHttpsProxy targetHttpsProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionTargetHttpsProxiesSetSslCertificatesRequest})))

(defn routes-list
  "Retrieves the list of Route resources available to the specified project.
https://cloud.google.com/products/compute/v1/docs/routes/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (routes-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/routes",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn routes-get
  "Returns the specified Route resource.
https://cloud.google.com/products/compute/v1/docs/routes/get

project <string> Project ID for this request.
route <string> Name of the Route resource to return."
  [project route]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/routes/{route}",
     :uri-template-args {:project project, :route route},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn routes-insert
  "Creates a Route resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/routes/insert

project <string> Project ID for this request.
Route:
{:nextHopIp string,
 :description string,
 :tags [string],
 :nextHopNetwork string,
 :nextHopInstance string,
 :creationTimestamp string,
 :nextHopPeering string,
 :nextHopIlb string,
 :routeStatus [ACTIVE DROPPED INACTIVE PENDING],
 :name string,
 :selfLink string,
 :warnings
 [{:code
   [CLEANUP_FAILED
    DEPRECATED_RESOURCE_USED
    DEPRECATED_TYPE_USED
    DISK_SIZE_LARGER_THAN_IMAGE_SIZE
    EXPERIMENTAL_TYPE_USED
    EXTERNAL_API_WARNING
    FIELD_VALUE_OVERRIDEN
    INJECTED_KERNELS_DEPRECATED
    INVALID_HEALTH_CHECK_FOR_DYNAMIC_WIEGHTED_LB
    LARGE_DEPLOYMENT_WARNING
    LIST_OVERHEAD_QUOTA_EXCEED
    MISSING_TYPE_DEPENDENCY
    NEXT_HOP_ADDRESS_NOT_ASSIGNED
    NEXT_HOP_CANNOT_IP_FORWARD
    NEXT_HOP_INSTANCE_HAS_NO_IPV6_INTERFACE
    NEXT_HOP_INSTANCE_NOT_FOUND
    NEXT_HOP_INSTANCE_NOT_ON_NETWORK
    NEXT_HOP_NOT_RUNNING
    NOT_CRITICAL_ERROR
    NO_RESULTS_ON_PAGE
    PARTIAL_SUCCESS
    REQUIRED_TOS_AGREEMENT
    RESOURCE_IN_USE_BY_OTHER_RESOURCE_WARNING
    RESOURCE_NOT_DELETED
    SCHEMA_VALIDATION_IGNORED
    SINGLE_INSTANCE_PROPERTY_TEMPLATE
    UNDECLARED_PROPERTIES
    UNREACHABLE],
   :message string,
   :data [{:key string, :value string}]}],
 :nextHopVpnTunnel string,
 :priority integer,
 :id string,
 :kind string,
 :nextHopHub string,
 :network string,
 :asPaths
 [{:pathSegmentType
   [AS_CONFED_SEQUENCE AS_CONFED_SET AS_SEQUENCE AS_SET],
   :asLists [integer]}],
 :nextHopGateway string,
 :routeType [BGP STATIC SUBNET TRANSIT],
 :destRange string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project Route] (routes-insert project Route nil))
  ([project Route optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/routes",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Route})))

(defn routes-delete
  "Deletes the specified Route resource.
https://cloud.google.com/products/compute/v1/docs/routes/delete

project <string> Project ID for this request.
route <string> Name of the Route resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project route] (routes-delete project route nil))
  ([project route optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/routes/{route}",
       :uri-template-args {:project project, :route route},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionInstantSnapshots-list
  "Retrieves the list of InstantSnapshot resources contained within the specified region.
https://cloud.google.com/products/compute/v1/docs/regionInstantSnapshots/list

project <string> Project ID for this request.
region <string> The name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionInstantSnapshots-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instantSnapshots",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionInstantSnapshots-get
  "Returns the specified InstantSnapshot resource in the specified region.
https://cloud.google.com/products/compute/v1/docs/regionInstantSnapshots/get

project <string> Project ID for this request.
region <string> The name of the region for this request.
instantSnapshot <string> Name of the InstantSnapshot resource to return."
  [project region instantSnapshot]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instantSnapshots/{instantSnapshot}",
     :uri-template-args
     {:instantSnapshot instantSnapshot,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionInstantSnapshots-insert
  "Creates an instant snapshot in the specified region.
https://cloud.google.com/products/compute/v1/docs/regionInstantSnapshots/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
InstantSnapshot:
{:description string,
 :labels object,
 :sourceDiskId string,
 :architecture [ARCHITECTURE_UNSPECIFIED ARM64 X86_64],
 :creationTimestamp string,
 :resourceStatus {:storageSizeBytes string},
 :satisfiesPzi boolean,
 :zone string,
 :name string,
 :sourceDisk string,
 :selfLink string,
 :region string,
 :selfLinkWithId string,
 :satisfiesPzs boolean,
 :status [CREATING DELETING FAILED READY UNAVAILABLE],
 :id string,
 :kind string,
 :diskSizeGb string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region InstantSnapshot]
    (regionInstantSnapshots-insert project region InstantSnapshot nil))
  ([project region InstantSnapshot optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instantSnapshots",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstantSnapshot})))

(defn regionInstantSnapshots-setLabels
  "Sets the labels on a instantSnapshot in the given region. To learn more about labels, read the Labeling Resources documentation.
https://cloud.google.com/products/compute/v1/docs/regionInstantSnapshots/setLabels

project <string> Project ID for this request.
region <string> The region for this request.
resource <string> Name or id of the resource for this request.
RegionSetLabelsRequest:
{:labels object, :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region resource RegionSetLabelsRequest]
    (regionInstantSnapshots-setLabels
      project
      region
      resource
      RegionSetLabelsRequest
      nil))
  ([project region resource RegionSetLabelsRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instantSnapshots/{resource}/setLabels",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionSetLabelsRequest})))

(defn regionInstantSnapshots-delete
  "Deletes the specified InstantSnapshot resource. Keep in mind that deleting a single instantSnapshot might not necessarily delete all the data on that instantSnapshot. If any data on the instantSnapshot that is marked for deletion is needed for subsequent instantSnapshots, the data will be moved to the next corresponding instantSnapshot. For more information, see Deleting instantSnapshots.
https://cloud.google.com/products/compute/v1/docs/regionInstantSnapshots/delete

project <string> Project ID for this request.
region <string> The name of the region for this request.
instantSnapshot <string> Name of the InstantSnapshot resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region instantSnapshot]
    (regionInstantSnapshots-delete project region instantSnapshot nil))
  ([project region instantSnapshot optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instantSnapshots/{instantSnapshot}",
       :uri-template-args
       {:instantSnapshot instantSnapshot,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionInstantSnapshots-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/regionInstantSnapshots/getIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project region resource]
    (regionInstantSnapshots-getIamPolicy project region resource nil))
  ([project region resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instantSnapshots/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionInstantSnapshots-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/regionInstantSnapshots/setIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
RegionSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project region resource RegionSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instantSnapshots/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionSetPolicyRequest}))

(defn regionInstantSnapshots-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/regionInstantSnapshots/testIamPermissions

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project region resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instantSnapshots/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn addresses-list
  "Retrieves a list of addresses contained within the specified region.
https://cloud.google.com/products/compute/v1/docs/addresses/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (addresses-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/addresses",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn addresses-aggregatedList
  "Retrieves an aggregated list of addresses. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/addresses/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (addresses-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/addresses",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn addresses-get
  "Returns the specified address resource.
https://cloud.google.com/products/compute/v1/docs/addresses/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
address <string> Name of the address resource to return."
  [project region address]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/addresses/{address}",
     :uri-template-args
     {:address address, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn addresses-insert
  "Creates an address resource in the specified project by using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/addresses/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
Address:
{:description string,
 :labels object,
 :address string,
 :creationTimestamp string,
 :purpose
 [DNS_RESOLVER
  GCE_ENDPOINT
  IPSEC_INTERCONNECT
  NAT_AUTO
  PRIVATE_SERVICE_CONNECT
  SERVERLESS
  SHARED_LOADBALANCER_VIP
  VPC_PEERING],
 :name string,
 :selfLink string,
 :ipVersion [IPV4 IPV6 UNSPECIFIED_VERSION],
 :region string,
 :prefixLength integer,
 :status [IN_USE RESERVED RESERVING],
 :id string,
 :kind string,
 :networkTier
 [FIXED_STANDARD PREMIUM STANDARD STANDARD_OVERRIDES_FIXED_STANDARD],
 :network string,
 :addressType [EXTERNAL INTERNAL UNSPECIFIED_TYPE],
 :subnetwork string,
 :users [string],
 :labelFingerprint string,
 :ipv6EndpointType [NETLB VM]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region Address]
    (addresses-insert project region Address nil))
  ([project region Address optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/addresses",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Address})))

(defn addresses-delete
  "Deletes the specified address resource.
https://cloud.google.com/products/compute/v1/docs/addresses/delete

project <string> Project ID for this request.
region <string> Name of the region for this request.
address <string> Name of the address resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region address]
    (addresses-delete project region address nil))
  ([project region address optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/addresses/{address}",
       :uri-template-args
       {:address address, :project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn addresses-move
  "Moves the specified address resource.
https://cloud.google.com/products/compute/v1/docs/addresses/move

project <string> Source project ID which the Address is moved from.
region <string> Name of the region for this request.
address <string> Name of the address resource to move.
RegionAddressesMoveRequest:
{:destinationAddress string, :description string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region address RegionAddressesMoveRequest]
    (addresses-move
      project
      region
      address
      RegionAddressesMoveRequest
      nil))
  ([project region address RegionAddressesMoveRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/addresses/{address}/move",
       :uri-template-args
       {:address address, :project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionAddressesMoveRequest})))

(defn addresses-setLabels
  "Sets the labels on an Address. To learn more about labels, read the Labeling Resources documentation.
https://cloud.google.com/products/compute/v1/docs/addresses/setLabels

project <string> Project ID for this request.
region <string> The region for this request.
resource <string> Name or id of the resource for this request.
RegionSetLabelsRequest:
{:labels object, :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region resource RegionSetLabelsRequest]
    (addresses-setLabels
      project
      region
      resource
      RegionSetLabelsRequest
      nil))
  ([project region resource RegionSetLabelsRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/addresses/{resource}/setLabels",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionSetLabelsRequest})))

(defn nodeGroups-get
  "Returns the specified NodeGroup. Get a list of available NodeGroups by making a list() request. Note: the \"nodes\" field should not be used. Use nodeGroups.listNodes instead.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/get

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the node group to return."
  [project zone nodeGroup]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}",
     :uri-template-args
     {:nodeGroup nodeGroup, :project project, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn nodeGroups-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/setIamPolicy

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.
ZoneSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project zone resource ZoneSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :resource resource, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body ZoneSetPolicyRequest}))

(defn nodeGroups-insert
  "Creates a NodeGroup resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/insert

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
initialNodeCount <integer> Initial count of nodes in the node group.
NodeGroup:
{:description string,
 :locationHint string,
 :creationTimestamp string,
 :zone string,
 :name string,
 :shareSettings
 {:shareType
  [LOCAL ORGANIZATION SHARE_TYPE_UNSPECIFIED SPECIFIC_PROJECTS],
  :projectMap object},
 :selfLink string,
 :size integer,
 :nodeTemplate string,
 :maintenancePolicy
 [DEFAULT
  MAINTENANCE_POLICY_UNSPECIFIED
  MIGRATE_WITHIN_NODE_GROUP
  RESTART_IN_PLACE],
 :status [CREATING DELETING INVALID READY],
 :id string,
 :kind string,
 :maintenanceWindow
 {:startTime string,
  :maintenanceDuration {:seconds string, :nanos integer}},
 :autoscalingPolicy
 {:mode [MODE_UNSPECIFIED OFF ON ONLY_SCALE_OUT],
  :minNodes integer,
  :maxNodes integer},
 :maintenanceInterval [AS_NEEDED RECURRENT],
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone initialNodeCount NodeGroup]
    (nodeGroups-insert project zone initialNodeCount NodeGroup nil))
  ([project zone initialNodeCount NodeGroup optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups",
       :uri-template-args {:project project, :zone zone},
       :query-params
       (merge {:initialNodeCount initialNodeCount} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NodeGroup})))

(defn nodeGroups-patch
  "Updates the specified node group.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/patch

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the NodeGroup resource to update.
NodeGroup:
{:description string,
 :locationHint string,
 :creationTimestamp string,
 :zone string,
 :name string,
 :shareSettings
 {:shareType
  [LOCAL ORGANIZATION SHARE_TYPE_UNSPECIFIED SPECIFIC_PROJECTS],
  :projectMap object},
 :selfLink string,
 :size integer,
 :nodeTemplate string,
 :maintenancePolicy
 [DEFAULT
  MAINTENANCE_POLICY_UNSPECIFIED
  MIGRATE_WITHIN_NODE_GROUP
  RESTART_IN_PLACE],
 :status [CREATING DELETING INVALID READY],
 :id string,
 :kind string,
 :maintenanceWindow
 {:startTime string,
  :maintenanceDuration {:seconds string, :nanos integer}},
 :autoscalingPolicy
 {:mode [MODE_UNSPECIFIED OFF ON ONLY_SCALE_OUT],
  :minNodes integer,
  :maxNodes integer},
 :maintenanceInterval [AS_NEEDED RECURRENT],
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone nodeGroup NodeGroup]
    (nodeGroups-patch project zone nodeGroup NodeGroup nil))
  ([project zone nodeGroup NodeGroup optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}",
       :uri-template-args
       {:nodeGroup nodeGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NodeGroup})))

(defn nodeGroups-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/testIamPermissions

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project zone resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :resource resource, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn nodeGroups-aggregatedList
  "Retrieves an aggregated list of node groups. Note: use nodeGroups.listNodes for more details about each group. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (nodeGroups-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/nodeGroups",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeGroups-addNodes
  "Adds specified number of nodes to the node group.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/addNodes

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the NodeGroup resource.
NodeGroupsAddNodesRequest:
{:additionalNodeCount integer}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone nodeGroup NodeGroupsAddNodesRequest]
    (nodeGroups-addNodes
      project
      zone
      nodeGroup
      NodeGroupsAddNodesRequest
      nil))
  ([project zone nodeGroup NodeGroupsAddNodesRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}/addNodes",
       :uri-template-args
       {:nodeGroup nodeGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NodeGroupsAddNodesRequest})))

(defn nodeGroups-performMaintenance
  "Perform maintenance on a subset of nodes in the node group.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/performMaintenance

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the node group scoping this request.
NodeGroupsPerformMaintenanceRequest:
{:nodes [string], :startTime string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone nodeGroup NodeGroupsPerformMaintenanceRequest]
    (nodeGroups-performMaintenance
      project
      zone
      nodeGroup
      NodeGroupsPerformMaintenanceRequest
      nil))
  ([project
    zone
    nodeGroup
    NodeGroupsPerformMaintenanceRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}/performMaintenance",
       :uri-template-args
       {:nodeGroup nodeGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NodeGroupsPerformMaintenanceRequest})))

(defn nodeGroups-delete
  "Deletes the specified NodeGroup resource.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/delete

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the NodeGroup resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone nodeGroup]
    (nodeGroups-delete project zone nodeGroup nil))
  ([project zone nodeGroup optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}",
       :uri-template-args
       {:nodeGroup nodeGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn nodeGroups-deleteNodes
  "Deletes specified nodes from the node group.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/deleteNodes

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the NodeGroup resource whose nodes will be deleted.
NodeGroupsDeleteNodesRequest:
{:nodes [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone nodeGroup NodeGroupsDeleteNodesRequest]
    (nodeGroups-deleteNodes
      project
      zone
      nodeGroup
      NodeGroupsDeleteNodesRequest
      nil))
  ([project zone nodeGroup NodeGroupsDeleteNodesRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}/deleteNodes",
       :uri-template-args
       {:nodeGroup nodeGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NodeGroupsDeleteNodesRequest})))

(defn nodeGroups-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/getIamPolicy

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project zone resource]
    (nodeGroups-getIamPolicy project zone resource nil))
  ([project zone resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :resource resource, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeGroups-list
  "Retrieves a list of node groups available to the specified project. Note: use nodeGroups.listNodes for more details about each group.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/list

project <string> Project ID for this request.
zone <string> The name of the zone for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone] (nodeGroups-list project zone nil))
  ([project zone optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeGroups-setNodeTemplate
  "Updates the node template of the node group.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/setNodeTemplate

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the NodeGroup resource to update.
NodeGroupsSetNodeTemplateRequest:
{:nodeTemplate string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone nodeGroup NodeGroupsSetNodeTemplateRequest]
    (nodeGroups-setNodeTemplate
      project
      zone
      nodeGroup
      NodeGroupsSetNodeTemplateRequest
      nil))
  ([project zone nodeGroup NodeGroupsSetNodeTemplateRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}/setNodeTemplate",
       :uri-template-args
       {:nodeGroup nodeGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NodeGroupsSetNodeTemplateRequest})))

(defn nodeGroups-simulateMaintenanceEvent
  "Simulates maintenance event on specified nodes from the node group.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/simulateMaintenanceEvent

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the NodeGroup resource whose nodes will go under maintenance simulation.
NodeGroupsSimulateMaintenanceEventRequest:
{:nodes [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone nodeGroup NodeGroupsSimulateMaintenanceEventRequest]
    (nodeGroups-simulateMaintenanceEvent
      project
      zone
      nodeGroup
      NodeGroupsSimulateMaintenanceEventRequest
      nil))
  ([project
    zone
    nodeGroup
    NodeGroupsSimulateMaintenanceEventRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}/simulateMaintenanceEvent",
       :uri-template-args
       {:nodeGroup nodeGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NodeGroupsSimulateMaintenanceEventRequest})))

(defn nodeGroups-listNodes
  "Lists nodes in the node group.
https://cloud.google.com/products/compute/v1/docs/nodeGroups/listNodes

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeGroup <string> Name of the NodeGroup resource whose nodes you want to list.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone nodeGroup]
    (nodeGroups-listNodes project zone nodeGroup nil))
  ([project zone nodeGroup optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeGroups/{nodeGroup}/listNodes",
       :uri-template-args
       {:nodeGroup nodeGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn networkAttachments-get
  "Returns the specified NetworkAttachment resource in the given scope.
https://cloud.google.com/products/compute/v1/docs/networkAttachments/get

project <string> Project ID for this request.
region <string> Name of the region of this request.
networkAttachment <string> Name of the NetworkAttachment resource to return."
  [project region networkAttachment]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkAttachments/{networkAttachment}",
     :uri-template-args
     {:networkAttachment networkAttachment,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn networkAttachments-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/networkAttachments/setIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
RegionSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project region resource RegionSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkAttachments/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionSetPolicyRequest}))

(defn networkAttachments-insert
  "Creates a NetworkAttachment in the specified project in the given scope using the parameters that are included in the request.
https://cloud.google.com/products/compute/v1/docs/networkAttachments/insert

project <string> Project ID for this request.
region <string> Name of the region of this request.
NetworkAttachment:
{:description string,
 :connectionEndpoints
 [{:status
   [ACCEPTED
    CLOSED
    NEEDS_ATTENTION
    PENDING
    REJECTED
    STATUS_UNSPECIFIED],
   :projectIdOrNum string,
   :subnetwork string,
   :ipAddress string,
   :ipv6Address string,
   :secondaryIpCidrRanges [string],
   :subnetworkCidrRange string}],
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :connectionPreference [ACCEPT_AUTOMATIC ACCEPT_MANUAL INVALID],
 :selfLinkWithId string,
 :producerAcceptLists [string],
 :id string,
 :subnetworks [string],
 :kind string,
 :network string,
 :producerRejectLists [string],
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000). end_interface: MixerMutationRequestBuilder"
  ([project region NetworkAttachment]
    (networkAttachments-insert project region NetworkAttachment nil))
  ([project region NetworkAttachment optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkAttachments",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NetworkAttachment})))

(defn networkAttachments-patch
  "Patches the specified NetworkAttachment resource with the data included in the request. This method supports PATCH semantics and uses JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/networkAttachments/patch

project <string> Project ID for this request.
region <string> Name of the region for this request.
networkAttachment <string> Name of the NetworkAttachment resource to patch.
NetworkAttachment:
{:description string,
 :connectionEndpoints
 [{:status
   [ACCEPTED
    CLOSED
    NEEDS_ATTENTION
    PENDING
    REJECTED
    STATUS_UNSPECIFIED],
   :projectIdOrNum string,
   :subnetwork string,
   :ipAddress string,
   :ipv6Address string,
   :secondaryIpCidrRanges [string],
   :subnetworkCidrRange string}],
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :connectionPreference [ACCEPT_AUTOMATIC ACCEPT_MANUAL INVALID],
 :selfLinkWithId string,
 :producerAcceptLists [string],
 :id string,
 :subnetworks [string],
 :kind string,
 :network string,
 :producerRejectLists [string],
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000). end_interface: MixerMutationRequestBuilder"
  ([project region networkAttachment NetworkAttachment]
    (networkAttachments-patch
      project
      region
      networkAttachment
      NetworkAttachment
      nil))
  ([project region networkAttachment NetworkAttachment optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkAttachments/{networkAttachment}",
       :uri-template-args
       {:networkAttachment networkAttachment,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NetworkAttachment})))

(defn networkAttachments-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/networkAttachments/testIamPermissions

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project region resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkAttachments/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn networkAttachments-aggregatedList
  "Retrieves the list of all NetworkAttachment resources, regional and global, available to the specified project. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/networkAttachments/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (networkAttachments-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/networkAttachments",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn networkAttachments-delete
  "Deletes the specified NetworkAttachment in the given scope
https://cloud.google.com/products/compute/v1/docs/networkAttachments/delete

project <string> Project ID for this request.
region <string> Name of the region of this request.
networkAttachment <string> Name of the NetworkAttachment resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000). end_interface: MixerMutationRequestBuilder"
  ([project region networkAttachment]
    (networkAttachments-delete project region networkAttachment nil))
  ([project region networkAttachment optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkAttachments/{networkAttachment}",
       :uri-template-args
       {:networkAttachment networkAttachment,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn networkAttachments-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/networkAttachments/getIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project region resource]
    (networkAttachments-getIamPolicy project region resource nil))
  ([project region resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkAttachments/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn networkAttachments-list
  "Lists the NetworkAttachments for a project in the given scope.
https://cloud.google.com/products/compute/v1/docs/networkAttachments/list

project <string> Project ID for this request.
region <string> Name of the region of this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (networkAttachments-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkAttachments",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn urlMaps-invalidateCache
  "Initiates a cache invalidation operation, invalidating the specified path, scoped to the specified UrlMap. For more information, see [Invalidating cached content](/cdn/docs/invalidating-cached-content).
https://cloud.google.com/products/compute/v1/docs/urlMaps/invalidateCache

project <string> Project ID for this request.
urlMap <string> Name of the UrlMap scoping this request.
CacheInvalidationRule:
{:path string, :host string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project urlMap CacheInvalidationRule]
    (urlMaps-invalidateCache project urlMap CacheInvalidationRule nil))
  ([project urlMap CacheInvalidationRule optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/urlMaps/{urlMap}/invalidateCache",
       :uri-template-args {:project project, :urlMap urlMap},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body CacheInvalidationRule})))

(defn urlMaps-get
  "Returns the specified UrlMap resource.
https://cloud.google.com/products/compute/v1/docs/urlMaps/get

project <string> Project ID for this request.
urlMap <string> Name of the UrlMap resource to return."
  [project urlMap]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/urlMaps/{urlMap}",
     :uri-template-args {:project project, :urlMap urlMap},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn urlMaps-insert
  "Creates a UrlMap resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/urlMaps/insert

project <string> Project ID for this request.
UrlMap:
{:description string,
 :pathMatchers
 [{:name string,
   :description string,
   :defaultService string,
   :defaultRouteAction
   {:weightedBackendServices
    [{:backendService string,
      :weight integer,
      :headerAction
      {:requestHeadersToRemove [string],
       :requestHeadersToAdd [HttpHeaderOption],
       :responseHeadersToRemove [string],
       :responseHeadersToAdd [HttpHeaderOption]}}],
    :urlRewrite
    {:pathPrefixRewrite string,
     :hostRewrite string,
     :pathTemplateRewrite string},
    :timeout {:seconds string, :nanos integer},
    :retryPolicy
    {:retryConditions [string],
     :numRetries integer,
     :perTryTimeout {:seconds string, :nanos integer}},
    :requestMirrorPolicy {:backendService string},
    :corsPolicy
    {:allowOrigins [string],
     :allowOriginRegexes [string],
     :allowMethods [string],
     :allowHeaders [string],
     :exposeHeaders [string],
     :maxAge integer,
     :allowCredentials boolean,
     :disabled boolean},
    :faultInjectionPolicy
    {:delay {:fixedDelay Duration, :percentage number},
     :abort {:httpStatus integer, :percentage number}},
    :maxStreamDuration {:seconds string, :nanos integer}},
   :defaultUrlRedirect
   {:hostRedirect string,
    :pathRedirect string,
    :prefixRedirect string,
    :redirectResponseCode
    [FOUND
     MOVED_PERMANENTLY_DEFAULT
     PERMANENT_REDIRECT
     SEE_OTHER
     TEMPORARY_REDIRECT],
    :httpsRedirect boolean,
    :stripQuery boolean},
   :pathRules
   [{:service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :paths [string]}],
   :routeRules
   [{:priority integer,
     :description string,
     :matchRules
     [{:prefixMatch string,
       :fullPathMatch string,
       :regexMatch string,
       :ignoreCase boolean,
       :headerMatches
       [{:headerName string,
         :exactMatch string,
         :regexMatch string,
         :rangeMatch Int64RangeMatch,
         :presentMatch boolean,
         :prefixMatch string,
         :suffixMatch string,
         :invertMatch boolean}],
       :queryParameterMatches
       [{:name string,
         :presentMatch boolean,
         :exactMatch string,
         :regexMatch string}],
       :metadataFilters
       [{:filterMatchCriteria [MATCH_ALL MATCH_ANY NOT_SET],
         :filterLabels [MetadataFilterLabelMatch]}],
       :pathTemplateMatch string}],
     :service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :headerAction
     {:requestHeadersToRemove [string],
      :requestHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}],
      :responseHeadersToRemove [string],
      :responseHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}]}}],
   :headerAction
   {:requestHeadersToRemove [string],
    :requestHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}],
    :responseHeadersToRemove [string],
    :responseHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}]}}],
 :creationTimestamp string,
 :tests
 [{:description string,
   :host string,
   :path string,
   :headers [{:name string, :value string}],
   :service string,
   :expectedOutputUrl string,
   :expectedRedirectResponseCode integer}],
 :name string,
 :defaultUrlRedirect
 {:hostRedirect string,
  :pathRedirect string,
  :prefixRedirect string,
  :redirectResponseCode
  [FOUND
   MOVED_PERMANENTLY_DEFAULT
   PERMANENT_REDIRECT
   SEE_OTHER
   TEMPORARY_REDIRECT],
  :httpsRedirect boolean,
  :stripQuery boolean},
 :selfLink string,
 :headerAction
 {:requestHeadersToRemove [string],
  :requestHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}],
  :responseHeadersToRemove [string],
  :responseHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}]},
 :region string,
 :defaultRouteAction
 {:weightedBackendServices
  [{:backendService string,
    :weight integer,
    :headerAction
    {:requestHeadersToRemove [string],
     :requestHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}],
     :responseHeadersToRemove [string],
     :responseHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}]}}],
  :urlRewrite
  {:pathPrefixRewrite string,
   :hostRewrite string,
   :pathTemplateRewrite string},
  :timeout {:seconds string, :nanos integer},
  :retryPolicy
  {:retryConditions [string],
   :numRetries integer,
   :perTryTimeout {:seconds string, :nanos integer}},
  :requestMirrorPolicy {:backendService string},
  :corsPolicy
  {:allowOrigins [string],
   :allowOriginRegexes [string],
   :allowMethods [string],
   :allowHeaders [string],
   :exposeHeaders [string],
   :maxAge integer,
   :allowCredentials boolean,
   :disabled boolean},
  :faultInjectionPolicy
  {:delay
   {:fixedDelay {:seconds string, :nanos integer}, :percentage number},
   :abort {:httpStatus integer, :percentage number}},
  :maxStreamDuration {:seconds string, :nanos integer}},
 :defaultService string,
 :id string,
 :kind string,
 :hostRules
 [{:description string, :hosts [string], :pathMatcher string}],
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project UrlMap] (urlMaps-insert project UrlMap nil))
  ([project UrlMap optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/urlMaps",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body UrlMap})))

(defn urlMaps-validate
  "Runs static validation for the UrlMap. In particular, the tests of the provided UrlMap will be run. Calling this method does NOT create the UrlMap.
https://cloud.google.com/products/compute/v1/docs/urlMaps/validate

project <string> Project ID for this request.
urlMap <string> Name of the UrlMap resource to be validated as.
UrlMapsValidateRequest:
{:resource
 {:description string,
  :pathMatchers
  [{:name string,
    :description string,
    :defaultService string,
    :defaultRouteAction
    {:weightedBackendServices
     [{:backendService string,
       :weight integer,
       :headerAction HttpHeaderAction}],
     :urlRewrite
     {:pathPrefixRewrite string,
      :hostRewrite string,
      :pathTemplateRewrite string},
     :timeout {:seconds string, :nanos integer},
     :retryPolicy
     {:retryConditions [string],
      :numRetries integer,
      :perTryTimeout Duration},
     :requestMirrorPolicy {:backendService string},
     :corsPolicy
     {:allowOrigins [string],
      :allowOriginRegexes [string],
      :allowMethods [string],
      :allowHeaders [string],
      :exposeHeaders [string],
      :maxAge integer,
      :allowCredentials boolean,
      :disabled boolean},
     :faultInjectionPolicy
     {:delay HttpFaultDelay, :abort HttpFaultAbort},
     :maxStreamDuration {:seconds string, :nanos integer}},
    :defaultUrlRedirect
    {:hostRedirect string,
     :pathRedirect string,
     :prefixRedirect string,
     :redirectResponseCode
     [FOUND
      MOVED_PERMANENTLY_DEFAULT
      PERMANENT_REDIRECT
      SEE_OTHER
      TEMPORARY_REDIRECT],
     :httpsRedirect boolean,
     :stripQuery boolean},
    :pathRules
    [{:service string,
      :routeAction
      {:weightedBackendServices [WeightedBackendService],
       :urlRewrite UrlRewrite,
       :timeout Duration,
       :retryPolicy HttpRetryPolicy,
       :requestMirrorPolicy RequestMirrorPolicy,
       :corsPolicy CorsPolicy,
       :faultInjectionPolicy HttpFaultInjection,
       :maxStreamDuration Duration},
      :urlRedirect
      {:hostRedirect string,
       :pathRedirect string,
       :prefixRedirect string,
       :redirectResponseCode
       [FOUND
        MOVED_PERMANENTLY_DEFAULT
        PERMANENT_REDIRECT
        SEE_OTHER
        TEMPORARY_REDIRECT],
       :httpsRedirect boolean,
       :stripQuery boolean},
      :paths [string]}],
    :routeRules
    [{:priority integer,
      :description string,
      :matchRules
      [{:prefixMatch string,
        :fullPathMatch string,
        :regexMatch string,
        :ignoreCase boolean,
        :headerMatches [HttpHeaderMatch],
        :queryParameterMatches [HttpQueryParameterMatch],
        :metadataFilters [MetadataFilter],
        :pathTemplateMatch string}],
      :service string,
      :routeAction
      {:weightedBackendServices [WeightedBackendService],
       :urlRewrite UrlRewrite,
       :timeout Duration,
       :retryPolicy HttpRetryPolicy,
       :requestMirrorPolicy RequestMirrorPolicy,
       :corsPolicy CorsPolicy,
       :faultInjectionPolicy HttpFaultInjection,
       :maxStreamDuration Duration},
      :urlRedirect
      {:hostRedirect string,
       :pathRedirect string,
       :prefixRedirect string,
       :redirectResponseCode
       [FOUND
        MOVED_PERMANENTLY_DEFAULT
        PERMANENT_REDIRECT
        SEE_OTHER
        TEMPORARY_REDIRECT],
       :httpsRedirect boolean,
       :stripQuery boolean},
      :headerAction
      {:requestHeadersToRemove [string],
       :requestHeadersToAdd [HttpHeaderOption],
       :responseHeadersToRemove [string],
       :responseHeadersToAdd [HttpHeaderOption]}}],
    :headerAction
    {:requestHeadersToRemove [string],
     :requestHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}],
     :responseHeadersToRemove [string],
     :responseHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}]}}],
  :creationTimestamp string,
  :tests
  [{:description string,
    :host string,
    :path string,
    :headers [{:name string, :value string}],
    :service string,
    :expectedOutputUrl string,
    :expectedRedirectResponseCode integer}],
  :name string,
  :defaultUrlRedirect
  {:hostRedirect string,
   :pathRedirect string,
   :prefixRedirect string,
   :redirectResponseCode
   [FOUND
    MOVED_PERMANENTLY_DEFAULT
    PERMANENT_REDIRECT
    SEE_OTHER
    TEMPORARY_REDIRECT],
   :httpsRedirect boolean,
   :stripQuery boolean},
  :selfLink string,
  :headerAction
  {:requestHeadersToRemove [string],
   :requestHeadersToAdd
   [{:headerName string, :headerValue string, :replace boolean}],
   :responseHeadersToRemove [string],
   :responseHeadersToAdd
   [{:headerName string, :headerValue string, :replace boolean}]},
  :region string,
  :defaultRouteAction
  {:weightedBackendServices
   [{:backendService string,
     :weight integer,
     :headerAction
     {:requestHeadersToRemove [string],
      :requestHeadersToAdd [HttpHeaderOption],
      :responseHeadersToRemove [string],
      :responseHeadersToAdd [HttpHeaderOption]}}],
   :urlRewrite
   {:pathPrefixRewrite string,
    :hostRewrite string,
    :pathTemplateRewrite string},
   :timeout {:seconds string, :nanos integer},
   :retryPolicy
   {:retryConditions [string],
    :numRetries integer,
    :perTryTimeout {:seconds string, :nanos integer}},
   :requestMirrorPolicy {:backendService string},
   :corsPolicy
   {:allowOrigins [string],
    :allowOriginRegexes [string],
    :allowMethods [string],
    :allowHeaders [string],
    :exposeHeaders [string],
    :maxAge integer,
    :allowCredentials boolean,
    :disabled boolean},
   :faultInjectionPolicy
   {:delay {:fixedDelay Duration, :percentage number},
    :abort {:httpStatus integer, :percentage number}},
   :maxStreamDuration {:seconds string, :nanos integer}},
  :defaultService string,
  :id string,
  :kind string,
  :hostRules
  [{:description string, :hosts [string], :pathMatcher string}],
  :fingerprint string},
 :loadBalancingSchemes
 [[EXTERNAL EXTERNAL_MANAGED LOAD_BALANCING_SCHEME_UNSPECIFIED]]}"
  [project urlMap UrlMapsValidateRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/urlMaps/{urlMap}/validate",
     :uri-template-args {:project project, :urlMap urlMap},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body UrlMapsValidateRequest}))

(defn urlMaps-patch
  "Patches the specified UrlMap resource with the data included in the request. This method supports PATCH semantics and uses the JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/urlMaps/patch

project <string> Project ID for this request.
urlMap <string> Name of the UrlMap resource to patch.
UrlMap:
{:description string,
 :pathMatchers
 [{:name string,
   :description string,
   :defaultService string,
   :defaultRouteAction
   {:weightedBackendServices
    [{:backendService string,
      :weight integer,
      :headerAction
      {:requestHeadersToRemove [string],
       :requestHeadersToAdd [HttpHeaderOption],
       :responseHeadersToRemove [string],
       :responseHeadersToAdd [HttpHeaderOption]}}],
    :urlRewrite
    {:pathPrefixRewrite string,
     :hostRewrite string,
     :pathTemplateRewrite string},
    :timeout {:seconds string, :nanos integer},
    :retryPolicy
    {:retryConditions [string],
     :numRetries integer,
     :perTryTimeout {:seconds string, :nanos integer}},
    :requestMirrorPolicy {:backendService string},
    :corsPolicy
    {:allowOrigins [string],
     :allowOriginRegexes [string],
     :allowMethods [string],
     :allowHeaders [string],
     :exposeHeaders [string],
     :maxAge integer,
     :allowCredentials boolean,
     :disabled boolean},
    :faultInjectionPolicy
    {:delay {:fixedDelay Duration, :percentage number},
     :abort {:httpStatus integer, :percentage number}},
    :maxStreamDuration {:seconds string, :nanos integer}},
   :defaultUrlRedirect
   {:hostRedirect string,
    :pathRedirect string,
    :prefixRedirect string,
    :redirectResponseCode
    [FOUND
     MOVED_PERMANENTLY_DEFAULT
     PERMANENT_REDIRECT
     SEE_OTHER
     TEMPORARY_REDIRECT],
    :httpsRedirect boolean,
    :stripQuery boolean},
   :pathRules
   [{:service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :paths [string]}],
   :routeRules
   [{:priority integer,
     :description string,
     :matchRules
     [{:prefixMatch string,
       :fullPathMatch string,
       :regexMatch string,
       :ignoreCase boolean,
       :headerMatches
       [{:headerName string,
         :exactMatch string,
         :regexMatch string,
         :rangeMatch Int64RangeMatch,
         :presentMatch boolean,
         :prefixMatch string,
         :suffixMatch string,
         :invertMatch boolean}],
       :queryParameterMatches
       [{:name string,
         :presentMatch boolean,
         :exactMatch string,
         :regexMatch string}],
       :metadataFilters
       [{:filterMatchCriteria [MATCH_ALL MATCH_ANY NOT_SET],
         :filterLabels [MetadataFilterLabelMatch]}],
       :pathTemplateMatch string}],
     :service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :headerAction
     {:requestHeadersToRemove [string],
      :requestHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}],
      :responseHeadersToRemove [string],
      :responseHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}]}}],
   :headerAction
   {:requestHeadersToRemove [string],
    :requestHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}],
    :responseHeadersToRemove [string],
    :responseHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}]}}],
 :creationTimestamp string,
 :tests
 [{:description string,
   :host string,
   :path string,
   :headers [{:name string, :value string}],
   :service string,
   :expectedOutputUrl string,
   :expectedRedirectResponseCode integer}],
 :name string,
 :defaultUrlRedirect
 {:hostRedirect string,
  :pathRedirect string,
  :prefixRedirect string,
  :redirectResponseCode
  [FOUND
   MOVED_PERMANENTLY_DEFAULT
   PERMANENT_REDIRECT
   SEE_OTHER
   TEMPORARY_REDIRECT],
  :httpsRedirect boolean,
  :stripQuery boolean},
 :selfLink string,
 :headerAction
 {:requestHeadersToRemove [string],
  :requestHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}],
  :responseHeadersToRemove [string],
  :responseHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}]},
 :region string,
 :defaultRouteAction
 {:weightedBackendServices
  [{:backendService string,
    :weight integer,
    :headerAction
    {:requestHeadersToRemove [string],
     :requestHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}],
     :responseHeadersToRemove [string],
     :responseHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}]}}],
  :urlRewrite
  {:pathPrefixRewrite string,
   :hostRewrite string,
   :pathTemplateRewrite string},
  :timeout {:seconds string, :nanos integer},
  :retryPolicy
  {:retryConditions [string],
   :numRetries integer,
   :perTryTimeout {:seconds string, :nanos integer}},
  :requestMirrorPolicy {:backendService string},
  :corsPolicy
  {:allowOrigins [string],
   :allowOriginRegexes [string],
   :allowMethods [string],
   :allowHeaders [string],
   :exposeHeaders [string],
   :maxAge integer,
   :allowCredentials boolean,
   :disabled boolean},
  :faultInjectionPolicy
  {:delay
   {:fixedDelay {:seconds string, :nanos integer}, :percentage number},
   :abort {:httpStatus integer, :percentage number}},
  :maxStreamDuration {:seconds string, :nanos integer}},
 :defaultService string,
 :id string,
 :kind string,
 :hostRules
 [{:description string, :hosts [string], :pathMatcher string}],
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project urlMap UrlMap] (urlMaps-patch project urlMap UrlMap nil))
  ([project urlMap UrlMap optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/urlMaps/{urlMap}",
       :uri-template-args {:project project, :urlMap urlMap},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body UrlMap})))

(defn urlMaps-aggregatedList
  "Retrieves the list of all UrlMap resources, regional and global, available to the specified project. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/urlMaps/aggregatedList

project <string> Name of the project scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (urlMaps-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/urlMaps",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn urlMaps-update
  "Updates the specified UrlMap resource with the data included in the request.
https://cloud.google.com/products/compute/v1/docs/urlMaps/update

project <string> Project ID for this request.
urlMap <string> Name of the UrlMap resource to update.
UrlMap:
{:description string,
 :pathMatchers
 [{:name string,
   :description string,
   :defaultService string,
   :defaultRouteAction
   {:weightedBackendServices
    [{:backendService string,
      :weight integer,
      :headerAction
      {:requestHeadersToRemove [string],
       :requestHeadersToAdd [HttpHeaderOption],
       :responseHeadersToRemove [string],
       :responseHeadersToAdd [HttpHeaderOption]}}],
    :urlRewrite
    {:pathPrefixRewrite string,
     :hostRewrite string,
     :pathTemplateRewrite string},
    :timeout {:seconds string, :nanos integer},
    :retryPolicy
    {:retryConditions [string],
     :numRetries integer,
     :perTryTimeout {:seconds string, :nanos integer}},
    :requestMirrorPolicy {:backendService string},
    :corsPolicy
    {:allowOrigins [string],
     :allowOriginRegexes [string],
     :allowMethods [string],
     :allowHeaders [string],
     :exposeHeaders [string],
     :maxAge integer,
     :allowCredentials boolean,
     :disabled boolean},
    :faultInjectionPolicy
    {:delay {:fixedDelay Duration, :percentage number},
     :abort {:httpStatus integer, :percentage number}},
    :maxStreamDuration {:seconds string, :nanos integer}},
   :defaultUrlRedirect
   {:hostRedirect string,
    :pathRedirect string,
    :prefixRedirect string,
    :redirectResponseCode
    [FOUND
     MOVED_PERMANENTLY_DEFAULT
     PERMANENT_REDIRECT
     SEE_OTHER
     TEMPORARY_REDIRECT],
    :httpsRedirect boolean,
    :stripQuery boolean},
   :pathRules
   [{:service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :paths [string]}],
   :routeRules
   [{:priority integer,
     :description string,
     :matchRules
     [{:prefixMatch string,
       :fullPathMatch string,
       :regexMatch string,
       :ignoreCase boolean,
       :headerMatches
       [{:headerName string,
         :exactMatch string,
         :regexMatch string,
         :rangeMatch Int64RangeMatch,
         :presentMatch boolean,
         :prefixMatch string,
         :suffixMatch string,
         :invertMatch boolean}],
       :queryParameterMatches
       [{:name string,
         :presentMatch boolean,
         :exactMatch string,
         :regexMatch string}],
       :metadataFilters
       [{:filterMatchCriteria [MATCH_ALL MATCH_ANY NOT_SET],
         :filterLabels [MetadataFilterLabelMatch]}],
       :pathTemplateMatch string}],
     :service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :headerAction
     {:requestHeadersToRemove [string],
      :requestHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}],
      :responseHeadersToRemove [string],
      :responseHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}]}}],
   :headerAction
   {:requestHeadersToRemove [string],
    :requestHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}],
    :responseHeadersToRemove [string],
    :responseHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}]}}],
 :creationTimestamp string,
 :tests
 [{:description string,
   :host string,
   :path string,
   :headers [{:name string, :value string}],
   :service string,
   :expectedOutputUrl string,
   :expectedRedirectResponseCode integer}],
 :name string,
 :defaultUrlRedirect
 {:hostRedirect string,
  :pathRedirect string,
  :prefixRedirect string,
  :redirectResponseCode
  [FOUND
   MOVED_PERMANENTLY_DEFAULT
   PERMANENT_REDIRECT
   SEE_OTHER
   TEMPORARY_REDIRECT],
  :httpsRedirect boolean,
  :stripQuery boolean},
 :selfLink string,
 :headerAction
 {:requestHeadersToRemove [string],
  :requestHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}],
  :responseHeadersToRemove [string],
  :responseHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}]},
 :region string,
 :defaultRouteAction
 {:weightedBackendServices
  [{:backendService string,
    :weight integer,
    :headerAction
    {:requestHeadersToRemove [string],
     :requestHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}],
     :responseHeadersToRemove [string],
     :responseHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}]}}],
  :urlRewrite
  {:pathPrefixRewrite string,
   :hostRewrite string,
   :pathTemplateRewrite string},
  :timeout {:seconds string, :nanos integer},
  :retryPolicy
  {:retryConditions [string],
   :numRetries integer,
   :perTryTimeout {:seconds string, :nanos integer}},
  :requestMirrorPolicy {:backendService string},
  :corsPolicy
  {:allowOrigins [string],
   :allowOriginRegexes [string],
   :allowMethods [string],
   :allowHeaders [string],
   :exposeHeaders [string],
   :maxAge integer,
   :allowCredentials boolean,
   :disabled boolean},
  :faultInjectionPolicy
  {:delay
   {:fixedDelay {:seconds string, :nanos integer}, :percentage number},
   :abort {:httpStatus integer, :percentage number}},
  :maxStreamDuration {:seconds string, :nanos integer}},
 :defaultService string,
 :id string,
 :kind string,
 :hostRules
 [{:description string, :hosts [string], :pathMatcher string}],
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project urlMap UrlMap] (urlMaps-update project urlMap UrlMap nil))
  ([project urlMap UrlMap optional]
    (client/api-request
      {:method "PUT",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/urlMaps/{urlMap}",
       :uri-template-args {:project project, :urlMap urlMap},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body UrlMap})))

(defn urlMaps-delete
  "Deletes the specified UrlMap resource.
https://cloud.google.com/products/compute/v1/docs/urlMaps/delete

project <string> Project ID for this request.
urlMap <string> Name of the UrlMap resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project urlMap] (urlMaps-delete project urlMap nil))
  ([project urlMap optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/urlMaps/{urlMap}",
       :uri-template-args {:project project, :urlMap urlMap},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn urlMaps-list
  "Retrieves the list of UrlMap resources available to the specified project.
https://cloud.google.com/products/compute/v1/docs/urlMaps/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (urlMaps-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/urlMaps",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeTemplates-list
  "Retrieves a list of node templates available to the specified project.
https://cloud.google.com/products/compute/v1/docs/nodeTemplates/list

project <string> Project ID for this request.
region <string> The name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (nodeTemplates-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/nodeTemplates",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeTemplates-aggregatedList
  "Retrieves an aggregated list of node templates. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/nodeTemplates/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (nodeTemplates-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/nodeTemplates",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeTemplates-get
  "Returns the specified node template.
https://cloud.google.com/products/compute/v1/docs/nodeTemplates/get

project <string> Project ID for this request.
region <string> The name of the region for this request.
nodeTemplate <string> Name of the node template to return."
  [project region nodeTemplate]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/nodeTemplates/{nodeTemplate}",
     :uri-template-args
     {:nodeTemplate nodeTemplate, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn nodeTemplates-insert
  "Creates a NodeTemplate resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/nodeTemplates/insert

project <string> Project ID for this request.
region <string> The name of the region for this request.
NodeTemplate:
{:description string,
 :nodeAffinityLabels object,
 :nodeTypeFlexibility {:cpus string, :memory string, :localSsd string},
 :cpuOvercommitType [CPU_OVERCOMMIT_TYPE_UNSPECIFIED ENABLED NONE],
 :nodeType string,
 :creationTimestamp string,
 :name string,
 :statusMessage string,
 :selfLink string,
 :region string,
 :status [CREATING DELETING INVALID READY],
 :id string,
 :kind string,
 :disks [{:diskType string, :diskSizeGb integer, :diskCount integer}],
 :serverBinding
 {:type
  [RESTART_NODE_ON_ANY_SERVER
   RESTART_NODE_ON_MINIMAL_SERVERS
   SERVER_BINDING_TYPE_UNSPECIFIED]},
 :accelerators [{:acceleratorType string, :acceleratorCount integer}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region NodeTemplate]
    (nodeTemplates-insert project region NodeTemplate nil))
  ([project region NodeTemplate optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/nodeTemplates",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NodeTemplate})))

(defn nodeTemplates-delete
  "Deletes the specified NodeTemplate resource.
https://cloud.google.com/products/compute/v1/docs/nodeTemplates/delete

project <string> Project ID for this request.
region <string> The name of the region for this request.
nodeTemplate <string> Name of the NodeTemplate resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region nodeTemplate]
    (nodeTemplates-delete project region nodeTemplate nil))
  ([project region nodeTemplate optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/nodeTemplates/{nodeTemplate}",
       :uri-template-args
       {:nodeTemplate nodeTemplate, :project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn nodeTemplates-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/nodeTemplates/getIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project region resource]
    (nodeTemplates-getIamPolicy project region resource nil))
  ([project region resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/nodeTemplates/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeTemplates-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/nodeTemplates/setIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
RegionSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project region resource RegionSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/nodeTemplates/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionSetPolicyRequest}))

(defn nodeTemplates-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/nodeTemplates/testIamPermissions

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project region resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/nodeTemplates/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn firewalls-list
  "Retrieves the list of firewall rules available to the specified project.
https://cloud.google.com/products/compute/v1/docs/firewalls/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (firewalls-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/firewalls",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn firewalls-get
  "Returns the specified firewall.
https://cloud.google.com/products/compute/v1/docs/firewalls/get

project <string> Project ID for this request.
firewall <string> Name of the firewall rule to return."
  [project firewall]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/firewalls/{firewall}",
     :uri-template-args {:firewall firewall, :project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn firewalls-insert
  "Creates a firewall rule in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/firewalls/insert

project <string> Project ID for this request.
Firewall:
{:description string,
 :disabled boolean,
 :creationTimestamp string,
 :name string,
 :destinationRanges [string],
 :selfLink string,
 :targetTags [string],
 :sourceTags [string],
 :priority integer,
 :id string,
 :kind string,
 :denied [{:IPProtocol string, :ports [string]}],
 :sourceServiceAccounts [string],
 :targetServiceAccounts [string],
 :network string,
 :logConfig
 {:enable boolean,
  :metadata [EXCLUDE_ALL_METADATA INCLUDE_ALL_METADATA]},
 :direction [EGRESS INGRESS],
 :sourceRanges [string],
 :allowed [{:IPProtocol string, :ports [string]}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project Firewall] (firewalls-insert project Firewall nil))
  ([project Firewall optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/firewalls",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Firewall})))

(defn firewalls-delete
  "Deletes the specified firewall.
https://cloud.google.com/products/compute/v1/docs/firewalls/delete

project <string> Project ID for this request.
firewall <string> Name of the firewall rule to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project firewall] (firewalls-delete project firewall nil))
  ([project firewall optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/firewalls/{firewall}",
       :uri-template-args {:firewall firewall, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn firewalls-update
  "Updates the specified firewall rule with the data included in the request. Note that all fields will be updated if using PUT, even fields that are not specified. To update individual fields, please use PATCH instead.
https://cloud.google.com/products/compute/v1/docs/firewalls/update

project <string> Project ID for this request.
firewall <string> Name of the firewall rule to update.
Firewall:
{:description string,
 :disabled boolean,
 :creationTimestamp string,
 :name string,
 :destinationRanges [string],
 :selfLink string,
 :targetTags [string],
 :sourceTags [string],
 :priority integer,
 :id string,
 :kind string,
 :denied [{:IPProtocol string, :ports [string]}],
 :sourceServiceAccounts [string],
 :targetServiceAccounts [string],
 :network string,
 :logConfig
 {:enable boolean,
  :metadata [EXCLUDE_ALL_METADATA INCLUDE_ALL_METADATA]},
 :direction [EGRESS INGRESS],
 :sourceRanges [string],
 :allowed [{:IPProtocol string, :ports [string]}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project firewall Firewall]
    (firewalls-update project firewall Firewall nil))
  ([project firewall Firewall optional]
    (client/api-request
      {:method "PUT",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/firewalls/{firewall}",
       :uri-template-args {:firewall firewall, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Firewall})))

(defn firewalls-patch
  "Updates the specified firewall rule with the data included in the request. This method supports PATCH semantics and uses the JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/firewalls/patch

project <string> Project ID for this request.
firewall <string> Name of the firewall rule to patch.
Firewall:
{:description string,
 :disabled boolean,
 :creationTimestamp string,
 :name string,
 :destinationRanges [string],
 :selfLink string,
 :targetTags [string],
 :sourceTags [string],
 :priority integer,
 :id string,
 :kind string,
 :denied [{:IPProtocol string, :ports [string]}],
 :sourceServiceAccounts [string],
 :targetServiceAccounts [string],
 :network string,
 :logConfig
 {:enable boolean,
  :metadata [EXCLUDE_ALL_METADATA INCLUDE_ALL_METADATA]},
 :direction [EGRESS INGRESS],
 :sourceRanges [string],
 :allowed [{:IPProtocol string, :ports [string]}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project firewall Firewall]
    (firewalls-patch project firewall Firewall nil))
  ([project firewall Firewall optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/firewalls/{firewall}",
       :uri-template-args {:firewall firewall, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Firewall})))

(defn instanceGroups-get
  "Returns the specified zonal instance group. Get a list of available zonal instance groups by making a list() request. For managed instance groups, use the instanceGroupManagers or regionInstanceGroupManagers methods instead.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/get

project <string> Project ID for this request.
zone <string> The name of the zone where the instance group is located.
instanceGroup <string> The name of the instance group."
  [project zone instanceGroup]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}",
     :uri-template-args
     {:instanceGroup instanceGroup, :project project, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn instanceGroups-insert
  "Creates an instance group in the specified project using the parameters that are included in the request.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/insert

project <string> Project ID for this request.
zone <string> The name of the zone where you want to create the instance group.
InstanceGroup:
{:description string,
 :creationTimestamp string,
 :zone string,
 :name string,
 :selfLink string,
 :size integer,
 :region string,
 :id string,
 :kind string,
 :network string,
 :fingerprint string,
 :subnetwork string,
 :namedPorts [{:name string, :port integer}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone InstanceGroup]
    (instanceGroups-insert project zone InstanceGroup nil))
  ([project zone InstanceGroup optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroups",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroup})))

(defn instanceGroups-addInstances
  "Adds a list of instances to the specified instance group. All of the instances in the instance group must be in the same network/subnetwork. Read Adding instances for more information.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/addInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the instance group is located.
instanceGroup <string> The name of the instance group where you are adding instances.
InstanceGroupsAddInstancesRequest:
{:instances [{:instance string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone instanceGroup InstanceGroupsAddInstancesRequest]
    (instanceGroups-addInstances
      project
      zone
      instanceGroup
      InstanceGroupsAddInstancesRequest
      nil))
  ([project
    zone
    instanceGroup
    InstanceGroupsAddInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}/addInstances",
       :uri-template-args
       {:instanceGroup instanceGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupsAddInstancesRequest})))

(defn instanceGroups-aggregatedList
  "Retrieves the list of instance groups and sorts them by zone. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (instanceGroups-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/instanceGroups",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceGroups-delete
  "Deletes the specified instance group. The instances in the group are not deleted. Note that instance group must not belong to a backend service. Read Deleting an instance group for more information.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/delete

project <string> Project ID for this request.
zone <string> The name of the zone where the instance group is located.
instanceGroup <string> The name of the instance group to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone instanceGroup]
    (instanceGroups-delete project zone instanceGroup nil))
  ([project zone instanceGroup optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}",
       :uri-template-args
       {:instanceGroup instanceGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn instanceGroups-setNamedPorts
  "Sets the named ports for the specified instance group.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/setNamedPorts

project <string> Project ID for this request.
zone <string> The name of the zone where the instance group is located.
instanceGroup <string> The name of the instance group where the named ports are updated.
InstanceGroupsSetNamedPortsRequest:
{:namedPorts [{:name string, :port integer}], :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone instanceGroup InstanceGroupsSetNamedPortsRequest]
    (instanceGroups-setNamedPorts
      project
      zone
      instanceGroup
      InstanceGroupsSetNamedPortsRequest
      nil))
  ([project
    zone
    instanceGroup
    InstanceGroupsSetNamedPortsRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}/setNamedPorts",
       :uri-template-args
       {:instanceGroup instanceGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupsSetNamedPortsRequest})))

(defn instanceGroups-list
  "Retrieves the list of zonal instance group resources contained within the specified zone. For managed instance groups, use the instanceGroupManagers or regionInstanceGroupManagers methods instead.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/list

project <string> Project ID for this request.
zone <string> The name of the zone where the instance group is located.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone] (instanceGroups-list project zone nil))
  ([project zone optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroups",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceGroups-listInstances
  "Lists the instances in the specified instance group. The orderBy query parameter is not supported. The filter query parameter is supported, but only for expressions that use `eq` (equal) or `ne` (not equal) operators.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/listInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the instance group is located.
instanceGroup <string> The name of the instance group from which you want to generate a list of included instances.
InstanceGroupsListInstancesRequest:
{:instanceState [ALL RUNNING]}

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone instanceGroup InstanceGroupsListInstancesRequest]
    (instanceGroups-listInstances
      project
      zone
      instanceGroup
      InstanceGroupsListInstancesRequest
      nil))
  ([project
    zone
    instanceGroup
    InstanceGroupsListInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}/listInstances",
       :uri-template-args
       {:instanceGroup instanceGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"],
       :body InstanceGroupsListInstancesRequest})))

(defn instanceGroups-removeInstances
  "Removes one or more instances from the specified instance group, but does not delete those instances. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration before the VM instance is removed or deleted.
https://cloud.google.com/products/compute/v1/docs/instanceGroups/removeInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the instance group is located.
instanceGroup <string> The name of the instance group where the specified instances will be removed.
InstanceGroupsRemoveInstancesRequest:
{:instances [{:instance string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone instanceGroup InstanceGroupsRemoveInstancesRequest]
    (instanceGroups-removeInstances
      project
      zone
      instanceGroup
      InstanceGroupsRemoveInstancesRequest
      nil))
  ([project
    zone
    instanceGroup
    InstanceGroupsRemoveInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}/removeInstances",
       :uri-template-args
       {:instanceGroup instanceGroup, :project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupsRemoveInstancesRequest})))

(defn regionUrlMaps-list
  "Retrieves the list of UrlMap resources available to the specified project in the specified region.
https://cloud.google.com/products/compute/v1/docs/regionUrlMaps/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionUrlMaps-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/urlMaps",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionUrlMaps-get
  "Returns the specified UrlMap resource.
https://cloud.google.com/products/compute/v1/docs/regionUrlMaps/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
urlMap <string> Name of the UrlMap resource to return."
  [project region urlMap]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/urlMaps/{urlMap}",
     :uri-template-args
     {:project project, :region region, :urlMap urlMap},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionUrlMaps-insert
  "Creates a UrlMap resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionUrlMaps/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
UrlMap:
{:description string,
 :pathMatchers
 [{:name string,
   :description string,
   :defaultService string,
   :defaultRouteAction
   {:weightedBackendServices
    [{:backendService string,
      :weight integer,
      :headerAction
      {:requestHeadersToRemove [string],
       :requestHeadersToAdd [HttpHeaderOption],
       :responseHeadersToRemove [string],
       :responseHeadersToAdd [HttpHeaderOption]}}],
    :urlRewrite
    {:pathPrefixRewrite string,
     :hostRewrite string,
     :pathTemplateRewrite string},
    :timeout {:seconds string, :nanos integer},
    :retryPolicy
    {:retryConditions [string],
     :numRetries integer,
     :perTryTimeout {:seconds string, :nanos integer}},
    :requestMirrorPolicy {:backendService string},
    :corsPolicy
    {:allowOrigins [string],
     :allowOriginRegexes [string],
     :allowMethods [string],
     :allowHeaders [string],
     :exposeHeaders [string],
     :maxAge integer,
     :allowCredentials boolean,
     :disabled boolean},
    :faultInjectionPolicy
    {:delay {:fixedDelay Duration, :percentage number},
     :abort {:httpStatus integer, :percentage number}},
    :maxStreamDuration {:seconds string, :nanos integer}},
   :defaultUrlRedirect
   {:hostRedirect string,
    :pathRedirect string,
    :prefixRedirect string,
    :redirectResponseCode
    [FOUND
     MOVED_PERMANENTLY_DEFAULT
     PERMANENT_REDIRECT
     SEE_OTHER
     TEMPORARY_REDIRECT],
    :httpsRedirect boolean,
    :stripQuery boolean},
   :pathRules
   [{:service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :paths [string]}],
   :routeRules
   [{:priority integer,
     :description string,
     :matchRules
     [{:prefixMatch string,
       :fullPathMatch string,
       :regexMatch string,
       :ignoreCase boolean,
       :headerMatches
       [{:headerName string,
         :exactMatch string,
         :regexMatch string,
         :rangeMatch Int64RangeMatch,
         :presentMatch boolean,
         :prefixMatch string,
         :suffixMatch string,
         :invertMatch boolean}],
       :queryParameterMatches
       [{:name string,
         :presentMatch boolean,
         :exactMatch string,
         :regexMatch string}],
       :metadataFilters
       [{:filterMatchCriteria [MATCH_ALL MATCH_ANY NOT_SET],
         :filterLabels [MetadataFilterLabelMatch]}],
       :pathTemplateMatch string}],
     :service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :headerAction
     {:requestHeadersToRemove [string],
      :requestHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}],
      :responseHeadersToRemove [string],
      :responseHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}]}}],
   :headerAction
   {:requestHeadersToRemove [string],
    :requestHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}],
    :responseHeadersToRemove [string],
    :responseHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}]}}],
 :creationTimestamp string,
 :tests
 [{:description string,
   :host string,
   :path string,
   :headers [{:name string, :value string}],
   :service string,
   :expectedOutputUrl string,
   :expectedRedirectResponseCode integer}],
 :name string,
 :defaultUrlRedirect
 {:hostRedirect string,
  :pathRedirect string,
  :prefixRedirect string,
  :redirectResponseCode
  [FOUND
   MOVED_PERMANENTLY_DEFAULT
   PERMANENT_REDIRECT
   SEE_OTHER
   TEMPORARY_REDIRECT],
  :httpsRedirect boolean,
  :stripQuery boolean},
 :selfLink string,
 :headerAction
 {:requestHeadersToRemove [string],
  :requestHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}],
  :responseHeadersToRemove [string],
  :responseHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}]},
 :region string,
 :defaultRouteAction
 {:weightedBackendServices
  [{:backendService string,
    :weight integer,
    :headerAction
    {:requestHeadersToRemove [string],
     :requestHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}],
     :responseHeadersToRemove [string],
     :responseHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}]}}],
  :urlRewrite
  {:pathPrefixRewrite string,
   :hostRewrite string,
   :pathTemplateRewrite string},
  :timeout {:seconds string, :nanos integer},
  :retryPolicy
  {:retryConditions [string],
   :numRetries integer,
   :perTryTimeout {:seconds string, :nanos integer}},
  :requestMirrorPolicy {:backendService string},
  :corsPolicy
  {:allowOrigins [string],
   :allowOriginRegexes [string],
   :allowMethods [string],
   :allowHeaders [string],
   :exposeHeaders [string],
   :maxAge integer,
   :allowCredentials boolean,
   :disabled boolean},
  :faultInjectionPolicy
  {:delay
   {:fixedDelay {:seconds string, :nanos integer}, :percentage number},
   :abort {:httpStatus integer, :percentage number}},
  :maxStreamDuration {:seconds string, :nanos integer}},
 :defaultService string,
 :id string,
 :kind string,
 :hostRules
 [{:description string, :hosts [string], :pathMatcher string}],
 :fingerprint string}

optional:
requestId <string> begin_interface: MixerMutationRequestBuilder Request ID to support idempotency."
  ([project region UrlMap]
    (regionUrlMaps-insert project region UrlMap nil))
  ([project region UrlMap optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/urlMaps",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body UrlMap})))

(defn regionUrlMaps-delete
  "Deletes the specified UrlMap resource.
https://cloud.google.com/products/compute/v1/docs/regionUrlMaps/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
urlMap <string> Name of the UrlMap resource to delete.

optional:
requestId <string> begin_interface: MixerMutationRequestBuilder Request ID to support idempotency."
  ([project region urlMap]
    (regionUrlMaps-delete project region urlMap nil))
  ([project region urlMap optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/urlMaps/{urlMap}",
       :uri-template-args
       {:project project, :region region, :urlMap urlMap},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionUrlMaps-update
  "Updates the specified UrlMap resource with the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionUrlMaps/update

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
urlMap <string> Name of the UrlMap resource to update.
UrlMap:
{:description string,
 :pathMatchers
 [{:name string,
   :description string,
   :defaultService string,
   :defaultRouteAction
   {:weightedBackendServices
    [{:backendService string,
      :weight integer,
      :headerAction
      {:requestHeadersToRemove [string],
       :requestHeadersToAdd [HttpHeaderOption],
       :responseHeadersToRemove [string],
       :responseHeadersToAdd [HttpHeaderOption]}}],
    :urlRewrite
    {:pathPrefixRewrite string,
     :hostRewrite string,
     :pathTemplateRewrite string},
    :timeout {:seconds string, :nanos integer},
    :retryPolicy
    {:retryConditions [string],
     :numRetries integer,
     :perTryTimeout {:seconds string, :nanos integer}},
    :requestMirrorPolicy {:backendService string},
    :corsPolicy
    {:allowOrigins [string],
     :allowOriginRegexes [string],
     :allowMethods [string],
     :allowHeaders [string],
     :exposeHeaders [string],
     :maxAge integer,
     :allowCredentials boolean,
     :disabled boolean},
    :faultInjectionPolicy
    {:delay {:fixedDelay Duration, :percentage number},
     :abort {:httpStatus integer, :percentage number}},
    :maxStreamDuration {:seconds string, :nanos integer}},
   :defaultUrlRedirect
   {:hostRedirect string,
    :pathRedirect string,
    :prefixRedirect string,
    :redirectResponseCode
    [FOUND
     MOVED_PERMANENTLY_DEFAULT
     PERMANENT_REDIRECT
     SEE_OTHER
     TEMPORARY_REDIRECT],
    :httpsRedirect boolean,
    :stripQuery boolean},
   :pathRules
   [{:service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :paths [string]}],
   :routeRules
   [{:priority integer,
     :description string,
     :matchRules
     [{:prefixMatch string,
       :fullPathMatch string,
       :regexMatch string,
       :ignoreCase boolean,
       :headerMatches
       [{:headerName string,
         :exactMatch string,
         :regexMatch string,
         :rangeMatch Int64RangeMatch,
         :presentMatch boolean,
         :prefixMatch string,
         :suffixMatch string,
         :invertMatch boolean}],
       :queryParameterMatches
       [{:name string,
         :presentMatch boolean,
         :exactMatch string,
         :regexMatch string}],
       :metadataFilters
       [{:filterMatchCriteria [MATCH_ALL MATCH_ANY NOT_SET],
         :filterLabels [MetadataFilterLabelMatch]}],
       :pathTemplateMatch string}],
     :service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :headerAction
     {:requestHeadersToRemove [string],
      :requestHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}],
      :responseHeadersToRemove [string],
      :responseHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}]}}],
   :headerAction
   {:requestHeadersToRemove [string],
    :requestHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}],
    :responseHeadersToRemove [string],
    :responseHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}]}}],
 :creationTimestamp string,
 :tests
 [{:description string,
   :host string,
   :path string,
   :headers [{:name string, :value string}],
   :service string,
   :expectedOutputUrl string,
   :expectedRedirectResponseCode integer}],
 :name string,
 :defaultUrlRedirect
 {:hostRedirect string,
  :pathRedirect string,
  :prefixRedirect string,
  :redirectResponseCode
  [FOUND
   MOVED_PERMANENTLY_DEFAULT
   PERMANENT_REDIRECT
   SEE_OTHER
   TEMPORARY_REDIRECT],
  :httpsRedirect boolean,
  :stripQuery boolean},
 :selfLink string,
 :headerAction
 {:requestHeadersToRemove [string],
  :requestHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}],
  :responseHeadersToRemove [string],
  :responseHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}]},
 :region string,
 :defaultRouteAction
 {:weightedBackendServices
  [{:backendService string,
    :weight integer,
    :headerAction
    {:requestHeadersToRemove [string],
     :requestHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}],
     :responseHeadersToRemove [string],
     :responseHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}]}}],
  :urlRewrite
  {:pathPrefixRewrite string,
   :hostRewrite string,
   :pathTemplateRewrite string},
  :timeout {:seconds string, :nanos integer},
  :retryPolicy
  {:retryConditions [string],
   :numRetries integer,
   :perTryTimeout {:seconds string, :nanos integer}},
  :requestMirrorPolicy {:backendService string},
  :corsPolicy
  {:allowOrigins [string],
   :allowOriginRegexes [string],
   :allowMethods [string],
   :allowHeaders [string],
   :exposeHeaders [string],
   :maxAge integer,
   :allowCredentials boolean,
   :disabled boolean},
  :faultInjectionPolicy
  {:delay
   {:fixedDelay {:seconds string, :nanos integer}, :percentage number},
   :abort {:httpStatus integer, :percentage number}},
  :maxStreamDuration {:seconds string, :nanos integer}},
 :defaultService string,
 :id string,
 :kind string,
 :hostRules
 [{:description string, :hosts [string], :pathMatcher string}],
 :fingerprint string}

optional:
requestId <string> begin_interface: MixerMutationRequestBuilder Request ID to support idempotency."
  ([project region urlMap UrlMap]
    (regionUrlMaps-update project region urlMap UrlMap nil))
  ([project region urlMap UrlMap optional]
    (client/api-request
      {:method "PUT",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/urlMaps/{urlMap}",
       :uri-template-args
       {:project project, :region region, :urlMap urlMap},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body UrlMap})))

(defn regionUrlMaps-patch
  "Patches the specified UrlMap resource with the data included in the request. This method supports PATCH semantics and uses JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/regionUrlMaps/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
urlMap <string> Name of the UrlMap resource to patch.
UrlMap:
{:description string,
 :pathMatchers
 [{:name string,
   :description string,
   :defaultService string,
   :defaultRouteAction
   {:weightedBackendServices
    [{:backendService string,
      :weight integer,
      :headerAction
      {:requestHeadersToRemove [string],
       :requestHeadersToAdd [HttpHeaderOption],
       :responseHeadersToRemove [string],
       :responseHeadersToAdd [HttpHeaderOption]}}],
    :urlRewrite
    {:pathPrefixRewrite string,
     :hostRewrite string,
     :pathTemplateRewrite string},
    :timeout {:seconds string, :nanos integer},
    :retryPolicy
    {:retryConditions [string],
     :numRetries integer,
     :perTryTimeout {:seconds string, :nanos integer}},
    :requestMirrorPolicy {:backendService string},
    :corsPolicy
    {:allowOrigins [string],
     :allowOriginRegexes [string],
     :allowMethods [string],
     :allowHeaders [string],
     :exposeHeaders [string],
     :maxAge integer,
     :allowCredentials boolean,
     :disabled boolean},
    :faultInjectionPolicy
    {:delay {:fixedDelay Duration, :percentage number},
     :abort {:httpStatus integer, :percentage number}},
    :maxStreamDuration {:seconds string, :nanos integer}},
   :defaultUrlRedirect
   {:hostRedirect string,
    :pathRedirect string,
    :prefixRedirect string,
    :redirectResponseCode
    [FOUND
     MOVED_PERMANENTLY_DEFAULT
     PERMANENT_REDIRECT
     SEE_OTHER
     TEMPORARY_REDIRECT],
    :httpsRedirect boolean,
    :stripQuery boolean},
   :pathRules
   [{:service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :paths [string]}],
   :routeRules
   [{:priority integer,
     :description string,
     :matchRules
     [{:prefixMatch string,
       :fullPathMatch string,
       :regexMatch string,
       :ignoreCase boolean,
       :headerMatches
       [{:headerName string,
         :exactMatch string,
         :regexMatch string,
         :rangeMatch Int64RangeMatch,
         :presentMatch boolean,
         :prefixMatch string,
         :suffixMatch string,
         :invertMatch boolean}],
       :queryParameterMatches
       [{:name string,
         :presentMatch boolean,
         :exactMatch string,
         :regexMatch string}],
       :metadataFilters
       [{:filterMatchCriteria [MATCH_ALL MATCH_ANY NOT_SET],
         :filterLabels [MetadataFilterLabelMatch]}],
       :pathTemplateMatch string}],
     :service string,
     :routeAction
     {:weightedBackendServices
      [{:backendService string,
        :weight integer,
        :headerAction HttpHeaderAction}],
      :urlRewrite
      {:pathPrefixRewrite string,
       :hostRewrite string,
       :pathTemplateRewrite string},
      :timeout {:seconds string, :nanos integer},
      :retryPolicy
      {:retryConditions [string],
       :numRetries integer,
       :perTryTimeout Duration},
      :requestMirrorPolicy {:backendService string},
      :corsPolicy
      {:allowOrigins [string],
       :allowOriginRegexes [string],
       :allowMethods [string],
       :allowHeaders [string],
       :exposeHeaders [string],
       :maxAge integer,
       :allowCredentials boolean,
       :disabled boolean},
      :faultInjectionPolicy
      {:delay HttpFaultDelay, :abort HttpFaultAbort},
      :maxStreamDuration {:seconds string, :nanos integer}},
     :urlRedirect
     {:hostRedirect string,
      :pathRedirect string,
      :prefixRedirect string,
      :redirectResponseCode
      [FOUND
       MOVED_PERMANENTLY_DEFAULT
       PERMANENT_REDIRECT
       SEE_OTHER
       TEMPORARY_REDIRECT],
      :httpsRedirect boolean,
      :stripQuery boolean},
     :headerAction
     {:requestHeadersToRemove [string],
      :requestHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}],
      :responseHeadersToRemove [string],
      :responseHeadersToAdd
      [{:headerName string, :headerValue string, :replace boolean}]}}],
   :headerAction
   {:requestHeadersToRemove [string],
    :requestHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}],
    :responseHeadersToRemove [string],
    :responseHeadersToAdd
    [{:headerName string, :headerValue string, :replace boolean}]}}],
 :creationTimestamp string,
 :tests
 [{:description string,
   :host string,
   :path string,
   :headers [{:name string, :value string}],
   :service string,
   :expectedOutputUrl string,
   :expectedRedirectResponseCode integer}],
 :name string,
 :defaultUrlRedirect
 {:hostRedirect string,
  :pathRedirect string,
  :prefixRedirect string,
  :redirectResponseCode
  [FOUND
   MOVED_PERMANENTLY_DEFAULT
   PERMANENT_REDIRECT
   SEE_OTHER
   TEMPORARY_REDIRECT],
  :httpsRedirect boolean,
  :stripQuery boolean},
 :selfLink string,
 :headerAction
 {:requestHeadersToRemove [string],
  :requestHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}],
  :responseHeadersToRemove [string],
  :responseHeadersToAdd
  [{:headerName string, :headerValue string, :replace boolean}]},
 :region string,
 :defaultRouteAction
 {:weightedBackendServices
  [{:backendService string,
    :weight integer,
    :headerAction
    {:requestHeadersToRemove [string],
     :requestHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}],
     :responseHeadersToRemove [string],
     :responseHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}]}}],
  :urlRewrite
  {:pathPrefixRewrite string,
   :hostRewrite string,
   :pathTemplateRewrite string},
  :timeout {:seconds string, :nanos integer},
  :retryPolicy
  {:retryConditions [string],
   :numRetries integer,
   :perTryTimeout {:seconds string, :nanos integer}},
  :requestMirrorPolicy {:backendService string},
  :corsPolicy
  {:allowOrigins [string],
   :allowOriginRegexes [string],
   :allowMethods [string],
   :allowHeaders [string],
   :exposeHeaders [string],
   :maxAge integer,
   :allowCredentials boolean,
   :disabled boolean},
  :faultInjectionPolicy
  {:delay
   {:fixedDelay {:seconds string, :nanos integer}, :percentage number},
   :abort {:httpStatus integer, :percentage number}},
  :maxStreamDuration {:seconds string, :nanos integer}},
 :defaultService string,
 :id string,
 :kind string,
 :hostRules
 [{:description string, :hosts [string], :pathMatcher string}],
 :fingerprint string}

optional:
requestId <string> begin_interface: MixerMutationRequestBuilder Request ID to support idempotency."
  ([project region urlMap UrlMap]
    (regionUrlMaps-patch project region urlMap UrlMap nil))
  ([project region urlMap UrlMap optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/urlMaps/{urlMap}",
       :uri-template-args
       {:project project, :region region, :urlMap urlMap},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body UrlMap})))

(defn regionUrlMaps-validate
  "Runs static validation for the UrlMap. In particular, the tests of the provided UrlMap will be run. Calling this method does NOT create the UrlMap.
https://cloud.google.com/products/compute/v1/docs/regionUrlMaps/validate

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
urlMap <string> Name of the UrlMap resource to be validated as.
RegionUrlMapsValidateRequest:
{:resource
 {:description string,
  :pathMatchers
  [{:name string,
    :description string,
    :defaultService string,
    :defaultRouteAction
    {:weightedBackendServices
     [{:backendService string,
       :weight integer,
       :headerAction HttpHeaderAction}],
     :urlRewrite
     {:pathPrefixRewrite string,
      :hostRewrite string,
      :pathTemplateRewrite string},
     :timeout {:seconds string, :nanos integer},
     :retryPolicy
     {:retryConditions [string],
      :numRetries integer,
      :perTryTimeout Duration},
     :requestMirrorPolicy {:backendService string},
     :corsPolicy
     {:allowOrigins [string],
      :allowOriginRegexes [string],
      :allowMethods [string],
      :allowHeaders [string],
      :exposeHeaders [string],
      :maxAge integer,
      :allowCredentials boolean,
      :disabled boolean},
     :faultInjectionPolicy
     {:delay HttpFaultDelay, :abort HttpFaultAbort},
     :maxStreamDuration {:seconds string, :nanos integer}},
    :defaultUrlRedirect
    {:hostRedirect string,
     :pathRedirect string,
     :prefixRedirect string,
     :redirectResponseCode
     [FOUND
      MOVED_PERMANENTLY_DEFAULT
      PERMANENT_REDIRECT
      SEE_OTHER
      TEMPORARY_REDIRECT],
     :httpsRedirect boolean,
     :stripQuery boolean},
    :pathRules
    [{:service string,
      :routeAction
      {:weightedBackendServices [WeightedBackendService],
       :urlRewrite UrlRewrite,
       :timeout Duration,
       :retryPolicy HttpRetryPolicy,
       :requestMirrorPolicy RequestMirrorPolicy,
       :corsPolicy CorsPolicy,
       :faultInjectionPolicy HttpFaultInjection,
       :maxStreamDuration Duration},
      :urlRedirect
      {:hostRedirect string,
       :pathRedirect string,
       :prefixRedirect string,
       :redirectResponseCode
       [FOUND
        MOVED_PERMANENTLY_DEFAULT
        PERMANENT_REDIRECT
        SEE_OTHER
        TEMPORARY_REDIRECT],
       :httpsRedirect boolean,
       :stripQuery boolean},
      :paths [string]}],
    :routeRules
    [{:priority integer,
      :description string,
      :matchRules
      [{:prefixMatch string,
        :fullPathMatch string,
        :regexMatch string,
        :ignoreCase boolean,
        :headerMatches [HttpHeaderMatch],
        :queryParameterMatches [HttpQueryParameterMatch],
        :metadataFilters [MetadataFilter],
        :pathTemplateMatch string}],
      :service string,
      :routeAction
      {:weightedBackendServices [WeightedBackendService],
       :urlRewrite UrlRewrite,
       :timeout Duration,
       :retryPolicy HttpRetryPolicy,
       :requestMirrorPolicy RequestMirrorPolicy,
       :corsPolicy CorsPolicy,
       :faultInjectionPolicy HttpFaultInjection,
       :maxStreamDuration Duration},
      :urlRedirect
      {:hostRedirect string,
       :pathRedirect string,
       :prefixRedirect string,
       :redirectResponseCode
       [FOUND
        MOVED_PERMANENTLY_DEFAULT
        PERMANENT_REDIRECT
        SEE_OTHER
        TEMPORARY_REDIRECT],
       :httpsRedirect boolean,
       :stripQuery boolean},
      :headerAction
      {:requestHeadersToRemove [string],
       :requestHeadersToAdd [HttpHeaderOption],
       :responseHeadersToRemove [string],
       :responseHeadersToAdd [HttpHeaderOption]}}],
    :headerAction
    {:requestHeadersToRemove [string],
     :requestHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}],
     :responseHeadersToRemove [string],
     :responseHeadersToAdd
     [{:headerName string, :headerValue string, :replace boolean}]}}],
  :creationTimestamp string,
  :tests
  [{:description string,
    :host string,
    :path string,
    :headers [{:name string, :value string}],
    :service string,
    :expectedOutputUrl string,
    :expectedRedirectResponseCode integer}],
  :name string,
  :defaultUrlRedirect
  {:hostRedirect string,
   :pathRedirect string,
   :prefixRedirect string,
   :redirectResponseCode
   [FOUND
    MOVED_PERMANENTLY_DEFAULT
    PERMANENT_REDIRECT
    SEE_OTHER
    TEMPORARY_REDIRECT],
   :httpsRedirect boolean,
   :stripQuery boolean},
  :selfLink string,
  :headerAction
  {:requestHeadersToRemove [string],
   :requestHeadersToAdd
   [{:headerName string, :headerValue string, :replace boolean}],
   :responseHeadersToRemove [string],
   :responseHeadersToAdd
   [{:headerName string, :headerValue string, :replace boolean}]},
  :region string,
  :defaultRouteAction
  {:weightedBackendServices
   [{:backendService string,
     :weight integer,
     :headerAction
     {:requestHeadersToRemove [string],
      :requestHeadersToAdd [HttpHeaderOption],
      :responseHeadersToRemove [string],
      :responseHeadersToAdd [HttpHeaderOption]}}],
   :urlRewrite
   {:pathPrefixRewrite string,
    :hostRewrite string,
    :pathTemplateRewrite string},
   :timeout {:seconds string, :nanos integer},
   :retryPolicy
   {:retryConditions [string],
    :numRetries integer,
    :perTryTimeout {:seconds string, :nanos integer}},
   :requestMirrorPolicy {:backendService string},
   :corsPolicy
   {:allowOrigins [string],
    :allowOriginRegexes [string],
    :allowMethods [string],
    :allowHeaders [string],
    :exposeHeaders [string],
    :maxAge integer,
    :allowCredentials boolean,
    :disabled boolean},
   :faultInjectionPolicy
   {:delay {:fixedDelay Duration, :percentage number},
    :abort {:httpStatus integer, :percentage number}},
   :maxStreamDuration {:seconds string, :nanos integer}},
  :defaultService string,
  :id string,
  :kind string,
  :hostRules
  [{:description string, :hosts [string], :pathMatcher string}],
  :fingerprint string}}"
  [project region urlMap RegionUrlMapsValidateRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/urlMaps/{urlMap}/validate",
     :uri-template-args
     {:project project, :region region, :urlMap urlMap},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionUrlMapsValidateRequest}))

(defn resourcePolicies-get
  "Retrieves all information of the specified resource policy.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
resourcePolicy <string> Name of the resource policy to retrieve."
  [project region resourcePolicy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/resourcePolicies/{resourcePolicy}",
     :uri-template-args
     {:project project,
      :region region,
      :resourcePolicy resourcePolicy},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn resourcePolicies-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/setIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
RegionSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project region resource RegionSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/resourcePolicies/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionSetPolicyRequest}))

(defn resourcePolicies-insert
  "Creates a new resource policy.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
ResourcePolicy:
{:description string,
 :creationTimestamp string,
 :resourceStatus
 {:instanceSchedulePolicy
  {:nextRunStartTime string, :lastRunStartTime string}},
 :diskConsistencyGroupPolicy object,
 :name string,
 :selfLink string,
 :region string,
 :instanceSchedulePolicy
 {:vmStartSchedule {:schedule string},
  :vmStopSchedule {:schedule string},
  :timeZone string,
  :startTime string,
  :expirationTime string},
 :status [CREATING DELETING EXPIRED INVALID READY],
 :id string,
 :kind string,
 :groupPlacementPolicy
 {:vmCount integer,
  :availabilityDomainCount integer,
  :collocation [COLLOCATED UNSPECIFIED_COLLOCATION]},
 :snapshotSchedulePolicy
 {:schedule
  {:hourlySchedule
   {:hoursInCycle integer, :startTime string, :duration string},
   :dailySchedule
   {:daysInCycle integer, :startTime string, :duration string},
   :weeklySchedule
   {:dayOfWeeks
    [{:day
      [FRIDAY
       INVALID
       MONDAY
       SATURDAY
       SUNDAY
       THURSDAY
       TUESDAY
       WEDNESDAY],
      :startTime string,
      :duration string}]}},
  :retentionPolicy
  {:maxRetentionDays integer,
   :onSourceDiskDelete
   [APPLY_RETENTION_POLICY
    KEEP_AUTO_SNAPSHOTS
    UNSPECIFIED_ON_SOURCE_DISK_DELETE]},
  :snapshotProperties
  {:labels object,
   :storageLocations [string],
   :guestFlush boolean,
   :chainName string}}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region ResourcePolicy]
    (resourcePolicies-insert project region ResourcePolicy nil))
  ([project region ResourcePolicy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/resourcePolicies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body ResourcePolicy})))

(defn resourcePolicies-patch
  "Modify the specified resource policy.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/patch

project <string> Project ID for this request.
region <string> Name of the region for this request.
resourcePolicy <string> Id of the resource policy to patch.
ResourcePolicy:
{:description string,
 :creationTimestamp string,
 :resourceStatus
 {:instanceSchedulePolicy
  {:nextRunStartTime string, :lastRunStartTime string}},
 :diskConsistencyGroupPolicy object,
 :name string,
 :selfLink string,
 :region string,
 :instanceSchedulePolicy
 {:vmStartSchedule {:schedule string},
  :vmStopSchedule {:schedule string},
  :timeZone string,
  :startTime string,
  :expirationTime string},
 :status [CREATING DELETING EXPIRED INVALID READY],
 :id string,
 :kind string,
 :groupPlacementPolicy
 {:vmCount integer,
  :availabilityDomainCount integer,
  :collocation [COLLOCATED UNSPECIFIED_COLLOCATION]},
 :snapshotSchedulePolicy
 {:schedule
  {:hourlySchedule
   {:hoursInCycle integer, :startTime string, :duration string},
   :dailySchedule
   {:daysInCycle integer, :startTime string, :duration string},
   :weeklySchedule
   {:dayOfWeeks
    [{:day
      [FRIDAY
       INVALID
       MONDAY
       SATURDAY
       SUNDAY
       THURSDAY
       TUESDAY
       WEDNESDAY],
      :startTime string,
      :duration string}]}},
  :retentionPolicy
  {:maxRetentionDays integer,
   :onSourceDiskDelete
   [APPLY_RETENTION_POLICY
    KEEP_AUTO_SNAPSHOTS
    UNSPECIFIED_ON_SOURCE_DISK_DELETE]},
  :snapshotProperties
  {:labels object,
   :storageLocations [string],
   :guestFlush boolean,
   :chainName string}}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
updateMask <string> update_mask indicates fields to be updated as part of this request."
  ([project region resourcePolicy ResourcePolicy]
    (resourcePolicies-patch
      project
      region
      resourcePolicy
      ResourcePolicy
      nil))
  ([project region resourcePolicy ResourcePolicy optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/resourcePolicies/{resourcePolicy}",
       :uri-template-args
       {:project project,
        :region region,
        :resourcePolicy resourcePolicy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body ResourcePolicy})))

(defn resourcePolicies-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/testIamPermissions

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project region resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/resourcePolicies/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn resourcePolicies-aggregatedList
  "Retrieves an aggregated list of resource policies. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (resourcePolicies-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/resourcePolicies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn resourcePolicies-delete
  "Deletes the specified resource policy.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/delete

project <string> Project ID for this request.
region <string> Name of the region for this request.
resourcePolicy <string> Name of the resource policy to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region resourcePolicy]
    (resourcePolicies-delete project region resourcePolicy nil))
  ([project region resourcePolicy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/resourcePolicies/{resourcePolicy}",
       :uri-template-args
       {:project project,
        :region region,
        :resourcePolicy resourcePolicy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn resourcePolicies-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/getIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project region resource]
    (resourcePolicies-getIamPolicy project region resource nil))
  ([project region resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/resourcePolicies/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn resourcePolicies-list
  "A list all the resource policies that have been configured for the specified project in specified region.
https://cloud.google.com/products/compute/v1/docs/resourcePolicies/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (resourcePolicies-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/resourcePolicies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionAutoscalers-list
  "Retrieves a list of autoscalers contained within the specified region.
https://cloud.google.com/products/compute/v1/docs/regionAutoscalers/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionAutoscalers-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/autoscalers",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionAutoscalers-get
  "Returns the specified autoscaler.
https://cloud.google.com/products/compute/v1/docs/regionAutoscalers/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
autoscaler <string> Name of the autoscaler to return."
  [project region autoscaler]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/autoscalers/{autoscaler}",
     :uri-template-args
     {:autoscaler autoscaler, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionAutoscalers-insert
  "Creates an autoscaler in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionAutoscalers/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
Autoscaler:
{:description string,
 :statusDetails
 [{:message string,
   :type
   [ALL_INSTANCES_UNHEALTHY
    BACKEND_SERVICE_DOES_NOT_EXIST
    CAPPED_AT_MAX_NUM_REPLICAS
    CUSTOM_METRIC_DATA_POINTS_TOO_SPARSE
    CUSTOM_METRIC_INVALID
    MIN_EQUALS_MAX
    MISSING_CUSTOM_METRIC_DATA_POINTS
    MISSING_LOAD_BALANCING_DATA_POINTS
    MODE_OFF
    MODE_ONLY_SCALE_OUT
    MODE_ONLY_UP
    MORE_THAN_ONE_BACKEND_SERVICE
    NOT_ENOUGH_QUOTA_AVAILABLE
    REGION_RESOURCE_STOCKOUT
    SCALING_TARGET_DOES_NOT_EXIST
    SCHEDULED_INSTANCES_GREATER_THAN_AUTOSCALER_MAX
    SCHEDULED_INSTANCES_LESS_THAN_AUTOSCALER_MIN
    UNKNOWN
    UNSUPPORTED_MAX_RATE_LOAD_BALANCING_CONFIGURATION
    ZONE_RESOURCE_STOCKOUT]}],
 :creationTimestamp string,
 :zone string,
 :name string,
 :selfLink string,
 :region string,
 :recommendedSize integer,
 :scalingScheduleStatus object,
 :status [ACTIVE DELETING ERROR PENDING],
 :id string,
 :kind string,
 :autoscalingPolicy
 {:coolDownPeriodSec integer,
  :mode [OFF ON ONLY_SCALE_OUT ONLY_UP],
  :minNumReplicas integer,
  :maxNumReplicas integer,
  :cpuUtilization
  {:utilizationTarget number,
   :predictiveMethod [NONE OPTIMIZE_AVAILABILITY]},
  :scaleInControl
  {:maxScaledInReplicas
   {:fixed integer, :percent integer, :calculated integer},
   :timeWindowSec integer},
  :scalingSchedules object,
  :loadBalancingUtilization {:utilizationTarget number},
  :customMetricUtilizations
  [{:metric string,
    :filter string,
    :utilizationTarget number,
    :singleInstanceAssignment number,
    :utilizationTargetType
    [DELTA_PER_MINUTE DELTA_PER_SECOND GAUGE]}]},
 :target string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region Autoscaler]
    (regionAutoscalers-insert project region Autoscaler nil))
  ([project region Autoscaler optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/autoscalers",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Autoscaler})))

(defn regionAutoscalers-update
  "Updates an autoscaler in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionAutoscalers/update

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
Autoscaler:
{:description string,
 :statusDetails
 [{:message string,
   :type
   [ALL_INSTANCES_UNHEALTHY
    BACKEND_SERVICE_DOES_NOT_EXIST
    CAPPED_AT_MAX_NUM_REPLICAS
    CUSTOM_METRIC_DATA_POINTS_TOO_SPARSE
    CUSTOM_METRIC_INVALID
    MIN_EQUALS_MAX
    MISSING_CUSTOM_METRIC_DATA_POINTS
    MISSING_LOAD_BALANCING_DATA_POINTS
    MODE_OFF
    MODE_ONLY_SCALE_OUT
    MODE_ONLY_UP
    MORE_THAN_ONE_BACKEND_SERVICE
    NOT_ENOUGH_QUOTA_AVAILABLE
    REGION_RESOURCE_STOCKOUT
    SCALING_TARGET_DOES_NOT_EXIST
    SCHEDULED_INSTANCES_GREATER_THAN_AUTOSCALER_MAX
    SCHEDULED_INSTANCES_LESS_THAN_AUTOSCALER_MIN
    UNKNOWN
    UNSUPPORTED_MAX_RATE_LOAD_BALANCING_CONFIGURATION
    ZONE_RESOURCE_STOCKOUT]}],
 :creationTimestamp string,
 :zone string,
 :name string,
 :selfLink string,
 :region string,
 :recommendedSize integer,
 :scalingScheduleStatus object,
 :status [ACTIVE DELETING ERROR PENDING],
 :id string,
 :kind string,
 :autoscalingPolicy
 {:coolDownPeriodSec integer,
  :mode [OFF ON ONLY_SCALE_OUT ONLY_UP],
  :minNumReplicas integer,
  :maxNumReplicas integer,
  :cpuUtilization
  {:utilizationTarget number,
   :predictiveMethod [NONE OPTIMIZE_AVAILABILITY]},
  :scaleInControl
  {:maxScaledInReplicas
   {:fixed integer, :percent integer, :calculated integer},
   :timeWindowSec integer},
  :scalingSchedules object,
  :loadBalancingUtilization {:utilizationTarget number},
  :customMetricUtilizations
  [{:metric string,
    :filter string,
    :utilizationTarget number,
    :singleInstanceAssignment number,
    :utilizationTargetType
    [DELTA_PER_MINUTE DELTA_PER_SECOND GAUGE]}]},
 :target string}

optional:
autoscaler <string> Name of the autoscaler to update.
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region Autoscaler]
    (regionAutoscalers-update project region Autoscaler nil))
  ([project region Autoscaler optional]
    (client/api-request
      {:method "PUT",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/autoscalers",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Autoscaler})))

(defn regionAutoscalers-patch
  "Updates an autoscaler in the specified project using the data included in the request. This method supports PATCH semantics and uses the JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/regionAutoscalers/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
Autoscaler:
{:description string,
 :statusDetails
 [{:message string,
   :type
   [ALL_INSTANCES_UNHEALTHY
    BACKEND_SERVICE_DOES_NOT_EXIST
    CAPPED_AT_MAX_NUM_REPLICAS
    CUSTOM_METRIC_DATA_POINTS_TOO_SPARSE
    CUSTOM_METRIC_INVALID
    MIN_EQUALS_MAX
    MISSING_CUSTOM_METRIC_DATA_POINTS
    MISSING_LOAD_BALANCING_DATA_POINTS
    MODE_OFF
    MODE_ONLY_SCALE_OUT
    MODE_ONLY_UP
    MORE_THAN_ONE_BACKEND_SERVICE
    NOT_ENOUGH_QUOTA_AVAILABLE
    REGION_RESOURCE_STOCKOUT
    SCALING_TARGET_DOES_NOT_EXIST
    SCHEDULED_INSTANCES_GREATER_THAN_AUTOSCALER_MAX
    SCHEDULED_INSTANCES_LESS_THAN_AUTOSCALER_MIN
    UNKNOWN
    UNSUPPORTED_MAX_RATE_LOAD_BALANCING_CONFIGURATION
    ZONE_RESOURCE_STOCKOUT]}],
 :creationTimestamp string,
 :zone string,
 :name string,
 :selfLink string,
 :region string,
 :recommendedSize integer,
 :scalingScheduleStatus object,
 :status [ACTIVE DELETING ERROR PENDING],
 :id string,
 :kind string,
 :autoscalingPolicy
 {:coolDownPeriodSec integer,
  :mode [OFF ON ONLY_SCALE_OUT ONLY_UP],
  :minNumReplicas integer,
  :maxNumReplicas integer,
  :cpuUtilization
  {:utilizationTarget number,
   :predictiveMethod [NONE OPTIMIZE_AVAILABILITY]},
  :scaleInControl
  {:maxScaledInReplicas
   {:fixed integer, :percent integer, :calculated integer},
   :timeWindowSec integer},
  :scalingSchedules object,
  :loadBalancingUtilization {:utilizationTarget number},
  :customMetricUtilizations
  [{:metric string,
    :filter string,
    :utilizationTarget number,
    :singleInstanceAssignment number,
    :utilizationTargetType
    [DELTA_PER_MINUTE DELTA_PER_SECOND GAUGE]}]},
 :target string}

optional:
autoscaler <string> Name of the autoscaler to patch.
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region Autoscaler]
    (regionAutoscalers-patch project region Autoscaler nil))
  ([project region Autoscaler optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/autoscalers",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Autoscaler})))

(defn regionAutoscalers-delete
  "Deletes the specified autoscaler.
https://cloud.google.com/products/compute/v1/docs/regionAutoscalers/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
autoscaler <string> Name of the autoscaler to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region autoscaler]
    (regionAutoscalers-delete project region autoscaler nil))
  ([project region autoscaler optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/autoscalers/{autoscaler}",
       :uri-template-args
       {:autoscaler autoscaler, :project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn httpHealthChecks-list
  "Retrieves the list of HttpHealthCheck resources available to the specified project.
https://cloud.google.com/products/compute/v1/docs/httpHealthChecks/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (httpHealthChecks-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/httpHealthChecks",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn httpHealthChecks-get
  "Returns the specified HttpHealthCheck resource.
https://cloud.google.com/products/compute/v1/docs/httpHealthChecks/get

project <string> Project ID for this request.
httpHealthCheck <string> Name of the HttpHealthCheck resource to return."
  [project httpHealthCheck]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/httpHealthChecks/{httpHealthCheck}",
     :uri-template-args
     {:httpHealthCheck httpHealthCheck, :project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn httpHealthChecks-insert
  "Creates a HttpHealthCheck resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/httpHealthChecks/insert

project <string> Project ID for this request.
HttpHealthCheck:
{:description string,
 :creationTimestamp string,
 :healthyThreshold integer,
 :name string,
 :selfLink string,
 :port integer,
 :host string,
 :requestPath string,
 :id string,
 :kind string,
 :checkIntervalSec integer,
 :timeoutSec integer,
 :unhealthyThreshold integer}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project HttpHealthCheck]
    (httpHealthChecks-insert project HttpHealthCheck nil))
  ([project HttpHealthCheck optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/httpHealthChecks",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body HttpHealthCheck})))

(defn httpHealthChecks-delete
  "Deletes the specified HttpHealthCheck resource.
https://cloud.google.com/products/compute/v1/docs/httpHealthChecks/delete

project <string> Project ID for this request.
httpHealthCheck <string> Name of the HttpHealthCheck resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project httpHealthCheck]
    (httpHealthChecks-delete project httpHealthCheck nil))
  ([project httpHealthCheck optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/httpHealthChecks/{httpHealthCheck}",
       :uri-template-args
       {:httpHealthCheck httpHealthCheck, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn httpHealthChecks-update
  "Updates a HttpHealthCheck resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/httpHealthChecks/update

project <string> Project ID for this request.
httpHealthCheck <string> Name of the HttpHealthCheck resource to update.
HttpHealthCheck:
{:description string,
 :creationTimestamp string,
 :healthyThreshold integer,
 :name string,
 :selfLink string,
 :port integer,
 :host string,
 :requestPath string,
 :id string,
 :kind string,
 :checkIntervalSec integer,
 :timeoutSec integer,
 :unhealthyThreshold integer}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project httpHealthCheck HttpHealthCheck]
    (httpHealthChecks-update
      project
      httpHealthCheck
      HttpHealthCheck
      nil))
  ([project httpHealthCheck HttpHealthCheck optional]
    (client/api-request
      {:method "PUT",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/httpHealthChecks/{httpHealthCheck}",
       :uri-template-args
       {:httpHealthCheck httpHealthCheck, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body HttpHealthCheck})))

(defn httpHealthChecks-patch
  "Updates a HttpHealthCheck resource in the specified project using the data included in the request. This method supports PATCH semantics and uses the JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/httpHealthChecks/patch

project <string> Project ID for this request.
httpHealthCheck <string> Name of the HttpHealthCheck resource to patch.
HttpHealthCheck:
{:description string,
 :creationTimestamp string,
 :healthyThreshold integer,
 :name string,
 :selfLink string,
 :port integer,
 :host string,
 :requestPath string,
 :id string,
 :kind string,
 :checkIntervalSec integer,
 :timeoutSec integer,
 :unhealthyThreshold integer}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project httpHealthCheck HttpHealthCheck]
    (httpHealthChecks-patch
      project
      httpHealthCheck
      HttpHealthCheck
      nil))
  ([project httpHealthCheck HttpHealthCheck optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/httpHealthChecks/{httpHealthCheck}",
       :uri-template-args
       {:httpHealthCheck httpHealthCheck, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body HttpHealthCheck})))

(defn regionNotificationEndpoints-list
  "Lists the NotificationEndpoints for a project in the given region.
https://cloud.google.com/products/compute/v1/docs/regionNotificationEndpoints/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region]
    (regionNotificationEndpoints-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/notificationEndpoints",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionNotificationEndpoints-get
  "Returns the specified NotificationEndpoint resource in the given region.
https://cloud.google.com/products/compute/v1/docs/regionNotificationEndpoints/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
notificationEndpoint <string> Name of the NotificationEndpoint resource to return."
  [project region notificationEndpoint]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/notificationEndpoints/{notificationEndpoint}",
     :uri-template-args
     {:notificationEndpoint notificationEndpoint,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionNotificationEndpoints-insert
  "Create a NotificationEndpoint in the specified project in the given region using the parameters that are included in the request.
https://cloud.google.com/products/compute/v1/docs/regionNotificationEndpoints/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
NotificationEndpoint:
{:kind string,
 :id string,
 :creationTimestamp string,
 :name string,
 :description string,
 :selfLink string,
 :region string,
 :grpcSettings
 {:endpoint string,
  :retryDurationSec integer,
  :payloadName string,
  :authority string,
  :resendInterval {:seconds string, :nanos integer}}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region NotificationEndpoint]
    (regionNotificationEndpoints-insert
      project
      region
      NotificationEndpoint
      nil))
  ([project region NotificationEndpoint optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/notificationEndpoints",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NotificationEndpoint})))

(defn regionNotificationEndpoints-delete
  "Deletes the specified NotificationEndpoint in the given region
https://cloud.google.com/products/compute/v1/docs/regionNotificationEndpoints/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
notificationEndpoint <string> Name of the NotificationEndpoint resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region notificationEndpoint]
    (regionNotificationEndpoints-delete
      project
      region
      notificationEndpoint
      nil))
  ([project region notificationEndpoint optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/notificationEndpoints/{notificationEndpoint}",
       :uri-template-args
       {:notificationEndpoint notificationEndpoint,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionSslCertificates-list
  "Retrieves the list of SslCertificate resources available to the specified project in the specified region.
https://cloud.google.com/products/compute/v1/docs/regionSslCertificates/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionSslCertificates-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/sslCertificates",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionSslCertificates-get
  "Returns the specified SslCertificate resource in the specified region. Get a list of available SSL certificates by making a list() request.
https://cloud.google.com/products/compute/v1/docs/regionSslCertificates/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
sslCertificate <string> Name of the SslCertificate resource to return."
  [project region sslCertificate]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/sslCertificates/{sslCertificate}",
     :uri-template-args
     {:project project,
      :region region,
      :sslCertificate sslCertificate},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionSslCertificates-insert
  "Creates a SslCertificate resource in the specified project and region using the data included in the request
https://cloud.google.com/products/compute/v1/docs/regionSslCertificates/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
SslCertificate:
{:description string,
 :creationTimestamp string,
 :name string,
 :selfManaged {:certificate string, :privateKey string},
 :privateKey string,
 :selfLink string,
 :type [MANAGED SELF_MANAGED TYPE_UNSPECIFIED],
 :region string,
 :certificate string,
 :subjectAlternativeNames [string],
 :id string,
 :kind string,
 :managed
 {:domains [string],
  :status
  [ACTIVE
   MANAGED_CERTIFICATE_STATUS_UNSPECIFIED
   PROVISIONING
   PROVISIONING_FAILED
   PROVISIONING_FAILED_PERMANENTLY
   RENEWAL_FAILED],
  :domainStatus object},
 :expireTime string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region SslCertificate]
    (regionSslCertificates-insert project region SslCertificate nil))
  ([project region SslCertificate optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/sslCertificates",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body SslCertificate})))

(defn regionSslCertificates-delete
  "Deletes the specified SslCertificate resource in the region.
https://cloud.google.com/products/compute/v1/docs/regionSslCertificates/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
sslCertificate <string> Name of the SslCertificate resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region sslCertificate]
    (regionSslCertificates-delete project region sslCertificate nil))
  ([project region sslCertificate optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/sslCertificates/{sslCertificate}",
       :uri-template-args
       {:project project,
        :region region,
        :sslCertificate sslCertificate},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn nodeTypes-list
  "Retrieves a list of node types available to the specified project.
https://cloud.google.com/products/compute/v1/docs/nodeTypes/list

project <string> Project ID for this request.
zone <string> The name of the zone for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone] (nodeTypes-list project zone nil))
  ([project zone optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeTypes",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeTypes-aggregatedList
  "Retrieves an aggregated list of node types. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/nodeTypes/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (nodeTypes-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/nodeTypes",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn nodeTypes-get
  "Returns the specified node type.
https://cloud.google.com/products/compute/v1/docs/nodeTypes/get

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
nodeType <string> Name of the node type to return."
  [project zone nodeType]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/nodeTypes/{nodeType}",
     :uri-template-args
     {:nodeType nodeType, :project project, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn targetTcpProxies-list
  "Retrieves the list of TargetTcpProxy resources available to the specified project.
https://cloud.google.com/products/compute/v1/docs/targetTcpProxies/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (targetTcpProxies-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetTcpProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetTcpProxies-aggregatedList
  "Retrieves the list of all TargetTcpProxy resources, regional and global, available to the specified project. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/targetTcpProxies/aggregatedList

project <string> Name of the project scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (targetTcpProxies-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/targetTcpProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetTcpProxies-get
  "Returns the specified TargetTcpProxy resource.
https://cloud.google.com/products/compute/v1/docs/targetTcpProxies/get

project <string> Project ID for this request.
targetTcpProxy <string> Name of the TargetTcpProxy resource to return."
  [project targetTcpProxy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetTcpProxies/{targetTcpProxy}",
     :uri-template-args
     {:project project, :targetTcpProxy targetTcpProxy},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn targetTcpProxies-insert
  "Creates a TargetTcpProxy resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/targetTcpProxies/insert

project <string> Project ID for this request.
TargetTcpProxy:
{:description string,
 :service string,
 :proxyHeader [NONE PROXY_V1],
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :id string,
 :kind string,
 :proxyBind boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project TargetTcpProxy]
    (targetTcpProxies-insert project TargetTcpProxy nil))
  ([project TargetTcpProxy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetTcpProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetTcpProxy})))

(defn targetTcpProxies-delete
  "Deletes the specified TargetTcpProxy resource.
https://cloud.google.com/products/compute/v1/docs/targetTcpProxies/delete

project <string> Project ID for this request.
targetTcpProxy <string> Name of the TargetTcpProxy resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetTcpProxy]
    (targetTcpProxies-delete project targetTcpProxy nil))
  ([project targetTcpProxy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetTcpProxies/{targetTcpProxy}",
       :uri-template-args
       {:project project, :targetTcpProxy targetTcpProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn targetTcpProxies-setBackendService
  "Changes the BackendService for TargetTcpProxy.
https://cloud.google.com/products/compute/v1/docs/targetTcpProxies/setBackendService

project <string> Project ID for this request.
targetTcpProxy <string> Name of the TargetTcpProxy resource whose BackendService resource is to be set.
TargetTcpProxiesSetBackendServiceRequest:
{:service string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetTcpProxy TargetTcpProxiesSetBackendServiceRequest]
    (targetTcpProxies-setBackendService
      project
      targetTcpProxy
      TargetTcpProxiesSetBackendServiceRequest
      nil))
  ([project
    targetTcpProxy
    TargetTcpProxiesSetBackendServiceRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetTcpProxies/{targetTcpProxy}/setBackendService",
       :uri-template-args
       {:project project, :targetTcpProxy targetTcpProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetTcpProxiesSetBackendServiceRequest})))

(defn targetTcpProxies-setProxyHeader
  "Changes the ProxyHeaderType for TargetTcpProxy.
https://cloud.google.com/products/compute/v1/docs/targetTcpProxies/setProxyHeader

project <string> Project ID for this request.
targetTcpProxy <string> Name of the TargetTcpProxy resource whose ProxyHeader is to be set.
TargetTcpProxiesSetProxyHeaderRequest:
{:proxyHeader [NONE PROXY_V1]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetTcpProxy TargetTcpProxiesSetProxyHeaderRequest]
    (targetTcpProxies-setProxyHeader
      project
      targetTcpProxy
      TargetTcpProxiesSetProxyHeaderRequest
      nil))
  ([project
    targetTcpProxy
    TargetTcpProxiesSetProxyHeaderRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetTcpProxies/{targetTcpProxy}/setProxyHeader",
       :uri-template-args
       {:project project, :targetTcpProxy targetTcpProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetTcpProxiesSetProxyHeaderRequest})))

(defn storagePools-get
  "Returns a specified storage pool. Gets a list of available storage pools by making a list() request.
https://cloud.google.com/products/compute/v1/docs/storagePools/get

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
storagePool <string> Name of the storage pool to return."
  [project zone storagePool]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools/{storagePool}",
     :uri-template-args
     {:project project, :storagePool storagePool, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn storagePools-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/storagePools/setIamPolicy

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.
ZoneSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project zone resource ZoneSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :resource resource, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body ZoneSetPolicyRequest}))

(defn storagePools-insert
  "Creates a storage pool in the specified project using the data in the request.
https://cloud.google.com/products/compute/v1/docs/storagePools/insert

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
StoragePool:
{:description string,
 :labels object,
 :performanceProvisioningType [ADVANCED STANDARD UNSPECIFIED],
 :creationTimestamp string,
 :resourceStatus
 {:totalProvisionedDiskIops string,
  :diskCount string,
  :lastResizeTimestamp string,
  :poolUsedCapacityBytes string,
  :totalProvisionedDiskThroughput string,
  :poolUsedIops string,
  :maxTotalProvisionedDiskCapacityGb string,
  :poolUserWrittenBytes string,
  :poolUsedThroughput string,
  :totalProvisionedDiskCapacityGb string},
 :storagePoolType string,
 :zone string,
 :name string,
 :selfLink string,
 :state [CREATING DELETING FAILED READY],
 :capacityProvisioningType [ADVANCED STANDARD UNSPECIFIED],
 :selfLinkWithId string,
 :status
 {:totalProvisionedDiskIops string,
  :diskCount string,
  :lastResizeTimestamp string,
  :poolUsedCapacityBytes string,
  :totalProvisionedDiskThroughput string,
  :poolUsedIops string,
  :maxTotalProvisionedDiskCapacityGb string,
  :poolUserWrittenBytes string,
  :poolUsedThroughput string,
  :totalProvisionedDiskCapacityGb string},
 :poolProvisionedCapacityGb string,
 :id string,
 :kind string,
 :poolProvisionedIops string,
 :poolProvisionedThroughput string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone StoragePool]
    (storagePools-insert project zone StoragePool nil))
  ([project zone StoragePool optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body StoragePool})))

(defn storagePools-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/storagePools/testIamPermissions

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project zone resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :resource resource, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn storagePools-aggregatedList
  "Retrieves an aggregated list of storage pools. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/storagePools/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (storagePools-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/storagePools",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn storagePools-update
  "Updates the specified storagePool with the data included in the request. The update is performed only on selected fields included as part of update-mask. Only the following fields can be modified: size_tb and provisioned_iops.
https://cloud.google.com/products/compute/v1/docs/storagePools/update

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
storagePool <string> The storagePool name for this request.
StoragePool:
{:description string,
 :labels object,
 :performanceProvisioningType [ADVANCED STANDARD UNSPECIFIED],
 :creationTimestamp string,
 :resourceStatus
 {:totalProvisionedDiskIops string,
  :diskCount string,
  :lastResizeTimestamp string,
  :poolUsedCapacityBytes string,
  :totalProvisionedDiskThroughput string,
  :poolUsedIops string,
  :maxTotalProvisionedDiskCapacityGb string,
  :poolUserWrittenBytes string,
  :poolUsedThroughput string,
  :totalProvisionedDiskCapacityGb string},
 :storagePoolType string,
 :zone string,
 :name string,
 :selfLink string,
 :state [CREATING DELETING FAILED READY],
 :capacityProvisioningType [ADVANCED STANDARD UNSPECIFIED],
 :selfLinkWithId string,
 :status
 {:totalProvisionedDiskIops string,
  :diskCount string,
  :lastResizeTimestamp string,
  :poolUsedCapacityBytes string,
  :totalProvisionedDiskThroughput string,
  :poolUsedIops string,
  :maxTotalProvisionedDiskCapacityGb string,
  :poolUserWrittenBytes string,
  :poolUsedThroughput string,
  :totalProvisionedDiskCapacityGb string},
 :poolProvisionedCapacityGb string,
 :id string,
 :kind string,
 :poolProvisionedIops string,
 :poolProvisionedThroughput string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
updateMask <string> update_mask indicates fields to be updated as part of this request."
  ([project zone storagePool StoragePool]
    (storagePools-update project zone storagePool StoragePool nil))
  ([project zone storagePool StoragePool optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools/{storagePool}",
       :uri-template-args
       {:project project, :storagePool storagePool, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body StoragePool})))

(defn storagePools-delete
  "Deletes the specified storage pool. Deleting a storagePool removes its data permanently and is irreversible. However, deleting a storagePool does not delete any snapshots previously made from the storagePool. You must separately delete snapshots.
https://cloud.google.com/products/compute/v1/docs/storagePools/delete

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
storagePool <string> Name of the storage pool to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone storagePool]
    (storagePools-delete project zone storagePool nil))
  ([project zone storagePool optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools/{storagePool}",
       :uri-template-args
       {:project project, :storagePool storagePool, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn storagePools-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/storagePools/getIamPolicy

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project zone resource]
    (storagePools-getIamPolicy project zone resource nil))
  ([project zone resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :resource resource, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn storagePools-list
  "Retrieves a list of storage pools contained within the specified zone.
https://cloud.google.com/products/compute/v1/docs/storagePools/list

project <string> Project ID for this request.
zone <string> The name of the zone for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone] (storagePools-list project zone nil))
  ([project zone optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn storagePools-listDisks
  "Lists the disks in a specified storage pool.
https://cloud.google.com/products/compute/v1/docs/storagePools/listDisks

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
storagePool <string> Name of the storage pool to list disks of.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone storagePool]
    (storagePools-listDisks project zone storagePool nil))
  ([project zone storagePool optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/storagePools/{storagePool}/listDisks",
       :uri-template-args
       {:project project, :storagePool storagePool, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionCommitments-list
  "Retrieves a list of commitments contained within the specified region.
https://cloud.google.com/products/compute/v1/docs/regionCommitments/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionCommitments-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/commitments",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionCommitments-aggregatedList
  "Retrieves an aggregated list of commitments by region. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/regionCommitments/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (regionCommitments-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/commitments",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionCommitments-get
  "Returns the specified commitment resource.
https://cloud.google.com/products/compute/v1/docs/regionCommitments/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
commitment <string> Name of the commitment to return."
  [project region commitment]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/commitments/{commitment}",
     :uri-template-args
     {:commitment commitment, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionCommitments-insert
  "Creates a commitment in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionCommitments/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
Commitment:
{:existingReservations [string],
 :description string,
 :category [CATEGORY_UNSPECIFIED LICENSE MACHINE],
 :creationTimestamp string,
 :name string,
 :mergeSourceCommitments [string],
 :statusMessage string,
 :endTimestamp string,
 :selfLink string,
 :type
 [ACCELERATOR_OPTIMIZED
  ACCELERATOR_OPTIMIZED_A3
  ACCELERATOR_OPTIMIZED_A3_MEGA
  COMPUTE_OPTIMIZED
  COMPUTE_OPTIMIZED_C2D
  COMPUTE_OPTIMIZED_C3
  COMPUTE_OPTIMIZED_C3D
  COMPUTE_OPTIMIZED_H3
  GENERAL_PURPOSE
  GENERAL_PURPOSE_E2
  GENERAL_PURPOSE_N2
  GENERAL_PURPOSE_N2D
  GENERAL_PURPOSE_N4
  GENERAL_PURPOSE_T2D
  GRAPHICS_OPTIMIZED
  MEMORY_OPTIMIZED
  MEMORY_OPTIMIZED_M3
  STORAGE_OPTIMIZED_Z3
  TYPE_UNSPECIFIED],
 :licenseResource
 {:license string, :amount string, :coresPerLicense string},
 :splitSourceCommitment string,
 :region string,
 :status [ACTIVE CANCELLED CREATING EXPIRED NOT_YET_ACTIVE],
 :id string,
 :kind string,
 :startTimestamp string,
 :resources
 [{:type [ACCELERATOR LOCAL_SSD MEMORY UNSPECIFIED VCPU],
   :amount string,
   :acceleratorType string}],
 :reservations
 [{:description string,
   :resourcePolicies object,
   :creationTimestamp string,
   :resourceStatus
   {:specificSkuAllocation {:sourceInstanceTemplateId string}},
   :zone string,
   :name string,
   :shareSettings
   {:shareType
    [LOCAL ORGANIZATION SHARE_TYPE_UNSPECIFIED SPECIFIC_PROJECTS],
    :projectMap object},
   :specificReservationRequired boolean,
   :commitment string,
   :selfLink string,
   :satisfiesPzs boolean,
   :specificReservation
   {:instanceProperties
    {:machineType string,
     :guestAccelerators
     [{:acceleratorType string, :acceleratorCount integer}],
     :minCpuPlatform string,
     :localSsds [{:diskSizeGb string, :interface [NVME SCSI]}],
     :locationHint string},
    :count string,
    :inUseCount string,
    :assuredCount string,
    :sourceInstanceTemplate string},
   :status [CREATING DELETING INVALID READY UPDATING],
   :id string,
   :kind string,
   :aggregateReservation
   {:vmFamily
    [VM_FAMILY_CLOUD_TPU_LITE_DEVICE_CT5L
     VM_FAMILY_CLOUD_TPU_LITE_POD_SLICE_CT5LP
     VM_FAMILY_CLOUD_TPU_POD_SLICE_CT4P],
    :reservedResources
    [{:accelerator
      {:acceleratorCount integer, :acceleratorType string}}],
    :inUseResources
    [{:accelerator
      {:acceleratorCount integer, :acceleratorType string}}],
    :workloadType [BATCH SERVING UNSPECIFIED]}}],
 :plan [INVALID THIRTY_SIX_MONTH TWELVE_MONTH],
 :autoRenew boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region Commitment]
    (regionCommitments-insert project region Commitment nil))
  ([project region Commitment optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/commitments",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Commitment})))

(defn regionCommitments-update
  "Updates the specified commitment with the data included in the request. Update is performed only on selected fields included as part of update-mask. Only the following fields can be modified: auto_renew.
https://cloud.google.com/products/compute/v1/docs/regionCommitments/update

project <string> Project ID for this request.
region <string> Name of the region for this request.
commitment <string> Name of the commitment for which auto renew is being updated.
Commitment:
{:existingReservations [string],
 :description string,
 :category [CATEGORY_UNSPECIFIED LICENSE MACHINE],
 :creationTimestamp string,
 :name string,
 :mergeSourceCommitments [string],
 :statusMessage string,
 :endTimestamp string,
 :selfLink string,
 :type
 [ACCELERATOR_OPTIMIZED
  ACCELERATOR_OPTIMIZED_A3
  ACCELERATOR_OPTIMIZED_A3_MEGA
  COMPUTE_OPTIMIZED
  COMPUTE_OPTIMIZED_C2D
  COMPUTE_OPTIMIZED_C3
  COMPUTE_OPTIMIZED_C3D
  COMPUTE_OPTIMIZED_H3
  GENERAL_PURPOSE
  GENERAL_PURPOSE_E2
  GENERAL_PURPOSE_N2
  GENERAL_PURPOSE_N2D
  GENERAL_PURPOSE_N4
  GENERAL_PURPOSE_T2D
  GRAPHICS_OPTIMIZED
  MEMORY_OPTIMIZED
  MEMORY_OPTIMIZED_M3
  STORAGE_OPTIMIZED_Z3
  TYPE_UNSPECIFIED],
 :licenseResource
 {:license string, :amount string, :coresPerLicense string},
 :splitSourceCommitment string,
 :region string,
 :status [ACTIVE CANCELLED CREATING EXPIRED NOT_YET_ACTIVE],
 :id string,
 :kind string,
 :startTimestamp string,
 :resources
 [{:type [ACCELERATOR LOCAL_SSD MEMORY UNSPECIFIED VCPU],
   :amount string,
   :acceleratorType string}],
 :reservations
 [{:description string,
   :resourcePolicies object,
   :creationTimestamp string,
   :resourceStatus
   {:specificSkuAllocation {:sourceInstanceTemplateId string}},
   :zone string,
   :name string,
   :shareSettings
   {:shareType
    [LOCAL ORGANIZATION SHARE_TYPE_UNSPECIFIED SPECIFIC_PROJECTS],
    :projectMap object},
   :specificReservationRequired boolean,
   :commitment string,
   :selfLink string,
   :satisfiesPzs boolean,
   :specificReservation
   {:instanceProperties
    {:machineType string,
     :guestAccelerators
     [{:acceleratorType string, :acceleratorCount integer}],
     :minCpuPlatform string,
     :localSsds [{:diskSizeGb string, :interface [NVME SCSI]}],
     :locationHint string},
    :count string,
    :inUseCount string,
    :assuredCount string,
    :sourceInstanceTemplate string},
   :status [CREATING DELETING INVALID READY UPDATING],
   :id string,
   :kind string,
   :aggregateReservation
   {:vmFamily
    [VM_FAMILY_CLOUD_TPU_LITE_DEVICE_CT5L
     VM_FAMILY_CLOUD_TPU_LITE_POD_SLICE_CT5LP
     VM_FAMILY_CLOUD_TPU_POD_SLICE_CT4P],
    :reservedResources
    [{:accelerator
      {:acceleratorCount integer, :acceleratorType string}}],
    :inUseResources
    [{:accelerator
      {:acceleratorCount integer, :acceleratorType string}}],
    :workloadType [BATCH SERVING UNSPECIFIED]}}],
 :plan [INVALID THIRTY_SIX_MONTH TWELVE_MONTH],
 :autoRenew boolean}

optional:
paths <string> 
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
updateMask <string> update_mask indicates fields to be updated as part of this request."
  ([project region commitment Commitment]
    (regionCommitments-update
      project
      region
      commitment
      Commitment
      nil))
  ([project region commitment Commitment optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/commitments/{commitment}",
       :uri-template-args
       {:commitment commitment, :project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Commitment})))

(defn regionDiskTypes-list
  "Retrieves a list of regional disk types available to the specified project.
https://cloud.google.com/products/compute/v1/docs/regionDiskTypes/list

project <string> Project ID for this request.
region <string> The name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionDiskTypes-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/diskTypes",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionDiskTypes-get
  "Returns the specified regional disk type.
https://cloud.google.com/products/compute/v1/docs/regionDiskTypes/get

project <string> Project ID for this request.
region <string> The name of the region for this request.
diskType <string> Name of the disk type to return."
  [project region diskType]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/diskTypes/{diskType}",
     :uri-template-args
     {:diskType diskType, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionInstanceGroupManagers-listManagedInstances
  "Lists the instances in the managed instance group and instances that are scheduled to be created. The list includes any current actions that the group has scheduled for its instances. The orderBy query parameter is not supported. The `pageToken` query parameter is supported only if the group's `listManagedInstancesResults` field is set to `PAGINATED`.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/listManagedInstances

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> The name of the managed instance group.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region instanceGroupManager]
    (regionInstanceGroupManagers-listManagedInstances
      project
      region
      instanceGroupManager
      nil))
  ([project region instanceGroupManager optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/listManagedInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionInstanceGroupManagers-setInstanceTemplate
  "Sets the instance template to use when creating new instances or recreating instances in this group. Existing instances are not affected.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/setInstanceTemplate

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> The name of the managed instance group.
RegionInstanceGroupManagersSetTemplateRequest:
{:instanceTemplate string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersSetTemplateRequest]
    (regionInstanceGroupManagers-setInstanceTemplate
      project
      region
      instanceGroupManager
      RegionInstanceGroupManagersSetTemplateRequest
      nil))
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersSetTemplateRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/setInstanceTemplate",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionInstanceGroupManagersSetTemplateRequest})))

(defn regionInstanceGroupManagers-listErrors
  "Lists all errors thrown by actions on instances for a given regional managed instance group. The filter and orderBy query parameters are not supported.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/listErrors

project <string> Project ID for this request.
region <string> Name of the region scoping this request. This should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It must be a string that meets the requirements in RFC1035, or an unsigned long integer: must match regexp pattern: (?:[a-z](?:[-a-z0-9]{0,61}[a-z0-9])?)|1-9{0,19}.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region instanceGroupManager]
    (regionInstanceGroupManagers-listErrors
      project
      region
      instanceGroupManager
      nil))
  ([project region instanceGroupManager optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/listErrors",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionInstanceGroupManagers-get
  "Returns all of the details about the specified managed instance group.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> Name of the managed instance group to return."
  [project region instanceGroupManager]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}",
     :uri-template-args
     {:instanceGroupManager instanceGroupManager,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionInstanceGroupManagers-insert
  "Creates a managed instance group using the information that you specify in the request. After the group is created, instances in the group are created using the specified instance template. This operation is marked as DONE when the group is created even if the instances in the group have not yet been created. You must separately verify the status of the individual instances with the listmanagedinstances method. A regional managed instance group can contain up to 2000 instances.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
InstanceGroupManager:
{:description string,
 :updatePolicy
 {:type [OPPORTUNISTIC PROACTIVE],
  :instanceRedistributionType [NONE PROACTIVE],
  :minimalAction [NONE REFRESH REPLACE RESTART],
  :mostDisruptiveAllowedAction [NONE REFRESH REPLACE RESTART],
  :maxSurge {:fixed integer, :percent integer, :calculated integer},
  :maxUnavailable
  {:fixed integer, :percent integer, :calculated integer},
  :replacementMethod [RECREATE SUBSTITUTE]},
 :instanceLifecyclePolicy
 {:forceUpdateOnRepair [NO YES],
  :defaultActionOnFailure [DO_NOTHING REPAIR]},
 :creationTimestamp string,
 :satisfiesPzi boolean,
 :zone string,
 :name string,
 :selfLink string,
 :targetPools [string],
 :region string,
 :currentActions
 {:recreating integer,
  :restarting integer,
  :starting integer,
  :creating integer,
  :deleting integer,
  :stopping integer,
  :suspending integer,
  :refreshing integer,
  :abandoning integer,
  :resuming integer,
  :verifying integer,
  :none integer,
  :creatingWithoutRetries integer},
 :autoHealingPolicies
 [{:healthCheck string, :initialDelaySec integer}],
 :listManagedInstancesResults [PAGELESS PAGINATED],
 :satisfiesPzs boolean,
 :status
 {:isStable boolean,
  :allInstancesConfig {:effective boolean, :currentRevision string},
  :versionTarget {:isReached boolean},
  :stateful
  {:hasStatefulConfig boolean,
   :perInstanceConfigs {:allEffective boolean}},
  :autoscaler string},
 :instanceGroup string,
 :id string,
 :kind string,
 :allInstancesConfig {:properties {:metadata object, :labels object}},
 :distributionPolicy
 {:zones [{:zone string}],
  :targetShape [ANY ANY_SINGLE_ZONE BALANCED EVEN]},
 :statefulPolicy
 {:preservedState
  {:disks object, :internalIPs object, :externalIPs object}},
 :baseInstanceName string,
 :versions
 [{:name string,
   :instanceTemplate string,
   :targetSize
   {:fixed integer, :percent integer, :calculated integer}}],
 :targetSize integer,
 :instanceTemplate string,
 :fingerprint string,
 :namedPorts [{:name string, :port integer}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region InstanceGroupManager]
    (regionInstanceGroupManagers-insert
      project
      region
      InstanceGroupManager
      nil))
  ([project region InstanceGroupManager optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManager})))

(defn regionInstanceGroupManagers-abandonInstances
  "Flags the specified instances to be immediately removed from the managed instance group. Abandoning an instance does not delete the instance, but it does remove the instance from any target pools that are applied by the managed instance group. This method reduces the targetSize of the managed instance group by the number of instances that you abandon. This operation is marked as DONE when the action is scheduled even if the instances have not yet been removed from the group. You must separately verify the status of the abandoning action with the listmanagedinstances method. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration has elapsed before the VM instance is removed or deleted. You can specify a maximum of 1000 instances with this method per request.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/abandonInstances

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> Name of the managed instance group.
RegionInstanceGroupManagersAbandonInstancesRequest:
{:instances [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersAbandonInstancesRequest]
    (regionInstanceGroupManagers-abandonInstances
      project
      region
      instanceGroupManager
      RegionInstanceGroupManagersAbandonInstancesRequest
      nil))
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersAbandonInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/abandonInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionInstanceGroupManagersAbandonInstancesRequest})))

(defn regionInstanceGroupManagers-updatePerInstanceConfigs
  "Inserts or updates per-instance configurations for the managed instance group. perInstanceConfig.name serves as a key used to distinguish whether to perform insert or patch.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/updatePerInstanceConfigs

project <string> Project ID for this request.
region <string> Name of the region scoping this request, should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.
RegionInstanceGroupManagerUpdateInstanceConfigReq:
{:perInstanceConfigs
 [{:name string,
   :preservedState
   {:disks object,
    :metadata object,
    :internalIPs object,
    :externalIPs object},
   :status
   [APPLYING DELETING EFFECTIVE NONE UNAPPLIED UNAPPLIED_DELETION],
   :fingerprint string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagerUpdateInstanceConfigReq]
    (regionInstanceGroupManagers-updatePerInstanceConfigs
      project
      region
      instanceGroupManager
      RegionInstanceGroupManagerUpdateInstanceConfigReq
      nil))
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagerUpdateInstanceConfigReq
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/updatePerInstanceConfigs",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionInstanceGroupManagerUpdateInstanceConfigReq})))

(defn regionInstanceGroupManagers-recreateInstances
  "Flags the specified VM instances in the managed instance group to be immediately recreated. Each instance is recreated using the group's current configuration. This operation is marked as DONE when the flag is set even if the instances have not yet been recreated. You must separately verify the status of each instance by checking its currentAction field; for more information, see Checking the status of managed instances. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration has elapsed before the VM instance is removed or deleted. You can specify a maximum of 1000 instances with this method per request.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/recreateInstances

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> Name of the managed instance group.
RegionInstanceGroupManagersRecreateRequest:
{:instances [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersRecreateRequest]
    (regionInstanceGroupManagers-recreateInstances
      project
      region
      instanceGroupManager
      RegionInstanceGroupManagersRecreateRequest
      nil))
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersRecreateRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/recreateInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionInstanceGroupManagersRecreateRequest})))

(defn regionInstanceGroupManagers-applyUpdatesToInstances
  "Apply updates to selected instances the managed instance group.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/applyUpdatesToInstances

project <string> Project ID for this request.
region <string> Name of the region scoping this request, should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group, should conform to RFC1035.
RegionInstanceGroupManagersApplyUpdatesRequest:
{:instances [string],
 :minimalAction [NONE REFRESH REPLACE RESTART],
 :mostDisruptiveAllowedAction [NONE REFRESH REPLACE RESTART],
 :allInstances boolean}"
  [project
   region
   instanceGroupManager
   RegionInstanceGroupManagersApplyUpdatesRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/applyUpdatesToInstances",
     :uri-template-args
     {:instanceGroupManager instanceGroupManager,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionInstanceGroupManagersApplyUpdatesRequest}))

(defn regionInstanceGroupManagers-patch
  "Updates a managed instance group using the information that you specify in the request. This operation is marked as DONE when the group is patched even if the instances in the group are still in the process of being patched. You must separately verify the status of the individual instances with the listmanagedinstances method. This method supports PATCH semantics and uses the JSON merge patch format and processing rules. If you update your group to specify a new template or instance configuration, it's possible that your intended specification for each VM in the group is different from the current state of that VM. To learn how to apply an updated configuration to the VMs in a MIG, see Updating instances in a MIG.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> The name of the instance group manager.
InstanceGroupManager:
{:description string,
 :updatePolicy
 {:type [OPPORTUNISTIC PROACTIVE],
  :instanceRedistributionType [NONE PROACTIVE],
  :minimalAction [NONE REFRESH REPLACE RESTART],
  :mostDisruptiveAllowedAction [NONE REFRESH REPLACE RESTART],
  :maxSurge {:fixed integer, :percent integer, :calculated integer},
  :maxUnavailable
  {:fixed integer, :percent integer, :calculated integer},
  :replacementMethod [RECREATE SUBSTITUTE]},
 :instanceLifecyclePolicy
 {:forceUpdateOnRepair [NO YES],
  :defaultActionOnFailure [DO_NOTHING REPAIR]},
 :creationTimestamp string,
 :satisfiesPzi boolean,
 :zone string,
 :name string,
 :selfLink string,
 :targetPools [string],
 :region string,
 :currentActions
 {:recreating integer,
  :restarting integer,
  :starting integer,
  :creating integer,
  :deleting integer,
  :stopping integer,
  :suspending integer,
  :refreshing integer,
  :abandoning integer,
  :resuming integer,
  :verifying integer,
  :none integer,
  :creatingWithoutRetries integer},
 :autoHealingPolicies
 [{:healthCheck string, :initialDelaySec integer}],
 :listManagedInstancesResults [PAGELESS PAGINATED],
 :satisfiesPzs boolean,
 :status
 {:isStable boolean,
  :allInstancesConfig {:effective boolean, :currentRevision string},
  :versionTarget {:isReached boolean},
  :stateful
  {:hasStatefulConfig boolean,
   :perInstanceConfigs {:allEffective boolean}},
  :autoscaler string},
 :instanceGroup string,
 :id string,
 :kind string,
 :allInstancesConfig {:properties {:metadata object, :labels object}},
 :distributionPolicy
 {:zones [{:zone string}],
  :targetShape [ANY ANY_SINGLE_ZONE BALANCED EVEN]},
 :statefulPolicy
 {:preservedState
  {:disks object, :internalIPs object, :externalIPs object}},
 :baseInstanceName string,
 :versions
 [{:name string,
   :instanceTemplate string,
   :targetSize
   {:fixed integer, :percent integer, :calculated integer}}],
 :targetSize integer,
 :instanceTemplate string,
 :fingerprint string,
 :namedPorts [{:name string, :port integer}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region instanceGroupManager InstanceGroupManager]
    (regionInstanceGroupManagers-patch
      project
      region
      instanceGroupManager
      InstanceGroupManager
      nil))
  ([project region instanceGroupManager InstanceGroupManager optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManager})))

(defn regionInstanceGroupManagers-deletePerInstanceConfigs
  "Deletes selected per-instance configurations for the managed instance group.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/deletePerInstanceConfigs

project <string> Project ID for this request.
region <string> Name of the region scoping this request, should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.
RegionInstanceGroupManagerDeleteInstanceConfigReq:
{:names [string]}"
  [project
   region
   instanceGroupManager
   RegionInstanceGroupManagerDeleteInstanceConfigReq]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/deletePerInstanceConfigs",
     :uri-template-args
     {:instanceGroupManager instanceGroupManager,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionInstanceGroupManagerDeleteInstanceConfigReq}))

(defn regionInstanceGroupManagers-delete
  "Deletes the specified managed instance group and all of the instances in that group.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> Name of the managed instance group to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region instanceGroupManager]
    (regionInstanceGroupManagers-delete
      project
      region
      instanceGroupManager
      nil))
  ([project region instanceGroupManager optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionInstanceGroupManagers-resize
  "Changes the intended size of the managed instance group. If you increase the size, the group creates new instances using the current instance template. If you decrease the size, the group deletes one or more instances. The resize operation is marked DONE if the resize request is successful. The underlying actions take additional time. You must separately verify the status of the creating or deleting actions with the listmanagedinstances method. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration has elapsed before the VM instance is removed or deleted.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/resize

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> Name of the managed instance group.
size <integer> Number of instances that should exist in this instance group manager.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region instanceGroupManager size]
    (regionInstanceGroupManagers-resize
      project
      region
      instanceGroupManager
      size
      nil))
  ([project region instanceGroupManager size optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/resize",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {:size size} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionInstanceGroupManagers-listPerInstanceConfigs
  "Lists all of the per-instance configurations defined for the managed instance group. The orderBy query parameter is not supported.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/listPerInstanceConfigs

project <string> Project ID for this request.
region <string> Name of the region scoping this request, should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region instanceGroupManager]
    (regionInstanceGroupManagers-listPerInstanceConfigs
      project
      region
      instanceGroupManager
      nil))
  ([project region instanceGroupManager optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/listPerInstanceConfigs",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionInstanceGroupManagers-setTargetPools
  "Modifies the target pools to which all new instances in this group are assigned. Existing instances in the group are not affected.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/setTargetPools

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> Name of the managed instance group.
RegionInstanceGroupManagersSetTargetPoolsRequest:
{:targetPools [string], :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersSetTargetPoolsRequest]
    (regionInstanceGroupManagers-setTargetPools
      project
      region
      instanceGroupManager
      RegionInstanceGroupManagersSetTargetPoolsRequest
      nil))
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersSetTargetPoolsRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/setTargetPools",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionInstanceGroupManagersSetTargetPoolsRequest})))

(defn regionInstanceGroupManagers-list
  "Retrieves the list of managed instance groups that are contained within the specified region.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region]
    (regionInstanceGroupManagers-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionInstanceGroupManagers-patchPerInstanceConfigs
  "Inserts or patches per-instance configurations for the managed instance group. perInstanceConfig.name serves as a key used to distinguish whether to perform insert or patch.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/patchPerInstanceConfigs

project <string> Project ID for this request.
region <string> Name of the region scoping this request, should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.
RegionInstanceGroupManagerPatchInstanceConfigReq:
{:perInstanceConfigs
 [{:name string,
   :preservedState
   {:disks object,
    :metadata object,
    :internalIPs object,
    :externalIPs object},
   :status
   [APPLYING DELETING EFFECTIVE NONE UNAPPLIED UNAPPLIED_DELETION],
   :fingerprint string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagerPatchInstanceConfigReq]
    (regionInstanceGroupManagers-patchPerInstanceConfigs
      project
      region
      instanceGroupManager
      RegionInstanceGroupManagerPatchInstanceConfigReq
      nil))
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagerPatchInstanceConfigReq
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/patchPerInstanceConfigs",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionInstanceGroupManagerPatchInstanceConfigReq})))

(defn regionInstanceGroupManagers-createInstances
  "Creates instances with per-instance configurations in this regional managed instance group. Instances are created using the current instance template. The create instances operation is marked DONE if the createInstances request is successful. The underlying actions take additional time. You must separately verify the status of the creating or actions with the listmanagedinstances method.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/createInstances

project <string> Project ID for this request.
region <string> The name of the region where the managed instance group is located. It should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.
RegionInstanceGroupManagersCreateInstancesRequest:
{:instances
 [{:name string,
   :preservedState
   {:disks object,
    :metadata object,
    :internalIPs object,
    :externalIPs object},
   :status
   [APPLYING DELETING EFFECTIVE NONE UNAPPLIED UNAPPLIED_DELETION],
   :fingerprint string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersCreateInstancesRequest]
    (regionInstanceGroupManagers-createInstances
      project
      region
      instanceGroupManager
      RegionInstanceGroupManagersCreateInstancesRequest
      nil))
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersCreateInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/createInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionInstanceGroupManagersCreateInstancesRequest})))

(defn regionInstanceGroupManagers-deleteInstances
  "Flags the specified instances in the managed instance group to be immediately deleted. The instances are also removed from any target pools of which they were a member. This method reduces the targetSize of the managed instance group by the number of instances that you delete. The deleteInstances operation is marked DONE if the deleteInstances request is successful. The underlying actions take additional time. You must separately verify the status of the deleting action with the listmanagedinstances method. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration has elapsed before the VM instance is removed or deleted. You can specify a maximum of 1000 instances with this method per request.
https://cloud.google.com/products/compute/v1/docs/regionInstanceGroupManagers/deleteInstances

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
instanceGroupManager <string> Name of the managed instance group.
RegionInstanceGroupManagersDeleteInstancesRequest:
{:instances [string], :skipInstancesOnValidationError boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersDeleteInstancesRequest]
    (regionInstanceGroupManagers-deleteInstances
      project
      region
      instanceGroupManager
      RegionInstanceGroupManagersDeleteInstancesRequest
      nil))
  ([project
    region
    instanceGroupManager
    RegionInstanceGroupManagersDeleteInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/deleteInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionInstanceGroupManagersDeleteInstancesRequest})))

(defn regionHealthChecks-list
  "Retrieves the list of HealthCheck resources available to the specified project.
https://cloud.google.com/products/compute/v1/docs/regionHealthChecks/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionHealthChecks-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/healthChecks",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionHealthChecks-get
  "Returns the specified HealthCheck resource.
https://cloud.google.com/products/compute/v1/docs/regionHealthChecks/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
healthCheck <string> Name of the HealthCheck resource to return."
  [project region healthCheck]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/healthChecks/{healthCheck}",
     :uri-template-args
     {:healthCheck healthCheck, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionHealthChecks-insert
  "Creates a HealthCheck resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionHealthChecks/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
HealthCheck:
{:description string,
 :httpHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :sslHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :request string,
  :response string,
  :proxyHeader [NONE PROXY_V1]},
 :creationTimestamp string,
 :tcpHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :request string,
  :response string,
  :proxyHeader [NONE PROXY_V1]},
 :healthyThreshold integer,
 :name string,
 :selfLink string,
 :type [GRPC HTTP HTTP2 HTTPS INVALID SSL TCP],
 :region string,
 :http2HealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :id string,
 :httpsHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :kind string,
 :checkIntervalSec integer,
 :timeoutSec integer,
 :unhealthyThreshold integer,
 :grpcHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :grpcServiceName string},
 :logConfig {:enable boolean}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region HealthCheck]
    (regionHealthChecks-insert project region HealthCheck nil))
  ([project region HealthCheck optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/healthChecks",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body HealthCheck})))

(defn regionHealthChecks-delete
  "Deletes the specified HealthCheck resource.
https://cloud.google.com/products/compute/v1/docs/regionHealthChecks/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
healthCheck <string> Name of the HealthCheck resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region healthCheck]
    (regionHealthChecks-delete project region healthCheck nil))
  ([project region healthCheck optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/healthChecks/{healthCheck}",
       :uri-template-args
       {:healthCheck healthCheck, :project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionHealthChecks-update
  "Updates a HealthCheck resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionHealthChecks/update

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
healthCheck <string> Name of the HealthCheck resource to update.
HealthCheck:
{:description string,
 :httpHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :sslHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :request string,
  :response string,
  :proxyHeader [NONE PROXY_V1]},
 :creationTimestamp string,
 :tcpHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :request string,
  :response string,
  :proxyHeader [NONE PROXY_V1]},
 :healthyThreshold integer,
 :name string,
 :selfLink string,
 :type [GRPC HTTP HTTP2 HTTPS INVALID SSL TCP],
 :region string,
 :http2HealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :id string,
 :httpsHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :kind string,
 :checkIntervalSec integer,
 :timeoutSec integer,
 :unhealthyThreshold integer,
 :grpcHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :grpcServiceName string},
 :logConfig {:enable boolean}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region healthCheck HealthCheck]
    (regionHealthChecks-update
      project
      region
      healthCheck
      HealthCheck
      nil))
  ([project region healthCheck HealthCheck optional]
    (client/api-request
      {:method "PUT",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/healthChecks/{healthCheck}",
       :uri-template-args
       {:healthCheck healthCheck, :project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body HealthCheck})))

(defn regionHealthChecks-patch
  "Updates a HealthCheck resource in the specified project using the data included in the request. This method supports PATCH semantics and uses the JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/regionHealthChecks/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
healthCheck <string> Name of the HealthCheck resource to patch.
HealthCheck:
{:description string,
 :httpHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :sslHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :request string,
  :response string,
  :proxyHeader [NONE PROXY_V1]},
 :creationTimestamp string,
 :tcpHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :request string,
  :response string,
  :proxyHeader [NONE PROXY_V1]},
 :healthyThreshold integer,
 :name string,
 :selfLink string,
 :type [GRPC HTTP HTTP2 HTTPS INVALID SSL TCP],
 :region string,
 :http2HealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :id string,
 :httpsHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :host string,
  :requestPath string,
  :proxyHeader [NONE PROXY_V1],
  :response string},
 :kind string,
 :checkIntervalSec integer,
 :timeoutSec integer,
 :unhealthyThreshold integer,
 :grpcHealthCheck
 {:port integer,
  :portName string,
  :portSpecification [USE_FIXED_PORT USE_NAMED_PORT USE_SERVING_PORT],
  :grpcServiceName string},
 :logConfig {:enable boolean}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region healthCheck HealthCheck]
    (regionHealthChecks-patch
      project
      region
      healthCheck
      HealthCheck
      nil))
  ([project region healthCheck HealthCheck optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/healthChecks/{healthCheck}",
       :uri-template-args
       {:healthCheck healthCheck, :project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body HealthCheck})))

(defn images-get
  "Returns the specified image.
https://cloud.google.com/products/compute/v1/docs/images/get

project <string> Project ID for this request.
image <string> Name of the image resource to return."
  [project image]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/{image}",
     :uri-template-args {:image image, :project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn images-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/images/setIamPolicy

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.
GlobalSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project resource GlobalSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/{resource}/setIamPolicy",
     :uri-template-args {:project project, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body GlobalSetPolicyRequest}))

(defn images-insert
  "Creates an image in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/images/insert

project <string> Project ID for this request.
Image:
{:description string,
 :labels object,
 :sourceSnapshotEncryptionKey
 {:rawKey string,
  :rsaEncryptedKey string,
  :kmsKeyName string,
  :sha256 string,
  :kmsKeyServiceAccount string},
 :guestOsFeatures
 [{:type
   [FEATURE_TYPE_UNSPECIFIED
    GVNIC
    IDPF
    MULTI_IP_SUBNET
    SECURE_BOOT
    SEV_CAPABLE
    SEV_LIVE_MIGRATABLE
    SEV_LIVE_MIGRATABLE_V2
    SEV_SNP_CAPABLE
    UEFI_COMPATIBLE
    VIRTIO_SCSI_MULTIQUEUE
    WINDOWS]}],
 :family string,
 :sourceImage string,
 :sourceDiskId string,
 :architecture [ARCHITECTURE_UNSPECIFIED ARM64 X86_64],
 :creationTimestamp string,
 :satisfiesPzi boolean,
 :sourceImageEncryptionKey
 {:rawKey string,
  :rsaEncryptedKey string,
  :kmsKeyName string,
  :sha256 string,
  :kmsKeyServiceAccount string},
 :imageEncryptionKey
 {:rawKey string,
  :rsaEncryptedKey string,
  :kmsKeyName string,
  :sha256 string,
  :kmsKeyServiceAccount string},
 :name string,
 :sourceDisk string,
 :sourceImageId string,
 :licenses [string],
 :selfLink string,
 :sourceSnapshot string,
 :sourceDiskEncryptionKey
 {:rawKey string,
  :rsaEncryptedKey string,
  :kmsKeyName string,
  :sha256 string,
  :kmsKeyServiceAccount string},
 :satisfiesPzs boolean,
 :status [DELETING FAILED PENDING READY],
 :deprecated
 {:state [ACTIVE DELETED DEPRECATED OBSOLETE],
  :replacement string,
  :deprecated string,
  :obsolete string,
  :deleted string},
 :id string,
 :kind string,
 :sourceSnapshotId string,
 :rawDisk {:source string, :sha1Checksum string, :containerType [TAR]},
 :sourceType [RAW],
 :diskSizeGb string,
 :licenseCodes [string],
 :storageLocations [string],
 :enableConfidentialCompute boolean,
 :shieldedInstanceInitialState
 {:pk {:content string, :fileType [BIN UNDEFINED X509]},
  :keks [{:content string, :fileType [BIN UNDEFINED X509]}],
  :dbs [{:content string, :fileType [BIN UNDEFINED X509]}],
  :dbxs [{:content string, :fileType [BIN UNDEFINED X509]}]},
 :archiveSizeBytes string,
 :labelFingerprint string}

optional:
forceCreate <boolean> Force image creation if true.
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project Image] (images-insert project Image nil))
  ([project Image optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/images",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/devstorage.full_control"
        "https://www.googleapis.com/auth/devstorage.read_only"
        "https://www.googleapis.com/auth/devstorage.read_write"],
       :body Image})))

(defn images-getFromFamily
  "Returns the latest image that is part of an image family and is not deprecated. For more information on image families, see Public image families documentation.
https://cloud.google.com/products/compute/v1/docs/images/getFromFamily

project <string> The image project that the image belongs to. For example, to get a CentOS image, specify centos-cloud as the image project.
family <string> Name of the image family to search for."
  [project family]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/family/{family}",
     :uri-template-args {:family family, :project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn images-patch
  "Patches the specified image with the data included in the request. Only the following fields can be modified: family, description, deprecation status.
https://cloud.google.com/products/compute/v1/docs/images/patch

project <string> Project ID for this request.
image <string> Name of the image resource to patch.
Image:
{:description string,
 :labels object,
 :sourceSnapshotEncryptionKey
 {:rawKey string,
  :rsaEncryptedKey string,
  :kmsKeyName string,
  :sha256 string,
  :kmsKeyServiceAccount string},
 :guestOsFeatures
 [{:type
   [FEATURE_TYPE_UNSPECIFIED
    GVNIC
    IDPF
    MULTI_IP_SUBNET
    SECURE_BOOT
    SEV_CAPABLE
    SEV_LIVE_MIGRATABLE
    SEV_LIVE_MIGRATABLE_V2
    SEV_SNP_CAPABLE
    UEFI_COMPATIBLE
    VIRTIO_SCSI_MULTIQUEUE
    WINDOWS]}],
 :family string,
 :sourceImage string,
 :sourceDiskId string,
 :architecture [ARCHITECTURE_UNSPECIFIED ARM64 X86_64],
 :creationTimestamp string,
 :satisfiesPzi boolean,
 :sourceImageEncryptionKey
 {:rawKey string,
  :rsaEncryptedKey string,
  :kmsKeyName string,
  :sha256 string,
  :kmsKeyServiceAccount string},
 :imageEncryptionKey
 {:rawKey string,
  :rsaEncryptedKey string,
  :kmsKeyName string,
  :sha256 string,
  :kmsKeyServiceAccount string},
 :name string,
 :sourceDisk string,
 :sourceImageId string,
 :licenses [string],
 :selfLink string,
 :sourceSnapshot string,
 :sourceDiskEncryptionKey
 {:rawKey string,
  :rsaEncryptedKey string,
  :kmsKeyName string,
  :sha256 string,
  :kmsKeyServiceAccount string},
 :satisfiesPzs boolean,
 :status [DELETING FAILED PENDING READY],
 :deprecated
 {:state [ACTIVE DELETED DEPRECATED OBSOLETE],
  :replacement string,
  :deprecated string,
  :obsolete string,
  :deleted string},
 :id string,
 :kind string,
 :sourceSnapshotId string,
 :rawDisk {:source string, :sha1Checksum string, :containerType [TAR]},
 :sourceType [RAW],
 :diskSizeGb string,
 :licenseCodes [string],
 :storageLocations [string],
 :enableConfidentialCompute boolean,
 :shieldedInstanceInitialState
 {:pk {:content string, :fileType [BIN UNDEFINED X509]},
  :keks [{:content string, :fileType [BIN UNDEFINED X509]}],
  :dbs [{:content string, :fileType [BIN UNDEFINED X509]}],
  :dbxs [{:content string, :fileType [BIN UNDEFINED X509]}]},
 :archiveSizeBytes string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project image Image] (images-patch project image Image nil))
  ([project image Image optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/{image}",
       :uri-template-args {:image image, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body Image})))

(defn images-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/images/testIamPermissions

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/{resource}/testIamPermissions",
     :uri-template-args {:project project, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn images-setLabels
  "Sets the labels on an image. To learn more about labels, read the Labeling Resources documentation.
https://cloud.google.com/products/compute/v1/docs/images/setLabels

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.
GlobalSetLabelsRequest:
{:labels object, :labelFingerprint string}"
  [project resource GlobalSetLabelsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/{resource}/setLabels",
     :uri-template-args {:project project, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body GlobalSetLabelsRequest}))

(defn images-delete
  "Deletes the specified image.
https://cloud.google.com/products/compute/v1/docs/images/delete

project <string> Project ID for this request.
image <string> Name of the image resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project image] (images-delete project image nil))
  ([project image optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/{image}",
       :uri-template-args {:image image, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn images-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/images/getIamPolicy

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project resource] (images-getIamPolicy project resource nil))
  ([project resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/{resource}/getIamPolicy",
       :uri-template-args {:project project, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn images-list
  "Retrieves the list of custom images available to the specified project. Custom images are images you create that belong to your project. This method does not get any images that belong to other projects, including publicly-available images, like Debian 8. If you want to get a list of publicly-available images, use this method to make a request to the respective image project, such as debian-cloud or windows-cloud.
https://cloud.google.com/products/compute/v1/docs/images/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (images-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/images",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn images-deprecate
  "Sets the deprecation status of an image. If an empty request body is given, clears the deprecation status instead.
https://cloud.google.com/products/compute/v1/docs/images/deprecate

project <string> Project ID for this request.
image <string> Image name.
DeprecationStatus:
{:state [ACTIVE DELETED DEPRECATED OBSOLETE],
 :replacement string,
 :deprecated string,
 :obsolete string,
 :deleted string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project image DeprecationStatus]
    (images-deprecate project image DeprecationStatus nil))
  ([project image DeprecationStatus optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/images/{image}/deprecate",
       :uri-template-args {:image image, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body DeprecationStatus})))

(defn regionSecurityPolicies-removeRule
  "Deletes a rule at the specified priority.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/removeRule

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
securityPolicy <string> Name of the security policy to update.

optional:
priority <integer> The priority of the rule to remove from the security policy."
  ([project region securityPolicy]
    (regionSecurityPolicies-removeRule
      project
      region
      securityPolicy
      nil))
  ([project region securityPolicy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies/{securityPolicy}/removeRule",
       :uri-template-args
       {:project project,
        :region region,
        :securityPolicy securityPolicy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionSecurityPolicies-get
  "List all of the ordered rules present in a single specified policy.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
securityPolicy <string> Name of the security policy to get."
  [project region securityPolicy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies/{securityPolicy}",
     :uri-template-args
     {:project project,
      :region region,
      :securityPolicy securityPolicy},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionSecurityPolicies-insert
  "Creates a new policy in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
SecurityPolicy:
{:description string,
 :labels object,
 :userDefinedFields
 [{:name string,
   :base [IPV4 IPV6 TCP UDP],
   :offset integer,
   :size integer,
   :mask string}],
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :type [CLOUD_ARMOR CLOUD_ARMOR_EDGE CLOUD_ARMOR_NETWORK],
 :rules
 [{:description string,
   :headerAction
   {:requestHeadersToAdds [{:headerName string, :headerValue string}]},
   :networkMatch
   {:userDefinedFields [{:name string, :values [string]}],
    :srcIpRanges [string],
    :destIpRanges [string],
    :ipProtocols [string],
    :srcPorts [string],
    :destPorts [string],
    :srcRegionCodes [string],
    :srcAsns [integer]},
   :redirectOptions
   {:type [EXTERNAL_302 GOOGLE_RECAPTCHA], :target string},
   :preconfiguredWafConfig
   {:exclusions
    [{:targetRuleSet string,
      :targetRuleIds [string],
      :requestHeadersToExclude
      [{:val string,
        :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
      :requestCookiesToExclude
      [{:val string,
        :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
      :requestQueryParamsToExclude
      [{:val string,
        :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
      :requestUrisToExclude
      [{:val string,
        :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}]}]},
   :preview boolean,
   :rateLimitOptions
   {:enforceOnKeyConfigs
    [{:enforceOnKeyType
      [ALL
       HTTP_COOKIE
       HTTP_HEADER
       HTTP_PATH
       IP
       REGION_CODE
       SNI
       TLS_JA3_FINGERPRINT
       USER_IP
       XFF_IP],
      :enforceOnKeyName string}],
    :enforceOnKey
    [ALL
     HTTP_COOKIE
     HTTP_HEADER
     HTTP_PATH
     IP
     REGION_CODE
     SNI
     TLS_JA3_FINGERPRINT
     USER_IP
     XFF_IP],
    :exceedAction string,
    :banDurationSec integer,
    :enforceOnKeyName string,
    :rateLimitThreshold {:count integer, :intervalSec integer},
    :banThreshold {:count integer, :intervalSec integer},
    :conformAction string,
    :exceedRedirectOptions
    {:type [EXTERNAL_302 GOOGLE_RECAPTCHA], :target string}},
   :priority integer,
   :kind string,
   :action string,
   :match
   {:expr
    {:expression string,
     :title string,
     :description string,
     :location string},
    :exprOptions
    {:recaptchaOptions
     {:actionTokenSiteKeys [string], :sessionTokenSiteKeys [string]}},
    :versionedExpr [SRC_IPS_V1],
    :config {:srcIpRanges [string]}}}],
 :region string,
 :recaptchaOptionsConfig {:redirectSiteKey string},
 :id string,
 :adaptiveProtectionConfig
 {:layer7DdosDefenseConfig
  {:enable boolean,
   :ruleVisibility [PREMIUM STANDARD],
   :thresholdConfigs
   [{:name string,
     :autoDeployLoadThreshold number,
     :autoDeployConfidenceThreshold number,
     :autoDeployImpactedBaselineThreshold number,
     :autoDeployExpirationSec integer}]}},
 :kind string,
 :ddosProtectionConfig {:ddosProtection [ADVANCED STANDARD]},
 :advancedOptionsConfig
 {:jsonParsing [DISABLED STANDARD STANDARD_WITH_GRAPHQL],
  :jsonCustomConfig {:contentTypes [string]},
  :logLevel [NORMAL VERBOSE],
  :userIpRequestHeaders [string]},
 :fingerprint string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
validateOnly <boolean> If true, the request will not be committed."
  ([project region SecurityPolicy]
    (regionSecurityPolicies-insert project region SecurityPolicy nil))
  ([project region SecurityPolicy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body SecurityPolicy})))

(defn regionSecurityPolicies-patch
  "Patches the specified policy with the data included in the request. To clear fields in the policy, leave the fields empty and specify them in the updateMask. This cannot be used to be update the rules in the policy. Please use the per rule methods like addRule, patchRule, and removeRule instead.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
securityPolicy <string> Name of the security policy to update.
SecurityPolicy:
{:description string,
 :labels object,
 :userDefinedFields
 [{:name string,
   :base [IPV4 IPV6 TCP UDP],
   :offset integer,
   :size integer,
   :mask string}],
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :type [CLOUD_ARMOR CLOUD_ARMOR_EDGE CLOUD_ARMOR_NETWORK],
 :rules
 [{:description string,
   :headerAction
   {:requestHeadersToAdds [{:headerName string, :headerValue string}]},
   :networkMatch
   {:userDefinedFields [{:name string, :values [string]}],
    :srcIpRanges [string],
    :destIpRanges [string],
    :ipProtocols [string],
    :srcPorts [string],
    :destPorts [string],
    :srcRegionCodes [string],
    :srcAsns [integer]},
   :redirectOptions
   {:type [EXTERNAL_302 GOOGLE_RECAPTCHA], :target string},
   :preconfiguredWafConfig
   {:exclusions
    [{:targetRuleSet string,
      :targetRuleIds [string],
      :requestHeadersToExclude
      [{:val string,
        :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
      :requestCookiesToExclude
      [{:val string,
        :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
      :requestQueryParamsToExclude
      [{:val string,
        :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
      :requestUrisToExclude
      [{:val string,
        :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}]}]},
   :preview boolean,
   :rateLimitOptions
   {:enforceOnKeyConfigs
    [{:enforceOnKeyType
      [ALL
       HTTP_COOKIE
       HTTP_HEADER
       HTTP_PATH
       IP
       REGION_CODE
       SNI
       TLS_JA3_FINGERPRINT
       USER_IP
       XFF_IP],
      :enforceOnKeyName string}],
    :enforceOnKey
    [ALL
     HTTP_COOKIE
     HTTP_HEADER
     HTTP_PATH
     IP
     REGION_CODE
     SNI
     TLS_JA3_FINGERPRINT
     USER_IP
     XFF_IP],
    :exceedAction string,
    :banDurationSec integer,
    :enforceOnKeyName string,
    :rateLimitThreshold {:count integer, :intervalSec integer},
    :banThreshold {:count integer, :intervalSec integer},
    :conformAction string,
    :exceedRedirectOptions
    {:type [EXTERNAL_302 GOOGLE_RECAPTCHA], :target string}},
   :priority integer,
   :kind string,
   :action string,
   :match
   {:expr
    {:expression string,
     :title string,
     :description string,
     :location string},
    :exprOptions
    {:recaptchaOptions
     {:actionTokenSiteKeys [string], :sessionTokenSiteKeys [string]}},
    :versionedExpr [SRC_IPS_V1],
    :config {:srcIpRanges [string]}}}],
 :region string,
 :recaptchaOptionsConfig {:redirectSiteKey string},
 :id string,
 :adaptiveProtectionConfig
 {:layer7DdosDefenseConfig
  {:enable boolean,
   :ruleVisibility [PREMIUM STANDARD],
   :thresholdConfigs
   [{:name string,
     :autoDeployLoadThreshold number,
     :autoDeployConfidenceThreshold number,
     :autoDeployImpactedBaselineThreshold number,
     :autoDeployExpirationSec integer}]}},
 :kind string,
 :ddosProtectionConfig {:ddosProtection [ADVANCED STANDARD]},
 :advancedOptionsConfig
 {:jsonParsing [DISABLED STANDARD STANDARD_WITH_GRAPHQL],
  :jsonCustomConfig {:contentTypes [string]},
  :logLevel [NORMAL VERBOSE],
  :userIpRequestHeaders [string]},
 :fingerprint string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
updateMask <string> Indicates fields to be cleared as part of this request."
  ([project region securityPolicy SecurityPolicy]
    (regionSecurityPolicies-patch
      project
      region
      securityPolicy
      SecurityPolicy
      nil))
  ([project region securityPolicy SecurityPolicy optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies/{securityPolicy}",
       :uri-template-args
       {:project project,
        :region region,
        :securityPolicy securityPolicy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body SecurityPolicy})))

(defn regionSecurityPolicies-delete
  "Deletes the specified policy.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
securityPolicy <string> Name of the security policy to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region securityPolicy]
    (regionSecurityPolicies-delete project region securityPolicy nil))
  ([project region securityPolicy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies/{securityPolicy}",
       :uri-template-args
       {:project project,
        :region region,
        :securityPolicy securityPolicy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionSecurityPolicies-patchRule
  "Patches a rule at the specified priority. To clear fields in the rule, leave the fields empty and specify them in the updateMask.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/patchRule

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
securityPolicy <string> Name of the security policy to update.
SecurityPolicyRule:
{:description string,
 :headerAction
 {:requestHeadersToAdds [{:headerName string, :headerValue string}]},
 :networkMatch
 {:userDefinedFields [{:name string, :values [string]}],
  :srcIpRanges [string],
  :destIpRanges [string],
  :ipProtocols [string],
  :srcPorts [string],
  :destPorts [string],
  :srcRegionCodes [string],
  :srcAsns [integer]},
 :redirectOptions
 {:type [EXTERNAL_302 GOOGLE_RECAPTCHA], :target string},
 :preconfiguredWafConfig
 {:exclusions
  [{:targetRuleSet string,
    :targetRuleIds [string],
    :requestHeadersToExclude
    [{:val string,
      :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
    :requestCookiesToExclude
    [{:val string,
      :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
    :requestQueryParamsToExclude
    [{:val string,
      :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
    :requestUrisToExclude
    [{:val string,
      :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}]}]},
 :preview boolean,
 :rateLimitOptions
 {:enforceOnKeyConfigs
  [{:enforceOnKeyType
    [ALL
     HTTP_COOKIE
     HTTP_HEADER
     HTTP_PATH
     IP
     REGION_CODE
     SNI
     TLS_JA3_FINGERPRINT
     USER_IP
     XFF_IP],
    :enforceOnKeyName string}],
  :enforceOnKey
  [ALL
   HTTP_COOKIE
   HTTP_HEADER
   HTTP_PATH
   IP
   REGION_CODE
   SNI
   TLS_JA3_FINGERPRINT
   USER_IP
   XFF_IP],
  :exceedAction string,
  :banDurationSec integer,
  :enforceOnKeyName string,
  :rateLimitThreshold {:count integer, :intervalSec integer},
  :banThreshold {:count integer, :intervalSec integer},
  :conformAction string,
  :exceedRedirectOptions
  {:type [EXTERNAL_302 GOOGLE_RECAPTCHA], :target string}},
 :priority integer,
 :kind string,
 :action string,
 :match
 {:expr
  {:expression string,
   :title string,
   :description string,
   :location string},
  :exprOptions
  {:recaptchaOptions
   {:actionTokenSiteKeys [string], :sessionTokenSiteKeys [string]}},
  :versionedExpr [SRC_IPS_V1],
  :config {:srcIpRanges [string]}}}

optional:
priority <integer> The priority of the rule to patch.
updateMask <string> Indicates fields to be cleared as part of this request.
validateOnly <boolean> If true, the request will not be committed."
  ([project region securityPolicy SecurityPolicyRule]
    (regionSecurityPolicies-patchRule
      project
      region
      securityPolicy
      SecurityPolicyRule
      nil))
  ([project region securityPolicy SecurityPolicyRule optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies/{securityPolicy}/patchRule",
       :uri-template-args
       {:project project,
        :region region,
        :securityPolicy securityPolicy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body SecurityPolicyRule})))

(defn regionSecurityPolicies-addRule
  "Inserts a rule into a security policy.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/addRule

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
securityPolicy <string> Name of the security policy to update.
SecurityPolicyRule:
{:description string,
 :headerAction
 {:requestHeadersToAdds [{:headerName string, :headerValue string}]},
 :networkMatch
 {:userDefinedFields [{:name string, :values [string]}],
  :srcIpRanges [string],
  :destIpRanges [string],
  :ipProtocols [string],
  :srcPorts [string],
  :destPorts [string],
  :srcRegionCodes [string],
  :srcAsns [integer]},
 :redirectOptions
 {:type [EXTERNAL_302 GOOGLE_RECAPTCHA], :target string},
 :preconfiguredWafConfig
 {:exclusions
  [{:targetRuleSet string,
    :targetRuleIds [string],
    :requestHeadersToExclude
    [{:val string,
      :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
    :requestCookiesToExclude
    [{:val string,
      :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
    :requestQueryParamsToExclude
    [{:val string,
      :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}],
    :requestUrisToExclude
    [{:val string,
      :op [CONTAINS ENDS_WITH EQUALS EQUALS_ANY STARTS_WITH]}]}]},
 :preview boolean,
 :rateLimitOptions
 {:enforceOnKeyConfigs
  [{:enforceOnKeyType
    [ALL
     HTTP_COOKIE
     HTTP_HEADER
     HTTP_PATH
     IP
     REGION_CODE
     SNI
     TLS_JA3_FINGERPRINT
     USER_IP
     XFF_IP],
    :enforceOnKeyName string}],
  :enforceOnKey
  [ALL
   HTTP_COOKIE
   HTTP_HEADER
   HTTP_PATH
   IP
   REGION_CODE
   SNI
   TLS_JA3_FINGERPRINT
   USER_IP
   XFF_IP],
  :exceedAction string,
  :banDurationSec integer,
  :enforceOnKeyName string,
  :rateLimitThreshold {:count integer, :intervalSec integer},
  :banThreshold {:count integer, :intervalSec integer},
  :conformAction string,
  :exceedRedirectOptions
  {:type [EXTERNAL_302 GOOGLE_RECAPTCHA], :target string}},
 :priority integer,
 :kind string,
 :action string,
 :match
 {:expr
  {:expression string,
   :title string,
   :description string,
   :location string},
  :exprOptions
  {:recaptchaOptions
   {:actionTokenSiteKeys [string], :sessionTokenSiteKeys [string]}},
  :versionedExpr [SRC_IPS_V1],
  :config {:srcIpRanges [string]}}}

optional:
validateOnly <boolean> If true, the request will not be committed."
  ([project region securityPolicy SecurityPolicyRule]
    (regionSecurityPolicies-addRule
      project
      region
      securityPolicy
      SecurityPolicyRule
      nil))
  ([project region securityPolicy SecurityPolicyRule optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies/{securityPolicy}/addRule",
       :uri-template-args
       {:project project,
        :region region,
        :securityPolicy securityPolicy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body SecurityPolicyRule})))

(defn regionSecurityPolicies-list
  "List all the policies that have been configured for the specified project and region.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionSecurityPolicies-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionSecurityPolicies-getRule
  "Gets a rule at the specified priority.
https://cloud.google.com/products/compute/v1/docs/regionSecurityPolicies/getRule

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
securityPolicy <string> Name of the security policy to which the queried rule belongs.

optional:
priority <integer> The priority of the rule to get from the security policy."
  ([project region securityPolicy]
    (regionSecurityPolicies-getRule project region securityPolicy nil))
  ([project region securityPolicy optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/securityPolicies/{securityPolicy}/getRule",
       :uri-template-args
       {:project project,
        :region region,
        :securityPolicy securityPolicy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionOperations-list
  "Retrieves a list of Operation resources contained within the specified region.
https://cloud.google.com/products/compute/v1/docs/regionOperations/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionOperations-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/operations",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionOperations-get
  "Retrieves the specified region-specific Operations resource.
https://cloud.google.com/products/compute/v1/docs/regionOperations/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
operation <string> Name of the Operations resource to return."
  [project region operation]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/operations/{operation}",
     :uri-template-args
     {:operation operation, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionOperations-wait
  "Waits for the specified Operation resource to return as `DONE` or for the request to approach the 2 minute deadline, and retrieves the specified Operation resource. This method differs from the `GET` method in that it waits for no more than the default deadline (2 minutes) and then returns the current state of the operation, which might be `DONE` or still in progress. This method is called on a best-effort basis. Specifically: - In uncommon cases, when the server is overloaded, the request might return before the default deadline is reached, or might return after zero seconds. - If the default deadline is reached, there is no guarantee that the operation is actually done when the method returns. Be prepared to retry if the operation is not `DONE`. 
https://cloud.google.com/products/compute/v1/docs/regionOperations/wait

project <string> Project ID for this request.
region <string> Name of the region for this request.
operation <string> Name of the Operations resource to return."
  [project region operation]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/operations/{operation}/wait",
     :uri-template-args
     {:operation operation, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionOperations-delete
  "Deletes the specified region-specific Operations resource.
https://cloud.google.com/products/compute/v1/docs/regionOperations/delete

project <string> Project ID for this request.
region <string> Name of the region for this request.
operation <string> Name of the Operations resource to delete."
  [project region operation]
  (client/api-request
    {:method "DELETE",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/operations/{operation}",
     :uri-template-args
     {:operation operation, :project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"]}))

(defn vpnGateways-list
  "Retrieves a list of VPN gateways available to the specified project and region.
https://cloud.google.com/products/compute/v1/docs/vpnGateways/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (vpnGateways-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/vpnGateways",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn vpnGateways-aggregatedList
  "Retrieves an aggregated list of VPN gateways. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/vpnGateways/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (vpnGateways-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/vpnGateways",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn vpnGateways-get
  "Returns the specified VPN gateway.
https://cloud.google.com/products/compute/v1/docs/vpnGateways/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
vpnGateway <string> Name of the VPN gateway to return."
  [project region vpnGateway]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/vpnGateways/{vpnGateway}",
     :uri-template-args
     {:project project, :region region, :vpnGateway vpnGateway},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn vpnGateways-getStatus
  "Returns the status for the specified VPN gateway.
https://cloud.google.com/products/compute/v1/docs/vpnGateways/getStatus

project <string> Project ID for this request.
region <string> Name of the region for this request.
vpnGateway <string> Name of the VPN gateway to return."
  [project region vpnGateway]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/vpnGateways/{vpnGateway}/getStatus",
     :uri-template-args
     {:project project, :region region, :vpnGateway vpnGateway},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn vpnGateways-insert
  "Creates a VPN gateway in the specified project and region using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/vpnGateways/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
VpnGateway:
{:description string,
 :labels object,
 :vpnInterfaces
 [{:id integer,
   :ipAddress string,
   :ipv6Address string,
   :interconnectAttachment string}],
 :creationTimestamp string,
 :stackType [IPV4_IPV6 IPV4_ONLY IPV6_ONLY],
 :name string,
 :selfLink string,
 :region string,
 :id string,
 :kind string,
 :gatewayIpVersion [IPV4 IPV6],
 :network string,
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region VpnGateway]
    (vpnGateways-insert project region VpnGateway nil))
  ([project region VpnGateway optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/vpnGateways",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body VpnGateway})))

(defn vpnGateways-delete
  "Deletes the specified VPN gateway.
https://cloud.google.com/products/compute/v1/docs/vpnGateways/delete

project <string> Project ID for this request.
region <string> Name of the region for this request.
vpnGateway <string> Name of the VPN gateway to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region vpnGateway]
    (vpnGateways-delete project region vpnGateway nil))
  ([project region vpnGateway optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/vpnGateways/{vpnGateway}",
       :uri-template-args
       {:project project, :region region, :vpnGateway vpnGateway},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn vpnGateways-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/vpnGateways/testIamPermissions

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project region resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/vpnGateways/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn vpnGateways-setLabels
  "Sets the labels on a VpnGateway. To learn more about labels, read the Labeling Resources documentation.
https://cloud.google.com/products/compute/v1/docs/vpnGateways/setLabels

project <string> Project ID for this request.
region <string> The region for this request.
resource <string> Name or id of the resource for this request.
RegionSetLabelsRequest:
{:labels object, :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region resource RegionSetLabelsRequest]
    (vpnGateways-setLabels
      project
      region
      resource
      RegionSetLabelsRequest
      nil))
  ([project region resource RegionSetLabelsRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/vpnGateways/{resource}/setLabels",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionSetLabelsRequest})))

(defn diskTypes-list
  "Retrieves a list of disk types available to the specified project.
https://cloud.google.com/products/compute/v1/docs/diskTypes/list

project <string> Project ID for this request.
zone <string> The name of the zone for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone] (diskTypes-list project zone nil))
  ([project zone optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/diskTypes",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn diskTypes-aggregatedList
  "Retrieves an aggregated list of disk types. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/diskTypes/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (diskTypes-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/diskTypes",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn diskTypes-get
  "Returns the specified disk type.
https://cloud.google.com/products/compute/v1/docs/diskTypes/get

project <string> Project ID for this request.
zone <string> The name of the zone for this request.
diskType <string> Name of the disk type to return."
  [project zone diskType]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/diskTypes/{diskType}",
     :uri-template-args
     {:diskType diskType, :project project, :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn publicAdvertisedPrefixes-list
  "Lists the PublicAdvertisedPrefixes for a project.
https://cloud.google.com/products/compute/v1/docs/publicAdvertisedPrefixes/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (publicAdvertisedPrefixes-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/publicAdvertisedPrefixes",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn publicAdvertisedPrefixes-get
  "Returns the specified PublicAdvertisedPrefix resource.
https://cloud.google.com/products/compute/v1/docs/publicAdvertisedPrefixes/get

project <string> Project ID for this request.
publicAdvertisedPrefix <string> Name of the PublicAdvertisedPrefix resource to return."
  [project publicAdvertisedPrefix]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/publicAdvertisedPrefixes/{publicAdvertisedPrefix}",
     :uri-template-args
     {:project project,
      :publicAdvertisedPrefix publicAdvertisedPrefix},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn publicAdvertisedPrefixes-insert
  "Creates a PublicAdvertisedPrefix in the specified project using the parameters that are included in the request.
https://cloud.google.com/products/compute/v1/docs/publicAdvertisedPrefixes/insert

project <string> Project ID for this request.
PublicAdvertisedPrefix:
{:description string,
 :dnsVerificationIp string,
 :creationTimestamp string,
 :name string,
 :ipCidrRange string,
 :selfLink string,
 :byoipApiVersion [V1 V2],
 :pdpScope [GLOBAL GLOBAL_AND_REGIONAL REGIONAL],
 :status
 [ANNOUNCED_TO_INTERNET
  INITIAL
  PREFIX_CONFIGURATION_COMPLETE
  PREFIX_CONFIGURATION_IN_PROGRESS
  PREFIX_REMOVAL_IN_PROGRESS
  PTR_CONFIGURED
  READY_TO_ANNOUNCE
  REVERSE_DNS_LOOKUP_FAILED
  VALIDATED],
 :id string,
 :kind string,
 :publicDelegatedPrefixs
 [{:name string,
   :region string,
   :project string,
   :status string,
   :ipRange string}],
 :sharedSecret string,
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project PublicAdvertisedPrefix]
    (publicAdvertisedPrefixes-insert
      project
      PublicAdvertisedPrefix
      nil))
  ([project PublicAdvertisedPrefix optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/publicAdvertisedPrefixes",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body PublicAdvertisedPrefix})))

(defn publicAdvertisedPrefixes-delete
  "Deletes the specified PublicAdvertisedPrefix
https://cloud.google.com/products/compute/v1/docs/publicAdvertisedPrefixes/delete

project <string> Project ID for this request.
publicAdvertisedPrefix <string> Name of the PublicAdvertisedPrefix resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project publicAdvertisedPrefix]
    (publicAdvertisedPrefixes-delete
      project
      publicAdvertisedPrefix
      nil))
  ([project publicAdvertisedPrefix optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/publicAdvertisedPrefixes/{publicAdvertisedPrefix}",
       :uri-template-args
       {:project project,
        :publicAdvertisedPrefix publicAdvertisedPrefix},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn publicAdvertisedPrefixes-patch
  "Patches the specified Router resource with the data included in the request. This method supports PATCH semantics and uses JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/publicAdvertisedPrefixes/patch

project <string> Project ID for this request.
publicAdvertisedPrefix <string> Name of the PublicAdvertisedPrefix resource to patch.
PublicAdvertisedPrefix:
{:description string,
 :dnsVerificationIp string,
 :creationTimestamp string,
 :name string,
 :ipCidrRange string,
 :selfLink string,
 :byoipApiVersion [V1 V2],
 :pdpScope [GLOBAL GLOBAL_AND_REGIONAL REGIONAL],
 :status
 [ANNOUNCED_TO_INTERNET
  INITIAL
  PREFIX_CONFIGURATION_COMPLETE
  PREFIX_CONFIGURATION_IN_PROGRESS
  PREFIX_REMOVAL_IN_PROGRESS
  PTR_CONFIGURED
  READY_TO_ANNOUNCE
  REVERSE_DNS_LOOKUP_FAILED
  VALIDATED],
 :id string,
 :kind string,
 :publicDelegatedPrefixs
 [{:name string,
   :region string,
   :project string,
   :status string,
   :ipRange string}],
 :sharedSecret string,
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project publicAdvertisedPrefix PublicAdvertisedPrefix]
    (publicAdvertisedPrefixes-patch
      project
      publicAdvertisedPrefix
      PublicAdvertisedPrefix
      nil))
  ([project publicAdvertisedPrefix PublicAdvertisedPrefix optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/publicAdvertisedPrefixes/{publicAdvertisedPrefix}",
       :uri-template-args
       {:project project,
        :publicAdvertisedPrefix publicAdvertisedPrefix},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body PublicAdvertisedPrefix})))

(defn publicAdvertisedPrefixes-announce
  "Announces the specified PublicAdvertisedPrefix
https://cloud.google.com/products/compute/v1/docs/publicAdvertisedPrefixes/announce

project <string> Project ID for this request.
publicAdvertisedPrefix <string> The name of the public advertised prefix. It should comply with RFC1035.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project publicAdvertisedPrefix]
    (publicAdvertisedPrefixes-announce
      project
      publicAdvertisedPrefix
      nil))
  ([project publicAdvertisedPrefix optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/publicAdvertisedPrefixes/{publicAdvertisedPrefix}/announce",
       :uri-template-args
       {:project project,
        :publicAdvertisedPrefix publicAdvertisedPrefix},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn publicAdvertisedPrefixes-withdraw
  "Withdraws the specified PublicAdvertisedPrefix
https://cloud.google.com/products/compute/v1/docs/publicAdvertisedPrefixes/withdraw

project <string> Project ID for this request.
publicAdvertisedPrefix <string> The name of the public advertised prefix. It should comply with RFC1035.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project publicAdvertisedPrefix]
    (publicAdvertisedPrefixes-withdraw
      project
      publicAdvertisedPrefix
      nil))
  ([project publicAdvertisedPrefix optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/publicAdvertisedPrefixes/{publicAdvertisedPrefix}/withdraw",
       :uri-template-args
       {:project project,
        :publicAdvertisedPrefix publicAdvertisedPrefix},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn instanceGroupManagers-listManagedInstances
  "Lists all of the instances in the managed instance group. Each instance in the list has a currentAction, which indicates the action that the managed instance group is performing on the instance. For example, if the group is still creating an instance, the currentAction is CREATING. If a previous action failed, the list displays the errors for that failed action. The orderBy query parameter is not supported. The `pageToken` query parameter is supported only if the group's `listManagedInstancesResults` field is set to `PAGINATED`.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/listManagedInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone instanceGroupManager]
    (instanceGroupManagers-listManagedInstances
      project
      zone
      instanceGroupManager
      nil))
  ([project zone instanceGroupManager optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/listManagedInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceGroupManagers-setInstanceTemplate
  "Specifies the instance template to use when creating new instances in this group. The templates for existing instances in the group do not change unless you run recreateInstances, run applyUpdatesToInstances, or set the group's updatePolicy.type to PROACTIVE.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/setInstanceTemplate

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group.
InstanceGroupManagersSetInstanceTemplateRequest:
{:instanceTemplate string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersSetInstanceTemplateRequest]
    (instanceGroupManagers-setInstanceTemplate
      project
      zone
      instanceGroupManager
      InstanceGroupManagersSetInstanceTemplateRequest
      nil))
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersSetInstanceTemplateRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setInstanceTemplate",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManagersSetInstanceTemplateRequest})))

(defn instanceGroupManagers-listErrors
  "Lists all errors thrown by actions on instances for a given managed instance group. The filter and orderBy query parameters are not supported.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/listErrors

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located. It should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It must be a string that meets the requirements in RFC1035, or an unsigned long integer: must match regexp pattern: (?:[a-z](?:[-a-z0-9]{0,61}[a-z0-9])?)|1-9{0,19}.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone instanceGroupManager]
    (instanceGroupManagers-listErrors
      project
      zone
      instanceGroupManager
      nil))
  ([project zone instanceGroupManager optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/listErrors",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceGroupManagers-get
  "Returns all of the details about the specified managed instance group.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/get

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group."
  [project zone instanceGroupManager]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}",
     :uri-template-args
     {:instanceGroupManager instanceGroupManager,
      :project project,
      :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn instanceGroupManagers-insert
  "Creates a managed instance group using the information that you specify in the request. After the group is created, instances in the group are created using the specified instance template. This operation is marked as DONE when the group is created even if the instances in the group have not yet been created. You must separately verify the status of the individual instances with the listmanagedinstances method. A managed instance group can have up to 1000 VM instances per group. Please contact Cloud Support if you need an increase in this limit.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/insert

project <string> Project ID for this request.
zone <string> The name of the zone where you want to create the managed instance group.
InstanceGroupManager:
{:description string,
 :updatePolicy
 {:type [OPPORTUNISTIC PROACTIVE],
  :instanceRedistributionType [NONE PROACTIVE],
  :minimalAction [NONE REFRESH REPLACE RESTART],
  :mostDisruptiveAllowedAction [NONE REFRESH REPLACE RESTART],
  :maxSurge {:fixed integer, :percent integer, :calculated integer},
  :maxUnavailable
  {:fixed integer, :percent integer, :calculated integer},
  :replacementMethod [RECREATE SUBSTITUTE]},
 :instanceLifecyclePolicy
 {:forceUpdateOnRepair [NO YES],
  :defaultActionOnFailure [DO_NOTHING REPAIR]},
 :creationTimestamp string,
 :satisfiesPzi boolean,
 :zone string,
 :name string,
 :selfLink string,
 :targetPools [string],
 :region string,
 :currentActions
 {:recreating integer,
  :restarting integer,
  :starting integer,
  :creating integer,
  :deleting integer,
  :stopping integer,
  :suspending integer,
  :refreshing integer,
  :abandoning integer,
  :resuming integer,
  :verifying integer,
  :none integer,
  :creatingWithoutRetries integer},
 :autoHealingPolicies
 [{:healthCheck string, :initialDelaySec integer}],
 :listManagedInstancesResults [PAGELESS PAGINATED],
 :satisfiesPzs boolean,
 :status
 {:isStable boolean,
  :allInstancesConfig {:effective boolean, :currentRevision string},
  :versionTarget {:isReached boolean},
  :stateful
  {:hasStatefulConfig boolean,
   :perInstanceConfigs {:allEffective boolean}},
  :autoscaler string},
 :instanceGroup string,
 :id string,
 :kind string,
 :allInstancesConfig {:properties {:metadata object, :labels object}},
 :distributionPolicy
 {:zones [{:zone string}],
  :targetShape [ANY ANY_SINGLE_ZONE BALANCED EVEN]},
 :statefulPolicy
 {:preservedState
  {:disks object, :internalIPs object, :externalIPs object}},
 :baseInstanceName string,
 :versions
 [{:name string,
   :instanceTemplate string,
   :targetSize
   {:fixed integer, :percent integer, :calculated integer}}],
 :targetSize integer,
 :instanceTemplate string,
 :fingerprint string,
 :namedPorts [{:name string, :port integer}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone InstanceGroupManager]
    (instanceGroupManagers-insert
      project
      zone
      InstanceGroupManager
      nil))
  ([project zone InstanceGroupManager optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManager})))

(defn instanceGroupManagers-abandonInstances
  "Flags the specified instances to be removed from the managed instance group. Abandoning an instance does not delete the instance, but it does remove the instance from any target pools that are applied by the managed instance group. This method reduces the targetSize of the managed instance group by the number of instances that you abandon. This operation is marked as DONE when the action is scheduled even if the instances have not yet been removed from the group. You must separately verify the status of the abandoning action with the listmanagedinstances method. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration has elapsed before the VM instance is removed or deleted. You can specify a maximum of 1000 instances with this method per request.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/abandonInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group.
InstanceGroupManagersAbandonInstancesRequest:
{:instances [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersAbandonInstancesRequest]
    (instanceGroupManagers-abandonInstances
      project
      zone
      instanceGroupManager
      InstanceGroupManagersAbandonInstancesRequest
      nil))
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersAbandonInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/abandonInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManagersAbandonInstancesRequest})))

(defn instanceGroupManagers-updatePerInstanceConfigs
  "Inserts or updates per-instance configurations for the managed instance group. perInstanceConfig.name serves as a key used to distinguish whether to perform insert or patch.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/updatePerInstanceConfigs

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located. It should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.
InstanceGroupManagersUpdatePerInstanceConfigsReq:
{:perInstanceConfigs
 [{:name string,
   :preservedState
   {:disks object,
    :metadata object,
    :internalIPs object,
    :externalIPs object},
   :status
   [APPLYING DELETING EFFECTIVE NONE UNAPPLIED UNAPPLIED_DELETION],
   :fingerprint string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersUpdatePerInstanceConfigsReq]
    (instanceGroupManagers-updatePerInstanceConfigs
      project
      zone
      instanceGroupManager
      InstanceGroupManagersUpdatePerInstanceConfigsReq
      nil))
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersUpdatePerInstanceConfigsReq
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/updatePerInstanceConfigs",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManagersUpdatePerInstanceConfigsReq})))

(defn instanceGroupManagers-recreateInstances
  "Flags the specified VM instances in the managed instance group to be immediately recreated. Each instance is recreated using the group's current configuration. This operation is marked as DONE when the flag is set even if the instances have not yet been recreated. You must separately verify the status of each instance by checking its currentAction field; for more information, see Checking the status of managed instances. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration has elapsed before the VM instance is removed or deleted. You can specify a maximum of 1000 instances with this method per request.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/recreateInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group.
InstanceGroupManagersRecreateInstancesRequest:
{:instances [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersRecreateInstancesRequest]
    (instanceGroupManagers-recreateInstances
      project
      zone
      instanceGroupManager
      InstanceGroupManagersRecreateInstancesRequest
      nil))
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersRecreateInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/recreateInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManagersRecreateInstancesRequest})))

(defn instanceGroupManagers-applyUpdatesToInstances
  "Applies changes to selected instances on the managed instance group. This method can be used to apply new overrides and/or new versions.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/applyUpdatesToInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located. Should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group, should conform to RFC1035.
InstanceGroupManagersApplyUpdatesRequest:
{:instances [string],
 :minimalAction [NONE REFRESH REPLACE RESTART],
 :mostDisruptiveAllowedAction [NONE REFRESH REPLACE RESTART],
 :allInstances boolean}"
  [project
   zone
   instanceGroupManager
   InstanceGroupManagersApplyUpdatesRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/applyUpdatesToInstances",
     :uri-template-args
     {:instanceGroupManager instanceGroupManager,
      :project project,
      :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body InstanceGroupManagersApplyUpdatesRequest}))

(defn instanceGroupManagers-patch
  "Updates a managed instance group using the information that you specify in the request. This operation is marked as DONE when the group is patched even if the instances in the group are still in the process of being patched. You must separately verify the status of the individual instances with the listManagedInstances method. This method supports PATCH semantics and uses the JSON merge patch format and processing rules. If you update your group to specify a new template or instance configuration, it's possible that your intended specification for each VM in the group is different from the current state of that VM. To learn how to apply an updated configuration to the VMs in a MIG, see Updating instances in a MIG.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/patch

project <string> Project ID for this request.
zone <string> The name of the zone where you want to create the managed instance group.
instanceGroupManager <string> The name of the instance group manager.
InstanceGroupManager:
{:description string,
 :updatePolicy
 {:type [OPPORTUNISTIC PROACTIVE],
  :instanceRedistributionType [NONE PROACTIVE],
  :minimalAction [NONE REFRESH REPLACE RESTART],
  :mostDisruptiveAllowedAction [NONE REFRESH REPLACE RESTART],
  :maxSurge {:fixed integer, :percent integer, :calculated integer},
  :maxUnavailable
  {:fixed integer, :percent integer, :calculated integer},
  :replacementMethod [RECREATE SUBSTITUTE]},
 :instanceLifecyclePolicy
 {:forceUpdateOnRepair [NO YES],
  :defaultActionOnFailure [DO_NOTHING REPAIR]},
 :creationTimestamp string,
 :satisfiesPzi boolean,
 :zone string,
 :name string,
 :selfLink string,
 :targetPools [string],
 :region string,
 :currentActions
 {:recreating integer,
  :restarting integer,
  :starting integer,
  :creating integer,
  :deleting integer,
  :stopping integer,
  :suspending integer,
  :refreshing integer,
  :abandoning integer,
  :resuming integer,
  :verifying integer,
  :none integer,
  :creatingWithoutRetries integer},
 :autoHealingPolicies
 [{:healthCheck string, :initialDelaySec integer}],
 :listManagedInstancesResults [PAGELESS PAGINATED],
 :satisfiesPzs boolean,
 :status
 {:isStable boolean,
  :allInstancesConfig {:effective boolean, :currentRevision string},
  :versionTarget {:isReached boolean},
  :stateful
  {:hasStatefulConfig boolean,
   :perInstanceConfigs {:allEffective boolean}},
  :autoscaler string},
 :instanceGroup string,
 :id string,
 :kind string,
 :allInstancesConfig {:properties {:metadata object, :labels object}},
 :distributionPolicy
 {:zones [{:zone string}],
  :targetShape [ANY ANY_SINGLE_ZONE BALANCED EVEN]},
 :statefulPolicy
 {:preservedState
  {:disks object, :internalIPs object, :externalIPs object}},
 :baseInstanceName string,
 :versions
 [{:name string,
   :instanceTemplate string,
   :targetSize
   {:fixed integer, :percent integer, :calculated integer}}],
 :targetSize integer,
 :instanceTemplate string,
 :fingerprint string,
 :namedPorts [{:name string, :port integer}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone instanceGroupManager InstanceGroupManager]
    (instanceGroupManagers-patch
      project
      zone
      instanceGroupManager
      InstanceGroupManager
      nil))
  ([project zone instanceGroupManager InstanceGroupManager optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManager})))

(defn instanceGroupManagers-aggregatedList
  "Retrieves the list of managed instance groups and groups them by zone. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (instanceGroupManagers-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/instanceGroupManagers",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceGroupManagers-deletePerInstanceConfigs
  "Deletes selected per-instance configurations for the managed instance group.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/deletePerInstanceConfigs

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located. It should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.
InstanceGroupManagersDeletePerInstanceConfigsReq:
{:names [string]}"
  [project
   zone
   instanceGroupManager
   InstanceGroupManagersDeletePerInstanceConfigsReq]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/deletePerInstanceConfigs",
     :uri-template-args
     {:instanceGroupManager instanceGroupManager,
      :project project,
      :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body InstanceGroupManagersDeletePerInstanceConfigsReq}))

(defn instanceGroupManagers-delete
  "Deletes the specified managed instance group and all of the instances in that group. Note that the instance group must not belong to a backend service. Read Deleting an instance group for more information.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/delete

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone instanceGroupManager]
    (instanceGroupManagers-delete
      project
      zone
      instanceGroupManager
      nil))
  ([project zone instanceGroupManager optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn instanceGroupManagers-resize
  "Resizes the managed instance group. If you increase the size, the group creates new instances using the current instance template. If you decrease the size, the group deletes instances. The resize operation is marked DONE when the resize actions are scheduled even if the group has not yet added or deleted any instances. You must separately verify the status of the creating or deleting actions with the listmanagedinstances method. When resizing down, the instance group arbitrarily chooses the order in which VMs are deleted. The group takes into account some VM attributes when making the selection including: + The status of the VM instance. + The health of the VM instance. + The instance template version the VM is based on. + For regional managed instance groups, the location of the VM instance. This list is subject to change. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration has elapsed before the VM instance is removed or deleted.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/resize

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group.
size <integer> The number of running instances that the managed instance group should maintain at any given time. The group automatically adds or removes instances to maintain the number of instances specified by this parameter.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project zone instanceGroupManager size]
    (instanceGroupManagers-resize
      project
      zone
      instanceGroupManager
      size
      nil))
  ([project zone instanceGroupManager size optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resize",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {:size size} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn instanceGroupManagers-listPerInstanceConfigs
  "Lists all of the per-instance configurations defined for the managed instance group. The orderBy query parameter is not supported.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/listPerInstanceConfigs

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located. It should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone instanceGroupManager]
    (instanceGroupManagers-listPerInstanceConfigs
      project
      zone
      instanceGroupManager
      nil))
  ([project zone instanceGroupManager optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/listPerInstanceConfigs",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceGroupManagers-setTargetPools
  "Modifies the target pools to which all instances in this managed instance group are assigned. The target pools automatically apply to all of the instances in the managed instance group. This operation is marked DONE when you make the request even if the instances have not yet been added to their target pools. The change might take some time to apply to all of the instances in the group depending on the size of the group.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/setTargetPools

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group.
InstanceGroupManagersSetTargetPoolsRequest:
{:targetPools [string], :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersSetTargetPoolsRequest]
    (instanceGroupManagers-setTargetPools
      project
      zone
      instanceGroupManager
      InstanceGroupManagersSetTargetPoolsRequest
      nil))
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersSetTargetPoolsRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setTargetPools",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManagersSetTargetPoolsRequest})))

(defn instanceGroupManagers-list
  "Retrieves a list of managed instance groups that are contained within the specified project and zone.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/list

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project zone] (instanceGroupManagers-list project zone nil))
  ([project zone optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers",
       :uri-template-args {:project project, :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceGroupManagers-patchPerInstanceConfigs
  "Inserts or patches per-instance configurations for the managed instance group. perInstanceConfig.name serves as a key used to distinguish whether to perform insert or patch.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/patchPerInstanceConfigs

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located. It should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.
InstanceGroupManagersPatchPerInstanceConfigsReq:
{:perInstanceConfigs
 [{:name string,
   :preservedState
   {:disks object,
    :metadata object,
    :internalIPs object,
    :externalIPs object},
   :status
   [APPLYING DELETING EFFECTIVE NONE UNAPPLIED UNAPPLIED_DELETION],
   :fingerprint string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersPatchPerInstanceConfigsReq]
    (instanceGroupManagers-patchPerInstanceConfigs
      project
      zone
      instanceGroupManager
      InstanceGroupManagersPatchPerInstanceConfigsReq
      nil))
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersPatchPerInstanceConfigsReq
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/patchPerInstanceConfigs",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManagersPatchPerInstanceConfigsReq})))

(defn instanceGroupManagers-createInstances
  "Creates instances with per-instance configurations in this managed instance group. Instances are created using the current instance template. The create instances operation is marked DONE if the createInstances request is successful. The underlying actions take additional time. You must separately verify the status of the creating or actions with the listmanagedinstances method.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/createInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located. It should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. It should conform to RFC1035.
InstanceGroupManagersCreateInstancesRequest:
{:instances
 [{:name string,
   :preservedState
   {:disks object,
    :metadata object,
    :internalIPs object,
    :externalIPs object},
   :status
   [APPLYING DELETING EFFECTIVE NONE UNAPPLIED UNAPPLIED_DELETION],
   :fingerprint string}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersCreateInstancesRequest]
    (instanceGroupManagers-createInstances
      project
      zone
      instanceGroupManager
      InstanceGroupManagersCreateInstancesRequest
      nil))
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersCreateInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/createInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManagersCreateInstancesRequest})))

(defn instanceGroupManagers-deleteInstances
  "Flags the specified instances in the managed instance group for immediate deletion. The instances are also removed from any target pools of which they were a member. This method reduces the targetSize of the managed instance group by the number of instances that you delete. This operation is marked as DONE when the action is scheduled even if the instances are still being deleted. You must separately verify the status of the deleting action with the listmanagedinstances method. If the group is part of a backend service that has enabled connection draining, it can take up to 60 seconds after the connection draining duration has elapsed before the VM instance is removed or deleted. You can specify a maximum of 1000 instances with this method per request.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagers/deleteInstances

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located.
instanceGroupManager <string> The name of the managed instance group.
InstanceGroupManagersDeleteInstancesRequest:
{:instances [string], :skipInstancesOnValidationError boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersDeleteInstancesRequest]
    (instanceGroupManagers-deleteInstances
      project
      zone
      instanceGroupManager
      InstanceGroupManagersDeleteInstancesRequest
      nil))
  ([project
    zone
    instanceGroupManager
    InstanceGroupManagersDeleteInstancesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/deleteInstances",
       :uri-template-args
       {:instanceGroupManager instanceGroupManager,
        :project project,
        :zone zone},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceGroupManagersDeleteInstancesRequest})))

(defn targetSslProxies-setBackendService
  "Changes the BackendService for TargetSslProxy.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/setBackendService

project <string> Project ID for this request.
targetSslProxy <string> Name of the TargetSslProxy resource whose BackendService resource is to be set.
TargetSslProxiesSetBackendServiceRequest:
{:service string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetSslProxy TargetSslProxiesSetBackendServiceRequest]
    (targetSslProxies-setBackendService
      project
      targetSslProxy
      TargetSslProxiesSetBackendServiceRequest
      nil))
  ([project
    targetSslProxy
    TargetSslProxiesSetBackendServiceRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies/{targetSslProxy}/setBackendService",
       :uri-template-args
       {:project project, :targetSslProxy targetSslProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetSslProxiesSetBackendServiceRequest})))

(defn targetSslProxies-get
  "Returns the specified TargetSslProxy resource.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/get

project <string> Project ID for this request.
targetSslProxy <string> Name of the TargetSslProxy resource to return."
  [project targetSslProxy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies/{targetSslProxy}",
     :uri-template-args
     {:project project, :targetSslProxy targetSslProxy},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn targetSslProxies-insert
  "Creates a TargetSslProxy resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/insert

project <string> Project ID for this request.
TargetSslProxy:
{:description string,
 :service string,
 :proxyHeader [NONE PROXY_V1],
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :certificateMap string,
 :id string,
 :kind string,
 :sslPolicy string,
 :sslCertificates [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project TargetSslProxy]
    (targetSslProxies-insert project TargetSslProxy nil))
  ([project TargetSslProxy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetSslProxy})))

(defn targetSslProxies-setSslPolicy
  "Sets the SSL policy for TargetSslProxy. The SSL policy specifies the server-side support for SSL features. This affects connections between clients and the load balancer. They do not affect the connection between the load balancer and the backends.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/setSslPolicy

project <string> Project ID for this request.
targetSslProxy <string> Name of the TargetSslProxy resource whose SSL policy is to be set. The name must be 1-63 characters long, and comply with RFC1035.
SslPolicyReference:
{:sslPolicy string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetSslProxy SslPolicyReference]
    (targetSslProxies-setSslPolicy
      project
      targetSslProxy
      SslPolicyReference
      nil))
  ([project targetSslProxy SslPolicyReference optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies/{targetSslProxy}/setSslPolicy",
       :uri-template-args
       {:project project, :targetSslProxy targetSslProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body SslPolicyReference})))

(defn targetSslProxies-setProxyHeader
  "Changes the ProxyHeaderType for TargetSslProxy.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/setProxyHeader

project <string> Project ID for this request.
targetSslProxy <string> Name of the TargetSslProxy resource whose ProxyHeader is to be set.
TargetSslProxiesSetProxyHeaderRequest:
{:proxyHeader [NONE PROXY_V1]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetSslProxy TargetSslProxiesSetProxyHeaderRequest]
    (targetSslProxies-setProxyHeader
      project
      targetSslProxy
      TargetSslProxiesSetProxyHeaderRequest
      nil))
  ([project
    targetSslProxy
    TargetSslProxiesSetProxyHeaderRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies/{targetSslProxy}/setProxyHeader",
       :uri-template-args
       {:project project, :targetSslProxy targetSslProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetSslProxiesSetProxyHeaderRequest})))

(defn targetSslProxies-setSslCertificates
  "Changes SslCertificates for TargetSslProxy.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/setSslCertificates

project <string> Project ID for this request.
targetSslProxy <string> Name of the TargetSslProxy resource whose SslCertificate resource is to be set.
TargetSslProxiesSetSslCertificatesRequest:
{:sslCertificates [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetSslProxy TargetSslProxiesSetSslCertificatesRequest]
    (targetSslProxies-setSslCertificates
      project
      targetSslProxy
      TargetSslProxiesSetSslCertificatesRequest
      nil))
  ([project
    targetSslProxy
    TargetSslProxiesSetSslCertificatesRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies/{targetSslProxy}/setSslCertificates",
       :uri-template-args
       {:project project, :targetSslProxy targetSslProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetSslProxiesSetSslCertificatesRequest})))

(defn targetSslProxies-delete
  "Deletes the specified TargetSslProxy resource.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/delete

project <string> Project ID for this request.
targetSslProxy <string> Name of the TargetSslProxy resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetSslProxy]
    (targetSslProxies-delete project targetSslProxy nil))
  ([project targetSslProxy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies/{targetSslProxy}",
       :uri-template-args
       {:project project, :targetSslProxy targetSslProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn targetSslProxies-list
  "Retrieves the list of TargetSslProxy resources available to the specified project.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (targetSslProxies-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetSslProxies-setCertificateMap
  "Changes the Certificate Map for TargetSslProxy.
https://cloud.google.com/products/compute/v1/docs/targetSslProxies/setCertificateMap

project <string> Project ID for this request.
targetSslProxy <string> Name of the TargetSslProxy resource whose CertificateMap is to be set. The name must be 1-63 characters long, and comply with RFC1035.
TargetSslProxiesSetCertificateMapRequest:
{:certificateMap string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetSslProxy TargetSslProxiesSetCertificateMapRequest]
    (targetSslProxies-setCertificateMap
      project
      targetSslProxy
      TargetSslProxiesSetCertificateMapRequest
      nil))
  ([project
    targetSslProxy
    TargetSslProxiesSetCertificateMapRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetSslProxies/{targetSslProxy}/setCertificateMap",
       :uri-template-args
       {:project project, :targetSslProxy targetSslProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetSslProxiesSetCertificateMapRequest})))

(defn packetMirrorings-get
  "Returns the specified PacketMirroring resource.
https://cloud.google.com/products/compute/v1/docs/packetMirrorings/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
packetMirroring <string> Name of the PacketMirroring resource to return."
  [project region packetMirroring]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/packetMirrorings/{packetMirroring}",
     :uri-template-args
     {:packetMirroring packetMirroring,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn packetMirrorings-insert
  "Creates a PacketMirroring resource in the specified project and region using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/packetMirrorings/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
PacketMirroring:
{:description string,
 :enable [FALSE TRUE],
 :creationTimestamp string,
 :mirroredResources
 {:subnetworks [{:url string, :canonicalUrl string}],
  :instances [{:url string, :canonicalUrl string}],
  :tags [string]},
 :name string,
 :selfLink string,
 :region string,
 :filter
 {:cidrRanges [string],
  :IPProtocols [string],
  :direction [BOTH EGRESS INGRESS]},
 :priority integer,
 :id string,
 :kind string,
 :network {:url string, :canonicalUrl string},
 :collectorIlb {:url string, :canonicalUrl string}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region PacketMirroring]
    (packetMirrorings-insert project region PacketMirroring nil))
  ([project region PacketMirroring optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/packetMirrorings",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body PacketMirroring})))

(defn packetMirrorings-patch
  "Patches the specified PacketMirroring resource with the data included in the request. This method supports PATCH semantics and uses JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/packetMirrorings/patch

project <string> Project ID for this request.
region <string> Name of the region for this request.
packetMirroring <string> Name of the PacketMirroring resource to patch.
PacketMirroring:
{:description string,
 :enable [FALSE TRUE],
 :creationTimestamp string,
 :mirroredResources
 {:subnetworks [{:url string, :canonicalUrl string}],
  :instances [{:url string, :canonicalUrl string}],
  :tags [string]},
 :name string,
 :selfLink string,
 :region string,
 :filter
 {:cidrRanges [string],
  :IPProtocols [string],
  :direction [BOTH EGRESS INGRESS]},
 :priority integer,
 :id string,
 :kind string,
 :network {:url string, :canonicalUrl string},
 :collectorIlb {:url string, :canonicalUrl string}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region packetMirroring PacketMirroring]
    (packetMirrorings-patch
      project
      region
      packetMirroring
      PacketMirroring
      nil))
  ([project region packetMirroring PacketMirroring optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/packetMirrorings/{packetMirroring}",
       :uri-template-args
       {:packetMirroring packetMirroring,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body PacketMirroring})))

(defn packetMirrorings-delete
  "Deletes the specified PacketMirroring resource.
https://cloud.google.com/products/compute/v1/docs/packetMirrorings/delete

project <string> Project ID for this request.
region <string> Name of the region for this request.
packetMirroring <string> Name of the PacketMirroring resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region packetMirroring]
    (packetMirrorings-delete project region packetMirroring nil))
  ([project region packetMirroring optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/packetMirrorings/{packetMirroring}",
       :uri-template-args
       {:packetMirroring packetMirroring,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn packetMirrorings-list
  "Retrieves a list of PacketMirroring resources available to the specified project and region.
https://cloud.google.com/products/compute/v1/docs/packetMirrorings/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (packetMirrorings-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/packetMirrorings",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn packetMirrorings-aggregatedList
  "Retrieves an aggregated list of packetMirrorings. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/packetMirrorings/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (packetMirrorings-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/packetMirrorings",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn packetMirrorings-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/packetMirrorings/testIamPermissions

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project region resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/packetMirrorings/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn instanceTemplates-list
  "Retrieves a list of instance templates that are contained within the specified project.
https://cloud.google.com/products/compute/v1/docs/instanceTemplates/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (instanceTemplates-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/instanceTemplates",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceTemplates-get
  "Returns the specified instance template.
https://cloud.google.com/products/compute/v1/docs/instanceTemplates/get

project <string> Project ID for this request.
instanceTemplate <string> The name of the instance template."
  [project instanceTemplate]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/instanceTemplates/{instanceTemplate}",
     :uri-template-args
     {:instanceTemplate instanceTemplate, :project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn instanceTemplates-insert
  "Creates an instance template in the specified project using the data that is included in the request. If you are creating a new template to update an existing instance group, your new instance template must use the same network or, if applicable, the same subnetwork as the original template.
https://cloud.google.com/products/compute/v1/docs/instanceTemplates/insert

project <string> Project ID for this request.
InstanceTemplate:
{:description string,
 :properties
 {:description string,
  :tags {:items [string], :fingerprint string},
  :labels object,
  :shieldedInstanceConfig
  {:enableSecureBoot boolean,
   :enableVtpm boolean,
   :enableIntegrityMonitoring boolean},
  :scheduling
  {:locationHint string,
   :preemptible boolean,
   :automaticRestart boolean,
   :nodeAffinities
   [{:key string,
     :operator [IN NOT_IN OPERATOR_UNSPECIFIED],
     :values [string]}],
   :provisioningModel [SPOT STANDARD],
   :onHostMaintenance [MIGRATE TERMINATE],
   :instanceTerminationAction
   [DELETE INSTANCE_TERMINATION_ACTION_UNSPECIFIED STOP],
   :terminationTime string,
   :minNodeCpus integer,
   :localSsdRecoveryTimeout {:seconds string, :nanos integer},
   :maxRunDuration {:seconds string, :nanos integer},
   :onInstanceStopAction {:discardLocalSsd boolean}},
  :resourcePolicies [string],
  :privateIpv6GoogleAccess
  [ENABLE_BIDIRECTIONAL_ACCESS_TO_GOOGLE
   ENABLE_OUTBOUND_VM_ACCESS_TO_GOOGLE
   INHERIT_FROM_SUBNETWORK],
  :canIpForward boolean,
  :resourceManagerTags object,
  :guestAccelerators
  [{:acceleratorType string, :acceleratorCount integer}],
  :networkPerformanceConfig
  {:totalEgressBandwidthTier [DEFAULT TIER_1]},
  :machineType string,
  :confidentialInstanceConfig
  {:enableConfidentialCompute boolean,
   :confidentialInstanceType
   [CONFIDENTIAL_INSTANCE_TYPE_UNSPECIFIED SEV SEV_SNP]},
  :keyRevocationActionType
  [KEY_REVOCATION_ACTION_TYPE_UNSPECIFIED NONE STOP],
  :advancedMachineFeatures
  {:enableNestedVirtualization boolean,
   :threadsPerCore integer,
   :visibleCoreCount integer,
   :enableUefiNetworking boolean,
   :performanceMonitoringUnit
   [ARCHITECTURAL
    ENHANCED
    PERFORMANCE_MONITORING_UNIT_UNSPECIFIED
    STANDARD]},
  :disks
  [{:interface [NVME SCSI],
    :guestOsFeatures
    [{:type
      [FEATURE_TYPE_UNSPECIFIED
       GVNIC
       IDPF
       MULTI_IP_SUBNET
       SECURE_BOOT
       SEV_CAPABLE
       SEV_LIVE_MIGRATABLE
       SEV_LIVE_MIGRATABLE_V2
       SEV_SNP_CAPABLE
       UEFI_COMPATIBLE
       VIRTIO_SCSI_MULTIQUEUE
       WINDOWS]}],
    :index integer,
    :deviceName string,
    :architecture [ARCHITECTURE_UNSPECIFIED ARM64 X86_64],
    :forceAttach boolean,
    :boot boolean,
    :mode [READ_ONLY READ_WRITE],
    :licenses [string],
    :type [PERSISTENT SCRATCH],
    :source string,
    :autoDelete boolean,
    :savedState [DISK_SAVED_STATE_UNSPECIFIED PRESERVED],
    :kind string,
    :initializeParams
    {:description string,
     :provisionedThroughput string,
     :labels object,
     :sourceSnapshotEncryptionKey
     {:rawKey string,
      :rsaEncryptedKey string,
      :kmsKeyName string,
      :sha256 string,
      :kmsKeyServiceAccount string},
     :resourcePolicies [string],
     :sourceImage string,
     :architecture [ARCHITECTURE_UNSPECIFIED ARM64 X86_64],
     :sourceImageEncryptionKey
     {:rawKey string,
      :rsaEncryptedKey string,
      :kmsKeyName string,
      :sha256 string,
      :kmsKeyServiceAccount string},
     :storagePool string,
     :resourceManagerTags object,
     :licenses [string],
     :sourceSnapshot string,
     :diskType string,
     :onUpdateAction
     [RECREATE_DISK RECREATE_DISK_IF_SOURCE_CHANGED USE_EXISTING_DISK],
     :diskSizeGb string,
     :diskName string,
     :replicaZones [string],
     :enableConfidentialCompute boolean,
     :provisionedIops string},
    :diskSizeGb string,
    :diskEncryptionKey
    {:rawKey string,
     :rsaEncryptedKey string,
     :kmsKeyName string,
     :sha256 string,
     :kmsKeyServiceAccount string},
    :shieldedInstanceInitialState
    {:pk {:content string, :fileType [BIN UNDEFINED X509]},
     :keks [{:content string, :fileType [BIN UNDEFINED X509]}],
     :dbs [{:content string, :fileType [BIN UNDEFINED X509]}],
     :dbxs [{:content string, :fileType [BIN UNDEFINED X509]}]}}],
  :reservationAffinity
  {:consumeReservationType
   [ANY_RESERVATION NO_RESERVATION SPECIFIC_RESERVATION UNSPECIFIED],
   :key string,
   :values [string]},
  :networkInterfaces
  [{:internalIpv6PrefixLength integer,
    :networkAttachment string,
    :ipv6AccessType [EXTERNAL INTERNAL],
    :stackType [IPV4_IPV6 IPV4_ONLY],
    :accessConfigs
    [{:publicPtrDomainName string,
      :name string,
      :natIP string,
      :type [DIRECT_IPV6 ONE_TO_ONE_NAT],
      :externalIpv6 string,
      :kind string,
      :networkTier
      [FIXED_STANDARD
       PREMIUM
       STANDARD
       STANDARD_OVERRIDES_FIXED_STANDARD],
      :setPublicPtr boolean,
      :externalIpv6PrefixLength integer,
      :securityPolicy string}],
    :name string,
    :aliasIpRanges
    [{:ipCidrRange string, :subnetworkRangeName string}],
    :nicType [GVNIC IDPF UNSPECIFIED_NIC_TYPE VIRTIO_NET],
    :ipv6Address string,
    :ipv6AccessConfigs
    [{:publicPtrDomainName string,
      :name string,
      :natIP string,
      :type [DIRECT_IPV6 ONE_TO_ONE_NAT],
      :externalIpv6 string,
      :kind string,
      :networkTier
      [FIXED_STANDARD
       PREMIUM
       STANDARD
       STANDARD_OVERRIDES_FIXED_STANDARD],
      :setPublicPtr boolean,
      :externalIpv6PrefixLength integer,
      :securityPolicy string}],
    :kind string,
    :queueCount integer,
    :network string,
    :networkIP string,
    :fingerprint string,
    :subnetwork string}],
  :metadata
  {:kind string,
   :fingerprint string,
   :items [{:key string, :value string}]},
  :serviceAccounts [{:email string, :scopes [string]}],
  :minCpuPlatform string},
 :sourceInstance string,
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :id string,
 :sourceInstanceParams
 {:diskConfigs
  [{:deviceName string,
    :instantiateFrom
    [ATTACH_READ_ONLY
     BLANK
     CUSTOM_IMAGE
     DEFAULT
     DO_NOT_INCLUDE
     SOURCE_IMAGE
     SOURCE_IMAGE_FAMILY],
    :autoDelete boolean,
    :customImage string}]},
 :kind string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project InstanceTemplate]
    (instanceTemplates-insert project InstanceTemplate nil))
  ([project InstanceTemplate optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/instanceTemplates",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceTemplate})))

(defn instanceTemplates-delete
  "Deletes the specified instance template. Deleting an instance template is permanent and cannot be undone. It is not possible to delete templates that are already in use by a managed instance group.
https://cloud.google.com/products/compute/v1/docs/instanceTemplates/delete

project <string> Project ID for this request.
instanceTemplate <string> The name of the instance template to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project instanceTemplate]
    (instanceTemplates-delete project instanceTemplate nil))
  ([project instanceTemplate optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/instanceTemplates/{instanceTemplate}",
       :uri-template-args
       {:instanceTemplate instanceTemplate, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn instanceTemplates-aggregatedList
  "Retrieves the list of all InstanceTemplates resources, regional and global, available to the specified project. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/instanceTemplates/aggregatedList

project <string> Name of the project scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (instanceTemplates-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/instanceTemplates",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceTemplates-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/instanceTemplates/getIamPolicy

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project resource]
    (instanceTemplates-getIamPolicy project resource nil))
  ([project resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/instanceTemplates/{resource}/getIamPolicy",
       :uri-template-args {:project project, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceTemplates-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/instanceTemplates/setIamPolicy

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.
GlobalSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project resource GlobalSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/instanceTemplates/{resource}/setIamPolicy",
     :uri-template-args {:project project, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body GlobalSetPolicyRequest}))

(defn instanceTemplates-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/instanceTemplates/testIamPermissions

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/instanceTemplates/{resource}/testIamPermissions",
     :uri-template-args {:project project, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn regionNetworkFirewallPolicies-cloneRules
  "Copies rules to the specified network firewall policy.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/cloneRules

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to update.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
sourceFirewallPolicy <string> The firewall policy from which to copy rules."
  ([project region firewallPolicy]
    (regionNetworkFirewallPolicies-cloneRules
      project
      region
      firewallPolicy
      nil))
  ([project region firewallPolicy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}/cloneRules",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionNetworkFirewallPolicies-removeRule
  "Deletes a rule of the specified priority.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/removeRule

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to update.

optional:
priority <integer> The priority of the rule to remove from the firewall policy.
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region firewallPolicy]
    (regionNetworkFirewallPolicies-removeRule
      project
      region
      firewallPolicy
      nil))
  ([project region firewallPolicy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}/removeRule",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionNetworkFirewallPolicies-get
  "Returns the specified network firewall policy.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to get."
  [project region firewallPolicy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}",
     :uri-template-args
     {:firewallPolicy firewallPolicy,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionNetworkFirewallPolicies-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/setIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
RegionSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project region resource RegionSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionSetPolicyRequest}))

(defn regionNetworkFirewallPolicies-insert
  "Creates a new network firewall policy in the specified project and region.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
FirewallPolicy:
{:description string,
 :ruleTupleCount integer,
 :parent string,
 :creationTimestamp string,
 :displayName string,
 :name string,
 :selfLink string,
 :associations
 [{:name string,
   :attachmentTarget string,
   :firewallPolicyId string,
   :shortName string,
   :displayName string}],
 :rules
 [{:description string,
   :securityProfileGroup string,
   :ruleTupleCount integer,
   :disabled boolean,
   :ruleName string,
   :tlsInspect boolean,
   :targetSecureTags [{:name string, :state [EFFECTIVE INEFFECTIVE]}],
   :enableLogging boolean,
   :priority integer,
   :kind string,
   :targetResources [string],
   :targetServiceAccounts [string],
   :action string,
   :direction [EGRESS INGRESS],
   :match
   {:destFqdns [string],
    :srcThreatIntelligences [string],
    :srcSecureTags [{:name string, :state [EFFECTIVE INEFFECTIVE]}],
    :srcRegionCodes [string],
    :srcFqdns [string],
    :destRegionCodes [string],
    :layer4Configs [{:ipProtocol string, :ports [string]}],
    :srcAddressGroups [string],
    :destIpRanges [string],
    :destThreatIntelligences [string],
    :destAddressGroups [string],
    :srcIpRanges [string]}}],
 :region string,
 :shortName string,
 :selfLinkWithId string,
 :id string,
 :kind string,
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region FirewallPolicy]
    (regionNetworkFirewallPolicies-insert
      project
      region
      FirewallPolicy
      nil))
  ([project region FirewallPolicy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body FirewallPolicy})))

(defn regionNetworkFirewallPolicies-getEffectiveFirewalls
  "Returns the effective firewalls on a given network.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/getEffectiveFirewalls

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
network <string> Network reference"
  [project region network]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/getEffectiveFirewalls",
     :uri-template-args {:project project, :region region},
     :query-params {:network network},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionNetworkFirewallPolicies-patch
  "Patches the specified network firewall policy.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to update.
FirewallPolicy:
{:description string,
 :ruleTupleCount integer,
 :parent string,
 :creationTimestamp string,
 :displayName string,
 :name string,
 :selfLink string,
 :associations
 [{:name string,
   :attachmentTarget string,
   :firewallPolicyId string,
   :shortName string,
   :displayName string}],
 :rules
 [{:description string,
   :securityProfileGroup string,
   :ruleTupleCount integer,
   :disabled boolean,
   :ruleName string,
   :tlsInspect boolean,
   :targetSecureTags [{:name string, :state [EFFECTIVE INEFFECTIVE]}],
   :enableLogging boolean,
   :priority integer,
   :kind string,
   :targetResources [string],
   :targetServiceAccounts [string],
   :action string,
   :direction [EGRESS INGRESS],
   :match
   {:destFqdns [string],
    :srcThreatIntelligences [string],
    :srcSecureTags [{:name string, :state [EFFECTIVE INEFFECTIVE]}],
    :srcRegionCodes [string],
    :srcFqdns [string],
    :destRegionCodes [string],
    :layer4Configs [{:ipProtocol string, :ports [string]}],
    :srcAddressGroups [string],
    :destIpRanges [string],
    :destThreatIntelligences [string],
    :destAddressGroups [string],
    :srcIpRanges [string]}}],
 :region string,
 :shortName string,
 :selfLinkWithId string,
 :id string,
 :kind string,
 :fingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region firewallPolicy FirewallPolicy]
    (regionNetworkFirewallPolicies-patch
      project
      region
      firewallPolicy
      FirewallPolicy
      nil))
  ([project region firewallPolicy FirewallPolicy optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body FirewallPolicy})))

(defn regionNetworkFirewallPolicies-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/testIamPermissions

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project region resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn regionNetworkFirewallPolicies-delete
  "Deletes the specified network firewall policy.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region firewallPolicy]
    (regionNetworkFirewallPolicies-delete
      project
      region
      firewallPolicy
      nil))
  ([project region firewallPolicy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionNetworkFirewallPolicies-removeAssociation
  "Removes an association for the specified network firewall policy.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/removeAssociation

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to update.

optional:
name <string> Name for the association that will be removed.
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region firewallPolicy]
    (regionNetworkFirewallPolicies-removeAssociation
      project
      region
      firewallPolicy
      nil))
  ([project region firewallPolicy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}/removeAssociation",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionNetworkFirewallPolicies-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/getIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project region resource]
    (regionNetworkFirewallPolicies-getIamPolicy
      project
      region
      resource
      nil))
  ([project region resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionNetworkFirewallPolicies-patchRule
  "Patches a rule of the specified priority.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/patchRule

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to update.
FirewallPolicyRule:
{:description string,
 :securityProfileGroup string,
 :ruleTupleCount integer,
 :disabled boolean,
 :ruleName string,
 :tlsInspect boolean,
 :targetSecureTags [{:name string, :state [EFFECTIVE INEFFECTIVE]}],
 :enableLogging boolean,
 :priority integer,
 :kind string,
 :targetResources [string],
 :targetServiceAccounts [string],
 :action string,
 :direction [EGRESS INGRESS],
 :match
 {:destFqdns [string],
  :srcThreatIntelligences [string],
  :srcSecureTags [{:name string, :state [EFFECTIVE INEFFECTIVE]}],
  :srcRegionCodes [string],
  :srcFqdns [string],
  :destRegionCodes [string],
  :layer4Configs [{:ipProtocol string, :ports [string]}],
  :srcAddressGroups [string],
  :destIpRanges [string],
  :destThreatIntelligences [string],
  :destAddressGroups [string],
  :srcIpRanges [string]}}

optional:
priority <integer> The priority of the rule to patch.
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region firewallPolicy FirewallPolicyRule]
    (regionNetworkFirewallPolicies-patchRule
      project
      region
      firewallPolicy
      FirewallPolicyRule
      nil))
  ([project region firewallPolicy FirewallPolicyRule optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}/patchRule",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body FirewallPolicyRule})))

(defn regionNetworkFirewallPolicies-addRule
  "Inserts a rule into a network firewall policy.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/addRule

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to update.
FirewallPolicyRule:
{:description string,
 :securityProfileGroup string,
 :ruleTupleCount integer,
 :disabled boolean,
 :ruleName string,
 :tlsInspect boolean,
 :targetSecureTags [{:name string, :state [EFFECTIVE INEFFECTIVE]}],
 :enableLogging boolean,
 :priority integer,
 :kind string,
 :targetResources [string],
 :targetServiceAccounts [string],
 :action string,
 :direction [EGRESS INGRESS],
 :match
 {:destFqdns [string],
  :srcThreatIntelligences [string],
  :srcSecureTags [{:name string, :state [EFFECTIVE INEFFECTIVE]}],
  :srcRegionCodes [string],
  :srcFqdns [string],
  :destRegionCodes [string],
  :layer4Configs [{:ipProtocol string, :ports [string]}],
  :srcAddressGroups [string],
  :destIpRanges [string],
  :destThreatIntelligences [string],
  :destAddressGroups [string],
  :srcIpRanges [string]}}

optional:
maxPriority <integer> When rule.priority is not specified, auto choose a unused priority between minPriority and maxPriority>. This field is exclusive with rule.priority.
minPriority <integer> When rule.priority is not specified, auto choose a unused priority between minPriority and maxPriority>. This field is exclusive with rule.priority.
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region firewallPolicy FirewallPolicyRule]
    (regionNetworkFirewallPolicies-addRule
      project
      region
      firewallPolicy
      FirewallPolicyRule
      nil))
  ([project region firewallPolicy FirewallPolicyRule optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}/addRule",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body FirewallPolicyRule})))

(defn regionNetworkFirewallPolicies-list
  "Lists all the network firewall policies that have been configured for the specified project in the given region.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region]
    (regionNetworkFirewallPolicies-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionNetworkFirewallPolicies-getRule
  "Gets a rule of the specified priority.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/getRule

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to which the queried rule belongs.

optional:
priority <integer> The priority of the rule to get from the firewall policy."
  ([project region firewallPolicy]
    (regionNetworkFirewallPolicies-getRule
      project
      region
      firewallPolicy
      nil))
  ([project region firewallPolicy optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}/getRule",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionNetworkFirewallPolicies-addAssociation
  "Inserts an association for the specified network firewall policy.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/addAssociation

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to update.
FirewallPolicyAssociation:
{:name string,
 :attachmentTarget string,
 :firewallPolicyId string,
 :shortName string,
 :displayName string}

optional:
replaceExistingAssociation <boolean> Indicates whether or not to replace it if an association already exists. This is false by default, in which case an error will be returned if an association already exists.
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region firewallPolicy FirewallPolicyAssociation]
    (regionNetworkFirewallPolicies-addAssociation
      project
      region
      firewallPolicy
      FirewallPolicyAssociation
      nil))
  ([project region firewallPolicy FirewallPolicyAssociation optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}/addAssociation",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body FirewallPolicyAssociation})))

(defn regionNetworkFirewallPolicies-getAssociation
  "Gets an association with the specified name.
https://cloud.google.com/products/compute/v1/docs/regionNetworkFirewallPolicies/getAssociation

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
firewallPolicy <string> Name of the firewall policy to which the queried association belongs.

optional:
name <string> The name of the association to get from the firewall policy."
  ([project region firewallPolicy]
    (regionNetworkFirewallPolicies-getAssociation
      project
      region
      firewallPolicy
      nil))
  ([project region firewallPolicy optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/firewallPolicies/{firewallPolicy}/getAssociation",
       :uri-template-args
       {:firewallPolicy firewallPolicy,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionTargetTcpProxies-list
  "Retrieves a list of TargetTcpProxy resources available to the specified project in a given region.
https://cloud.google.com/products/compute/v1/docs/regionTargetTcpProxies/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionTargetTcpProxies-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetTcpProxies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionTargetTcpProxies-get
  "Returns the specified TargetTcpProxy resource.
https://cloud.google.com/products/compute/v1/docs/regionTargetTcpProxies/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
targetTcpProxy <string> Name of the TargetTcpProxy resource to return."
  [project region targetTcpProxy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetTcpProxies/{targetTcpProxy}",
     :uri-template-args
     {:project project,
      :region region,
      :targetTcpProxy targetTcpProxy},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionTargetTcpProxies-insert
  "Creates a TargetTcpProxy resource in the specified project and region using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/regionTargetTcpProxies/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
TargetTcpProxy:
{:description string,
 :service string,
 :proxyHeader [NONE PROXY_V1],
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :id string,
 :kind string,
 :proxyBind boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region TargetTcpProxy]
    (regionTargetTcpProxies-insert project region TargetTcpProxy nil))
  ([project region TargetTcpProxy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetTcpProxies",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetTcpProxy})))

(defn regionTargetTcpProxies-delete
  "Deletes the specified TargetTcpProxy resource.
https://cloud.google.com/products/compute/v1/docs/regionTargetTcpProxies/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
targetTcpProxy <string> Name of the TargetTcpProxy resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region targetTcpProxy]
    (regionTargetTcpProxies-delete project region targetTcpProxy nil))
  ([project region targetTcpProxy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/targetTcpProxies/{targetTcpProxy}",
       :uri-template-args
       {:project project,
        :region region,
        :targetTcpProxy targetTcpProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn globalOrganizationOperations-list
  "Retrieves a list of Operation resources contained within the specified organization.
https://cloud.google.com/products/compute/v1/docs/globalOrganizationOperations/list

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
parentId <string> Parent ID for this request.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([] (globalOrganizationOperations-list nil))
  ([optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/locations/global/operations",
       :uri-template-args {},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn globalOrganizationOperations-get
  "Retrieves the specified Operations resource. Gets a list of operations by making a `list()` request.
https://cloud.google.com/products/compute/v1/docs/globalOrganizationOperations/get

operation <string> Name of the Operations resource to return.

optional:
parentId <string> Parent ID for this request."
  ([operation] (globalOrganizationOperations-get operation nil))
  ([operation optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/locations/global/operations/{operation}",
       :uri-template-args {:operation operation},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn globalOrganizationOperations-delete
  "Deletes the specified Operations resource.
https://cloud.google.com/products/compute/v1/docs/globalOrganizationOperations/delete

operation <string> Name of the Operations resource to delete.

optional:
parentId <string> Parent ID for this request."
  ([operation] (globalOrganizationOperations-delete operation nil))
  ([operation optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/locations/global/operations/{operation}",
       :uri-template-args {:operation operation},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regions-list
  "Retrieves the list of region resources available to the specified project. To decrease latency for this method, you can optionally omit any unneeded information from the response by using a field mask. This practice is especially recommended for unused quota information (the `items.quotas` field). To exclude one or more fields, set your request's `fields` query parameter to only include the fields you need. For example, to only include the `id` and `selfLink` fields, add the query parameter `?fields=id,selfLink` to your request. This method fails if the quota information is unavailable for the region and if the organization policy constraint compute.requireBasicQuotaInResponse is enforced. This constraint, when enforced, disables the fail-open behaviour when quota information (the `items.quotas` field) is unavailable for the region. It is recommended to use the default setting for the constraint unless your application requires the fail-closed behaviour for this method.
https://cloud.google.com/products/compute/v1/docs/regions/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (regions-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regions-get
  "Returns the specified Region resource. To decrease latency for this method, you can optionally omit any unneeded information from the response by using a field mask. This practice is especially recommended for unused quota information (the `quotas` field). To exclude one or more fields, set your request's `fields` query parameter to only include the fields you need. For example, to only include the `id` and `selfLink` fields, add the query parameter `?fields=id,selfLink` to your request.
https://cloud.google.com/products/compute/v1/docs/regions/get

project <string> Project ID for this request.
region <string> Name of the region resource to return."
  [project region]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}",
     :uri-template-args {:project project, :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionBackendServices-setSecurityPolicy
  "Sets the Google Cloud Armor security policy for the specified backend service. For more information, see Google Cloud Armor Overview
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/setSecurityPolicy

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
backendService <string> Name of the BackendService resource to which the security policy should be set. The name should conform to RFC1035.
SecurityPolicyReference:
{:securityPolicy string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region backendService SecurityPolicyReference]
    (regionBackendServices-setSecurityPolicy
      project
      region
      backendService
      SecurityPolicyReference
      nil))
  ([project region backendService SecurityPolicyReference optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{backendService}/setSecurityPolicy",
       :uri-template-args
       {:backendService backendService,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body SecurityPolicyReference})))

(defn regionBackendServices-getHealth
  "Gets the most recent health check results for this regional BackendService.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/getHealth

project <string> 
region <string> Name of the region scoping this request.
backendService <string> Name of the BackendService resource for which to get health.
ResourceGroupReference:
{:group string}"
  [project region backendService ResourceGroupReference]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{backendService}/getHealth",
     :uri-template-args
     {:backendService backendService,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body ResourceGroupReference}))

(defn regionBackendServices-get
  "Returns the specified regional BackendService resource.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
backendService <string> Name of the BackendService resource to return."
  [project region backendService]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{backendService}",
     :uri-template-args
     {:backendService backendService,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionBackendServices-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/setIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
RegionSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project region resource RegionSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{resource}/setIamPolicy",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body RegionSetPolicyRequest}))

(defn regionBackendServices-insert
  "Creates a regional BackendService resource in the specified project using the data included in the request. For more information, see Backend services overview.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
BackendService:
{:description string,
 :securitySettings
 {:clientTlsPolicy string,
  :subjectAltNames [string],
  :awsV4Authentication
  {:accessKeyId string,
   :accessKey string,
   :accessKeyVersion string,
   :originRegion string}},
 :customResponseHeaders [string],
 :edgeSecurityPolicy string,
 :creationTimestamp string,
 :usedBy [{:reference string}],
 :protocol [GRPC HTTP HTTP2 HTTPS SSL TCP UDP UNSPECIFIED],
 :enableCDN boolean,
 :connectionDraining {:drainingTimeoutSec integer},
 :name string,
 :subsetting {:policy [CONSISTENT_HASH_SUBSETTING NONE]},
 :maxStreamDuration {:seconds string, :nanos integer},
 :serviceLbPolicy string,
 :portName string,
 :cdnPolicy
 {:defaultTtl integer,
  :serveWhileStale integer,
  :signedUrlKeyNames [string],
  :maxTtl integer,
  :signedUrlCacheMaxAgeSec string,
  :requestCoalescing boolean,
  :negativeCaching boolean,
  :bypassCacheOnRequestHeaders [{:headerName string}],
  :negativeCachingPolicy [{:code integer, :ttl integer}],
  :cacheKeyPolicy
  {:includeProtocol boolean,
   :includeHost boolean,
   :includeQueryString boolean,
   :queryStringWhitelist [string],
   :queryStringBlacklist [string],
   :includeHttpHeaders [string],
   :includeNamedCookies [string]},
  :clientTtl integer,
  :cacheMode
  [CACHE_ALL_STATIC
   FORCE_CACHE_ALL
   INVALID_CACHE_MODE
   USE_ORIGIN_HEADERS]},
 :outlierDetection
 {:successRateStdevFactor integer,
  :successRateMinimumHosts integer,
  :enforcingSuccessRate integer,
  :maxEjectionPercent integer,
  :successRateRequestVolume integer,
  :interval {:seconds string, :nanos integer},
  :consecutiveGatewayFailure integer,
  :baseEjectionTime {:seconds string, :nanos integer},
  :enforcingConsecutiveGatewayFailure integer,
  :consecutiveErrors integer,
  :enforcingConsecutiveErrors integer},
 :selfLink string,
 :loadBalancingScheme
 [EXTERNAL
  EXTERNAL_MANAGED
  INTERNAL
  INTERNAL_MANAGED
  INTERNAL_SELF_MANAGED
  INVALID_LOAD_BALANCING_SCHEME],
 :connectionTrackingPolicy
 {:trackingMode [INVALID_TRACKING_MODE PER_CONNECTION PER_SESSION],
  :connectionPersistenceOnUnhealthyBackends
  [ALWAYS_PERSIST DEFAULT_FOR_PROTOCOL NEVER_PERSIST],
  :idleTimeoutSec integer,
  :enableStrongAffinity boolean},
 :consistentHash
 {:httpCookie
  {:name string, :path string, :ttl {:seconds string, :nanos integer}},
  :httpHeaderName string,
  :minimumRingSize string},
 :port integer,
 :healthChecks [string],
 :region string,
 :circuitBreakers
 {:maxRequestsPerConnection integer,
  :maxConnections integer,
  :maxPendingRequests integer,
  :maxRequests integer,
  :maxRetries integer},
 :iap
 {:enabled boolean,
  :oauth2ClientId string,
  :oauth2ClientSecret string,
  :oauth2ClientSecretSha256 string},
 :compressionMode [AUTOMATIC DISABLED],
 :localityLbPolicies
 [{:policy
   {:name
    [INVALID_LB_POLICY
     LEAST_REQUEST
     MAGLEV
     ORIGINAL_DESTINATION
     RANDOM
     RING_HASH
     ROUND_ROBIN
     WEIGHTED_MAGLEV]},
   :customPolicy {:name string, :data string}}],
 :id string,
 :kind string,
 :localityLbPolicy
 [INVALID_LB_POLICY
  LEAST_REQUEST
  MAGLEV
  ORIGINAL_DESTINATION
  RANDOM
  RING_HASH
  ROUND_ROBIN
  WEIGHTED_MAGLEV],
 :failoverPolicy
 {:disableConnectionDrainOnFailover boolean,
  :dropTrafficIfUnhealthy boolean,
  :failoverRatio number},
 :sessionAffinity
 [CLIENT_IP
  CLIENT_IP_NO_DESTINATION
  CLIENT_IP_PORT_PROTO
  CLIENT_IP_PROTO
  GENERATED_COOKIE
  HEADER_FIELD
  HTTP_COOKIE
  NONE],
 :timeoutSec integer,
 :network string,
 :backends
 [{:description string,
   :failover boolean,
   :capacityScaler number,
   :group string,
   :maxRatePerInstance number,
   :maxConnectionsPerEndpoint integer,
   :balancingMode [CONNECTION RATE UTILIZATION],
   :maxConnections integer,
   :maxRatePerEndpoint number,
   :maxConnectionsPerInstance integer,
   :preference [DEFAULT PREFERENCE_UNSPECIFIED PREFERRED],
   :maxUtilization number,
   :maxRate integer}],
 :customRequestHeaders [string],
 :logConfig
 {:enable boolean,
  :sampleRate number,
  :optionalMode [CUSTOM EXCLUDE_ALL_OPTIONAL INCLUDE_ALL_OPTIONAL],
  :optionalFields [string]},
 :fingerprint string,
 :affinityCookieTtlSec integer,
 :metadatas object,
 :securityPolicy string,
 :serviceBindings [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region BackendService]
    (regionBackendServices-insert project region BackendService nil))
  ([project region BackendService optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body BackendService})))

(defn regionBackendServices-listUsable
  "Retrieves an aggregated list of all usable backend services in the specified project in the given region.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/listUsable

project <string> Project ID for this request.
region <string> Name of the region scoping this request. It must be a string that meets the requirements in RFC1035.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region]
    (regionBackendServices-listUsable project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/listUsable",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionBackendServices-patch
  "Updates the specified regional BackendService resource with the data included in the request. For more information, see Understanding backend services This method supports PATCH semantics and uses the JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
backendService <string> Name of the BackendService resource to patch.
BackendService:
{:description string,
 :securitySettings
 {:clientTlsPolicy string,
  :subjectAltNames [string],
  :awsV4Authentication
  {:accessKeyId string,
   :accessKey string,
   :accessKeyVersion string,
   :originRegion string}},
 :customResponseHeaders [string],
 :edgeSecurityPolicy string,
 :creationTimestamp string,
 :usedBy [{:reference string}],
 :protocol [GRPC HTTP HTTP2 HTTPS SSL TCP UDP UNSPECIFIED],
 :enableCDN boolean,
 :connectionDraining {:drainingTimeoutSec integer},
 :name string,
 :subsetting {:policy [CONSISTENT_HASH_SUBSETTING NONE]},
 :maxStreamDuration {:seconds string, :nanos integer},
 :serviceLbPolicy string,
 :portName string,
 :cdnPolicy
 {:defaultTtl integer,
  :serveWhileStale integer,
  :signedUrlKeyNames [string],
  :maxTtl integer,
  :signedUrlCacheMaxAgeSec string,
  :requestCoalescing boolean,
  :negativeCaching boolean,
  :bypassCacheOnRequestHeaders [{:headerName string}],
  :negativeCachingPolicy [{:code integer, :ttl integer}],
  :cacheKeyPolicy
  {:includeProtocol boolean,
   :includeHost boolean,
   :includeQueryString boolean,
   :queryStringWhitelist [string],
   :queryStringBlacklist [string],
   :includeHttpHeaders [string],
   :includeNamedCookies [string]},
  :clientTtl integer,
  :cacheMode
  [CACHE_ALL_STATIC
   FORCE_CACHE_ALL
   INVALID_CACHE_MODE
   USE_ORIGIN_HEADERS]},
 :outlierDetection
 {:successRateStdevFactor integer,
  :successRateMinimumHosts integer,
  :enforcingSuccessRate integer,
  :maxEjectionPercent integer,
  :successRateRequestVolume integer,
  :interval {:seconds string, :nanos integer},
  :consecutiveGatewayFailure integer,
  :baseEjectionTime {:seconds string, :nanos integer},
  :enforcingConsecutiveGatewayFailure integer,
  :consecutiveErrors integer,
  :enforcingConsecutiveErrors integer},
 :selfLink string,
 :loadBalancingScheme
 [EXTERNAL
  EXTERNAL_MANAGED
  INTERNAL
  INTERNAL_MANAGED
  INTERNAL_SELF_MANAGED
  INVALID_LOAD_BALANCING_SCHEME],
 :connectionTrackingPolicy
 {:trackingMode [INVALID_TRACKING_MODE PER_CONNECTION PER_SESSION],
  :connectionPersistenceOnUnhealthyBackends
  [ALWAYS_PERSIST DEFAULT_FOR_PROTOCOL NEVER_PERSIST],
  :idleTimeoutSec integer,
  :enableStrongAffinity boolean},
 :consistentHash
 {:httpCookie
  {:name string, :path string, :ttl {:seconds string, :nanos integer}},
  :httpHeaderName string,
  :minimumRingSize string},
 :port integer,
 :healthChecks [string],
 :region string,
 :circuitBreakers
 {:maxRequestsPerConnection integer,
  :maxConnections integer,
  :maxPendingRequests integer,
  :maxRequests integer,
  :maxRetries integer},
 :iap
 {:enabled boolean,
  :oauth2ClientId string,
  :oauth2ClientSecret string,
  :oauth2ClientSecretSha256 string},
 :compressionMode [AUTOMATIC DISABLED],
 :localityLbPolicies
 [{:policy
   {:name
    [INVALID_LB_POLICY
     LEAST_REQUEST
     MAGLEV
     ORIGINAL_DESTINATION
     RANDOM
     RING_HASH
     ROUND_ROBIN
     WEIGHTED_MAGLEV]},
   :customPolicy {:name string, :data string}}],
 :id string,
 :kind string,
 :localityLbPolicy
 [INVALID_LB_POLICY
  LEAST_REQUEST
  MAGLEV
  ORIGINAL_DESTINATION
  RANDOM
  RING_HASH
  ROUND_ROBIN
  WEIGHTED_MAGLEV],
 :failoverPolicy
 {:disableConnectionDrainOnFailover boolean,
  :dropTrafficIfUnhealthy boolean,
  :failoverRatio number},
 :sessionAffinity
 [CLIENT_IP
  CLIENT_IP_NO_DESTINATION
  CLIENT_IP_PORT_PROTO
  CLIENT_IP_PROTO
  GENERATED_COOKIE
  HEADER_FIELD
  HTTP_COOKIE
  NONE],
 :timeoutSec integer,
 :network string,
 :backends
 [{:description string,
   :failover boolean,
   :capacityScaler number,
   :group string,
   :maxRatePerInstance number,
   :maxConnectionsPerEndpoint integer,
   :balancingMode [CONNECTION RATE UTILIZATION],
   :maxConnections integer,
   :maxRatePerEndpoint number,
   :maxConnectionsPerInstance integer,
   :preference [DEFAULT PREFERENCE_UNSPECIFIED PREFERRED],
   :maxUtilization number,
   :maxRate integer}],
 :customRequestHeaders [string],
 :logConfig
 {:enable boolean,
  :sampleRate number,
  :optionalMode [CUSTOM EXCLUDE_ALL_OPTIONAL INCLUDE_ALL_OPTIONAL],
  :optionalFields [string]},
 :fingerprint string,
 :affinityCookieTtlSec integer,
 :metadatas object,
 :securityPolicy string,
 :serviceBindings [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region backendService BackendService]
    (regionBackendServices-patch
      project
      region
      backendService
      BackendService
      nil))
  ([project region backendService BackendService optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{backendService}",
       :uri-template-args
       {:backendService backendService,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body BackendService})))

(defn regionBackendServices-testIamPermissions
  "Returns permissions that a caller has on the specified resource.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/testIamPermissions

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project region resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{resource}/testIamPermissions",
     :uri-template-args
     {:project project, :region region, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn regionBackendServices-update
  "Updates the specified regional BackendService resource with the data included in the request. For more information, see Backend services overview .
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/update

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
backendService <string> Name of the BackendService resource to update.
BackendService:
{:description string,
 :securitySettings
 {:clientTlsPolicy string,
  :subjectAltNames [string],
  :awsV4Authentication
  {:accessKeyId string,
   :accessKey string,
   :accessKeyVersion string,
   :originRegion string}},
 :customResponseHeaders [string],
 :edgeSecurityPolicy string,
 :creationTimestamp string,
 :usedBy [{:reference string}],
 :protocol [GRPC HTTP HTTP2 HTTPS SSL TCP UDP UNSPECIFIED],
 :enableCDN boolean,
 :connectionDraining {:drainingTimeoutSec integer},
 :name string,
 :subsetting {:policy [CONSISTENT_HASH_SUBSETTING NONE]},
 :maxStreamDuration {:seconds string, :nanos integer},
 :serviceLbPolicy string,
 :portName string,
 :cdnPolicy
 {:defaultTtl integer,
  :serveWhileStale integer,
  :signedUrlKeyNames [string],
  :maxTtl integer,
  :signedUrlCacheMaxAgeSec string,
  :requestCoalescing boolean,
  :negativeCaching boolean,
  :bypassCacheOnRequestHeaders [{:headerName string}],
  :negativeCachingPolicy [{:code integer, :ttl integer}],
  :cacheKeyPolicy
  {:includeProtocol boolean,
   :includeHost boolean,
   :includeQueryString boolean,
   :queryStringWhitelist [string],
   :queryStringBlacklist [string],
   :includeHttpHeaders [string],
   :includeNamedCookies [string]},
  :clientTtl integer,
  :cacheMode
  [CACHE_ALL_STATIC
   FORCE_CACHE_ALL
   INVALID_CACHE_MODE
   USE_ORIGIN_HEADERS]},
 :outlierDetection
 {:successRateStdevFactor integer,
  :successRateMinimumHosts integer,
  :enforcingSuccessRate integer,
  :maxEjectionPercent integer,
  :successRateRequestVolume integer,
  :interval {:seconds string, :nanos integer},
  :consecutiveGatewayFailure integer,
  :baseEjectionTime {:seconds string, :nanos integer},
  :enforcingConsecutiveGatewayFailure integer,
  :consecutiveErrors integer,
  :enforcingConsecutiveErrors integer},
 :selfLink string,
 :loadBalancingScheme
 [EXTERNAL
  EXTERNAL_MANAGED
  INTERNAL
  INTERNAL_MANAGED
  INTERNAL_SELF_MANAGED
  INVALID_LOAD_BALANCING_SCHEME],
 :connectionTrackingPolicy
 {:trackingMode [INVALID_TRACKING_MODE PER_CONNECTION PER_SESSION],
  :connectionPersistenceOnUnhealthyBackends
  [ALWAYS_PERSIST DEFAULT_FOR_PROTOCOL NEVER_PERSIST],
  :idleTimeoutSec integer,
  :enableStrongAffinity boolean},
 :consistentHash
 {:httpCookie
  {:name string, :path string, :ttl {:seconds string, :nanos integer}},
  :httpHeaderName string,
  :minimumRingSize string},
 :port integer,
 :healthChecks [string],
 :region string,
 :circuitBreakers
 {:maxRequestsPerConnection integer,
  :maxConnections integer,
  :maxPendingRequests integer,
  :maxRequests integer,
  :maxRetries integer},
 :iap
 {:enabled boolean,
  :oauth2ClientId string,
  :oauth2ClientSecret string,
  :oauth2ClientSecretSha256 string},
 :compressionMode [AUTOMATIC DISABLED],
 :localityLbPolicies
 [{:policy
   {:name
    [INVALID_LB_POLICY
     LEAST_REQUEST
     MAGLEV
     ORIGINAL_DESTINATION
     RANDOM
     RING_HASH
     ROUND_ROBIN
     WEIGHTED_MAGLEV]},
   :customPolicy {:name string, :data string}}],
 :id string,
 :kind string,
 :localityLbPolicy
 [INVALID_LB_POLICY
  LEAST_REQUEST
  MAGLEV
  ORIGINAL_DESTINATION
  RANDOM
  RING_HASH
  ROUND_ROBIN
  WEIGHTED_MAGLEV],
 :failoverPolicy
 {:disableConnectionDrainOnFailover boolean,
  :dropTrafficIfUnhealthy boolean,
  :failoverRatio number},
 :sessionAffinity
 [CLIENT_IP
  CLIENT_IP_NO_DESTINATION
  CLIENT_IP_PORT_PROTO
  CLIENT_IP_PROTO
  GENERATED_COOKIE
  HEADER_FIELD
  HTTP_COOKIE
  NONE],
 :timeoutSec integer,
 :network string,
 :backends
 [{:description string,
   :failover boolean,
   :capacityScaler number,
   :group string,
   :maxRatePerInstance number,
   :maxConnectionsPerEndpoint integer,
   :balancingMode [CONNECTION RATE UTILIZATION],
   :maxConnections integer,
   :maxRatePerEndpoint number,
   :maxConnectionsPerInstance integer,
   :preference [DEFAULT PREFERENCE_UNSPECIFIED PREFERRED],
   :maxUtilization number,
   :maxRate integer}],
 :customRequestHeaders [string],
 :logConfig
 {:enable boolean,
  :sampleRate number,
  :optionalMode [CUSTOM EXCLUDE_ALL_OPTIONAL INCLUDE_ALL_OPTIONAL],
  :optionalFields [string]},
 :fingerprint string,
 :affinityCookieTtlSec integer,
 :metadatas object,
 :securityPolicy string,
 :serviceBindings [string]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region backendService BackendService]
    (regionBackendServices-update
      project
      region
      backendService
      BackendService
      nil))
  ([project region backendService BackendService optional]
    (client/api-request
      {:method "PUT",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{backendService}",
       :uri-template-args
       {:backendService backendService,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body BackendService})))

(defn regionBackendServices-delete
  "Deletes the specified regional BackendService resource.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
backendService <string> Name of the BackendService resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region backendService]
    (regionBackendServices-delete project region backendService nil))
  ([project region backendService optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{backendService}",
       :uri-template-args
       {:backendService backendService,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionBackendServices-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/getIamPolicy

project <string> Project ID for this request.
region <string> The name of the region for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project region resource]
    (regionBackendServices-getIamPolicy project region resource nil))
  ([project region resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices/{resource}/getIamPolicy",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionBackendServices-list
  "Retrieves the list of regional BackendService resources available to the specified project in the given region.
https://cloud.google.com/products/compute/v1/docs/regionBackendServices/list

project <string> Project ID for this request.
region <string> Name of the region scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionBackendServices-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/backendServices",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionInstanceTemplates-list
  "Retrieves a list of instance templates that are contained within the specified project and region.
https://cloud.google.com/products/compute/v1/docs/regionInstanceTemplates/list

project <string> Project ID for this request.
region <string> The name of the regions for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionInstanceTemplates-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceTemplates",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionInstanceTemplates-get
  "Returns the specified instance template.
https://cloud.google.com/products/compute/v1/docs/regionInstanceTemplates/get

project <string> Project ID for this request.
region <string> The name of the region for this request.
instanceTemplate <string> The name of the instance template."
  [project region instanceTemplate]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceTemplates/{instanceTemplate}",
     :uri-template-args
     {:instanceTemplate instanceTemplate,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionInstanceTemplates-insert
  "Creates an instance template in the specified project and region using the global instance template whose URL is included in the request.
https://cloud.google.com/products/compute/v1/docs/regionInstanceTemplates/insert

project <string> Project ID for this request.
region <string> The name of the region for this request.
InstanceTemplate:
{:description string,
 :properties
 {:description string,
  :tags {:items [string], :fingerprint string},
  :labels object,
  :shieldedInstanceConfig
  {:enableSecureBoot boolean,
   :enableVtpm boolean,
   :enableIntegrityMonitoring boolean},
  :scheduling
  {:locationHint string,
   :preemptible boolean,
   :automaticRestart boolean,
   :nodeAffinities
   [{:key string,
     :operator [IN NOT_IN OPERATOR_UNSPECIFIED],
     :values [string]}],
   :provisioningModel [SPOT STANDARD],
   :onHostMaintenance [MIGRATE TERMINATE],
   :instanceTerminationAction
   [DELETE INSTANCE_TERMINATION_ACTION_UNSPECIFIED STOP],
   :terminationTime string,
   :minNodeCpus integer,
   :localSsdRecoveryTimeout {:seconds string, :nanos integer},
   :maxRunDuration {:seconds string, :nanos integer},
   :onInstanceStopAction {:discardLocalSsd boolean}},
  :resourcePolicies [string],
  :privateIpv6GoogleAccess
  [ENABLE_BIDIRECTIONAL_ACCESS_TO_GOOGLE
   ENABLE_OUTBOUND_VM_ACCESS_TO_GOOGLE
   INHERIT_FROM_SUBNETWORK],
  :canIpForward boolean,
  :resourceManagerTags object,
  :guestAccelerators
  [{:acceleratorType string, :acceleratorCount integer}],
  :networkPerformanceConfig
  {:totalEgressBandwidthTier [DEFAULT TIER_1]},
  :machineType string,
  :confidentialInstanceConfig
  {:enableConfidentialCompute boolean,
   :confidentialInstanceType
   [CONFIDENTIAL_INSTANCE_TYPE_UNSPECIFIED SEV SEV_SNP]},
  :keyRevocationActionType
  [KEY_REVOCATION_ACTION_TYPE_UNSPECIFIED NONE STOP],
  :advancedMachineFeatures
  {:enableNestedVirtualization boolean,
   :threadsPerCore integer,
   :visibleCoreCount integer,
   :enableUefiNetworking boolean,
   :performanceMonitoringUnit
   [ARCHITECTURAL
    ENHANCED
    PERFORMANCE_MONITORING_UNIT_UNSPECIFIED
    STANDARD]},
  :disks
  [{:interface [NVME SCSI],
    :guestOsFeatures
    [{:type
      [FEATURE_TYPE_UNSPECIFIED
       GVNIC
       IDPF
       MULTI_IP_SUBNET
       SECURE_BOOT
       SEV_CAPABLE
       SEV_LIVE_MIGRATABLE
       SEV_LIVE_MIGRATABLE_V2
       SEV_SNP_CAPABLE
       UEFI_COMPATIBLE
       VIRTIO_SCSI_MULTIQUEUE
       WINDOWS]}],
    :index integer,
    :deviceName string,
    :architecture [ARCHITECTURE_UNSPECIFIED ARM64 X86_64],
    :forceAttach boolean,
    :boot boolean,
    :mode [READ_ONLY READ_WRITE],
    :licenses [string],
    :type [PERSISTENT SCRATCH],
    :source string,
    :autoDelete boolean,
    :savedState [DISK_SAVED_STATE_UNSPECIFIED PRESERVED],
    :kind string,
    :initializeParams
    {:description string,
     :provisionedThroughput string,
     :labels object,
     :sourceSnapshotEncryptionKey
     {:rawKey string,
      :rsaEncryptedKey string,
      :kmsKeyName string,
      :sha256 string,
      :kmsKeyServiceAccount string},
     :resourcePolicies [string],
     :sourceImage string,
     :architecture [ARCHITECTURE_UNSPECIFIED ARM64 X86_64],
     :sourceImageEncryptionKey
     {:rawKey string,
      :rsaEncryptedKey string,
      :kmsKeyName string,
      :sha256 string,
      :kmsKeyServiceAccount string},
     :storagePool string,
     :resourceManagerTags object,
     :licenses [string],
     :sourceSnapshot string,
     :diskType string,
     :onUpdateAction
     [RECREATE_DISK RECREATE_DISK_IF_SOURCE_CHANGED USE_EXISTING_DISK],
     :diskSizeGb string,
     :diskName string,
     :replicaZones [string],
     :enableConfidentialCompute boolean,
     :provisionedIops string},
    :diskSizeGb string,
    :diskEncryptionKey
    {:rawKey string,
     :rsaEncryptedKey string,
     :kmsKeyName string,
     :sha256 string,
     :kmsKeyServiceAccount string},
    :shieldedInstanceInitialState
    {:pk {:content string, :fileType [BIN UNDEFINED X509]},
     :keks [{:content string, :fileType [BIN UNDEFINED X509]}],
     :dbs [{:content string, :fileType [BIN UNDEFINED X509]}],
     :dbxs [{:content string, :fileType [BIN UNDEFINED X509]}]}}],
  :reservationAffinity
  {:consumeReservationType
   [ANY_RESERVATION NO_RESERVATION SPECIFIC_RESERVATION UNSPECIFIED],
   :key string,
   :values [string]},
  :networkInterfaces
  [{:internalIpv6PrefixLength integer,
    :networkAttachment string,
    :ipv6AccessType [EXTERNAL INTERNAL],
    :stackType [IPV4_IPV6 IPV4_ONLY],
    :accessConfigs
    [{:publicPtrDomainName string,
      :name string,
      :natIP string,
      :type [DIRECT_IPV6 ONE_TO_ONE_NAT],
      :externalIpv6 string,
      :kind string,
      :networkTier
      [FIXED_STANDARD
       PREMIUM
       STANDARD
       STANDARD_OVERRIDES_FIXED_STANDARD],
      :setPublicPtr boolean,
      :externalIpv6PrefixLength integer,
      :securityPolicy string}],
    :name string,
    :aliasIpRanges
    [{:ipCidrRange string, :subnetworkRangeName string}],
    :nicType [GVNIC IDPF UNSPECIFIED_NIC_TYPE VIRTIO_NET],
    :ipv6Address string,
    :ipv6AccessConfigs
    [{:publicPtrDomainName string,
      :name string,
      :natIP string,
      :type [DIRECT_IPV6 ONE_TO_ONE_NAT],
      :externalIpv6 string,
      :kind string,
      :networkTier
      [FIXED_STANDARD
       PREMIUM
       STANDARD
       STANDARD_OVERRIDES_FIXED_STANDARD],
      :setPublicPtr boolean,
      :externalIpv6PrefixLength integer,
      :securityPolicy string}],
    :kind string,
    :queueCount integer,
    :network string,
    :networkIP string,
    :fingerprint string,
    :subnetwork string}],
  :metadata
  {:kind string,
   :fingerprint string,
   :items [{:key string, :value string}]},
  :serviceAccounts [{:email string, :scopes [string]}],
  :minCpuPlatform string},
 :sourceInstance string,
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :id string,
 :sourceInstanceParams
 {:diskConfigs
  [{:deviceName string,
    :instantiateFrom
    [ATTACH_READ_ONLY
     BLANK
     CUSTOM_IMAGE
     DEFAULT
     DO_NOT_INCLUDE
     SOURCE_IMAGE
     SOURCE_IMAGE_FAMILY],
    :autoDelete boolean,
    :customImage string}]},
 :kind string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region InstanceTemplate]
    (regionInstanceTemplates-insert
      project
      region
      InstanceTemplate
      nil))
  ([project region InstanceTemplate optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceTemplates",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InstanceTemplate})))

(defn regionInstanceTemplates-delete
  "Deletes the specified instance template. Deleting an instance template is permanent and cannot be undone.
https://cloud.google.com/products/compute/v1/docs/regionInstanceTemplates/delete

project <string> Project ID for this request.
region <string> The name of the region for this request.
instanceTemplate <string> The name of the instance template to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region instanceTemplate]
    (regionInstanceTemplates-delete
      project
      region
      instanceTemplate
      nil))
  ([project region instanceTemplate optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/instanceTemplates/{instanceTemplate}",
       :uri-template-args
       {:instanceTemplate instanceTemplate,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn interconnectAttachments-list
  "Retrieves the list of interconnect attachments contained within the specified region.
https://cloud.google.com/products/compute/v1/docs/interconnectAttachments/list

project <string> Project ID for this request.
region <string> Name of the region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (interconnectAttachments-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/interconnectAttachments",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn interconnectAttachments-aggregatedList
  "Retrieves an aggregated list of interconnect attachments. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/interconnectAttachments/aggregatedList

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (interconnectAttachments-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/interconnectAttachments",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn interconnectAttachments-get
  "Returns the specified interconnect attachment.
https://cloud.google.com/products/compute/v1/docs/interconnectAttachments/get

project <string> Project ID for this request.
region <string> Name of the region for this request.
interconnectAttachment <string> Name of the interconnect attachment to return."
  [project region interconnectAttachment]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}",
     :uri-template-args
     {:interconnectAttachment interconnectAttachment,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn interconnectAttachments-insert
  "Creates an InterconnectAttachment in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/interconnectAttachments/insert

project <string> Project ID for this request.
region <string> Name of the region for this request.
InterconnectAttachment:
{:candidateSubnets [string],
 :description string,
 :subnetLength integer,
 :labels object,
 :dataplaneVersion integer,
 :ipsecInternalAddresses [string],
 :creationTimestamp string,
 :mtu integer,
 :router string,
 :customerRouterIpAddress string,
 :candidateIpv6Subnets [string],
 :stackType [IPV4_IPV6 IPV4_ONLY],
 :cloudRouterIpv6InterfaceId string,
 :name string,
 :interconnect string,
 :selfLink string,
 :type [DEDICATED PARTNER PARTNER_PROVIDER],
 :vlanTag8021q integer,
 :state
 [ACTIVE
  DEFUNCT
  PARTNER_REQUEST_RECEIVED
  PENDING_CUSTOMER
  PENDING_PARTNER
  STATE_UNSPECIFIED
  UNPROVISIONED],
 :cloudRouterIpv6Address string,
 :region string,
 :pairingKey string,
 :encryption [IPSEC NONE],
 :bandwidth
 [BPS_100M
  BPS_10G
  BPS_1G
  BPS_200M
  BPS_20G
  BPS_2G
  BPS_300M
  BPS_400M
  BPS_500M
  BPS_50G
  BPS_50M
  BPS_5G],
 :customerRouterIpv6Address string,
 :satisfiesPzs boolean,
 :id string,
 :kind string,
 :partnerAsn string,
 :adminEnabled boolean,
 :customerRouterIpv6InterfaceId string,
 :operationalStatus [OS_ACTIVE OS_UNPROVISIONED],
 :configurationConstraints
 {:bgpMd5 [MD5_OPTIONAL MD5_REQUIRED MD5_UNSUPPORTED],
  :bgpPeerAsnRanges [{:min integer, :max integer}]},
 :partnerMetadata
 {:partnerName string, :interconnectName string, :portalUrl string},
 :cloudRouterIpAddress string,
 :remoteService string,
 :googleReferenceId string,
 :edgeAvailabilityDomain
 [AVAILABILITY_DOMAIN_1 AVAILABILITY_DOMAIN_2 AVAILABILITY_DOMAIN_ANY],
 :privateInterconnectInfo {:tag8021q integer},
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
validateOnly <boolean> If true, the request will not be committed."
  ([project region InterconnectAttachment]
    (interconnectAttachments-insert
      project
      region
      InterconnectAttachment
      nil))
  ([project region InterconnectAttachment optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/interconnectAttachments",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InterconnectAttachment})))

(defn interconnectAttachments-patch
  "Updates the specified interconnect attachment with the data included in the request. This method supports PATCH semantics and uses the JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/interconnectAttachments/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
interconnectAttachment <string> Name of the interconnect attachment to patch.
InterconnectAttachment:
{:candidateSubnets [string],
 :description string,
 :subnetLength integer,
 :labels object,
 :dataplaneVersion integer,
 :ipsecInternalAddresses [string],
 :creationTimestamp string,
 :mtu integer,
 :router string,
 :customerRouterIpAddress string,
 :candidateIpv6Subnets [string],
 :stackType [IPV4_IPV6 IPV4_ONLY],
 :cloudRouterIpv6InterfaceId string,
 :name string,
 :interconnect string,
 :selfLink string,
 :type [DEDICATED PARTNER PARTNER_PROVIDER],
 :vlanTag8021q integer,
 :state
 [ACTIVE
  DEFUNCT
  PARTNER_REQUEST_RECEIVED
  PENDING_CUSTOMER
  PENDING_PARTNER
  STATE_UNSPECIFIED
  UNPROVISIONED],
 :cloudRouterIpv6Address string,
 :region string,
 :pairingKey string,
 :encryption [IPSEC NONE],
 :bandwidth
 [BPS_100M
  BPS_10G
  BPS_1G
  BPS_200M
  BPS_20G
  BPS_2G
  BPS_300M
  BPS_400M
  BPS_500M
  BPS_50G
  BPS_50M
  BPS_5G],
 :customerRouterIpv6Address string,
 :satisfiesPzs boolean,
 :id string,
 :kind string,
 :partnerAsn string,
 :adminEnabled boolean,
 :customerRouterIpv6InterfaceId string,
 :operationalStatus [OS_ACTIVE OS_UNPROVISIONED],
 :configurationConstraints
 {:bgpMd5 [MD5_OPTIONAL MD5_REQUIRED MD5_UNSUPPORTED],
  :bgpPeerAsnRanges [{:min integer, :max integer}]},
 :partnerMetadata
 {:partnerName string, :interconnectName string, :portalUrl string},
 :cloudRouterIpAddress string,
 :remoteService string,
 :googleReferenceId string,
 :edgeAvailabilityDomain
 [AVAILABILITY_DOMAIN_1 AVAILABILITY_DOMAIN_2 AVAILABILITY_DOMAIN_ANY],
 :privateInterconnectInfo {:tag8021q integer},
 :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region interconnectAttachment InterconnectAttachment]
    (interconnectAttachments-patch
      project
      region
      interconnectAttachment
      InterconnectAttachment
      nil))
  ([project
    region
    interconnectAttachment
    InterconnectAttachment
    optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}",
       :uri-template-args
       {:interconnectAttachment interconnectAttachment,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body InterconnectAttachment})))

(defn interconnectAttachments-delete
  "Deletes the specified interconnect attachment.
https://cloud.google.com/products/compute/v1/docs/interconnectAttachments/delete

project <string> Project ID for this request.
region <string> Name of the region for this request.
interconnectAttachment <string> Name of the interconnect attachment to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region interconnectAttachment]
    (interconnectAttachments-delete
      project
      region
      interconnectAttachment
      nil))
  ([project region interconnectAttachment optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}",
       :uri-template-args
       {:interconnectAttachment interconnectAttachment,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn interconnectAttachments-setLabels
  "Sets the labels on an InterconnectAttachment. To learn more about labels, read the Labeling Resources documentation.
https://cloud.google.com/products/compute/v1/docs/interconnectAttachments/setLabels

project <string> Project ID for this request.
region <string> The region for this request.
resource <string> Name or id of the resource for this request.
RegionSetLabelsRequest:
{:labels object, :labelFingerprint string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region resource RegionSetLabelsRequest]
    (interconnectAttachments-setLabels
      project
      region
      resource
      RegionSetLabelsRequest
      nil))
  ([project region resource RegionSetLabelsRequest optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/interconnectAttachments/{resource}/setLabels",
       :uri-template-args
       {:project project, :region region, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionSetLabelsRequest})))

(defn snapshotSettings-get
  "Get snapshot settings.
https://cloud.google.com/products/compute/v1/docs/snapshotSettings/get

project <string> Project ID for this request."
  [project]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/snapshotSettings",
     :uri-template-args {:project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn snapshotSettings-patch
  "Patch snapshot settings.
https://cloud.google.com/products/compute/v1/docs/snapshotSettings/patch

project <string> Project ID for this request.
SnapshotSettings:
{:storageLocation
 {:policy
  [LOCAL_REGION
   NEAREST_MULTI_REGION
   SPECIFIC_LOCATIONS
   STORAGE_LOCATION_POLICY_UNSPECIFIED],
  :locations object}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
updateMask <string> update_mask indicates fields to be updated as part of this request."
  ([project SnapshotSettings]
    (snapshotSettings-patch project SnapshotSettings nil))
  ([project SnapshotSettings optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/snapshotSettings",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body SnapshotSettings})))

(defn licenses-get
  "Returns the specified License resource. *Caution* This resource is intended for use only by third-party partners who are creating Cloud Marketplace images. 
https://cloud.google.com/products/compute/v1/docs/licenses/get

project <string> Project ID for this request.
license <string> Name of the License resource to return."
  [project license]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/licenses/{license}",
     :uri-template-args {:license license, :project project},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn licenses-list
  "Retrieves the list of licenses available in the specified project. This method does not get any licenses that belong to other projects, including licenses attached to publicly-available images, like Debian 9. If you want to get a list of publicly-available licenses, use this method to make a request to the respective image project, such as debian-cloud or windows-cloud. *Caution* This resource is intended for use only by third-party partners who are creating Cloud Marketplace images. 
https://cloud.google.com/products/compute/v1/docs/licenses/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (licenses-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/licenses",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn licenses-delete
  "Deletes the specified license. *Caution* This resource is intended for use only by third-party partners who are creating Cloud Marketplace images. 
https://cloud.google.com/products/compute/v1/docs/licenses/delete

project <string> Project ID for this request.
license <string> Name of the license resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project license] (licenses-delete project license nil))
  ([project license optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/licenses/{license}",
       :uri-template-args {:license license, :project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn licenses-insert
  "Create a License resource in the specified project. *Caution* This resource is intended for use only by third-party partners who are creating Cloud Marketplace images. 
https://cloud.google.com/products/compute/v1/docs/licenses/insert

project <string> Project ID for this request.
License:
{:description string,
 :creationTimestamp string,
 :licenseCode string,
 :name string,
 :chargesUseFee boolean,
 :selfLink string,
 :transferable boolean,
 :id string,
 :kind string,
 :resourceRequirements
 {:minGuestCpuCount integer, :minMemoryMb integer}}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project License] (licenses-insert project License nil))
  ([project License optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/licenses",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/devstorage.full_control"
        "https://www.googleapis.com/auth/devstorage.read_only"
        "https://www.googleapis.com/auth/devstorage.read_write"],
       :body License})))

(defn licenses-getIamPolicy
  "Gets the access control policy for a resource. May be empty if no such policy or resource exists. *Caution* This resource is intended for use only by third-party partners who are creating Cloud Marketplace images. 
https://cloud.google.com/products/compute/v1/docs/licenses/getIamPolicy

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.

optional:
optionsRequestedPolicyVersion <integer> Requested IAM Policy version."
  ([project resource] (licenses-getIamPolicy project resource nil))
  ([project resource optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/licenses/{resource}/getIamPolicy",
       :uri-template-args {:project project, :resource resource},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn licenses-setIamPolicy
  "Sets the access control policy on the specified resource. Replaces any existing policy. *Caution* This resource is intended for use only by third-party partners who are creating Cloud Marketplace images. 
https://cloud.google.com/products/compute/v1/docs/licenses/setIamPolicy

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.
GlobalSetPolicyRequest:
{:policy
 {:version integer,
  :bindings
  [{:role string,
    :members [string],
    :condition
    {:expression string,
     :title string,
     :description string,
     :location string},
    :bindingId string}],
  :auditConfigs
  [{:exemptedMembers [string],
    :auditLogConfigs
    [{:logType [ADMIN_READ DATA_READ DATA_WRITE LOG_TYPE_UNSPECIFIED],
      :exemptedMembers [string],
      :ignoreChildExemptions boolean}],
    :service string}],
  :rules
  [{:description string,
    :permissions [string],
    :action [ALLOW ALLOW_WITH_LOG DENY DENY_WITH_LOG LOG NO_ACTION],
    :ins [string],
    :notIns [string],
    :conditions
    [{:iam
      [APPROVER
       ATTRIBUTION
       AUTHORITY
       CREDENTIALS_TYPE
       CREDS_ASSERTION
       JUSTIFICATION_TYPE
       NO_ATTR
       SECURITY_REALM],
      :sys [IP NAME NO_ATTR REGION SERVICE],
      :svc string,
      :op [DISCHARGED EQUALS IN NOT_EQUALS NOT_IN NO_OP],
      :values [string]}],
    :logConfigs
    [{:counter
      {:metric string,
       :field string,
       :customFields [LogConfigCounterOptionsCustomField]},
      :dataAccess {:logMode [LOG_FAIL_CLOSED LOG_MODE_UNSPECIFIED]},
      :cloudAudit
      {:logName [ADMIN_ACTIVITY DATA_ACCESS UNSPECIFIED_LOG_NAME]}}]}],
  :etag string},
 :bindings
 [{:role string,
   :members [string],
   :condition
   {:expression string,
    :title string,
    :description string,
    :location string},
   :bindingId string}],
 :etag string}"
  [project resource GlobalSetPolicyRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/licenses/{resource}/setIamPolicy",
     :uri-template-args {:project project, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"],
     :body GlobalSetPolicyRequest}))

(defn licenses-testIamPermissions
  "Returns permissions that a caller has on the specified resource. *Caution* This resource is intended for use only by third-party partners who are creating Cloud Marketplace images. 
https://cloud.google.com/products/compute/v1/docs/licenses/testIamPermissions

project <string> Project ID for this request.
resource <string> Name or id of the resource for this request.
TestPermissionsRequest:
{:permissions [string]}"
  [project resource TestPermissionsRequest]
  (client/api-request
    {:method "POST",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/licenses/{resource}/testIamPermissions",
     :uri-template-args {:project project, :resource resource},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"],
     :body TestPermissionsRequest}))

(defn networkEdgeSecurityServices-get
  "Gets a specified NetworkEdgeSecurityService.
https://cloud.google.com/products/compute/v1/docs/networkEdgeSecurityServices/get

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
networkEdgeSecurityService <string> Name of the network edge security service to get."
  [project region networkEdgeSecurityService]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEdgeSecurityServices/{networkEdgeSecurityService}",
     :uri-template-args
     {:networkEdgeSecurityService networkEdgeSecurityService,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn networkEdgeSecurityServices-insert
  "Creates a new service in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/networkEdgeSecurityServices/insert

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
NetworkEdgeSecurityService:
{:description string,
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :selfLinkWithId string,
 :id string,
 :kind string,
 :fingerprint string,
 :securityPolicy string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
validateOnly <boolean> If true, the request will not be committed."
  ([project region NetworkEdgeSecurityService]
    (networkEdgeSecurityServices-insert
      project
      region
      NetworkEdgeSecurityService
      nil))
  ([project region NetworkEdgeSecurityService optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEdgeSecurityServices",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NetworkEdgeSecurityService})))

(defn networkEdgeSecurityServices-delete
  "Deletes the specified service.
https://cloud.google.com/products/compute/v1/docs/networkEdgeSecurityServices/delete

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
networkEdgeSecurityService <string> Name of the network edge security service to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region networkEdgeSecurityService]
    (networkEdgeSecurityServices-delete
      project
      region
      networkEdgeSecurityService
      nil))
  ([project region networkEdgeSecurityService optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEdgeSecurityServices/{networkEdgeSecurityService}",
       :uri-template-args
       {:networkEdgeSecurityService networkEdgeSecurityService,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn networkEdgeSecurityServices-patch
  "Patches the specified policy with the data included in the request.
https://cloud.google.com/products/compute/v1/docs/networkEdgeSecurityServices/patch

project <string> Project ID for this request.
region <string> Name of the region scoping this request.
networkEdgeSecurityService <string> Name of the network edge security service to update.
NetworkEdgeSecurityService:
{:description string,
 :creationTimestamp string,
 :name string,
 :selfLink string,
 :region string,
 :selfLinkWithId string,
 :id string,
 :kind string,
 :fingerprint string,
 :securityPolicy string}

optional:
paths <string> 
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
updateMask <string> Indicates fields to be updated as part of this request."
  ([project
    region
    networkEdgeSecurityService
    NetworkEdgeSecurityService]
    (networkEdgeSecurityServices-patch
      project
      region
      networkEdgeSecurityService
      NetworkEdgeSecurityService
      nil))
  ([project
    region
    networkEdgeSecurityService
    NetworkEdgeSecurityService
    optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEdgeSecurityServices/{networkEdgeSecurityService}",
       :uri-template-args
       {:networkEdgeSecurityService networkEdgeSecurityService,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NetworkEdgeSecurityService})))

(defn networkEdgeSecurityServices-aggregatedList
  "Retrieves the list of all NetworkEdgeSecurityService resources available to the specified project. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/networkEdgeSecurityServices/aggregatedList

project <string> Name of the project scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (networkEdgeSecurityServices-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/networkEdgeSecurityServices",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionZones-list
  "Retrieves the list of Zone resources under the specific region available to the specified project.
https://cloud.google.com/products/compute/v1/docs/regionZones/list

project <string> Project ID for this request.
region <string> Region for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region] (regionZones-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/zones",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionNetworkEndpointGroups-list
  "Retrieves the list of regional network endpoint groups available to the specified project in the given region.
https://cloud.google.com/products/compute/v1/docs/regionNetworkEndpointGroups/list

project <string> Project ID for this request.
region <string> The name of the region where the network endpoint group is located. It should comply with RFC1035.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region]
    (regionNetworkEndpointGroups-list project region nil))
  ([project region optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEndpointGroups",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn regionNetworkEndpointGroups-get
  "Returns the specified network endpoint group.
https://cloud.google.com/products/compute/v1/docs/regionNetworkEndpointGroups/get

project <string> Project ID for this request.
region <string> The name of the region where the network endpoint group is located. It should comply with RFC1035.
networkEndpointGroup <string> The name of the network endpoint group. It should comply with RFC1035."
  [project region networkEndpointGroup]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEndpointGroups/{networkEndpointGroup}",
     :uri-template-args
     {:networkEndpointGroup networkEndpointGroup,
      :project project,
      :region region},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn regionNetworkEndpointGroups-insert
  "Creates a network endpoint group in the specified project using the parameters that are included in the request.
https://cloud.google.com/products/compute/v1/docs/regionNetworkEndpointGroups/insert

project <string> Project ID for this request.
region <string> The name of the region where you want to create the network endpoint group. It should comply with RFC1035.
NetworkEndpointGroup:
{:description string,
 :networkEndpointType
 [GCE_VM_IP
  GCE_VM_IP_PORT
  INTERNET_FQDN_PORT
  INTERNET_IP_PORT
  NON_GCP_PRIVATE_IP_PORT
  PRIVATE_SERVICE_CONNECT
  SERVERLESS],
 :creationTimestamp string,
 :zone string,
 :name string,
 :cloudFunction {:function string, :urlMask string},
 :selfLink string,
 :pscTargetService string,
 :cloudRun {:service string, :tag string, :urlMask string},
 :size integer,
 :region string,
 :pscData
 {:consumerPscAddress string,
  :pscConnectionId string,
  :pscConnectionStatus
  [ACCEPTED
   CLOSED
   NEEDS_ATTENTION
   PENDING
   REJECTED
   STATUS_UNSPECIFIED]},
 :id string,
 :kind string,
 :appEngine {:service string, :version string, :urlMask string},
 :annotations object,
 :network string,
 :defaultPort integer,
 :subnetwork string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region NetworkEndpointGroup]
    (regionNetworkEndpointGroups-insert
      project
      region
      NetworkEndpointGroup
      nil))
  ([project region NetworkEndpointGroup optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEndpointGroups",
       :uri-template-args {:project project, :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body NetworkEndpointGroup})))

(defn regionNetworkEndpointGroups-delete
  "Deletes the specified network endpoint group. Note that the NEG cannot be deleted if it is configured as a backend of a backend service.
https://cloud.google.com/products/compute/v1/docs/regionNetworkEndpointGroups/delete

project <string> Project ID for this request.
region <string> The name of the region where the network endpoint group is located. It should comply with RFC1035.
networkEndpointGroup <string> The name of the network endpoint group to delete. It should comply with RFC1035.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project region networkEndpointGroup]
    (regionNetworkEndpointGroups-delete
      project
      region
      networkEndpointGroup
      nil))
  ([project region networkEndpointGroup optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEndpointGroups/{networkEndpointGroup}",
       :uri-template-args
       {:networkEndpointGroup networkEndpointGroup,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn regionNetworkEndpointGroups-attachNetworkEndpoints
  "Attach a list of network endpoints to the specified network endpoint group.
https://cloud.google.com/products/compute/v1/docs/regionNetworkEndpointGroups/attachNetworkEndpoints

project <string> Project ID for this request.
region <string> The name of the region where you want to create the network endpoint group. It should comply with RFC1035.
networkEndpointGroup <string> The name of the network endpoint group where you are attaching network endpoints to. It should comply with RFC1035.
RegionNetworkEndpointGroupsAttachEndpointsRequest:
{:networkEndpoints
 [{:ipAddress string,
   :fqdn string,
   :port integer,
   :instance string,
   :annotations object}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project
    region
    networkEndpointGroup
    RegionNetworkEndpointGroupsAttachEndpointsRequest]
    (regionNetworkEndpointGroups-attachNetworkEndpoints
      project
      region
      networkEndpointGroup
      RegionNetworkEndpointGroupsAttachEndpointsRequest
      nil))
  ([project
    region
    networkEndpointGroup
    RegionNetworkEndpointGroupsAttachEndpointsRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEndpointGroups/{networkEndpointGroup}/attachNetworkEndpoints",
       :uri-template-args
       {:networkEndpointGroup networkEndpointGroup,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionNetworkEndpointGroupsAttachEndpointsRequest})))

(defn regionNetworkEndpointGroups-detachNetworkEndpoints
  "Detach the network endpoint from the specified network endpoint group.
https://cloud.google.com/products/compute/v1/docs/regionNetworkEndpointGroups/detachNetworkEndpoints

project <string> Project ID for this request.
region <string> The name of the region where the network endpoint group is located. It should comply with RFC1035.
networkEndpointGroup <string> The name of the network endpoint group you are detaching network endpoints from. It should comply with RFC1035.
RegionNetworkEndpointGroupsDetachEndpointsRequest:
{:networkEndpoints
 [{:ipAddress string,
   :fqdn string,
   :port integer,
   :instance string,
   :annotations object}]}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000). end_interface: MixerMutationRequestBuilder"
  ([project
    region
    networkEndpointGroup
    RegionNetworkEndpointGroupsDetachEndpointsRequest]
    (regionNetworkEndpointGroups-detachNetworkEndpoints
      project
      region
      networkEndpointGroup
      RegionNetworkEndpointGroupsDetachEndpointsRequest
      nil))
  ([project
    region
    networkEndpointGroup
    RegionNetworkEndpointGroupsDetachEndpointsRequest
    optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEndpointGroups/{networkEndpointGroup}/detachNetworkEndpoints",
       :uri-template-args
       {:networkEndpointGroup networkEndpointGroup,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body RegionNetworkEndpointGroupsDetachEndpointsRequest})))

(defn regionNetworkEndpointGroups-listNetworkEndpoints
  "Lists the network endpoints in the specified network endpoint group.
https://cloud.google.com/products/compute/v1/docs/regionNetworkEndpointGroups/listNetworkEndpoints

project <string> Project ID for this request.
region <string> The name of the region where the network endpoint group is located. It should comply with RFC1035.
networkEndpointGroup <string> The name of the network endpoint group from which you want to generate a list of included network endpoints. It should comply with RFC1035.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project region networkEndpointGroup]
    (regionNetworkEndpointGroups-listNetworkEndpoints
      project
      region
      networkEndpointGroup
      nil))
  ([project region networkEndpointGroup optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/regions/{region}/networkEndpointGroups/{networkEndpointGroup}/listNetworkEndpoints",
       :uri-template-args
       {:networkEndpointGroup networkEndpointGroup,
        :project project,
        :region region},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetHttpProxies-list
  "Retrieves the list of TargetHttpProxy resources available to the specified project.
https://cloud.google.com/products/compute/v1/docs/targetHttpProxies/list

project <string> Project ID for this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code."
  ([project] (targetHttpProxies-list project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetHttpProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn targetHttpProxies-get
  "Returns the specified TargetHttpProxy resource.
https://cloud.google.com/products/compute/v1/docs/targetHttpProxies/get

project <string> Project ID for this request.
targetHttpProxy <string> Name of the TargetHttpProxy resource to return."
  [project targetHttpProxy]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetHttpProxies/{targetHttpProxy}",
     :uri-template-args
     {:project project, :targetHttpProxy targetHttpProxy},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn targetHttpProxies-insert
  "Creates a TargetHttpProxy resource in the specified project using the data included in the request.
https://cloud.google.com/products/compute/v1/docs/targetHttpProxies/insert

project <string> Project ID for this request.
TargetHttpProxy:
{:description string,
 :creationTimestamp string,
 :urlMap string,
 :name string,
 :httpKeepAliveTimeoutSec integer,
 :selfLink string,
 :region string,
 :id string,
 :kind string,
 :fingerprint string,
 :proxyBind boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project TargetHttpProxy]
    (targetHttpProxies-insert project TargetHttpProxy nil))
  ([project TargetHttpProxy optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetHttpProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetHttpProxy})))

(defn targetHttpProxies-delete
  "Deletes the specified TargetHttpProxy resource.
https://cloud.google.com/products/compute/v1/docs/targetHttpProxies/delete

project <string> Project ID for this request.
targetHttpProxy <string> Name of the TargetHttpProxy resource to delete.

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetHttpProxy]
    (targetHttpProxies-delete project targetHttpProxy nil))
  ([project targetHttpProxy optional]
    (client/api-request
      {:method "DELETE",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetHttpProxies/{targetHttpProxy}",
       :uri-template-args
       {:project project, :targetHttpProxy targetHttpProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"]})))

(defn targetHttpProxies-patch
  "Patches the specified TargetHttpProxy resource with the data included in the request. This method supports PATCH semantics and uses JSON merge patch format and processing rules.
https://cloud.google.com/products/compute/v1/docs/targetHttpProxies/patch

project <string> Project ID for this request.
targetHttpProxy <string> Name of the TargetHttpProxy resource to patch.
TargetHttpProxy:
{:description string,
 :creationTimestamp string,
 :urlMap string,
 :name string,
 :httpKeepAliveTimeoutSec integer,
 :selfLink string,
 :region string,
 :id string,
 :kind string,
 :fingerprint string,
 :proxyBind boolean}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetHttpProxy TargetHttpProxy]
    (targetHttpProxies-patch
      project
      targetHttpProxy
      TargetHttpProxy
      nil))
  ([project targetHttpProxy TargetHttpProxy optional]
    (client/api-request
      {:method "PATCH",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/global/targetHttpProxies/{targetHttpProxy}",
       :uri-template-args
       {:project project, :targetHttpProxy targetHttpProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body TargetHttpProxy})))

(defn targetHttpProxies-setUrlMap
  "Changes the URL map for TargetHttpProxy.
https://cloud.google.com/products/compute/v1/docs/targetHttpProxies/setUrlMap

project <string> Project ID for this request.
targetHttpProxy <string> Name of the TargetHttpProxy to set a URL map for.
UrlMapReference:
{:urlMap string}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same request ID, the server can check if original operation with the same request ID was received, and if so, will ignore the second request. This prevents clients from accidentally creating duplicate commitments. The request ID must be a valid UUID with the exception that zero UUID is not supported ( 00000000-0000-0000-0000-000000000000)."
  ([project targetHttpProxy UrlMapReference]
    (targetHttpProxies-setUrlMap
      project
      targetHttpProxy
      UrlMapReference
      nil))
  ([project targetHttpProxy UrlMapReference optional]
    (client/api-request
      {:method "POST",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/targetHttpProxies/{targetHttpProxy}/setUrlMap",
       :uri-template-args
       {:project project, :targetHttpProxy targetHttpProxy},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"],
       :body UrlMapReference})))

(defn targetHttpProxies-aggregatedList
  "Retrieves the list of all TargetHttpProxy resources, regional and global, available to the specified project. To prevent failure, Google recommends that you set the `returnPartialSuccess` parameter to `true`.
https://cloud.google.com/products/compute/v1/docs/targetHttpProxies/aggregatedList

project <string> Name of the project scoping this request.

optional:
filter <string> A filter expression that filters resources listed in the response. Most Compute resources support two types of filter expressions: expressions that support regular expressions and expressions that follow API improvement proposal AIP-160. These two types of filter expressions cannot be mixed in one request. If you want to use AIP-160, your expression must specify the field name, an operator, and the value that you want to use for filtering. The value must be a string, a number, or a boolean. The operator must be either `=`, `!=`, `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute Engine instances, you can exclude instances named `example-instance` by specifying `name != example-instance`. The `:*` comparison can be used to test whether a key has been defined. For example, to find all objects with `owner` label use: ``` labels.owner:* ``` You can also filter nested fields. For example, you could specify `scheduling.automaticRestart = false` to include instances only if they are not scheduled for automatic restarts. You can use filtering on nested fields to filter based on resource labels. To filter on multiple expressions, provide each separate expression within parentheses. For example: ``` (scheduling.automaticRestart = true) (cpuPlatform = \"Intel Skylake\") ``` By default, each expression is an `AND` expression. However, you can include `AND` and `OR` expressions explicitly. For example: ``` (cpuPlatform = \"Intel Skylake\") OR (cpuPlatform = \"Intel Broadwell\") AND (scheduling.automaticRestart = true) ``` If you want to use a regular expression, use the `eq` (equal) or `ne` (not equal) operator against a single un-parenthesized expression with or without quotes or against multiple parenthesized expressions. Examples: `fieldname eq unquoted literal` `fieldname eq 'single quoted literal'` `fieldname eq \"double quoted literal\"` `(fieldname1 eq literal) (fieldname2 ne \"literal\")` The literal value is interpreted as a regular expression using Google RE2 library syntax. The literal value must match the entire field. For example, to filter for instances that do not end with name \"instance\", you would use `name ne .*instance`. You cannot combine constraints on multiple fields using regular expressions.
includeAllScopes <boolean> Indicates whether every visible scope for each scope type (zone, region, global) should be included in the response. For new resource types added after this field, the flag has no effect as new resource types will always include every visible scope for each scope type in response. For resource types which predate this field, if this flag is omitted or false, only scopes of the scope types where the resource type is expected to be found will be included.
maxResults <integer> The maximum number of results per page that should be returned. If the number of available results is larger than `maxResults`, Compute Engine returns a `nextPageToken` that can be used to get the next page of results in subsequent list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
orderBy <string> Sorts list results by a certain order. By default, results are returned in alphanumerical order based on the resource name. You can also sort results in descending order based on the creation timestamp using `orderBy=\"creationTimestamp desc\"`. This sorts results based on the `creationTimestamp` field in reverse chronological order (newest result first). Use this to sort resources like operations so that the newest operation is returned first. Currently, only sorting by `name` or `creationTimestamp desc` is supported.
returnPartialSuccess <boolean> Opt-in for partial success behavior which provides partial results in case of failure. The default value is false. For example, when partial success behavior is enabled, aggregatedList for a single zone scope either returns all resources in the zone or no resources, with an error code.
serviceProjectNumber <string> The Shared VPC service project id or service project number for which aggregated list request is invoked for subnetworks list-usable api."
  ([project] (targetHttpProxies-aggregatedList project nil))
  ([project optional]
    (client/api-request
      {:method "GET",
       :uri-template
       "https://compute.googleapis.com/compute/v1/projects/{project}/aggregated/targetHttpProxies",
       :uri-template-args {:project project},
       :query-params (merge {} optional),
       :scopes
       ["https://www.googleapis.com/auth/cloud-platform"
        "https://www.googleapis.com/auth/compute"
        "https://www.googleapis.com/auth/compute.readonly"]})))

(defn instanceGroupManagerResizeRequests-get
  "Returns all of the details about the specified resize request.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagerResizeRequests/get

project <string> Project ID for this request.
zone <string> Name of the href=\"/compute/docs/regions-zones/#available\">zone scoping this request. Name should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group. Name should conform to RFC1035 or be a resource ID.
resizeRequest <string> The name of the resize request. Name should conform to RFC1035 or be a resource ID."
  [project zone instanceGroupManager resizeRequest]
  (client/api-request
    {:method "GET",
     :uri-template
     "https://compute.googleapis.com/compute/v1/projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resizeRequests/{resizeRequest}",
     :uri-template-args
     {:instanceGroupManager instanceGroupManager,
      :project project,
      :resizeRequest resizeRequest,
      :zone zone},
     :query-params {},
     :scopes
     ["https://www.googleapis.com/auth/cloud-platform"
      "https://www.googleapis.com/auth/compute"
      "https://www.googleapis.com/auth/compute.readonly"]}))

(defn instanceGroupManagerResizeRequests-insert
  "Creates a new resize request that starts provisioning VMs immediately or queues VM creation.
https://cloud.google.com/products/compute/v1/docs/instanceGroupManagerResizeRequests/insert

project <string> Project ID for this request.
zone <string> The name of the zone where the managed instance group is located and where the resize request will be created. Name should conform to RFC1035.
instanceGroupManager <string> The name of the managed instance group to which the resize request will be added. Name should conform to RFC1035 or be a resource ID.
InstanceGroupManagerResizeRequest:
{:description string,
 :creationTimestamp string,
 :requestedRunDuration {:seconds string, :nanos integer},
 :zone string,
 :name string,
 :selfLink string,
 :state
 [ACCEPTED CANCELLED CREATING FAILED STATE_UNSPECIFIED SUCCEEDED],
 :selfLinkWithId string,
 :status
 {:error
  {:errors
   [{:code string,
     :location string,
     :message string,
     :errorDetails
     [{:errorInfo {:reason string, :domain string, :metadatas object},
       :quotaInfo
       {:metricName string,
        :limitName string,
        :dimensions object,
        :limit number,
        :futureLimit number,
        :rolloutStatus [IN_PROGRESS ROLLOUT_STATUS_UNSPECIFIED]},
       :help {:links [{:description string, :url string}]},
       :localizedMessage {:locale string, :message string}}]}]},
  :lastAttempt
  {:error
   {:errors
    [{:code string,
      :location string,
      :message string,
      :errorDetails
      [{:errorInfo {:reason string, :domain string, :metadatas object},
        :quotaInfo
        {:metricName string,
         :limitName string,
         :dimensions object,
         :limit number,
         :futureLimit number,
         :rolloutStatus [IN_PROGRESS ROLLOUT_STATUS_UNSPECIFIED]},
        :help {:links [{:description string, :url string}]},
        :localizedMessage {:locale string, :message string}}]}]}}},
 :id string,
 :kind string,
 :resizeBy integer}

optional:
requestId <string> An optional request ID to identify requests. Specify a unique request ID so that if you must retry your request, the server will know to ignore the request if it has already been completed. For example, consider a situation where you make an initial request and the request times out. If you make the request again with the same