(ns atlas.ontology.yorba-data
  "Yorba data ontology module - data field classification and invariants.

   This module provides data field classification and validation for the
   `:atlas/yorba-data` ontology.

   A yorba-data entity represents a single data field (not objects or collections):
   - Primitives: string, number, date, etc.
   - Identified by namespaced keyword (the dev-id)
   - Classified as: system, user, or public data

   Example:
     (registry/register!
      :user/email
      :atlas/yorba-data
      #{:domain/users :compliance/pii}
      {:yorba-data/type :yorba-data/user})

   Serialisation args and endpoint I/O can optionally reference these entities.

   Usage:
     (require '[atlas.ontology.yorba-data :as yd])
     (yd/load!)"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]
            [datascript.core :as d]))

;; =============================================================================
;; ONTOLOGY DEFINITION
;; =============================================================================

(def ontology-definition
  "The ontology definition for :atlas/yorba-data"
  {:ontology/for :atlas/yorba-data
   :ontology/keys [:yorba-data/type]})  ; Classification: system, user, or public

;; =============================================================================
;; HELPER FUNCTIONS
;; =============================================================================

(defn all-data-fields
  "Return all registered yorba-data dev-ids (excluding ontology definitions)."
  []
  (->> (entity/all-with-aspect :atlas/yorba-data)
       (remove #(entity/has-aspect? % :atlas/ontology))
       set))

(defn type-for
  "Get the data type classification for a data field dev-id.
   Returns one of: :yorba-data/system, :yorba-data/user, :yorba-data/public, or nil"
  [data-dev-id]
  (:yorba-data/type (entity/props-for data-dev-id)))

(defn data-fields-by-type
  "Get all data fields of a specific type.
   type: :yorba-data/system, :yorba-data/user, or :yorba-data/public"
  [data-type]
  (->> (all-data-fields)
       (filter #(= data-type (type-for %)))
       set))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-data properties.
   Called by atlas.datalog when building database.

   Property mappings:
   - :yorba-data/type → :yorba-data/type (classification)"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-data)
    (let [dev-id (:atlas/dev-id props)
          data-type (:yorba-data/type props)]
      (cond-> []
        data-type
        (conj [:db/add dev-id :yorba-data/type data-type])))))

(def datalog-schema
  "Datascript schema for yorba-data properties."
  {:yorba-data/type {:db/cardinality :db.cardinality/one}})

;; =============================================================================
;; DATALOG-BASED QUERIES
;; =============================================================================

(defn find-data-by-type
  "Find all data fields of a specific type (datalog query).

   type: :yorba-data/system, :yorba-data/user, or :yorba-data/public
   Returns: sequence of data field dev-ids"
  [data-type]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?data-id
                :in $ ?type
                :where
                [?e :atlas/dev-id ?data-id]
                [?e :yorba-data/type ?type]]
              db
              data-type)
         (map first)
         vec)))

;; =============================================================================
;; INVARIANTS
;; =============================================================================

;; NOTE: Value validation (type must be one of allowed values) should be
;; handled by clojure.spec at registration time.
;;
;; Registry invariants focus on structural/cross-ontology validation.

(def invariants
  "All structural invariants specific to yorba-data ontology.

   Currently empty - cross-ontology invariants with endpoints are defined
   in the yorba-endpoint module."
  [])

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-ontology!
  "Register the yorba-data ontology in the registry."
  []
  (registry/register!
   :atlas/yorba-data
   :atlas/ontology
   #{:atlas/yorba-data}
   ontology-definition))

(defn- register-invariants!
  "Register yorba-data invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-data properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-data ontology, invariants, and datalog support.

   This must be called before using yorba-data features:
   - Registering data field entities with :yorba-data/type
   - Using helper functions like type-for, data-fields-by-type
   - Running cross-ontology invariants with endpoints

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    (register-ontology!)
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-data module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-data module (primarily for testing).

   WARNING: This does not remove already-registered data fields from the
   registry or clean up datalog extensions. Use reset! on registry/registry
   and datalog/reset-extensions! for a full reset."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing).

   Use this before calling load! in test fixtures when you've also reset
   the registry. This ensures load! will re-register the ontology."
  []
  (reset! loaded? false))
