(ns atlas.ontology.yorba-mcp-tool
  "Yorba MCP tool ontology module - tool registration and datalog support.

   This module provides registration and querying for MCP (Model Context Protocol)
   tools in the `:atlas/yorba-mcp-tool` ontology.

   An MCP tool represents a function/capability exposed to LLMs:
   - Has a name (MCP protocol identifier)
   - Has a description (human-readable)
   - Declares required input args (collection of namespaced keywords)
   - Declares optional input args (collection of namespaced keywords)
   - Declares output args (collection of namespaced keywords)

   Example:
     (registry/register!
      :mcp-tool/validate-user-email
      :atlas/yorba-mcp-tool
      #{:domain/users :operation/validate :tier/admin :effect/read}
      {:mcp-tool/name \"validate_user_email\"
       :mcp-tool/description \"Validates user email format and existence\"
       :mcp-tool/req-input-args [:user/email]
       :mcp-tool/opt-input-args [:validation/strict?]
       :mcp-tool/output-args [:validation/result :validation/errors]})

   The input/output args can optionally reference yorba-data entities for
   additional semantic information and validation.

   Usage:
     (require '[atlas.ontology.yorba-mcp-tool :as mcp])
     (mcp/load!)"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]
            [datascript.core :as d]))

;; =============================================================================
;; ONTOLOGY DEFINITION
;; =============================================================================

(def ontology-definition
  "The ontology definition for :atlas/yorba-mcp-tool"
  {:ontology/for :atlas/yorba-mcp-tool
   :ontology/keys [:mcp-tool/name              ; MCP protocol tool name (string)
                   :mcp-tool/description       ; Human-readable description (string)
                   :mcp-tool/req-input-args    ; Required input keywords (collection)
                   :mcp-tool/opt-input-args    ; Optional input keywords (collection)
                   :mcp-tool/output-args]      ; Output keywords (collection)
   ;; Dataflow integration - map to standard entity attributes
   ;; Note: both req and opt inputs are consumed
   :dataflow/context-key [:mcp-tool/req-input-args :mcp-tool/opt-input-args]
   :dataflow/response-key :mcp-tool/output-args})

;; =============================================================================
;; HELPER FUNCTIONS
;; =============================================================================

(defn all-mcp-tools
  "Return all registered yorba-mcp-tool dev-ids (excluding ontology definitions)."
  []
  (->> (entity/all-with-aspect :atlas/yorba-mcp-tool)
       (remove #(entity/has-aspect? % :atlas/ontology))
       set))

(defn name-for
  "Get the MCP protocol name for a tool dev-id."
  [tool-dev-id]
  (:mcp-tool/name (entity/props-for tool-dev-id)))

(defn description-for
  "Get the description for a tool dev-id."
  [tool-dev-id]
  (:mcp-tool/description (entity/props-for tool-dev-id)))

(defn req-input-args-for
  "Get the required input arguments for a tool dev-id.
   Returns a collection of namespaced keywords."
  [tool-dev-id]
  (:mcp-tool/req-input-args (entity/props-for tool-dev-id)))

(defn opt-input-args-for
  "Get the optional input arguments for a tool dev-id.
   Returns a collection of namespaced keywords."
  [tool-dev-id]
  (:mcp-tool/opt-input-args (entity/props-for tool-dev-id)))

(defn all-input-args-for
  "Get all input arguments (required + optional) for a tool dev-id.
   Returns a collection of namespaced keywords."
  [tool-dev-id]
  (concat (req-input-args-for tool-dev-id)
          (opt-input-args-for tool-dev-id)))

(defn output-args-for
  "Get the output arguments for a tool dev-id.
   Returns a collection of namespaced keywords."
  [tool-dev-id]
  (:mcp-tool/output-args (entity/props-for tool-dev-id)))

(defn tool-details
  "Get all details for a tool dev-id in a single map."
  [tool-dev-id]
  (let [props (entity/props-for tool-dev-id)]
    {:dev-id tool-dev-id
     :identity (entity/identity-for tool-dev-id)
     :name (:mcp-tool/name props)
     :description (:mcp-tool/description props)
     :req-input-args (:mcp-tool/req-input-args props)
     :opt-input-args (:mcp-tool/opt-input-args props)
     :output-args (:mcp-tool/output-args props)}))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-mcp-tool properties.
   Called by atlas.datalog when building database.

   Property mappings:
   - :mcp-tool/name → :mcp-tool/name (tool name string)
   - :mcp-tool/description → :mcp-tool/description (description string)
   - :mcp-tool/req-input-args → :entity/consumes (many, for cross-ontology queries)
   - :mcp-tool/opt-input-args → :entity/consumes (many, for cross-ontology queries)
   - :mcp-tool/output-args → :entity/produces (many, for cross-ontology queries)"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-mcp-tool)
    (let [dev-id (:atlas/dev-id props)
          tool-name (:mcp-tool/name props)
          description (:mcp-tool/description props)
          req-inputs (:mcp-tool/req-input-args props)
          opt-inputs (:mcp-tool/opt-input-args props)
          outputs (:mcp-tool/output-args props)]
      (cond-> []
        ;; Tool name
        tool-name
        (conj [:db/add dev-id :mcp-tool/name tool-name])

        ;; Description
        description
        (conj [:db/add dev-id :mcp-tool/description description])

        ;; Required input args → entity/consumes (many-cardinality)
        req-inputs
        (concat (map (fn [input-kw]
                       [:db/add dev-id :entity/consumes input-kw])
                     req-inputs))

        ;; Optional input args → entity/consumes (many-cardinality)
        opt-inputs
        (concat (map (fn [input-kw]
                       [:db/add dev-id :entity/consumes input-kw])
                     opt-inputs))

        ;; Output args → entity/produces (many-cardinality)
        outputs
        (concat (map (fn [output-kw]
                       [:db/add dev-id :entity/produces output-kw])
                     outputs))))))

(def datalog-schema
  "Datascript schema for yorba-mcp-tool properties.

   Note: Both req-input-args and opt-input-args are mapped to :entity/consumes.
   :entity/consumes and :entity/produces are shared with execution-function
   and yorba-endpoint ontologies for cross-ontology data flow queries."
  {:mcp-tool/name {:db/cardinality :db.cardinality/one}
   :mcp-tool/description {:db/cardinality :db.cardinality/one}
   :entity/consumes {:db/cardinality :db.cardinality/many}
   :entity/produces {:db/cardinality :db.cardinality/many}})

;; =============================================================================
;; DATALOG-BASED QUERIES
;; =============================================================================

(defn find-tools-consuming
  "Find all MCP tools that consume a specific data keyword (datalog query).

   Example: (find-tools-consuming :user/email)
   Returns: sequence of tool dev-ids"
  [data-kw]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?tool-id
                :in $ ?data-kw
                :where
                [?e :atlas/dev-id ?tool-id]
                [?e :entity/consumes ?data-kw]]
              db
              data-kw)
         (map first)
         vec)))

(defn find-tools-producing
  "Find all MCP tools that produce a specific data keyword (datalog query).

   Example: (find-tools-producing :validation/result)
   Returns: sequence of tool dev-ids"
  [data-kw]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?tool-id
                :in $ ?data-kw
                :where
                [?e :atlas/dev-id ?tool-id]
                [?e :entity/produces ?data-kw]]
              db
              data-kw)
         (map first)
         vec)))

(defn find-data-flow-pairs
  "Find producer-consumer pairs where one tool's output matches another's input.

   Returns: sequence of maps with {:producer, :consumer, :data-keyword}"
  []
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?producer-id ?consumer-id ?data-kw
                :where
                [?producer :entity/produces ?data-kw]
                [?consumer :entity/consumes ?data-kw]
                [?producer :atlas/dev-id ?producer-id]
                [?consumer :atlas/dev-id ?consumer-id]
                ;; Only MCP tools (not endpoints/functions)
                [?producer :mcp-tool/name]
                [?consumer :mcp-tool/name]
                ;; Avoid self-references
                [(not= ?producer ?consumer)]]
              db)
         (map (fn [[producer consumer data-kw]]
                {:producer producer
                 :consumer consumer
                 :data-keyword data-kw}))
         vec)))

(defn find-tools-by-aspect
  "Find all MCP tools with a specific aspect (datalog query).

   Example: (find-tools-by-aspect :domain/users)
   Returns: sequence of tool dev-ids"
  [aspect]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?tool-id
                :in $ ?aspect
                :where
                [?e :atlas/dev-id ?tool-id]
                [?e :mcp-tool/name]  ; Ensure it's an MCP tool
                [?e :entity/aspect ?aspect]]
              db
              aspect)
         (map first)
         vec)))

;; =============================================================================
;; INVARIANTS
;; =============================================================================

;; NOTE: Value existence checks (e.g., "must have name", "must have description")
;; should be handled by clojure.spec at registration time.
;;
;; Registry invariants focus on structural/cross-ontology validation.

(def invariants
  "All structural invariants specific to yorba-mcp-tool ontology.

   Currently empty - can add cross-ontology invariants with yorba-data later
   (e.g., tools consuming sensitive data must have compliance aspects)."
  [])

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-ontology!
  "Register the yorba-mcp-tool ontology in the registry."
  []
  (registry/register!
   :atlas/yorba-mcp-tool
   :atlas/ontology
   #{:atlas/yorba-mcp-tool}
   ontology-definition))

(defn- register-invariants!
  "Register yorba-mcp-tool invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-mcp-tool properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-mcp-tool ontology, invariants, and datalog support.

   This must be called before using yorba-mcp-tool features:
   - Registering MCP tool entities
   - Using helper functions like name-for, input-args-for
   - Running datalog queries

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    (register-ontology!)
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-mcp-tool module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-mcp-tool module (primarily for testing).

   WARNING: This does not remove already-registered tools from the
   registry or clean up datalog extensions. Use reset! on registry/registry
   and datalog/reset-extensions! for a full reset."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing).

   Use this before calling load! in test fixtures when you've also reset
   the registry. This ensures load! will re-register the ontology."
  []
  (reset! loaded? false))
