(ns atlas.ontology.yorba-cache-index
  "Yorba cache index ontology module - cache indexing configuration and datalog support.

   This module provides cache index configuration and validation for the
   `:atlas/yorba-cache-index` ontology.

   A yorba-cache-index entity represents a cache index configuration:
   - References a yorba-serialisation for data serialisation format
   - Declares which data fields (args) are indexed for fast lookup
   - Used for query-optimised caching

   Example:
     (registry/register!
      :cache-index/user-by-email
      :atlas/yorba-cache-index
      #{:domain/users :tier/foundation}
      {:cache-index/serialisation :serialisation/user-json
       :cache-index/args [:user/email :user/status]})

   **Dependency:** This module depends on `atlas.ontology.yorba-serialisation`.
   Loading this module will automatically load its dependencies.

   Usage:
     (require '[atlas.ontology.yorba-cache-index :as yci])
     (yci/load!)"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]
            [atlas.ontology.yorba-serialisation :as yorba-ser]
            [datascript.core :as d]))

;; =============================================================================
;; ONTOLOGY DEFINITION
;; =============================================================================

(def ontology-definition
  "The ontology definition for :atlas/yorba-cache-index"
  {:ontology/for :atlas/yorba-cache-index
   :ontology/keys [:cache-index/serialisation  ; Reference to serialisation dev-id
                   :cache-index/args]})         ; Fields indexed (collection of keywords)

;; =============================================================================
;; HELPER FUNCTIONS
;; =============================================================================

(defn all-cache-indexes
  "Return all registered yorba-cache-index dev-ids (excluding ontology definitions)."
  []
  (->> (entity/all-with-aspect :atlas/yorba-cache-index)
       (remove #(entity/has-aspect? % :atlas/ontology))
       set))

(defn serialisation-for
  "Get the serialisation dev-id for a cache index.
   Uses the ontology key: :cache-index/serialisation"
  [cache-index-dev-id]
  (:cache-index/serialisation (entity/props-for cache-index-dev-id)))

(defn args-for
  "Get the indexed args (fields) for a cache index.
   Uses the ontology key: :cache-index/args"
  [cache-index-dev-id]
  (:cache-index/args (entity/props-for cache-index-dev-id)))

(defn cache-indexes-by-serialisation
  "Get all cache indexes that use a specific serialisation dev-id."
  [serialisation-dev-id]
  (->> (all-cache-indexes)
       (filter #(= serialisation-dev-id (serialisation-for %)))
       set))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-cache-index properties.
   Called by atlas.datalog when building database.

   Property mappings:
   - :cache-index/serialisation → :cache-index/serialisation (dev-id reference)
   - :cache-index/args → :cache-index/indexes (many, fields indexed for lookup)"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-cache-index)
    (let [dev-id (:atlas/dev-id props)
          ser-ref (:cache-index/serialisation props)
          args (:cache-index/args props)]
      (cond-> []
        ;; Serialisation reference
        ser-ref
        (conj [:db/add dev-id :cache-index/serialisation ser-ref])

        ;; Indexed fields (many-cardinality)
        args
        (concat (map (fn [arg-kw]
                       [:db/add dev-id :cache-index/indexes arg-kw])
                     args))))))

(def datalog-schema
  "Datascript schema for yorba-cache-index properties."
  {:cache-index/serialisation {:db/cardinality :db.cardinality/one}
   :cache-index/indexes {:db/cardinality :db.cardinality/many}})

;; =============================================================================
;; DATALOG-BASED QUERIES
;; =============================================================================

(defn find-indexes-using-serialisation
  "Find all cache indexes that use a specific serialisation (datalog query).

   Example: (find-indexes-using-serialisation :serialisation/user-json)
   Returns: sequence of cache index dev-ids"
  [serialisation-dev-id]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?index-id
                :in $ ?ser-id
                :where
                [?e :atlas/dev-id ?index-id]
                [?e :cache-index/serialisation ?ser-id]]
              db
              serialisation-dev-id)
         (map first)
         vec)))

(defn find-indexes-indexing
  "Find all cache indexes that index a specific data keyword (datalog query).

   Example: (find-indexes-indexing :user/email)
   Returns: sequence of cache index dev-ids"
  [data-kw]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?index-id
                :in $ ?data-kw
                :where
                [?e :atlas/dev-id ?index-id]
                [?e :cache-index/indexes ?data-kw]]
              db
              data-kw)
         (map first)
         vec)))

;; =============================================================================
;; INVARIANTS
;; =============================================================================

;; NOTE: Value existence checks (e.g., "must have serialisation", "must have args")
;; should be handled by clojure.spec at registration time.
;;
;; Registry invariants focus on structural/cross-ontology validation.

(defn invariant-cache-index-has-valid-serialisation
  "Every yorba-cache-index that references a serialisation must point to an existing entity.

   **Structural Check:** Validates cross-ontology references.

   This ensures that cache indexes don't point to non-existent serialisation
   configurations, which would cause runtime errors when the cache index tries
   to serialise/deserialise indexed data.

   Checks: :cache-index/serialisation property references a valid serialisation dev-id."
  []
  (let [cache-indexes (all-cache-indexes)
        valid-serialisations (yorba-ser/all-serialisations)
        violations (for [idx-id cache-indexes
                         :let [ser-ref (serialisation-for idx-id)]
                         :when (and ser-ref
                                    (not (valid-serialisations ser-ref)))]
                     {:cache-index idx-id
                      :missing-serialisation ser-ref})]
    (when (seq violations)
      {:invariant :cache-index-has-valid-serialisation
       :violation :missing-serialisation-reference
       :details violations
       :severity :error
       :message (str "Cache indexes reference non-existent serialisations: "
                     (mapv :missing-serialisation violations))})))

(def invariants
  "All structural invariants specific to yorba-cache-index ontology.

   These focus on cross-registry validation, not value existence checks.
   Value checks (e.g., serialisation presence, args presence) should be handled by clojure.spec."
  [invariant-cache-index-has-valid-serialisation])

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-ontology!
  "Register the yorba-cache-index ontology in the registry."
  []
  (registry/register!
   :atlas/yorba-cache-index
   :atlas/ontology
   #{:atlas/yorba-cache-index}
   ontology-definition))

(defn- register-invariants!
  "Register yorba-cache-index invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-cache-index properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-cache-index ontology, invariants, and datalog support.

   This will also load the yorba-serialisation module as a dependency.

   This must be called before using yorba-cache-index features:
   - Registering cache index entities with :cache-index/serialisation and :cache-index/args
   - Using helper functions like serialisation-for, args-for
   - Running invariants

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    ;; Load dependencies first
    (yorba-ser/load!)
    ;; Then register this module
    (register-ontology!)
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-cache-index module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-cache-index module (primarily for testing).

   WARNING: This does not remove already-registered cache indexes from the
   registry or clean up datalog extensions. Use reset! on registry/registry
   and datalog/reset-extensions! for a full reset.

   Note: This does NOT unload the yorba-serialisation dependency.
   Use yorba-ser/unload! separately if needed."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing).

   Use this before calling load! in test fixtures when you've also reset
   the registry. This ensures load! will re-register the ontology."
  []
  (reset! loaded? false))
